'use strict';
const vistaconfig = require("../../../lib/rpcvista/vistaconfig");
const vista = require("../../../lib/rpcvista/vista_surgery_services");
const auth = require("../authenticate");
const errorHandler  = require("../../common/errorHandler");
const mongoose = require('mongoose');

const RoomTypeConfig = mongoose.model('SpecialtyRoomTypeMapping');

const rpcUtils = require("./rpc_utilities");
const caseService = require("./case");

/**********************************
 *        Operating Rooms         *
 **********************************/

/** Get ALL rooms **/

exports.getRooms = function (loginOptions, callback) {

    getHospitalLocations(loginOptions, function(err, locations)
    {
        if (err){
            console.log("Error: " + err);
            callback(errorHandler.serverIssue(err), null);
        }
        else {
            auth.buildConfiguration(loginOptions, function (configuration) {
                vista.callRpc(
                    vistaconfig.logger,
                    configuration,
                    'DDR LISTER',
                    {'"FILE"': '131.7', '"FIELDS"': '1;2;3;4;9;10;99', '"FLAGS"': 'IP'},
                    function (error, result) {
                        if (result === undefined || result instanceof Error) {
                            callback(result, null);
                        }
                        else if (error) {
                            callback(errorHandler.serverIssue(error), null);
                        }
                        else {
                            let clob = rpcUtils.removeDddWrapper(result);
                            let data = clob.split("\n")
                            let parsed_data = [];
                            data.forEach(function (line) {
                                let line_data = line.split("^");
                                parsed_data.push({
                                    ien: line_data[0],
                                    name: locations[line_data[1]] || undefined,
                                    location: line_data[2],
                                    number: line_data[3],
                                    contact: line_data[4],
                                    another_number: line_data[6],
                                    last_number: line_data[7]
                                });
                            });
                            callback(null, parsed_data);
                        }
                    }
                );
            });
        }
    });
}

function getHospitalLocations(loginOptions, callback) {
    auth.buildConfiguration(loginOptions, function (configuration) {

        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '44', '"FIELDS"': '.01;1;2;6;7;9;10;42', '"FLAGS"': 'IP'},
            function (error, result) {
                if (result === undefined || result instanceof Error) {
                    callback(errorHandler.serverIssue(result), null);
                }
                else if (error) {
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let clob = rpcUtils.removeDddWrapper(result);
                    let data = clob.split("\n")
                    let parsed_data = {};
                    data.forEach(function (line) {
                        let line_data = line.split("^");
                        parsed_data[line_data[0]] = line_data[2];
                    });
                    callback(null, parsed_data);
                }
            }
        );
    });
}

exports.getRoomHours = function (loginOptions, roomId, callback) {
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {
                '"FILE"': '131.711',
                '"FIELDS"': '.01;1;2;3',
                '"FLAGS"': 'IP',
                '"IENS"': ',' + roomId + ',',
                '"XREF"': "#"
            },
            function (error, result) {
                if (result === undefined || result instanceof Error) {
                    callback(errorHandler.serverIssue(result), null);
                } else {
                    let clob = rpcUtils.removeDddWrapper(result);
                    let data = clob.split("\n")
                    let parsed_data = [];
                    data.forEach(function (line) {
                        let line_data = line.split("^");
                        parsed_data.push({
                            ien: line_data[0],
                            day_of_week: line_data[1],
                            start_time: line_data[2],
                            end_time: line_data[3]
                        });
                    });
                    callback(errorHandler.serverIssue(error), parsed_data);
                }
            }
        );
    });
}

/*Get OR Room details
 * name
 * service(medicine, surgery, psychiatry, rehab medicine, neurology, none)
 * location
 * phoneNum
 * comments
 * personResponsible
 * cleaningTime
 * inactive
 */
exports.getRoomDetailsById = function (loginOptions, id, callback) {
    console.log("Entering getRoomDetailsById ...");
    callback(null,
        {
            "name": "OR SURG ROOM 1",
            "service": "Surgery",
            "location": "EO Building1A",
            "phoneNum": "987-294-7715",
            "comments": "Surgery room primarily used open hear surgery.",
            "personResponsible": "George Misak",
            "cleaningTime": 30,
            "inactive": 'n'
        });
    // TODO implement logic
};

/**
 * Get Room details for room that's associated with teh specified case
 * @param loginOptions
 * @param caseId
 * @param callback
 */
exports.getRoomDetailsByCaseId = function (loginOptions, caseId, callback) {
    console.log("Entering getRoomDetailsByCaseId ...");
    callback(null,
        {
            "name": "OR SURG ROOM 1",
            "service": "Surgery",
            "location": "EO Building1A",
            "phoneNum": "987-294-7715",
            "comments": "Surgery room primarily used open hear surgery.",
            "personResponsible": "George Misak",
            "cleaningTime": 30,
            "inactive": 'n'
        });
    // TODO implement logic
};

function getRandomHour(start_hour, end_hour){
    return Math.floor(Math.random() * ((end_hour-start_hour)+1) + start_hour);
}

function getRandomMinute(){
    const minutes = ["00", "15", "30", "45"];
    return minutes[Math.floor(Math.random() * minutes.length)];
}

// add trailing 0 in case if individual number minutes
// only case should be 0
function addZero(i) {
    if (i < 10) {
        i = "0" + i;
    }
    return i;
}

/**
convert case object to event
**/
function mediateCaseToEvent(result_case){
    let start_date_time = new Date(result_case.scheduledStartTime);
    let end_date_time = new Date(result_case.scheduledEndTime);

    return {
        "ien": result_case.ien,
        "name": result_case.principalProcedure,
        "start_time": start_date_time.getHours() + ":" + addZero(start_date_time.getMinutes()),
        "end_time": end_date_time.getHours()+":"+addZero(end_date_time.getMinutes()),
        "type": "case"
    };
}

exports.getEventsForRoomByDay = function(loginOptions, roomId, day, callback){
    caseService.searchCases(loginOptions,
            {
                "orRoom": roomId,
                "date": day
            },
            function(error, result){
                if(error){
                    console.log(error);
                    callback(errorHandler.serverIssue(error));
                }
                else{
                    console.log("Result in events: "+result);
                    let event_results = [];
                    for(var i=0; i < result.length; i++){
                        let event_case = mediateCaseToEvent(result[i]);
                        event_case['room_id'] = roomId;
                        event_results.push(event_case);
                    }
                    callback(event_results);
                }
            }
        );
}

exports.getEventsForRoomByDayRandom = function(loginOptions, roomId, day, callback){


    const cases1 = [
        {
            "ien": 1,
            "name": "Thyroid surgery",
            "room_id": roomId,
            "type": "case"
        },{
            "ien": 2,
            "name": "Minimally invasive parathryoidectomy",
            "room_id": roomId,
            "type": "case"
        },
        {
            "ien": 3,
            "name": "Laparoscopic adrenalectomy"  ,
            "room_id": roomId,
            "type": "case"         
        }
    ];

    const cases2 = [
        {
            "ien": 4,
            "name": "General Surgery",
            "room_id": roomId,
            "type": "case"
        },{
            "ien": 5,
            "name": "Hand Surgery",
            "room_id": roomId,
            "type": "case"
        }
    ];



    const random_between_01 =(Math.floor(Math.random() * 2));
    console.log("Math Random", random_between_01);
    var result_cases = null;
    if(random_between_01 == 0 ){
        result_cases = cases1;
    }else{
        result_cases = cases2;
    }

    for (var i = result_cases.length - 1; i >= 0; i--) {
        const start_hour = getRandomHour(9,13);
        result_cases[i].start_time = start_hour + ":" + getRandomMinute();
        result_cases[i].end_time = getRandomHour(start_hour, 14) + ":" + getRandomMinute();
    }
    callback(null, result_cases);
}