'use strict';
const vistaconfig = require("../../../lib/rpcvista/vistaconfig");
const vista = require("../../../lib/rpcvista/vista_surgery_services");
const auth = require("../authenticate");
const CaseCancellationReason = require("../../model/caseCancelReason");
const errorHandler = require("../../common/errorHandler");

const rpcUtils = require("./rpc_utilities");

/**
 * Retrieves a list of values from VistA for the file and fields specified
 * @param loginOptions
 * @param fileNum
 * @param dataFields
 * @param dataFieldGroup
 */
function getVistaFileData(loginOptions, fileNum, dataFields, dataFieldGroup, callback) {
    // Validation

    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            //                              '.01;1;2;6;7;9;10;42'
            {'"FILE"': fileNum, '"FIELDS"': dataFields, '"FLAGS"': 'IP'},
            function (error, result) {
                if (result === undefined || result instanceof Error || result === "") {
                    callback(errorHandler.serverIssue(result), null);
                }
                else if (error) {
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let list = [];
                    switch(dataFieldGroup){
                        case ID_NAME_CODE_INACTIVE:
                            list = parseIdNameCodeInactive(result);
                            break;
                        case ID_NAME_INACTIVE:
                            list = parseIdNameInactive(result);
                        default:
                            break;
                    }
                    callback(null, list);
                }
            });
    })
}

/**
 * Parse data when querying for id, name, code, and inactive
 * @param data
 * @returns {{}}
 */
function parseIdNameCodeInactive(result){
    let data = rpcUtils.removeDddWrapper(result),
        returnArray = [],
        dataArray, fieldArray, i, x, fields;

    dataArray = data.split(vistaconfig.LINE_SEPARATOR)

    for (i = 0, x = dataArray.length; i < x; i++)
    {
        fields = dataArray[i];

        if (fields !== undefined  && fields !== "" && fields.includes("^")) {
            fieldArray = fields.split(vistaconfig.NODE_SEPARATOR);
            // EXAMPLE
            //3^ABRASION^ABRASION^AB^1
            returnArray.push({
                id: fieldArray[0],
                name: fieldArray[2],
                code: fieldArray[3],
                inactive: fieldArray[4]
            });
        }
    }

    return returnArray;
}

/**
 * Parse data when querying for id, name, and inactive
 * @param data
 * @returns {{}}
 */
function parseIdNameInactive(result){
    let jsonData = {},
        data = rpcUtils.removeDddWrapper(result),
        returnArray = [],
        dataArray, fieldArray,  i, x, fields;

        dataArray = data.split(vistaconfig.LINE_SEPARATOR);

    for (i = 0, x = dataArray.length; i < x; i++)
    {
        fields = dataArray[i];

        if (fields !== undefined  && fields !== "" && fields.includes("^")) {
            fieldArray = fields.split(vistaconfig.NODE_SEPARATOR);
            // EXAMPLE
            //3^RESTING^RESTING^
            returnArray.push({
                id: fieldArray[0],
                name: fieldArray[2],
                inactive: fieldArray[3]
            });
        }
    }

    return returnArray;
}

/**
 * GET a list of all people from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAllPeople = function(loginOptions, callback) {
    console.log("Entering getAllPeople ...");
    /*
     '[MAP]\r\nIEN^.01I^WID(W8)\r\n[BEGIN_diDATA]\r\n.5^POSTMASTER^\r\n.6^SHARED,MAIL^\r\n1^WVEHR,PATCH INSTALLER^\r\n2^SEPPALA,GREGG^\r\n3^DEVILLERS,NANCY^\r\n4^INTERFACE,IB EIV^\r\n5^CENTRAL,PAID^\r\n6^EDILOCKBOX,AUTOMATIC^\r\n7^PRCA,AUTOAUDIT^\r\n8^MANAGER,SYSTEM^   System Manager\r\n9^DEWAYNE,ROBERT^   ANALYST\r\n10^USER,TEST^   ANALYST\r\n11^DOCTOR,ONE^   Physician\r\n12^NURSE,ONE^   Nurse\r\n13^CLERK,ONE^   Medical Clerk\r\n14^JUTZI,WILLIAM^\r\n15^DOCTOR,TWO^   Physician\r\n16^DOCTOR,THREE^   Physician\r\n17^DOCTOR,FOUR^   Physician\r\n18^DOCTOR,FIVE^   Physician\r\n19^DOCTOR,SIX^   Physician\r\n20^DOCTOR,SEVEN^   Physician\r\n21^DOCTOR,EIGHT^   Physician\r\n22^DOCTOR,NINE^   Physician\r\n23^DOCTOR,TEN^   Physician\r\n24^DOCTOR,ELEVEN^   Physician\r\n26^DOCTOR,TWELVE^   Physician\r\n27^NURSE,TWO^   Nurse\r\n28^NURSE,THREE^   Nurse\r\n29^NURSE,FOUR^   Nurse\r\n30^NURSE,FIVE^   Nurse\r\n31^NURSE,SIX^   Nurse\r\n32^NURSE,SEVEN^   Nurse\r\n33^NURSE,EIGHT^   Nurse\r\n34^NURSE,NINE^   Nurse\r\n35^NURSE,TEN^   Nurse\r\n36^NURSE,ELEVEN^   Nurse\r\n37^NURSE,TWELVE^   Nurse\r\n38^NURSE,THIRTEEN^   Nurse\r\n39^NURSE,FOURTEEN^   Nurse\r\n40^NURSE,FIFTEEN^   Nurse\r\n41^CLERK,TWO^   Medical Clerk\r\n42^CLERK,THREE^   Medical Clerk\r\n43^CLERK,FOUR^   Medical Clerk\r\n44^CLERK,FIVE^   Medical Clerk\r\n45^CLERK,SIX^   Medical Clerk\r\n46^CLERK,SEVEN^   Medical Clerk\r\n47^CLERK,EIGHT^   Medical Clerk\r\n48^CLERK,NINE^   Medical Clerk\r\n49^CLERK,TEN^   Medical Clerk\r\n50^CLERK,ELEVEN^   Medical Clerk\r\n51^CLERK,TWELVE^   Medical Clerk\r\n52^CLERK,THIRTEEN^   Medical Clerk\r\n53^CLERK,FOURTEEN^   Medical Clerk\r\n54^CLERK,FIFTEEN^   Medical Clerk\r\n55^ZZ IT,MANAGER^   ANALYST\r\n56^USER,PATCH^   System Manager\r\n57^AUTHORIZER,IB MRA^\r\n58^COORDINATOR,CLINICAL^   CLINICAL COORDINATOR\r\n59^STARTUP,SYSTEM^\r\n60^FARLEY,CHRIS^   CLINICAL COORDINATOR\r\n61^MCKINZIE,MARCIA^\r\n62^LAB,SUPERUSER^   ANALYST\r\n63^AMUNDSON,MARK^   ANALYST\r\n64^FERGUSON,DAVID^\r\n65^MARSHALL,RICK^   ANALYST\r\n66^WHITTEN,DAVE^   ANALYST\r\n67^DAL MOLIN,JOSEPH^   ANALYST\r\n68^ANDERSON,MAGGIE^   ANALYST\r\n69^STOXEN,JAMES H^   ANALYST\r\n70^PATCHES,ALL D^\r\n71^LRLAB,POC^\r\n72^LRLAB,HL^\r\n73^XOBVTESTER,APPLICATION PROXY^\r\n74^VPFS,APPLICATION PROXY^\r\n75^RADIOLOGY,OUTSIDE SERVICE^\r\n76^TASKMAN,PROXY USER^\r\n77^CLINICAL,DEVICE PROXY SERVICE^\r\n78^EDPTRACKING,PROXY^\r\n79^NHIN,APPLICATION PROXY^\r\n80^JAMES,BOB^\r\n81^BENSON,GEORGE^\r\n82^ONE,PROVIDER^\r\n83^TWO,PROVIDER^\r\n84^ONE,NURSE^   Nurse\r\n85^ONE,PHARMACIST^\r\n86^USER,COMS^\r\n87^BARLOW,MIKE^\r\n88^SURGERY,VISTA^\r\n89^UNTIEDT,JAYME^\r\n90^LE,VU^\r\n91^GOKULNATH,RAHUL^\r\n92^BRAVO,JESUS^\r\n93^ONE,SURGEON^\r\n94^TWO,SURGEON^\r\n95^SCHUCK,FRED^\r\n96^THREE,SURGEON^\r\n97^FOUR,SURGEON^\r\n98^CODER,SURGERY^\r\n99^SURGERY,RISKDATA MANAGER^\r\n100^SURGEON,FIVE^\r\n101^SURGERY,APPLICATION COORDINATOR^\r\n102^ONE,ANESTHESIA^\r\n103^TWO,ANESTHESIA^\r\n104^SURGERY,CHEIF^\r\n[END_diDATA]\r\n'
     */
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'SR GET NEW PERSONS',
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let t = result.charAt("$\^");


                    // console.log(result);
                    let returnArray = [],
                        temp, node,
                        resultArray = result.split(vistaconfig.NODE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        // console.log("HAVE DATA!!!");
                        for (var i = 0; i < resultArray.length; i++){
                            temp = resultArray[i];
                            if (temp != undefined && temp !== null && temp !== "") {
                                node = temp.split(vistaconfig.SEMICOLON_SEPARATOR);
                                // console.log(node);
                                if (node !== null && node !== undefined && node.length > 0) {
                                    // Populate model

                                    returnArray.push({
                                        "id": node[0],
                                        "name": node[1],
                                        // "title": node[2].trim()
                                    });
                                }
                                else {
                                    console.log("Node is null");
                                }
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of strings containing all local specialties from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAllSpecialties = function(loginOptions, callback) {
    console.log("Entering getAllSpecialties ...");
    /*
     '[MAP]\r\nIEN^IX(1)I^.01I^11I\r\n[BEGIN_diDATA]\r\n1^GENERAL(OR WHEN NOT DEFINED BELOW)^GENERAL(OR WHEN NOT DEFINED BELOW)^\r\n2^GYNECOLOGY^GYNECOLOGY^\r\n3^NEUROSURGERY^NEUROSURGERY^\r\n4^OPHTHALMOLOGY^OPHTHALMOLOGY^\r\n11^ORAL SURGERY (DENTAL)^ORAL SURGERY (DENTAL)^\r\n5^ORTHOPEDICS^ORTHOPEDICS^\r\n6^OTORHINOLARYNGOLOGY (ENT)^OTORHINOLARYNGOLOGY (ENT)^\r\n13^PERIPHERAL VASCULAR^PERIPHERAL VASCULAR^\r\n7^PLASTIC SURGERY (INCLUDES HEAD AND NECK)^PLASTIC SURGERY (INCLUDES HEAD AND NECK)^\r\n12^PODIATRY^PODIATRY^\r\n8^PROCTOLOGY^PROCTOLOGY^\r\n9^THORACIC SURGERY (INC. CARDIAC SURG.)^THORACIC SURGERY (INC. CARDIAC SURG.)^\r\n10^UROLOGY^UROLOGY^\r\n[END_diDATA]\r\n'
     */
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '137.45','"FIELDS"' : '.01','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        // console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0){
                                // Populate list
                                returnArray.push({
                                    "id":node[0],
                                    "name": node[1]
                                });
                            }
                            else{
                                console.log("Node is null");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Operating Rooms from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAllORRooms = function(loginOptions, callback) {
    console.log("Entering getAllORRooms ...");
    getHospitalLocations(loginOptions, function(err, locations){
        auth.buildConfiguration(loginOptions, function (configuration) {
            vista.callRpc(
                vistaconfig.logger,
                configuration,
                'DDR LISTER',
                {'"FILE"': '131.7','"FIELDS"' : '.01;1;9','"FLAGS"': 'IP', '"INDEX"' : '#'},
                function (error, result) {
                    if (error) {
                        console.log("Error: " + error);
                        callback(errorHandler.serverIssue(error), null);
                    }
                    else {
                        let returnArray = [];
                        let node;
                        let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                        if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                            console.log("HAVE DATA!!!");
                            for (var i = 3; i < resultArray.length - 2; i++){
                                node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                                //console.log(node);
                                if (node !== null && node !== undefined && node.length > 0){
                                    // Populate model

                                    returnArray.push({
                                        "id": node[0],
                                        "name": locations[node[1]] || "",
                                        "location": node[3],
                                        "type": node[4]
                                    });
                                }
                                else{
                                    console.log("Node is null for data: " + element);
                                }
                            };
                        }
                        else{
                            console.log("Result Array empty for data: " + result);
                        }
                        callback(null, returnArray);
                    }
                });
        });
    });

}

function getHospitalLocations(loginOptions, callback) {
    auth.buildConfiguration(loginOptions, function (configuration) {

        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '44', '"FIELDS"': '.01;1;2;6;7;9;10;42', '"FLAGS"': 'IP'},
            function (error, result) {
                if (result === undefined || result instanceof Error) {
                    callback(result, null);
                }
                else if (error) {
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let clob = rpcUtils.removeDddWrapper(result);
                    let data = clob.split("\n")
                    let parsed_data = {};
                    data.forEach(function (line) {
                        let line_data = line.split("^");
                        parsed_data[line_data[0]] = line_data[2];
                    });
                    callback(error, parsed_data);
                }
            }
        );
    });
}

/**
 * GET a list of all Surgery Dispositions from VistA
 * @param loginOptions
 * @param callback
 */
exports.getSurgeryDispositions = function(loginOptions, callback) {
    console.log("Entering getSurgeryDispositions ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '131.6','"FIELDS"' : '.01;1;2','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model

                                returnArray.push({
                                    "id": node[0],
                                    "disposition": node[1],
                                    "code": node[3],
                                    "inactive": node[4]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Surgery Positions from VistA
 * @param loginOptions
 * @param callback
 */
exports.getSurgeryPositions = function(loginOptions, callback) {
    console.log("Entering getSurgeryDispositions ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '132','"FIELDS"' : '.01;4;5;10','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                // 20^LATERAL DECUBITUS, LEFT DOWN^LATERAL DECUBITUS, LEFT DOWN^LATERAL DECUBITUS^LDL^
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "majorPosition": node[3],
                                    "code": node[4],
                                    "inactive": node[5]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Special Equipment from VistA
 * @param loginOptions
 * @param callback
 */
exports.getSpecialEquipment = function(loginOptions, callback) {
    console.log("Entering getSpecialEquipment ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '131.3','"FIELDS"' : '.01;2;3','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                // 2^NEW SPECIAL EQUIPMENT TEST^NEW SPECIAL EQUIPMENT TEST^2^
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "number": node[3],
                                    "inactive": node[4],
                                    "surgicalSpecialties": []
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Planned Implants from VistA
 * @param loginOptions
 * @param callback
 */
exports.getPlannedImplants = function(loginOptions, callback) {
    console.log("Entering getSurgeryDispositions ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '131.5','"FIELDS"' : '.01;1;2;3;4','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                // 3^BETTER PLANNED IMPLANT^BETTER PLANNED IMPLANT^SMALL^^BETTER MODEL^BETTER VENDOR
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "number": node[3],
                                    "inactive": node[4],
                                    "model": node[5],
                                    "vendor": node[6],
                                    "surgicalSpecialties": []
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Special Supplies from VistA
 * @param loginOptions
 * @param callback
 */
exports.getSpecialSupplies = function(loginOptions, callback) {
    console.log("Entering getSurgeryDispositions ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '131.04','"FIELDS"' : '.01;1;2;3;4','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                // 2^NEW SPECIAL SUPPLIES NAME^NEW SPECIAL SUPPLIES NAME^LARGE^^NEW MODEL^NEW VENDOR
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "size": node[3],
                                    "inactive": node[4],
                                    "model": node[5],
                                    "vendor": node[6],
                                    "surgicalSpecialties": []
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Special Instruments from VistA
 * @param loginOptions
 * @param callback
 */
exports.getSpecialInstruments = function(loginOptions, callback) {
    console.log("Entering getSurgeryDispositions ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '131.02','"FIELDS"' : '.01;1;2','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                // 1^TEST SPECIAL INSTRUMENTS^TEST SPECIAL INSTRUMENTS^1^1
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "number": node[3],
                                    "inactive": node[4],
                                    "surgicalSpecialties": []
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Pharmacy Items from VistA
 * @param loginOptions
 * @param callback
 */
exports.getPharmacyItems = function(loginOptions, callback) {
    console.log("Entering getSurgeryDispositions ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '131.06','"FIELDS"' : '.01;1;2;3','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                // 1^1000^1000^10ML/H^1^TEST COMMENTS
                                returnArray.push({
                                    "id": node[0],
                                    "drugName": node[1],
                                    "dose": node[3],
                                    "inactive": node[4],
                                    "drugComments": node[5],
                                    "surgicalSpecialties": []
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * Gets a list of Hospital Locations from VistA
 * @param loginOptions
 * @param callback
 */
exports.getHospitalLocations = function(loginOptions, callback) {
    console.log("Entering getHospitalLocations ...");
    /*
     '[MAP]\r\nIEN^.01I^1I\r\n[BEGIN_diDATA]\r\n1^GENERAL PURPOSE OPERATING ROOM^GP\r\n2^ORTHOPEDIC OPERATING ROOM^OR\r\n3^CARDIAC OPERATING ROOM^CA\r\n4^NEUROSURGERY OPERATING ROOM^NE\r\n5^CARDIAC/NEURO OPERATING ROOM^CN\r\n6^AMBULATORY OPERATING ROOM^AM\r\n7^INTENSIVE CARE UNIT^IC\r\n8^ENDOSCOPY ROOM^EN\r\n9^CYSTOSCOPY ROOM^CY\r\n10^WARD^WA\r\n11^CLINIC^CL\r\n12^DEDICATED ROOM^DE\r\n13^OTHER LOCATION^OT\r\n[END_diDATA]\r\n'
     */
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '44','"FIELDS"' : '.01;1;2;7;9;10;14;15;16','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0){
                                // Populate model
                                //3^General^General^Gen^W^Ofc^M^123 Street^^82^

                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "abbreviation": node[3],
                                    "type": node[4],
                                    "visitLocation,": node[5],
                                    "service": node[6],
                                    "physicalLocation": node[7],
                                    "specialAmisStop": node[8],
                                    "visitCategory": node[9],
                                    "defaultProvider": node[10]
                                });
                            }
                            else{
                                console.log("Node is null for data: " + element);
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * Gets a list of Attending Codes from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAttendingCodes = function(loginOptions, callback) {
    console.log("Entering getAttendingCodes ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '132.9','"FIELDS"' : '.01;1;3','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                //1^0. STAFF ALONE^0. STAFF ALONE^OLD0^1
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "code": node[3],
                                    "inactive": node[4]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * Gets a list of Anesthesia Supervisor Codes from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAneSuperCodes = function(loginOptions, callback) {
    console.log("Entering getAttendingCodes ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '132.95','"FIELDS"' : '.01;1','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                //2^2. STAFF ASSISTED BY RESIDENT ^2. STAFF ASSISTED BY RESIDENT OR C.R.N.A.^2
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "code": node[3]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * Gets a list of ASA Class from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAsaClass = function(loginOptions, callback) {
    console.log("Entering getAsaClass ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '132.8','"FIELDS"' : '.01;1;2','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                //1^1^1^1-NO DISTURB.^This a normal healthy patient
                                returnArray.push({
                                    "id": node[0],
                                    "asaClass": node[1],
                                    "physcialStatus": node[3],
                                    "description": node[4]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}


/**
 * Gets a list of Medical Specialty from VistA
 * @param loginOptions
 * @param callback
 */
exports.getMedicalSpecialty = function(loginOptions, callback) {
    console.log("Entering getMedicalSpecialty ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '723','"FIELDS"' : '.01','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[4] != 1){
                                // Populate model
                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * GET a list of all Operating Room Types from VistA
 * @param loginOptions
 * @param callback
 */
/*exports.getAllORRoomTypes = function(loginOptions, callback) {
    console.log("Entering getAllORRoomTypes ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '134', '"FIELDS"': '.01;1', '"FLAGS"': 'IP', '"XREF"': '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0){
                                // Populate model

                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "code": node[2]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}*/

/**
 * Gets a list of Hosptial Locations from VistA
 * @param loginOptions
 * @param callback
 */
exports.getAllORRoomTypes = function(loginOptions, callback) {
    console.log("Entering getAllORRoomsTypes ...");
    /*
     '[MAP]\r\nIEN^.01I^1I\r\n[BEGIN_diDATA]\r\n1^GENERAL PURPOSE OPERATING ROOM^GP\r\n2^ORTHOPEDIC OPERATING ROOM^OR\r\n3^CARDIAC OPERATING ROOM^CA\r\n4^NEUROSURGERY OPERATING ROOM^NE\r\n5^CARDIAC/NEURO OPERATING ROOM^CN\r\n6^AMBULATORY OPERATING ROOM^AM\r\n7^INTENSIVE CARE UNIT^IC\r\n8^ENDOSCOPY ROOM^EN\r\n9^CYSTOSCOPY ROOM^CY\r\n10^WARD^WA\r\n11^CLINIC^CL\r\n12^DEDICATED ROOM^DE\r\n13^OTHER LOCATION^OT\r\n[END_diDATA]\r\n'
     */
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '44','"FIELDS"' : '.01;1;2;7;9;10;14;15;16','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    // console.log(result);
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            //console.log(node);
                            if (node !== null && node !== undefined && node.length > 0){
                                // Populate model
                                //3^General^General^Gen^W^Ofc^M^123 Street^^82^

                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1],
                                    "code": node[2],
                                    "abbreviation": node[3],
                                    "type": node[4],
                                    "visitLocation,": node[5],
                                    "service": node[6],
                                    "physicalLocation": node[7],
                                    "specialAmisStop": node[8],
                                    "visitCategory": node[9],
                                    "defaultProvider": node[10]
                                });
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            });
    });
}

/**
 * Gets a list of Case Cancellation Reasons from VistA
 * @param loginOptions
 * @param callback
 */
exports.getCaseCancellationReasons = function(loginOptions, callback){
    console.log("Entering getCaseCancellationReasons ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '135','"FIELDS"' : '.01;1;2;10','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function(error, result){
                if (error){
                    console.log("ERROR: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else if (result === undefined || result === null){
                    console.log("No result");
                    callback("No results", null);
                }
                else {
                    // name, code, avoidable, inactive
                    let returnArray = [];
                    let node;
                    let model;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        // resultArray.forEach(function(element)
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            console.log(node);
                            if (node !== null && node !== undefined && node.length > 0 && node[5] != 1){
                                // Populate model
                                model = new CaseCancellationReason();
                                model.setData({
                                    id: node[0],
                                    name: node[1],
                                    code: node[3],
                                    avoidable: node[4],
                                    inactive: node[5]
                                });

                                returnArray.push(model.sanitize());
                            }
                            else{
                                console.log("Node is null or inactive");
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            }
        );
    });
};


/**
 * Gets a list of Case Closure Reasons
 * @param loginOptions
 * @param callback
 */
exports.getCaseClosureReasons = function(loginOptions, callback){
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            {'"FILE"': '301.93','"FIELDS"' : '.01','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function(error, result){
                if (error){
                    console.log("ERROR: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else if (result === undefined || result === null){
                    console.log("No result");
                    callback("No results", null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        console.log("HAVE DATA!!!");
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            console.log(node);
                            if (node !== null && node !== undefined && node.length > 0){
                                // Populate model

                                returnArray.push({
                                    "id": node[0],
                                    "name": node[1]
                                });
                            }
                            else{
                                console.log("Node is null for data: " + element);
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            }
        );
    });
};

/**
 * Gets a list of ICD Codes
 * @param loginOptions
 * @param callback
 */
exports.getICDCodes = function(loginOptions, callback){
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'DDR LISTER',
            // {'"FILE"': '80','"FIELDS"' : '.01;1.1;1.11;1.2','"FLAGS"': 'IP', '"INDEX"' : '#'},
            {'"FILE"': '80','"FIELDS"' : '.01;1.1;1.11','"FLAGS"': 'IP', '"INDEX"' : '#'},
            function(error, result){
                if (error){
                    console.log("ERROR: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else if (result === undefined || result === null){
                    console.log("No result");
                    callback("No results", null);
                }
                else {
                    let returnArray = [];
                    let node;
                    let resultArray = result.split(vistaconfig.LINE_SEPARATOR);
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                        for (var i = 3; i < resultArray.length - 2; i++){
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                            if (node !== null && node !== undefined && node.length > 0){
                                // Populate model

                                returnArray.push({
                                    "id": node[0],
                                    "codeNumber": node[1],
                                    "codeSystem": node[2],
                                    "pdxExclusionCode": node[3]
                                });
                            }
                            else{
                                console.log("Node is null for data: " + element);
                            }
                        };
                    }
                    else{
                        console.log("Result Array empty for data: " + result);
                    }
                    callback(null, returnArray);
                }
            }
        );
    });
};

exports.getVistaFileData = getVistaFileData;
const ID_NAME_CODE_INACTIVE = 0;
const ID_NAME_INACTIVE = 1;
// exports.ID_NAME_CODE_INACTIVE = 2;

exports.ID_NAME_CODE_INACTIVE = ID_NAME_CODE_INACTIVE;
exports.ID_NAME_INACTIVE = ID_NAME_INACTIVE;