'use strict';
const vistaconfig = require("../../../lib/rpcvista/vistaconfig");
const vista = require("../../../lib/rpcvista/vista_surgery_services");
const auth = require("../authenticate");
const CPT = require("../../model/cpt");
const errorHandler = require("../../common/errorHandler");


/**
 * Gets a list of CPT codes from VistA
 * @param loginOptions
 * @param callback
 */
exports.getCPTList = function(loginOptions, callback){

    console.log("Entering getCPTList ...");
    auth.buildConfiguration(loginOptions, function(configuration){
         vista.callRpc(
             vistaconfig.logger,
             configuration,
             'DDR LISTER',
             {'"FILE"': '81','"FIELDS"' : '.01;2;6;3;5','"FLAGS"': 'IP', '"INDEX"' : '#'},
             function(error, result){
                // FIELDS: CPT CODE, SHORT NAME, SOURCE, INACTIVE FLAG
                if (error){
                    console.log("ERROR: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else if (result === undefined || result === null){
                    console.log("No result");
                    callback(errorHandler.notFound("No data result."), null);
                }
                else {
                    parseResultList(result, function(data){
                        callback(null, data);
                    });
                }
         });
     });
};

/**
 * Gets all CPT Durations [CPT, CPT-SURGEON, CPT-SPECIALTY] from all cases
 * @param loginOptions
 * @param cptList
 * @param set
 * @param criteria
 * @param callback
 */
function getCPTDurations(loginOptions, cptList, set, criteria, callback) {
    console.log("Entering getCPTDurations ...");
    // Handle arguments
    // Only loginOptions and callback was provided (initial call for full search)
    if (arguments.length === 2) {
        callback = cptList;
        cptList = {};
        set = 1;
        criteria = undefined;
    }
    // Only loginOptions, criteria, and callback are provided (initial call for specific search)
    else if (arguments.length === 3){
        criteria = cptList;
        callback = set;
        cptList = {};
        set = 1;
    }

    auth.buildConfiguration(loginOptions, function (configuration) {
        let node, resultArray;

        // Call 'SR GET ALL CASES'
        //TODO - change this implementation once a more appropriate RPC is created.
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'SR GET ALL CASES',
            (set * 50) - 50 + 1,
            set * 50,
            function (error, result) {
                // console.log(result);
                let complete = false;
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    resultArray = result.split("\r");
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 1) {
                        // console.log(resultArray);
                        for (let i = 0; i < resultArray.length - 1; i++) {
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);

                            if (node !== null && node !== undefined && node.length > 0 && node.length !== 1) {
                                cptList = processCPTDuration(node, criteria, cptList);
                            }
                            else {
                                console.log("Node is null for data: " + resultArray[i]);
                            }
                        }
                    }
                    else {
                        complete = true;
                    }
                }

                // Call recursively
                if (complete) {
                    callback(null, cptList);
                }
                else {
                    getCPTDurations(loginOptions, cptList, set + 1, criteria, callback);
                }
            });
    });
}

function processCPTDuration(node, criteria, cptList){
    if (arguments.length === 2){
        cptList = criteria;
        criteria = undefined;
    }

    let cpt, startTime, endTime, duration,specialty, surgeon, temp, tempSpecialty, tempSurgeon, addToList;

    // Populate model
    cpt = node[6];
    startTime = (node[7] !== undefined && node[7] !== "" ? vista.convertFromVistaDate(node[7]) : "");
    endTime = (node[8] !== undefined && node[8] !== "" ? vista.convertFromVistaDate(node[8]) : "");
    specialty = node[11];
    surgeon = node[9];

    addToList = (criteria === undefined
                || ((criteria.cptCode === undefined || criteria.cptCode === cpt)
                    && (criteria.specialty === undefined || criteria.specialty === specialty)
                    && (criteria.primarySurgeon === undefined || criteria.primarySurgeon === surgeon)));

    // TODO - Check for Non-OR cases and used TIME PROCEDURED STARTED AND ENDED once data is available from RPC

    // TODO - Include additional check once caseStatus is returned from GET ALL CASES
    if (startTime !== "" && endTime !== "" && addToList) {// && caseStatus !== "" && caseStatus == 1) {
        duration = (endTime - startTime) / 1000 / 60;

        // Add to cptList
        if (cptList[cpt] === undefined) {
            // console.log("adding new cpt: " + cpt);

            // All
            temp = {};
            temp[ALL_CATEGORY] = [];
            temp[ALL_CATEGORY].push({"duration":duration, "endDate": endTime});

            // Specialty
            tempSpecialty = {};
            if (specialty !== undefined) {
                // console.log("adding for specialty: " + specialty);
                tempSpecialty[specialty] = [];
                tempSpecialty[specialty].push({"duration":duration, "endDate": endTime});
            }
            temp[SPECIALTY_CATEGORY] = tempSpecialty;

            // Surgeon
            tempSurgeon = {};
            if (surgeon !== undefined)
            {
                // console.log("adding for surgeon: " + surgeon);
                tempSurgeon[surgeon] = [];
                tempSurgeon[surgeon].push({"duration":duration, "endDate": endTime});
            }
            temp[SURGEON_CATEGORY] = tempSurgeon;

            cptList[cpt] = temp;
        }
        else {
            // Update - add duration
            // console.log("found cpt: " + cpt);
            // console.log(cptList[cpt]);
            temp = cptList[cpt];
            // Updte ALL
            temp[ALL_CATEGORY].push({"duration":duration, "endDate": endTime});
            // Specialty
            if (specialty !== undefined){
                tempSpecialty = temp[SPECIALTY_CATEGORY] || {};
                if (tempSpecialty[specialty] === undefined){
                    tempSpecialty[specialty] = [];
                }
                
                tempSpecialty[specialty].push({"duration":duration, "endDate": endTime});
                temp[SPECIALTY_CATEGORY] = tempSpecialty;
            }
            
            // Surgeon
            if (surgeon !== undefined){
                tempSurgeon = temp[SURGEON_CATEGORY] || {};
                if (tempSurgeon[surgeon] === undefined){
                    tempSurgeon[surgeon] = [];
                }

                tempSurgeon[surgeon].push({"duration":duration, "endDate": endTime});
                temp[SURGEON_CATEGORY] = tempSurgeon;
            }

            cptList[cpt] = temp;
        }
    }
    else
    {
        console.log("No start time and end time available to calculate duration.")
    }

    return cptList;
}

function parseResultList(result, callback)
{
    let LINE_SEPARATOR = "\r\n";
    let  NODE_SEPARATOR = "^";
    let node;
    let model;
    let returnArray = [];
    let resultArray = result.split(LINE_SEPARATOR);
    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
        for (var i = 3; i < resultArray.length-2; i++){
            node = resultArray[i].split(NODE_SEPARATOR);
            if (node !== null && node !== undefined && node.length > 0){
                // Populate model
                model = new CPT();
                model.setData({
                    id: node[0],
                    cptCode: node[1],
                    name: node[3],
                    source: node[4], // C=CPT, H=HCPCS, L=Local
                    inactive: node[6],  // 1=inactive
                    cptCategory: node[5]
                });

                returnArray.push(model.sanitize());
            }
            else{
                console.log("Node is null for data: " + element);
            }
        };
    }
    else{
        console.log("Result Array empty for data: " + result);
    }
    callback(returnArray);

}

const ALL_CATEGORY = "ALL";
const SPECIALTY_CATEGORY = "SPECIALTY";
const SURGEON_CATEGORY = "SURGEON";


exports.getCPTDurations = getCPTDurations;

exports.ALL_CATEGORY = ALL_CATEGORY;
exports.SPECIALTY_CATEGORY = SPECIALTY_CATEGORY;
exports.SURGEON_CATEGORY = SURGEON_CATEGORY;