'use strict';
const vistaconfig = require("../../../lib/rpcvista/vistaconfig");
const vista = require("../../../lib/rpcvista/vista_surgery_services");
const auth = require("../authenticate");
const Case = require("../../model/case");
const NonOrCase = require("../../model/nonOrCase");
const OrCase = require("../../model/orCase");
const FullCase = require("../../model/fullCase");
const VistaConstants = require("../../model/vistaConstants");
const errorHandler = require("../../common/errorHandler");

/********************************
 *        General Cases         *
 ********************************/
/**
 * Retrieves specific case by case id (ien)
 * @param loginOptions
 * @param caseId
 * @param callback
 */
function findCaseById(loginOptions, caseId, callback) {
    console.log("Entering findCaseById ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
         vista.callRpc(
             vistaconfig.logger,
             configuration,
             'SR GET ONE CASE',
             caseId,
             function (error, result) {
                 if (error) {
                     console.log("Error: " + error);
                     callback(errorHandler.serverIssue(error), null);
                 }
                 else if (result === undefined || result === null || result === ""){
                     callback(errorHandler.notFound("Error: No case for ID: " + caseId));
                 }
                 else {
                     parseGetOneCaseResult(result, function(err, vistaCase){
                         if (err){
                             callback(errorHandler.serverIssue(err), null);
                         }
                         else {
                             callback(null, vistaCase);
                         }
                     });
                 }
         });
     });
}

/**
 * Creates a new case with the case data provided
 * @param loginOptions
 * @param caseData
 * @param callback
 */
function createNewCase(loginOptions, caseData, callback) {
    console.log("Entering createNewCase ...");

    // Case Data Validation
    if (caseData.dateOfOperation === undefined || caseData.dateOfOperation === null || caseData.dateOfOperation === ""
        || caseData.patient === undefined || caseData.patient === null || caseData.patient === ""
        || caseData.principalProcedure === undefined || caseData.principalProcedure === null || caseData.principalProcedure === "") {
        callback(errorHandler.invalidData("Error: Case is missing required data. Case creation requires a patient, principal procedure, and date of operation to be specified."), null);
    }
    else {

        auth.buildConfiguration(loginOptions, function (configuration) {
            if (caseData.caseCreateDate === undefined || caseData.caseCreateDate === ""){
                caseData.caseCreateDate = new Date();
            }

            buildFullCaseForVista(caseData, function (err, fullCase) {
                if (err) {
                    callback(errorHandler.serverIssue(err), null);
                }
                else {
                    vista.callRpc(
                        vistaconfig.logger,
                        configuration,
                        'SR CREATE CASE',
                        fullCase,
                        function (error, result) {
                            if (!error) {
                                console.log(result);
                                callback(null, {"ien": result});
                            }
                            else {
                                console.log("Error when creating case: " + error);
                                callback(errorHandler.serverIssue(error), null);
                            }
                        });
                }
            });
        });
    }
}

function createPlaceholderCase(loginOptions, caseData, callback) {
    console.log("Entering placeholderCase ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        formatPlaceholderForVista(caseData, function (err, fullCase) {
            console.log(fullCase);
            if (!err) {
                vista.callRpc(
                    vistaconfig.logger,
                    configuration,
                    'SR PUT PLHCASE',
                    caseData.patientId,
                    fullCase,
                    function (error, result) {
                        if (!error) {
                            console.log(result);
                            callback(null, result);
                        }
                        else {
                            console.log("Error when creating placeholder case: " + error);
                            callback(errorHandler.serverIssue(error), null);
                        }
                    });
            }
            else {
                console.log("Error when formating placeholder case: " + err);
                callback(errorHandler.serverIssue(err), null);
            }
        });
    });
}

exports.importCases = function (loginOptions, cases, callback) {
    let i;
    for (i = 0; i < cases.length; i++) {
        createNewCase(loginOptions, cases[i], function () {
            // Implement some logic
        });
    }
    callback(null, "SUCCESS");
}

/**
 * Creates a new case with the case data provided
 * @param loginOptions
 * @param caseData
 * @param callback
 */
function editExistingCase(loginOptions, id, caseData, callback) {
    console.log("Entering editExistingCase ...");
    auth.buildConfiguration(loginOptions, function (configuration) {
        buildFullCaseForVista(caseData, function (err, fullCase) {
            // TODO - call rpc to create
            if (err) {
                console.log("Error: " + err);
                callback(errorHandler.serverIssue(err), null);
            }
            else {

                //Update case string with IEN
                fullCase = fullCase + "^" + id;
                vista.callRpc(
                    vistaconfig.logger,
                    configuration,
                    'SR EDIT CASE',
                    fullCase,
                    function (error, result) {
                        if (!error) {
                            console.log(result);
                            callback(null, {"ien": result});
                        }
                        else {
                            console.log("Error when editing case: " + error);
                            callback(errorHandler.serverIssue(error), null);
                        }
                    });
            }
        });

        // callback(null, caseData);
    });
}

/*
 CASES
 -----
 date
 date range
 patient
 case status
 case type
 specialty
 provider
 primarySurgeon -
 firstAsst
 secondAsst
 attendingSurgeon
 or room
 cpt code
 other???


 SCHEDULED EVENTS
 ----------------
 operating room
 date
 date range
 */
function searchCases(loginOptions, searchData, returnArray, set, callback) {
    if (arguments.length === 3) {
        callback = returnArray;
        returnArray = [];
        set = 8;
    }

    auth.buildConfiguration(loginOptions, function (configuration) {
        let addToList = false;
        let vistaCase;
        let node;
        let resultArray;

        let date = searchData["date"] || undefined,
            dateRange = searchData["dateRange"] || undefined,
            patientId = searchData["patientId"] || undefined,
            caseStatus = searchData["caseStatus"] || undefined,
            caseType = searchData["caseType"] || undefined,
            specialty = searchData["specialty"] || undefined,
            provider = searchData["provider"] || undefined,
            primarySurgeon = searchData["primarySurgeon"] || undefined,
            firstAsst = searchData["firstAsst"] || undefined,
            secondAsst = searchData["secondAsst"] || undefined,
            attendingSurgeon = searchData["attendingSurgeon"] || undefined,
            orRoom = searchData["orRoom"] || undefined,
            cptCode = searchData["cptCode"] || undefined,
            concurrentCase = searchData["concurrentCase"] || undefined,
            staffMember = searchData["staffMember"] || undefined,
            ien = searchData["ien"] || undefined;

        // Convert date to date range for RPC
        if (dateRange === undefined && date !== undefined) {
            dateRange = {};
            dateRange.start = new Date(date);
            dateRange.end = new Date(new Date(date).getTime() + (1000*60*60*24));
        }
        else if (dateRange !== undefined && dateRange.start !== undefined && dateRange.end !== undefined) {
            dateRange.start = new Date(dateRange.start);
            dateRange.end = new Date(dateRange.end);
        }

        /*
        // Check for patient, or/non-or,  and date info
        if ((patientId !== undefined && patientId.length > 0)
            && (caseType !== undefined && caseType.length > 0)
            && dateRange !== undefined) {
            // Call 'SR GET ORNONOR CASES'
            if (caseType == "OR" || caseType == "PLACEHOLDER" || caseType == "REQUESTED" || caseType == "SCHEDULED") {
                // Build query
                vista.callRpc(
                    vistaconfig.logger,
                    configuration,
                    'SR GET ORNONOR CASES',
                    patientId,
                    vista.convertToVistaDateNoTime(new Date(dateRange.start)),
                    vista.convertToVistaDateNoTime(new Date(dateRange.end)),
                    "OR",
                    function (error, result) {
                        console.log(result);
                        if (error) {
                            console.log("Error: " + error);
                            callback(errorHandler.serverIssue(error), null);
                        }
                        else {
                            console.log("found or cases for patient by date");
                            // console.log(result);
                            resultArray = result.split("\r")
                            if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
                                console.log(resultArray);
                                for (let i = 0; i < resultArray.length; i++) {
                                    addToList = false;
                                    node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
                                    console.log(node);
                                    console.log("length");
                                    console.log(node.length);
                                    if (node !== null && node !== undefined && node.length > 0 && node.length !== 1) {

                                        /*
                                         Check for the following filters
                                         caseStatus
                                         caseType
                                         specialty
                                         primarySurgeon
                                         firstAsst
                                         secondAsst
                                         attendingSurgeon
                                         orRoom
                                         cptCode
                                         NOTE: Not checking for provider because a provider only exists in Non-OR cases
                                         */
                                        // Check for search criteria
                                        //TODO - UpdatecaseStatus and caseType checks for addList logic
                                        /*addToList = ((caseStatus === undefined || caseStatus.equalsIgnoreCase(node[44]))
                                         && (caseType === undefined || caseType.equalsIgnoreCase()));
                                         */
                                        // TODO - Remove the block comment starter for old search logic
                                        /*
                                        console.log();
                                        addToList = ((specialty === undefined || specialty.toUpperCase() === (node[7]).toUpperCase())
                                        && (primarySurgeon === undefined || primarySurgeon.toUpperCase() === (node[9]).toUpperCase())
                                        && (firstAsst === undefined || firstAsst.toUpperCase() === (node[10]).toUpperCase())
                                        && (secondAsst === undefined || secondAsst.toUpperCase() === (node[11]).toUpperCase())
                                        && (attendingSurgeon === undefined || attendingSurgeon.toUpperCase() === (node[12]).toUpperCase())
                                        && (orRoom === undefined || orRoom.toUpperCase() === (node[4]).toUpperCase())
                                        && (cptCode === undefined || cptCode.toUpperCase() === (node[27]).toUpperCase()));


                                        console.log(addToList);
                                        if (addToList) {
                                            // Populate model
                                            vistaCase = new OrCase();
                                            vistaCase.setData({
                                                "patient": node[0],
                                                "hospitalAdmissionStatus": node[1],
                                                "plannedAdmissionStatus": node[2],
                                                "serviceConnected": node[3],
                                                "orRoom": node[4],
                                                "caseScheduleType": node[5],
                                                "caseScheduleOrder": node[6],
                                                "surgerySpecialty": node[7],
                                                "preOpInfection": node[8],
                                                "primarySurgeon": node[9],
                                                "firstAsst": node[10],
                                                "secondAsst": node[11],
                                                "attendingSurgeon": node[12],
                                                "generalComments": node[13],
                                                "otherProcedures": node[14],
                                                "plannedPostopCare": node[15],
                                                "otherPreopDiagnosis": node[16],
                                                "reqAneTechnique": node[17],
                                                "reqFrozSect": node[18],
                                                "reqPreopXray": node[19],
                                                "intraoperativeXrays": node[20],
                                                "reqPhoto": node[21],
                                                "reqBloodKind": node[22],
                                                "caseCreateDate": vista.convertVistaDateToISODate(node[23]),
                                                "cancelDate": node[24],
                                                "primaryCancelReason": node[25],
                                                "referringPhysician": node[26],
                                                "principalProcedure": node[28],
                                                "plannedPrincOpProcCPT": node[27],
                                                "principalPreOpDiagnosis": node[29],
                                                "prinPreOpDiagICD": node[30],
                                                "requestBloodAvailability": node[31],
                                                "bloodProductInfo": node[32],
                                                "operationIndications": node[33],
                                                "preAdmissionTesting": node[34],
                                                "surgeryPosition": node[35],
                                                "principalPreOpICD": node[36],
                                                "procedureLaterality": node[37],
                                                "palliation": node[38],
                                                "specialEquipment": node[39],
                                                "plannedProsthImplant": node[40],
                                                "specialSupplies": node[41],
                                                "specialInstruments": node[42],
                                                "pharmacyItems": node[43]
                                            });

                                            returnArray.push(vistaCase.sanitize());
                                        }
                                    }
                                    else {
                                        console.log("Node is null for data: " + resultArray[i]);
                                    }
                                }
                            }
                            else {
                                console.log("Result Array empty for data: " + result);
                            }

                            // parseORCaseData(result, function(err, caseList){
                            //     callback(errorHandler.serverIssue(err), caseList);
                            // });
                            callback(errorHandler.serverIssue(error), returnArray);
                        }

                    }
                );
            }
            else if (caseType == "NON-OR") {
                console.log("Trying to search for NoN-OR case. OLD LOGIC");
            }
        }
        else {
        // TODO - REMOVE BLOCK COMMENT END BLOCK for old code
         */
            // Call 'SR GET ALL CASES'
            vista.callRpc(
                vistaconfig.logger,
                configuration,
                'SR GET ALL CASES',
                (set * 50) - 50 + 1,
                set * 50,
                function (error, result) {
                    let complete = false;
                    if (error) {
                        // Silently fail
                        console.log("Error: Silent Fail - " + error);
                        // callback(errorHandler.serverIssue(error), null);
                        // complete = true;
                    }
                    else {
                        // console.log(result);
                        resultArray = result.split("\r");
                        if (resultArray !== null && resultArray != undefined && resultArray.length > 1) {
                            // console.log(resultArray);
                            for (let i = 0; i < resultArray.length - 1; i++) {
                                addToList = false;
                                node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);

                                if (node !== null && node !== undefined && node.length > 0 && node.length !== 1) {

                                    /*
                                     Check for the following filters
                                     caseStatus
                                     caseType
                                     specialty
                                     primarySurgeon
                                     firstAsst
                                     secondAsst
                                     attendingSurgeon
                                     orRoom
                                     cptCode
                                     staffMember
                                     NOTE: Not checking for provider because a provider only exists in Non-OR cases
                                     */

                                    let opDate = (node[3] !== undefined && node[3] !== "" && resultArray[3] !== "NaN" ? vista.convertFromVistaDate(node[3]) : (node[4] !== undefined && node[4] !== "" && resultArray[4] !== "NaN" ? vista.convertFromVistaDate(node[4]) : undefined));

                                    // Check for search criteria
                                    addToList = ((patientId === undefined || patientId == node[1])
                                    && (specialty === undefined || specialty.toUpperCase() == (node[11]).toUpperCase())
                                    && (primarySurgeon === undefined || primarySurgeon.toUpperCase() == (node[14]).toUpperCase())
                                    && (provider === undefined || provider == (node[13]))
                                    && (firstAsst === undefined || firstAsst == (node[16]))
                                    && (secondAsst === undefined || secondAsst == (node[17]))
                                    && (attendingSurgeon === undefined || attendingSurgeon == (node[15]))
                                    && (orRoom === undefined || orRoom == (node[9]))
                                    && (cptCode === undefined || cptCode.toUpperCase() == (node[6]).toUpperCase())
                                    && (ien === undefined || ien == (node[0]))
                                    && (staffMember === undefined || staffMember === (node[14]) || staffMember === (node[16])
                                    || staffMember === (node[17]) || staffMember === (node[15]) || staffMember === (node[13]))
                                    && (concurrentCase === undefined || concurrentCase === (node[23]))
                                    && (caseType === undefined || caseType === node[22])
                                    && (caseStatus === undefined || caseStatus === node[21])
                                    && (dateRange === undefined || dateRange.start === undefined || dateRange.end === undefined || (opDate !== undefined && opDate >= dateRange.start && opDate <= dateRange.end)));

                                    if (addToList) {
                                        // Populate model
                                        vistaCase = new Case();
                                        vistaCase.setData({
                                            "ien": node[0],
                                            "patient": node[1],
                                            "caseCreateDate": (node[2] !== undefined && node[2] !== "" && resultArray[2] !== "NaN" ? vista.convertFromVistaDate(node[2]) : ""),
                                            "dateOfOperation": (node[3] !== undefined && node[3] !== "" && resultArray[3] !== "NaN" ? vista.convertFromVistaDate(node[3]) : ""),
                                            "procedureDate": (node[4] !== undefined && node[4] !== "" && resultArray[4] !== "NaN" ? vista.convertFromVistaDate(node[4]) : ""),
                                            "principalProcedure": node[5],
                                            "plannedPrincOpProcCPT": node[6],
                                            "scheduledStartTime": (node[7] !== undefined && node[7] !== "" && resultArray[7] !== "NaN" ? vista.convertFromVistaDate(node[7]) : ""),
                                            "scheduledEndTime": (node[8] !== undefined && node[8] !== "" && resultArray[8] !== "NaN" ? vista.convertFromVistaDate(node[8]) : ""),
                                            // "startTime": (node[7] !== undefined && node[7] !== "" && resultArray[^^^] !== "NaN" ? vista.convertFromVistaDate(node[7]) : ""),
                                            // "endTime": (node[8] !== undefined && node[8] !== "" && resultArray[^^^] !== "NaN" ? vista.convertFromVistaDate(node[8]) : ""),
                                            "orRoom": node[9],
                                            //'1' FOR SAME DAY;     '2' FOR ADMISSION;  '3' FOR HOSPITALIZED;
                                            "hospitalAdmissionStatus": node[10],//(node[10] === 1 ? "SAME DAY" : (node[10] === 2 ? "FOR ADMISSION" : "FOR HOSPITALIZED")),
                                            "surgerySpecialty": node[11],
                                            "referringPhysician": node[12],
                                            "provider": node[13],
                                            "primarySurgeon": node[14],
                                            "attendingSurgeon": node[15],
                                            "firstAsst": node[16],
                                            "secondAsst": node[17],
                                            // 'EM' FOR EMERGENCY;  'EL' FOR ELECTIVE;  'A' FOR ADD ON (NON-EMERGENT);  'S' FOR STANDBY;    'U' FOR URGENT;
                                            "caseScheduleType": node[18],   //(node[18] === 'A' ? "FOR ADD ON (NON-EMERGENT)" : (node[18]) === 'EL' ? "FOR ELECTIVE" : (node[18] === 'EM' ? "FOR EMERGENCY" : "FOR URGENT")),
                                            "primaryCancelReason": node[19],
                                            "requested": node[20],
                                            "caseStatus": node[21],
                                            "caseType": node[22],
                                            "concurrentCase": node[23]
                                        });

                                        returnArray.push(vistaCase.sanitize());
                                    }
                                }
                                else {

                                    console.log("Node is null for data: " + resultArray[i]);
                                }
                            }
                        }
                        else {
                            complete = true;
                        }
                    }

                    if (complete) {
                        callback(null, returnArray);
                    }
                    else {
                         // Call recursively
                        searchCases(loginOptions, searchData, returnArray, set + 1, callback);
                    }
                });
        // }
    });
}

/**
 *
 * @param loginOptions
 * @param extendedCase [startDate, endDate, extendedTime, staffMembers, orRoom]
 * @param returnArray
 * @param set
 * @param callback
 */
function getImpactedCases(loginOptions, extendedMinutes, extendedCase, returnArray, set, callback) {
    if (arguments.length === 4) {
        callback = returnArray;
        returnArray = [];
        set = 1;
    }

    auth.buildConfiguration(loginOptions, function (configuration) {
        let addToList = false;
        let vistaCase;
        let node;
        let resultArray;

        let extendedCaseStart = extendedCase["scheduledStartTime"] || undefined,
            extendedCaseEnd = extendedCase["scheduledEndTime"] || undefined,
            provider = extendedCase["provider"] || undefined,
            primarySurgeon = extendedCase["primarySurgeon"] || undefined,
            firstAsst = extendedCase["firstAsst"] || undefined,
            secondAsst = extendedCase["secondAsst"] || undefined,
            attendingSurgeon = extendedCase["attendingSurgeon"] || undefined,
            // TODO - check for additional staff members
            orRoom = extendedCase["orRoom"] || undefined,
            staffMembers = [],
            newEndtime;

        if (extendedCaseStart === undefined || extendedCaseStart === ""
            || extendedCaseEnd === undefined || extendedCaseEnd === ""){
            console.log("Date to extend is missing start date or end date: " + extendedCaseStart + " - " + extendedCaseEnd);
            callback("ERROR: Date to extend is missing start date or end date: " + extendedCaseStart + " - " + extendedCaseEnd, null);
        }

        // Calculate new endtime
        // dateRange.end = new Date(new Date(date).getTime() + (1000*60*60*24));
        newEndtime = new Date(extendedCaseEnd.getTime() + (extendedMinutes * 60 * 1000));
        // console.log("COMPARE: " + extendedCaseEnd + " == " + newEndtime);

        // Create staffMembers array
        if (provider !== undefined) {staffMembers.push(provider);}
        if (primarySurgeon !== undefined) {staffMembers.push(primarySurgeon);}
        if (firstAsst !== undefined) {staffMembers.push(firstAsst);}
        if (secondAsst !== undefined) {staffMembers.push(secondAsst);}
        if (attendingSurgeon !== undefined) {staffMembers.push(attendingSurgeon);}

        // Call 'SR GET ALL CASES'
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'SR GET ALL CASES',
            (set * 50) - 50 + 1,
            set * 50,
            function (error, result) {
                let complete = false;
                if (error) {
                    console.log("Error: " + error);
                    //callback(errorHandler.serverIssue(error), null);
                    complete = true;
                }
                else {
                    resultArray = result.split("\r");
                    if (resultArray !== null && resultArray != undefined && resultArray.length > 1) {
                        for (let i = 0; i < resultArray.length - 1; i++) {
                            addToList = false;
                            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);

                            if (node !== null && node !== undefined && node.length > 0 && node.length !== 1) {

                                // TODO only check scheduled cases once case status is fully implemented in RPCs
                                let startTime = node[7] || undefined,
                                    endTime = node[8] || undefined;
                                // Adjusted endtime  is > other case start time and < other case end time
                                // && (ORroom == OR room
                                //      || staff member
                                if (startTime !== undefined && endTime !== undefined) {
                                    startTime = vista.convertFromVistaDate(startTime);
                                    endTime = vista.convertFromVistaDate(endTime);

                                    addToList = (extendedCaseStart < startTime && startTime <= newEndtime)
                                        && ((node[14] === "" || staffMembers.indexOf(node[14]) > -1) // primarySurgeon
                                            || (node[13] === "" || staffMembers.indexOf(node[13]) > -1) // provider
                                            || (node[16] === "" || staffMembers.indexOf(node[16]) > -1) // firstAsst
                                            || (node[17] === "" || staffMembers.indexOf(node[17]) > -1) // secondAsst
                                            || (node[15] === "" || staffMembers.indexOf(node[15]) > -1) // attendingSurgeon
                                            || (orRoom === undefined || orRoom.toUpperCase() == (node[9]).toUpperCase())); // orRoom

                                    // TODO - check SSLQI once fully implemented

                                    if (addToList) {
                                        // Populate model
                                        vistaCase = new Case();
                                        vistaCase.setData({
                                            "ien": node[0],
                                            "patient": node[1],
                                            "caseCreateDate": (node[2] !== undefined && node[2] !== "" && resultArray[2] !== "NaN" ? vista.convertFromVistaDate(node[2]) : ""),
                                            "dateOfOperation": (node[3] !== undefined && node[3] !== "" && resultArray[3] !== "NaN" ? vista.convertFromVistaDate(node[3]) : ""),
                                            "procedureDate": (node[4] !== undefined && node[4] !== "" && resultArray[4] !== "NaN" ? vista.convertFromVistaDate(node[4]) : ""),
                                            "principalProcedure": node[5],
                                            "plannedPrincOpProcCPT": node[6],
                                            "scheduledStartTime": (node[7] !== undefined && node[7] !== "" && resultArray[7] !== "NaN" ? vista.convertFromVistaDate(node[7]) : ""),
                                            "scheduledEndTime": (node[8] !== undefined && node[8] !== "" && resultArray[8] !== "NaN" ? vista.convertFromVistaDate(node[8]) : ""),
                                            // "startTime": (node[7] !== undefined && node[7] !== "" && resultArray[^^^] !== "NaN" ? vista.convertFromVistaDate(node[7]) : ""),
                                            // "endTime": (node[8] !== undefined && node[8] !== "" && resultArray[^^^] !== "NaN" ? vista.convertFromVistaDate(node[8]) : ""),
                                            "orRoom": node[9],
                                            //'1' FOR SAME DAY;     '2' FOR ADMISSION;  '3' FOR HOSPITALIZED;
                                            "hospitalAdmissionStatus": (node[10] === 1 ? "FOR SAME DAY" : (node[10] === 2 ? "FOR ADMISSION" : "FOR HOSPITALIZED")),
                                            "surgerySpecialty": node[11],
                                            "referringPhysician": node[12],
                                            "provider": node[13],
                                            "primarySurgeon": node[14],
                                            "attendingSurgeon": node[15],
                                            "firstAsst": node[16],
                                            "secondAsst": node[17],
                                            // 'EM' FOR EMERGENCY;  'EL' FOR ELECTIVE;  'A' FOR ADD ON (NON-EMERGENT);  'S' FOR STANDBY;    'U' FOR URGENT;
                                            "caseScheduleType": (node[18] === 'A' ? "FOR ADD ON (NON-EMERGENT)" : (node[18]) === 'EL' ? "FOR ELECTIVE" : (node[18] === 'EM' ? "FOR EMERGENCY" : "FOR URGENT")),
                                            "primaryCancelReason": node[19],
                                            "requested": node[20],
                                            /*"caseStatus": (node[21] == 1 ? "COMPLETED" : (node[21] == 2 ? "NOT COMPLETED" : (node[21] == 3 ? "CANCELLED" : (node[21] == 4 ? "INACTIVE": (node[21] == 5 ? "DELETED" : node[21]))))),
                                             "caseType": (node[22] == 1 ? "REQUESTED" : (node[22] == 2 ? "SCHEDULED" : (node[22] == 3 ? "PLACEHOLDER" : (node[22] == 4 ? "NON-OR": node[22])))),    //node[22]
                                             */
                                            "caseStatus": "",
                                            "caseType": (node[20] === 1 ? "REQUESTED" : (node[13] === "" ? "SCHEDULED" : "NON-OR")),
                                            "concurrentCase": node[23]
                                        });

                                        returnArray.push(vistaCase.sanitize());
                                    }
                                }
                                else{
                                    console.log("No start or end time");
                                }
                            }
                            else {

                                console.log("Node is null for data: " + resultArray[i]);
                            }
                        }
                    }
                    else {
                        complete = true;
                    }
                }

                // Call recursively
                // if (returnArray.length < (set * 50) - 2)
                if (complete) {
                    callback(null, returnArray);
                }
                else {
                    getImpactedCases(loginOptions, extendedMinutes, extendedCase, returnArray, set + 1, callback);
                }
            });
    });
}

/** helper to create random numbers **/
function getRandomInt(min, max) {
    min = Math.ceil(min);
    max = Math.floor(max);
    return Math.floor(Math.random() * (max - min)) + min;
}

exports.orroomcases = function (loginoptions, room_id, callback) {
    let case_types = [
        "Major Abdominal Procedure",
        "Major Vascular Surgery",
        "Mastectomy (Radical)",
        "Mediport Insertion or Removal",
        "Prostate Surgery",
        "Removal of Hardware (Plates and Screws)",
        "Sinus Endoscopy",
        "Tonsillectomy",
        "Tubal Ligation",
        "Vasectomy",
        "Ventral Hernia"];


    let cases = [];
    let num_cases = getRandomInt(0, 2);
    console.log("Num cases: " + num_cases);
    for (let i = 0; i < num_cases; i++) {
        console.log("Case number: ", i);
        let vistaCase = new OrCase();
        let scheduledStartTime = new Date();
        let random_hour = getRandomInt(9, 15);
        console.log("Start Hour: " + random_hour);
        scheduledStartTime.setUTCHours(random_hour, 0);
        let scheduledEndTime = new Date();
        scheduledEndTime.setUTCHours(getRandomInt(scheduledStartTime.getUTCHours(), 17), 0);
        vistaCase.setData({
            "patient": null,
            "hospitalAdmissionStatus": null,
            "plannedAdmissionStatus": null,
            "serviceConnected": null,
            "orRoom": room_id,
            "caseScheduleType": null,
            "caseScheduleOrder": null,
            "surgerySpecialty": null,
            "preOpInfection": null,
            "primarySurgeon": null,
            "firstAsst": null,
            "secondAsst": null,
            "attendingSurgeon": null,
            "generalComments": null,
            "otherProcedures": null,
            "plannedPostopCare": null,
            "otherPreopDiagnosis": null,
            "reqAneTechnique": null,
            "reqFrozSect": null,
            "reqPreopXray": null,
            "intraoperativeXrays": null,
            "reqPhoto": null,
            "reqBloodKind": null,
            "caseCreateDate": null,
            "cancelDate": null,
            "primaryCancelReason": null,
            "referringPhysician": null,
            "principalProcedure": case_types[getRandomInt(0, 10)],
            "plannedPrincOpProcCPT": null,
            "principalPreOpDiagnosis": null,
            "prinPreOpDiagICD": null,
            "requestBloodAvailability": null,
            "bloodProductInfo": null,
            "operationIndications": null,
            "preAdmissionTesting": null,
            "surgeryPosition": null,
            "principalPreOpICD": null,
            "procedureLaterality": null,
            "palliation": null,
            "specialEquipment": null,
            "plannedProsthImplant": null,
            "specialSupplies": null,
            "specialInstruments": null,
            "pharmacyItems": null,
            "scheduledStartTime": scheduledStartTime,
            "scheduledEndTime": scheduledEndTime
        });
        cases.push(vistaCase.data);
    }
    console.log("Cases: ", cases);
    callback(null, cases);
}


/****************************************
 *          Helper Methods              *
 ****************************************/

function parseGetOneCaseResult(result, callback){
    if (result !== undefined && result !== null){
        let vistaCase, resultArray;
        resultArray = result.split("^");

        if (resultArray !== null && resultArray !== undefined && resultArray.length > 0) {
            vistaCase = new FullCase();
            vistaCase.setData(
                {
                    "ien": resultArray[0],
                    "patient": resultArray[3],
                    "hospitalAdmissionStatus": resultArray[4],
                    "serviceConnected": resultArray[5],
                    "associatedClinic": resultArray[6],
                    "resSupCode": resultArray[7],
                    "anesCareTimeBlock": parseMultipleField(resultArray[8]),
                    "generalComments": resultArray[9],
                    "princAnesthetist": resultArray[10],
                    "anesthesiologistSupvr": resultArray[11],
                    "anesSuperviseCode": resultArray[12],
                    "anesthesiaTechnique": parseMultipleField(resultArray[13]),
                    "otherProcedures": parseMultipleField(resultArray[14]),
                    "plannedPostopCare": resultArray[15],
                    "otherPostopDiags": parseMultipleField(resultArray[16]),
                    "caseCreateDate": (resultArray[17] !== undefined && resultArray[17] !== "" && resultArray[17] !== "NaN" ? vista.convertFromVistaDate(resultArray[17]) : ""),
                    "asaClass": resultArray[18],
                    "cancelDate": (resultArray[19] !== undefined && resultArray[19] !== "" && resultArray[19] !== "NaN" ? vista.convertFromVistaDate(resultArray[19]) : ""),
                    "primaryCancelReason": resultArray[20],
                    "diagnosticTherapeutic": resultArray[121],
                    "principalProcedure": resultArray[22],
                    "plannedPrincOpProcCPT": resultArray[23],
                    "principalDiagnosis": resultArray[24],
                    "specimens": resultArray[25],
                    "operationIndications": resultArray[26],
                    "operativeFindings": resultArray[27],
                    "briefClinHistory": resultArray[28],
                    "plannedPrinDiagCode": resultArray[29],
                    "location": resultArray[30],
                    "procedureDate": (resultArray[31] !== undefined && resultArray[31] !== "" && resultArray[31] !== "NaN" ? vista.convertFromVistaDate(resultArray[31]) : ""),
                    "procedureStartTime": (resultArray[32] !== undefined && resultArray[32] !== "" && resultArray[32] !== "NaN" ? vista.convertFromVistaDate(resultArray[32]) : ""),
                    "procedureEndTime": (resultArray[33] !== undefined && resultArray[33] !== "" && resultArray[33] !== "NaN" ? vista.convertFromVistaDate(resultArray[33]) : ""),
                    "provider": resultArray[34],
                    "attendingProvider": resultArray[35],
                    "medicalSpecialty": resultArray[36],
                    "procedureOccurence": parseMultipleField(resultArray[37]),
                    "procedureLaterality": resultArray[38],
                    "dictatedSummary": resultArray[39],
                    "cancelComments": resultArray[40],
                    "cancelTimeframe": resultArray[41],
                    "caseScheduleType": resultArray[42],
                    "concurrentCase": resultArray[43],
                    "dateOfOperation": (resultArray[44] !== undefined && resultArray[44] !== "" && resultArray[44] !== "NaN" ? vista.convertFromVistaDate(resultArray[44]) : ""),
                    "nonOrFlag": resultArray[45],
                    "orRoom": resultArray[46],
                    "principalPreOpDiagnosis": resultArray[47],
                    "reqAneTechnique": resultArray[48],
                    "surgeryPosition": parseMultipleField(resultArray[49]),
                    "plannedAdmissionStatus": resultArray[50],
                    "caseScheduleOrder": resultArray[51],
                    "surgerySpecialty": resultArray[52],
                    "preOpInfection": resultArray[53],
                    "primarySurgeon": resultArray[54],
                    "firstAsst": resultArray[55],
                    "secondAsst": resultArray[56],
                    "attendingSurgeon": resultArray[57],
                    "otherPreopDiagnosis": parseMultipleField(resultArray[58]),
                    "reqFrozSect": resultArray[59],
                    "reqPreopXray": resultArray[60],
                    "intraoperativeXrays": resultArray[61],
                    "reqPhoto": resultArray[62],
                    "reqBloodKind": parseMultipleField(resultArray[63]),
                    "referringPhysician": parseMultipleField(resultArray[64]),
                    "prinPreOpDiagICD": resultArray[65],
                    "requestBloodAvailability": resultArray[66],
                    "bloodProductInfo": resultArray[67],
                    "preAdmissionTesting": resultArray[68],
                    "palliation": resultArray[69],
                    "specialEquipment": parseMultipleField(resultArray[70]),
                    "plannedProsthImplant": parseMultipleField(resultArray[71]),
                    "specialSupplies": parseMultipleField(resultArray[72]),
                    "specialInstruments": parseMultipleField(resultArray[73]),
                    "pharmacyItems": parseMultipleField(resultArray[74]),
                    "bloodTypeXmatch": resultArray[75],
                    "scheduledEndTime": (resultArray[76] !== undefined && resultArray[76] !== "" && resultArray[76] !== "NaN" ? vista.convertFromVistaDate(resultArray[76]) : ""),
                    "scheduledStartTime": (resultArray[77] !== undefined && resultArray[77] !== "" && resultArray[77] !== "NaN" ? vista.convertFromVistaDate(resultArray[77]) : ""),
                    "spinalLevel": resultArray[78],
                    "surgeryScheduler": resultArray[79],
                    "caseStatus": resultArray[80],
                    "caseType": resultArray[81],
                    "preopSkinInteg": resultArray[82],
                    "toOrTranspDevice": resultArray[83],
                    "hairRemovalBy": resultArray[84],
                    "restraintPositionAids": parseMultipleField(resultArray[85]),
                    "skinPrepAgent": resultArray[86],
                    "skinPrepperOne": resultArray[87],
                    "preopMood": resultArray[88],
                    "preopConscious": resultArray[89],
                    "timePatientInHoldArea": resultArray[90],
                    "timePatientInOR": resultArray[91],
                    "timeOperationBegan": resultArray[92],
                    "timeOperationEnds": resultArray[93],
                    "timePatientOutOR": resultArray[94],
                    "bloodLoss": resultArray[95],
                    "totalUrineOutput": resultArray[96],
                    "opDisposition": resultArray[97],
                    "electroGroundPosition": resultArray[98],
                    "foleyCatheterSize": resultArray[99],
                    "foleyCatheterInsertedBy": resultArray[100],
                    "postopSkinInteg": resultArray[101],
                    "postopMood": resultArray[102],
                    "postopConscious": resultArray[103],
                    "packing": resultArray[104],
                    "patientEduAssessment": resultArray[105],
                    "woundClassifciation": resultArray[106],
                    "skinPrepperTwo": resultArray[107],
                    "skinPrepperThree": resultArray[108],
                    "secondSkinPrepAgent": resultArray[109],
                    "tubesAndDrains": resultArray[110],
                    "fromeCareAreaTranspDevice": resultArray[111],
                    "principalPostOpDiag": resultArray[112],
                    "dressing": resultArray[113],
                    "gastricOutput": resultArray[114],
                    "height": resultArray[115],
                    "heightMeasureDate": resultArray[116],
                    "weight": resultArray[117],
                    "hairRemovalMethod": resultArray[118],
                    "hariRemovalComments": resultArray[119],
                    "iuContamination": resultArray[120],
                    "iuSPSOrMgmt": resultArray[121],
                    "iuEmergencyCase": resultArray[122],
                    "iuNoBetterOption": resultArray[123],
                    "iuLoanerInstrument": resultArray[124],
                    "iuDecontamination": resultArray[125],
                    "reportGivenTo": resultArray[126],
                    "general": resultArray[127],
                    "diabetesMellitusChronic": resultArray[128],
                    "diabetesMellitusPreopMgmt": resultArray[129],
                    "tobaccoUse": resultArray[130],
                    "tobaccoUseTimeframe": resultArray[131],
                    "etohGTTwoDrinksPerDay": resultArray[132],
                    "positiveDrugScreening": resultArray[133],
                    "dyspnea": resultArray[134],
                    "preopSleepApnea": resultArray[135],
                    "dnrStatus": resultArray[136],
                    "preopFunctHealthStatus": resultArray[137],
                    "residence30DaysPreop": resultArray[138],
                    "ambulationDevicePreop": resultArray[139],
                    "homeless": resultArray[140],
                    "pulmonary": resultArray[141],
                    "ventilatorDependent": resultArray[142],
                    "historyOfCOPD": resultArray[143],
                    "currentPneumonia": resultArray[144],
                    "hepatobiliary": resultArray[145],
                    "ascites": resultArray[146],
                    "gastrointestinal": resultArray[147],
                    "esophagealVarices": resultArray[148],
                    "cardiac": resultArray[149],
                    "congestiveHeartFailurePreop": resultArray[150],
                    "priorMI": resultArray[151],
                    "pci": resultArray[152],
                    "priorHearSurgery": resultArray[153],
                    "anginaSeverity": resultArray[154],
                    "anginaTimeframe": resultArray[155],
                    "hypertension": resultArray[156],
                    "priorSurgSameOpField": resultArray[157],
                    "hxRadRxPlannedSurgField": resultArray[158],
                    "cvdRepair": resultArray[159],
                    "donorSerologyHIV": resultArray[160],
                    "vascular": resultArray[161],
                    "peripheralArterialDisease": resultArray[162],
                    "restPainGangrene": resultArray[163],
                    "renal": resultArray[164],
                    "renalFailure": resultArray[165],
                    "onDialysis": resultArray[166],
                    "centralNervSystemIllness": resultArray[167],
                    "impairedSensorium": resultArray[168],
                    "coma": resultArray[169],
                    "hemiplegiaHemiparesis": resultArray[170],
                    "historyOfCVD": resultArray[171],
                    "tumorInvolvingCNS": resultArray[172],
                    "impairedCognitiveFunction": resultArray[173],
                    "nutritionalImmuneOther": resultArray[174],
                    "disseminatedCancer": resultArray[175],
                    "openWound": resultArray[176],
                    "chronicSteroidUse": resultArray[177],
                    "tenPercDecrWeightLoss": resultArray[178],
                    "bleedingDisorders": resultArray[179],
                    "bleedingRiskDueToMed": resultArray[180],
                    "transfusionGTFourRBCUnits": resultArray[181],
                    "chemoForMaligLastNinety": resultArray[182],
                    "radiotherapyLastNinety": resultArray[183],
                    "preopSepsis": resultArray[184],
                    "pregnancy": "" // TODO -  Fix the following when RPC is fixed: resultArray[185]
                });
            callback(null, vistaCase.sanitize());
        }
        else {
            callback("Error: Unable to parse VistA data.", null);
        }
    }
    else{
        callback("Error: Data is null or undefined.", null);
    }

}

function parseMultipleField(value){
    let list = [];

    if (value !== undefined && value !== null && value !== ""){
        if (value.includes(";")){
            list = value.split(";");
        }
        else{
            list.push(value);
        }
    }

    return list;
}

function parseORCaseData(data, callback) {
    let returnArray = [];
    let vistaCase;
    let node;
    let resultArray = data.split("\r");//vistaconfig.LINE_SEPARATOR);

    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
        // console.log(resultArray);
        for (let i = 0; i < resultArray.length; i++) {
            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
            // console.log(node);
            if (node !== null && node !== undefined && node.length > 0 && node.length !== 1) {
                // Populate model
                vistaCase = new OrCase();
                vistaCase.setData({
                    "patient": node[0],
                    "hospitalAdmissionStatus": node[1],
                    "plannedAdmissionStatus": node[2],
                    "serviceConnected": node[3],
                    "orRoom": node[4],
                    "caseScheduleType": node[5],
                    "caseScheduleOrder": node[6],
                    "surgerySpecialty": node[7],
                    "preOpInfection": node[8],
                    "primarySurgeon": node[9],
                    "firstAsst": node[10],
                    "secondAsst": node[11],
                    "attendingSurgeon": node[12],
                    "generalComments": node[13],
                    "otherProcedures": node[14],
                    "plannedPostopCare": node[15],
                    "otherPreopDiagnosis": node[16],
                    "reqAneTechnique": node[17],
                    "reqFrozSect": node[18],
                    "reqPreopXray": node[19],
                    "intraoperativeXrays": node[20],
                    "reqPhoto": node[21],
                    "reqBloodKind": node[22],
                    "caseCreateDate": vista.convertVistaDateToISODate(node[23]),
                    "cancelDate": node[24],
                    "primaryCancelReason": node[25],
                    "referringPhysician": node[26],
                    "principalProcedure": node[27],
                    "plannedPrincOpProcCPT": node[28],
                    "principalPreOpDiagnosis": node[29],
                    "prinPreOpDiagICD": node[30],
                    "requestBloodAvailability": node[31],
                    "bloodProductInfo": node[32],
                    "operationIndications": node[33],
                    "preAdmissionTesting": node[34],
                    "surgeryPosition": node[35],
                    "principalPreOpICD": node[36],
                    "procedureLaterality": node[37],
                    "palliation": node[38],
                    "specialEquipment": node[39],
                    "plannedProsthImplant": node[40],
                    "specialSupplies": node[41],
                    "specialInstruments": node[42],
                    "pharmacyItems": node[43]
                });

                returnArray.push(vistaCase.sanitize());
            }
            else {
                console.log("Node is null for data: " + resultArray[i]);
            }
        }
    }
    else {
        console.log("Result Array empty for data: " + result);
    }

    callback(null, returnArray);
}

function parseNonORCaseData(data, callback) {
    let returnArray = [];
    let vistaCase;
    let node;
    let resultArray = data.split("\r");//vistaconfig.LINE_SEPARATOR);

    if (resultArray !== null && resultArray != undefined && resultArray.length > 0) {
        // console.log(resultArray);
        for (let i = 0; i < resultArray.length; i++) {
            node = resultArray[i].split(vistaconfig.NODE_SEPARATOR);
            if (node !== null && node !== undefined && node.length > 0 && node.length !== 1) {
                // Populate model
                vistaCase = new NonOrCase();
                vistaCase.setData({
                    "hospitalAdmissionStatus": node[1],
                    "serviceConnected": node[2],
                    "associatedClinic": node[3],
                    "dateOfOperation": node[4],
                    "resSupCode": node[5],
                    "anesCareTimeBlock": node[6],
                    "generalComments": node[7],
                    "princAnesthetist": node[8],
                    "anesthesiologistSupvr": node[9],
                    "anesSuperviseCode": node[10],
                    "anesthesiaTechnique": node[11],
                    "otherProcedures": node[12],
                    "plannedPostopCare": node[13],
                    "otherPostopDiags": node[14],
                    "caseCreateDate": node[15],
                    "asaClass": node[16],
                    "cancelDate": node[17],
                    "cancelTimeframe": node[18],
                    "primaryCancelReason": node[19],
                    "cancelComments": node[20],
                    "diagnosticTherapeutic": node[21],
                    "principalProcedure": node[22],
                    "plannedPrincOpProcCPT": node[23],
                    "principalDiagnosis": node[24],
                    "specimens": node[25],
                    "operationIndications": node[26],
                    "operativeFindings": node[27],
                    "briefClinHistory": node[28],
                    "plannedPrinDiagCode": node[29],
                    "location": node[30],
                    "procedureDate": node[31],
                    "procedureStartTime": node[32],
                    "procedureEndTime": node[33],
                    "provider": node[34],
                    "attendingProvider": node[35],
                    "medicalSpecialty": node[36],
                    "procedureOccurence": node[37],
                    "procedureLaterality": node[38],
                    "dictatedSummary": node[39]
                });

                returnArray.push(vistaCase.sanitize());
            }
            else {
                console.log("Node is null for data: " + resultArray[i]);
            }
        }
    }
    else {
        console.log("Result Array empty for data: " + result);
    }

    callback(null, returnArray);
}

function buildFullCaseForVista(data, callback) {

    let vistaCase = new FullCase();
    vistaCase.setData(data);
    let dateOfOperation = vistaCase.data.dateOfOperation,
        caseCreateDate = vistaCase.data.caseCreateDate,
        cancelDate = vistaCase.data.cancelDate,
        scheduledEndTime = vistaCase.data.scheduledEndTime,
        scheduledStartTime = vistaCase.data.scheduledStartTime,
        procedureDate = vistaCase.data.procedureDate,
        procedureStartTime = vistaCase.data.procedureStartTime,
        procedureEndTime = vistaCase.data.procedureEndTime;

    // Update date/time fields for VistA
    if (dateOfOperation !== undefined && dateOfOperation) {
        let temp = new Date(dateOfOperation);
        vistaCase.data.dateOfOperation = vista.convertToVistaDate(temp);
    }
    if (caseCreateDate !== undefined && caseCreateDate) {
        vistaCase.data.caseCreateDate = vista.convertToVistaDate(new Date(caseCreateDate));
    }
    if (cancelDate !== undefined && cancelDate) {
        vistaCase.data.cancelDate = vista.convertToVistaDate(new Date(cancelDate));
    }
    if (scheduledEndTime !== undefined && scheduledEndTime) {
        vistaCase.data.scheduledEndTime = vista.convertToVistaDate(new Date(scheduledEndTime));
    }
    if (scheduledStartTime !== undefined && scheduledStartTime) {
        vistaCase.data.scheduledStartTime = vista.convertToVistaDate(new Date(scheduledStartTime));
    }
    if (procedureDate !== undefined && procedureDate) {
        vistaCase.data.procedureDate = vista.convertToVistaDate(new Date(procedureDate));
    }
    if (procedureEndTime !== undefined && procedureEndTime) {
        vistaCase.data.procedureEndTime = vista.convertToVistaDate(new Date(procedureEndTime));
    }
    if (procedureStartTime !== undefined && procedureStartTime) {
        vistaCase.data.procedureStartTime = vista.convertToVistaDate(new Date(procedureStartTime));
    }

    formatForVista(vistaCase.sanitize(), callback);
};

function formatPlaceholderForVista(vistaCase, callback) {
    let vistaConstants = new VistaConstants(),
        vistaString = "",
        temp = "",
        i = 0,
        property,
        prop;

    // TODO remove when code is cleaned up
    const tempArray = ["hospitalAdmissionStatus", "serviceConnected", "orRoom", "caseScheduleType", "surgerySpecialty", "primarySurgeon", "principalProcedure"];
    // console.log(vistaCase);
    if (vistaCase !== undefined) {
        // TODO - Clean up code once we determine if we're using the 'SR PUT PLHCASE' routine or the generic create
        //for (property in vistaCase) {
        for (i = 0; i < tempArray.length; i++) {
            property = tempArray[i];
            vistaString += vistaConstants.schema[property] + CREATE_CASE_FIELD_DELIMETER + vistaCase[property] + ";";
            /*
             // Handle multiples
             if ((property === "anesthesiaTechnique" || property === "otherProcedures"
             || property === "otherPostopDiags" || property === "procedureOccurence"
             || property === "pharmacyItems" || property === "otherPreopDiagnosis"
             || property === "reqBloodKind" || property === "referringPhysician"
             || property === "specialEquipment" || property === "plannedProsthImplant"
             || property === "specialSupplies" || property === "specialInstruments")
             && vistaCase[property] !== undefined
             && Object.prototype.toString.call(vistaCase[property]) == '[object Array]') {
             temp = "";
             for (i = 0; i < vistaCase[property].length; i++) {
             temp += vistaCase[property][i] + ";";
             }
             vistaString += temp.slice(0, -1) + "^";
             }
             else if (property === "surgeryPosition" && vistaCase[property] !== undefined
             && Object.prototype.toString.call(vistaCase[property]) == '[object Array]') {
             for (i = 0; i < vistaCase[property].length; i++) {
             prop = vistaCase[property][i];
             if (prop !== undefined) {
             temp += prop.position + ":" + vista.convertToVistaDate(new Date(prop.date)) + ";";
             }
             }
             vistaString += temp.slice(0, -1);
             }
             else {
             vistaString += vistaConstants[property] + CREATE_CASE_FIELD_DELIMETER + vistaCase[property] + ";";
             }
             */
        }
    }

    callback(null, "\"" + vistaString.slice(0, -1) + "\"");

}

function formatForVista(vistaCase, callback) {
    let vistaString = "",
        temp = "",
        i = 0,
        property,
        prop;
    // console.log(vistaCase);
    if (vistaCase !== undefined) {
        for (property in vistaCase) {
            // console.log(property);
            // console.log(vistaCase[property]);
            // TODO - Handle multiple fields

            // Don't add ien to string for create/edit RPC
            if (property == "ien"){
                continue;
            }

            // Handle multiples
            if ((property === "anesthesiaTechnique" || property === "otherProcedures"
                || property === "otherPostopDiags" || property === "procedureOccurence"
                || property === "pharmacyItems" || property === "otherPreopDiagnosis"
                || property === "reqBloodKind" || property === "referringPhysician"
                || property === "specialEquipment" || property === "plannedProsthImplant"
                || property === "specialSupplies" || property === "specialInstruments"
                || property === "restraintPositionAids")
                && vistaCase[property] !== undefined
                && Object.prototype.toString.call(vistaCase[property]) == '[object Array]') {
                temp = "";
                for (i = 0; i < vistaCase[property].length; i++) {
                    temp += vistaCase[property][i] + ";";
                }
                vistaString += temp.slice(0, -1) + "^";
            }
            else if (property === "surgeryPosition" && vistaCase[property] !== undefined
                && Object.prototype.toString.call(vistaCase[property]) == '[object Array]') {
                for (i = 0; i < vistaCase[property].length; i++) {
                    prop = vistaCase[property][i];
                    if (prop !== undefined) {
                        temp += prop.position + ":" + vista.convertToVistaDate(new Date(prop.date)) + ";";
                    }
                }
                vistaString += temp.slice(0, -1) + "^";
            }
            else {
                vistaString += vistaCase[property] + "^";
            }
        }
    }
    // console.log('BEFORE');
    // console.log(vistaString);
    // console.log();
    // console.log('AFTER');
    // console.log(vistaString.slice(0, -1));
    callback(null, vistaString.slice(0, -1));

}

/********************************
 *          Constants           *
 ********************************/
//Case type
const REQUESTED = 'requested';
const SCHEDULED = 'scheduled';
const PLACEHOLDER = 'placeholder';
//Case status
const COMPLETED = 1;
const NOT_COMPLETED = 2;
const CANCELLED = 3;
const INACTIVE = 4;
const DELETED = 5;

// Cancel Timeframes
const CANCEL_TIMEFRAME_LESS_THAN_48 = 1;
const CANCEL_TIMEFRAME_MORE_THAN_48 = 2;

// Delimeters
const CREATE_CASE_FIELD_DELIMETER = "////";
const CREATE_CASE_MULTIPLE_DELIMETER = "";

// Export Constants
exports.REQUESTED = REQUESTED;
exports.SCHEDULED = SCHEDULED;
exports.PLACEHOLDER = PLACEHOLDER;
exports.COMPLETED = COMPLETED;
exports.NOT_COMPLETED = NOT_COMPLETED
exports.CANCELLED = CANCELLED;
exports.INACTIVE =  INACTIVE;
exports.DELETED = DELETED;
exports.CANCEL_TIMEFRAME_LESS_THAN_48 = CANCEL_TIMEFRAME_LESS_THAN_48;
exports.CANCEL_TIMEFRAME_MORE_THAN_48 = CANCEL_TIMEFRAME_MORE_THAN_48;

// Export Functions
exports.findCaseById = findCaseById;
exports.searchCases = searchCases;
exports.createNewCase = createNewCase;
exports.editExistingCase = editExistingCase;
exports.createPlaceholderCase = createPlaceholderCase;
exports.getImpactedCases = getImpactedCases;


/***************************************************************************************
 *        The following methods may be deleted in the future after code cleanup        *
 ***************************************************************************************/

/**
 * Retrieves list of OR cases for a specified patient and date range
 * @param loginOptions
 * @param patientId
 * @param startDate
 * @param endDate
 * @param callback
 */
exports.getPatientORCasesByDateRange = function (loginOptions, patientId, startDate, endDate, callback) {
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'SR GET ORNONOR CASES',
            patientId,
            vista.convertToVistaDateNoTime(startDate),
            vista.convertToVistaDateNoTime(endDate),
            "OR",
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    console.log("found or cases for patient by date");
                    // console.log(result);
                    parseORCaseData(result, function (err, caseList) {
                        callback(errorHandler.serverIssue(err), caseList);
                    });
                }

            }
        );
    });
}

/**
 * Retrieves list of OR cases for a specified patient and date range
 * @param loginOptions
 * @param patientId
 * @param startDate
 * @param endDate
 * @param callback
 */
exports.getPatientNonORCasesByDateRange = function (loginOptions, patientId, startDate, endDate, callback) {
    auth.buildConfiguration(loginOptions, function (configuration) {
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            'SR GET ORNONOR CASES',
            patientId,
            vista.convertToVistaDateNoTime(startDate),
            vista.convertToVistaDateNoTime(endDate),
            "NON-OR",
            function (error, result) {
                if (error) {
                    console.log("Error: " + error);
                    callback(errorHandler.serverIssue(error), null);
                }
                else {
                    console.log("found nor cases for patient by date");
                    //console.log(result);
                    parseNonORCaseData(result, function (err, caseList) {
                        callback(errorHandler.serverIssue(err), caseList);
                    });
                }

            }
        );
    });
}

/**
 * Retrieves a list of cases by case type for a specific time period
 * @param loginOptions
 * @param beginDate
 * @param endDate
 * @param caseType
 * @param callback
 */
exports.findCasesForDateRange = function (loginOptions, beginDate, endDate, caseType, callback) {
    //TODO - may need to handle different casetypes in different calls depending on RPC response/result
    // console.log("casetype: " + caseType);

    // Routine to call TODO - update 'REQUESTED CASES' and 'PLACEHOLDER CASES' once a routine exists
    //var routine = ((caseType === this.SCHEDULED)? 'SROSLST SCHED SURG':(caseType === this.REQUESTED ? 'REQUESTED CASES': 'PLACEHOLDER CASES'));
    let routine = 'SROSLST SCHED SURG';

    auth.buildConfiguration(loginOptions, function (configuration) {
        console.log(routine);
        vista.callRpc(
            vistaconfig.logger,
            configuration,
            routine,
            [vista.convertToVistaDate(beginDate), vista.convertToVistaDate(endDate)],
            function (error, result) {
                if (result === undefined || result instanceof Error) {
                    console.log('ERROR');
                    callback(result, null);
                } else {
                    /*
                     OR2^20^injury^3170104^24;DOCTOR,ELEVEN^^20:30^22:00
                     OR2^32^crown^3170511^93;ONE,SURGEON^^20:00^21:00
                     */
                    let resultList = result.split("\r\n");
                    let caseList = [];
                    if (resultList instanceof Array) {
                        let rowSplit, surgeonSplit, caseStartDateTime, caseEndDateTime, vistaCase;
                        for (let i = 0, x = resultList.length - 1; i < x; i++) {
                            if (resultList[i] === undefined || resultList[i] === null || resultList[i] === '') {
                                console.log('empty row - split tail entry: ' + (i == x));
                            }
                            else {
                                vistaCase = new OrCase();

                                rowSplit = resultList[i].split('^');
                                if (resultList != undefined) {
                                    surgeonSplit = rowSplit[4].split(';');
                                }

                                caseStartDateTime = vista.convertFromVistaDate(rowSplit[3] + '.' + rowSplit[6].replace(':', ''));
                                caseEndDateTime = vista.convertFromVistaDate(rowSplit[3] + '.' + rowSplit[7].replace(':', ''));

                                vistaCase.setData({
                                    ien: rowSplit[1],
                                    principalProcedure: rowSplit[2],
                                    dateOfOperation: caseStartDateTime,
                                    caseDuration: (caseEndDateTime - caseStartDateTime) / 1000 / 60,
                                    /*primarySurgeon: [{
                                     id: surgeonSplit[0],
                                     name: surgeonSplit[1]
                                     }],*/
                                    primarySurgeon: surgeonSplit[1],
                                    orRoom: rowSplit[0]
                                });
                                caseList[i] = vistaCase.sanitize();
                            }
                        }
                    }
                    callback(errorHandler.serverIssue(error), caseList);
                }
            }
        );
    });
};
