/**
 * SSLQIReservation Service
 * @author atignola
 */
'use strict';
const mongoose = require('mongoose');
const SSLQIReleaseRequest = mongoose.model('SSLQIReleaseRequest');
const sslqiReservationService = require('./sslqiReservation');



/**
 * Create new SSLQI release request
 * @param data
 * @param callback
 */
exports.create = function(data, callback){
    if (data.releaseStatus === undefined){
        data.releaseStatus = REQUESTED;
    }
    const releaseRequest = new SSLQIReleaseRequest(data);
    releaseRequest.save(function(err) {
        if (err) {
            console.log(err);
            callback(err, null);
        }
        else {
            callback(null, releaseRequest);
        }
    });
}

/**
 * Edit existing SSLQI reservation
 * @param data
 * @param callback
 */
exports.edit = function(id, data, callback) {
    if (data !== undefined && data !== null && Object.keys(data).length !== 0) {
        SSLQIReleaseRequest.findOne({_id: id}, function (err, existingDoc) {
            if (err) {
                callback(err, null);
            }
            else if (existingDoc === null || existingDoc === undefined) {
                callback("Entry does not exist.", null);
            }
            else {
                // Update SSLQI release request
                for (var property in data) {
                    if (property !== "_id") {
                        existingDoc[property] = data[property];
                    }
                }
                existingDoc.save(function (err) {
                    if (err) {
                        console.log(err);
                        callback(err, null);
                    }
                    else {
                        console.log("SUCCESS");
                        callback(null, "SUCCESS");
                    }
                })
            }
        });
    }
    else {
        callback("No data provided.", null);
    }
}

/**
 * Updates the status of the existing SSLQI release request
 * @param data
 * @param callback
 */
exports.updateStatus = function(id, status, callback) {
    SSLQIReleaseRequest.findOne({_id: id}, function (err, existingDoc) {
        if (err) {
            callback(err, null);
        }
        else if (existingDoc === null || existingDoc === undefined) {
            callback("Entry does not exist.", null);
        }
        else {
            // Update SSLQI release request
            existingDoc.releaseStatus = status;
            existingDoc.save(function (err) {
                if (err) {
                    callback(err, null);
                }
                else {
                    if (status === APPROVED){
                        let temp = {};
                        temp["caseId"] = existingDoc.caseId;
                        temp["requester"] = existingDoc.requester;
                        temp["startDT"] = existingDoc.startDT;
                        temp["endDT"] = existingDoc.endDT;
                        temp["comments"] = existingDoc.comments;

                        sslqiReservationService.edit(existingDoc.initialReservationId, temp, function(error, updatedReservation){
                            if (error){
                                callback(error, null);
                            }
                            else{
                                callback(null, updatedReservation);
                            }
                        });
                    }
                    else {
                        callback(null, existingDoc);
                    }
                }
            })
        }
    });
}

/**
 * Get SSLQI reservation by id
 * @param callback
 */
exports.getById = function(id, callback){
    SSLQIReleaseRequest.findOne({_id:id}, function(err, docs){
        if(!err){
            callback(null, docs);
        } else {
            callback(err, null);
        }
    });
}

/**
 * Retrieve by search criteria
 * @param criteria
 * @param callback
 */
exports.search = function (criteria, callback) {
    console.log("Entering search ...");
    let caseId = criteria.caseId|| undefined;
    let requester = criteria.requester || undefined;
    let initialReservationId = criteria.initialReservationId || undefined;
    let releaseStatus = criteria.releaseStatus || undefined;
    let query = {};

    // Build query
    if (caseId !== undefined) {
        query["caseId"] = caseId;
    }
    if (requester !== undefined) {
        query["requester"] = requester;
    }
    if (initialReservationId !== undefined) {
        query["initialReservationId"] = initialReservationId;
    }
    if (releaseStatus !== undefined) {
        query["releaseStatus"] = releaseStatus;
    }

    // Execute query
    SSLQIReleaseRequest.find(query, function (err, data) {
        if (!err) {
            callback(null, data);
        } else {
            console.log('error');
            callback(err, null);
        }
    });
};

/**
 Delete SSLQIReleaseRequest by ID
 @param id
 @param callback
 */
exports.deleteReleaseRequest = function(id, callback){
    SSLQIReleaseRequest.findOne({_id:id}, 'role', function(err, data){
        if (data !== null && data !== undefined && data["role"] !== "ALL" ){
            SSLQIReleaseRequest.remove({_id:id},function(err, removed){
                if(!err){callback(null, removed);}
                else {callback(err, null);}
            });
        }
        else{
            callback("Error: No data for the specified ID exists.", null);
        }

    });

}

const REQUESTED = 'REQUESTED';
const APPROVED = 'APPROVED';
const DENIED = 'DENIED';

exports.REQUESTED = REQUESTED;
exports.APPROVED = APPROVED;
exports.DENIED = DENIED;