'use strict';
const auth = require("../authenticate");
const cptService = require("../rpc/cpt");
const mongoose = require('mongoose');
const _ = require('underscore');
const CPTConfig = mongoose.model('CPTConfig');


/****************************
 *      CRUD Methods        *
 ****************************/
/**
 * Creates a CPTConfiguration
 * @param cptConfigData
 * @param callback
 */
exports.create = function(cptConfigData, callback){
    let newCptConfig = new CPTConfig(cptConfigData);
    newCptConfig.durationLastUpdated = new Date();
    newCptConfig.save(function (err) {
        if (err) {
            console.log(err);
            callback(err, null);
        }
        else {
            callback(null, newCptConfig);
        }
    });
}

/**
 * Edits an existing CPTConfiguration
 * @param cptCode
 * @param cptConfigData
 * @param callback
 */
exports.edit = function(cptCode, cptConfigData, callback) {
    CPTConfig.findOne({cptCode: cptCode}, function (err, existingRecord) {
        if (err) {
            console.log(err);
        }
        else if (existingRecord === null || existingRecord === undefined) {
            // Record does not exist, add new entry
            let newCptConfig = new CPTConfig(cptConfigData);
            newCptConfig.durationLastUpdated = new Date();
            newCptConfig.save(function (err) {
                if (err) {
                    console.log(err);
                    callback(err, null);
                }
                else {
                    console.log("Successfully inserted: " + cptCode);
                    callback(null, newCptConfig);
                }
            });
        }
        else {
            // Update existing record
            if (existingRecord.siteDuration !== cptConfigData.siteDuration){
                existingRecord.durationLastUpdated = new Date();
            }
            existingRecord.siteDuration = cptConfigData.siteDuration || "";
            existingRecord.complexityLevel = cptConfigData.complexityLevel || "";
            existingRecord.procedureCategory = cptConfigData.procedureCategory || "";
            existingRecord.siteId = cptConfigData.siteId || "";
            existingRecord.comments = cptConfigData.comments || "";
            existingRecord.save(function (err) {
                if (err) {
                    console.log(err);
                    callback(err, null);
                }
                else {
                    console.log("Successfully updated: " + cptCode);
                    callback(null, existingRecord);
                }
            })
        }
    });
}

/**
 * Removes a CPTConfiguration by CPT code
 * @param cptCode
 * @param callback
 */
exports.remove = function(cptCode, callback) {
    CPTConfig.findOne({cptCode: cptCode}, function (err, existingRecord) {
        if (err) {
            console.log(err);
        }
        else if (existingRecord === null || existingRecord === undefined) {
            // Record does not exist
            callback("Error: CPT configuration does not exist for CPT code: " + cptCode, null);
        }
        else {
            // Remove existing record
            CPTConfig.remove({cptCode: cptCode}, function(err, removed){
                if (err) {
                    console.log(err);
                    callback(err, null);
                }
                else {
                    console.log("Successfully removed: " + cptCode);
                    callback(null, removed);
                }
            });
        }
    });
}

/**
 * Get CPTConfig based on search criteria
 * @param criteria [cptCode, siteDuration, complexityLevel, procedureCategory, siteId]
 * @param callback
 */
exports.search = function (criteria, callback) {
    console.log("Entering search ...");
    let cptCode = criteria.cptCode || undefined;
    let surgeryDuration = criteria.siteDuration || undefined;
    let complexityLevel = criteria.complexityLevel || undefined;
    let procedureCategory = criteria.procedureCategory || undefined;
    let siteId = criteria.siteId || undefined;
    let query = {};

    // Build query
    if (cptCode !== undefined) {
        query["cptCode"] = cptCode;
    }
    if (surgeryDuration !== undefined) {
        query["siteDuration"] = surgeryDuration;
    }
    if (complexityLevel !== undefined) {
        query["complexityLevel"] = complexityLevel;
    }
    if (procedureCategory !== undefined) {
        query["procedureCategory"] = procedureCategory;
    }
    if (siteId !== undefined) {
        query["siteId"] = siteId;
    }
    // Execute query
    CPTConfig.find(query, function (err, data) {
        if (!err) {
            callback(null, data);
        } else {
            console.log('error');
            callback(err, null);
        }
    });
};

/**
 * Retrieves all CPTConfigurations from DB
 * @param callback
 */
exports.getAllCPTConfigs = function(callback){

    CPTConfig.find({}, function (err, docs) {
        if (!err){
            callback(null, docs);
        }
        else{
            callback(err, docs);
        }
    })
}

/**
 * Retrieves CPTConfiguration by CPT code
 * @param cptCode
 * @param callback
 */
exports.getByCPTCode = function(cptCode, callback) {
    CPTConfig.findOne({cptCode: cptCode}, function (err, config) {
        if (err) {
            console.log(err);
        }
        else if (config === null || config === undefined) {
            // Record does not exist
            callback("Error: CPT configuration does not exist for CPT code: " + cptCode, null);
        }
        else {
            callback(null, config);
        }
    });
}


/****************************************
 *      Duration Related Methods        *
 ****************************************/
/**
 * Retrieves the siteDuration for a CPT code
 * @param cptCode
 * @param callback
 */
exports.getDurationByCPT = function(cptCode, callback) {
    CPTConfig.findOne({cptCode: cptCode}, 'siteDuration', function (err, existingRecord) {
        if (err) {
            console.log(err);
        }
        else if (existingRecord === null || existingRecord === undefined) {
            // Record does not exist
            callback("Error: CPT configuration does not exist for CPT code: " + cptCode, null);
        }
        else {
            callback(null, existingRecord);
        }
    });
}

/**
 * Get CPTConfig based on search criteria
 * @param criteria [cptCode, siteDuration, complexityLevel, procedureCategory, siteId]
 * @param callback
 */
exports.getDurations = function (criteria, callback) {
    console.log("Entering search ...");
    let cptCode = criteria.cptCode || undefined;
    let specialty = criteria.specialty || undefined;
    let surgeon = criteria.surgeon || undefined;
    let cptDuration = {};

    CPTConfig.findOne({cptCode: cptCode}, function (err, config) {
        if (err) {
            console.log(err);
            callback(err, null);
        }
        else if (config === null || config === undefined) {
            // Record does not exist
            callback(null, {});
        }
        else {
            cptDuration["nationalDuration"] = config.nationalDuration || 0;
            cptDuration["siteDuration"] = config.siteDuration || 0;
            if (specialty !== undefined && config.specialtyDurations !== undefined){
                for (let i = 0, x = config.specialtyDurations.length; i < x; i++){
                    let temp = config.specialtyDurations[i];
                    if (temp[specialty] !== undefined){
                        cptDuration["specialtyDuration"] = temp[specialty] || 0;
                        break;
                    }
                }
            }
            else{
                cptDuration["specialtyDuration"] = 0;
            }
            if (surgeon !== undefined && config.surgeonDurations !== undefined){
                for (let i = 0, x = config.surgeonDurations.length; i < x; i++){
                    let temp = config.surgeonDurations[i];
                    if (temp[surgeon] !== undefined){
                        cptDuration["surgeonDuration"] = temp[surgeon] || 0;
                        break;
                    }
                }
            }
            else{
                cptDuration["surgeonDuration"] = 0;
            }

            callback(null, cptDuration);
        }
    });
};

/**
 * Get CPTConfig based on search criteria
 * @param criteria [cptCode, siteDuration, complexityLevel, procedureCategory, siteId]
 * @param callback
 */
exports.getDurationsForReport = function (criteria, callback) {
    console.log("Entering search ...");
    let cptCode = criteria.cptCode || undefined;
    let specialty = criteria.specialty || undefined;
    let surgeon = criteria.surgeon || undefined;
    let cptDuration = {};

    CPTConfig.findOne({cptCode: cptCode}, function (err, config) {
        if (err) {
            console.log(err);
            callback(err, null);
        }
        else if (config === null || config === undefined) {
            // Record does not exist
            callback(null, {});
        }
        else {
            cptDuration["nationalDuration"] = config.nationalDuration || 0;
            cptDuration["siteDuration"] = config.siteDuration || 0;
            if (specialty !== undefined && config.specialtyDurations !== undefined){
                for (let i = 0, x = config.specialtyDurations.length; i < x; i++){
                    let temp = config.specialtyDurations[i];
                    if (temp[specialty] !== undefined){
                        cptDuration["specialtyDuration"] = [];
                        cptDuration["specialtyDuration"].push(temp);
                        break;
                    }
                }
            }
            else{
                cptDuration["specialtyDuration"] = config.specialtyDurations;
            }
            if (surgeon !== undefined && config.surgeonDurations !== undefined){
                for (let i = 0, x = config.surgeonDurations.length; i < x; i++){
                    let temp = config.surgeonDurations[i];
                    if (temp[surgeon] !== undefined){
                        cptDuration["surgeonDuration"] = [];
                        cptDuration["surgeonDuration"].push(temp);
                        break;
                    }
                }
            }
            else{
                cptDuration["surgeonDuration"] = config.surgeonDurations;
            }

            callback(null, cptDuration);
        }
    });
};

/**
 * Gets all of the dates that a duration was last updated for CPT Config
 * @param callback
 */
exports.getAllCPTConfigDurationDates = function(callback){
    let foundData = false;
    CPTConfig.find().distinct('durationLastUpdated', function(err, result){
        foundData = true;
        console.log("found something");
        callback(err, result);
    });
}

/**
 * Updates the calculated CPT Duration in Mongo.
 * @param loginOptions
 */
exports.updateCPTDuration = function(durationObject, property){
    let newCptConfig, today = new Date(), average, specialtyAverages, surgeonAverages;
    // Get CPT averages
    average = calculateAverage(durationObject[cptService.ALL_CATEGORY]);
    specialtyAverages = calculateAverages(durationObject[cptService.SPECIALTY_CATEGORY]);
    surgeonAverages = calculateAverages(durationObject[cptService.SURGEON_CATEGORY]);

    // Update Mongo
    CPTConfig.findOne({cptCode: property}, function (err, existingRecord) {
        if (err) {
            console.log(err);
        }
        else if (existingRecord === null || existingRecord === undefined) {
            // Record does not exist, add new entry
            newCptConfig = new CPTConfig();
            newCptConfig.cptCode = property;
            newCptConfig.siteDuration = average;
            newCptConfig.surgeonDurations = surgeonAverages;
            newCptConfig.specialtyDurations = specialtyAverages;
            newCptConfig.durationLastUpdated = today;
            newCptConfig.save(function(err){
                if (err) {
                    console.log(err);
                }
                else {
                    console.log("Successfully inserted: " + property);
                }
            });
        }
        else {
            // Update existing record
            existingRecord.siteDuration = average;
            existingRecord.surgeonDurations = surgeonAverages;
            existingRecord.specialtyDurations = specialtyAverages;
            existingRecord.durationLastUpdated = today;
            existingRecord.save(function (err) {
                if (err) {
                    console.log(err);
                }
                else {
                    console.log("Successfully updated: " + property);
                }
            })
        }

    });
}

/*
Calculates the averages forspecialties/surgeons
 */
function calculateAverages(objects){
    console.log("Entering calculateAverages ...");
    let list = [], temp, tempObj, average, property;
    for (property in objects){
        temp = objects[property];
        average = calculateAverage(temp);
        tempObj = {};
        tempObj[property] = average;
        list.push(tempObj);
    }
    return list;
}

/*
Calculates the average for a list of durations/endDates
 */
function calculateAverage(durations){
    console.log("Entering calculateAverage ...");
    let i, x, sum = 0, average;
    durations = _.sortBy(durations, "endDate").reverse();

    if (durations.length < 10)
    {
        x = durations.length;
        for (i = 0; i < x; i++){
            sum += durations[i].duration;
        }
        average = sum/x;
    }
    else
    {
        let min = -1, max = -1, temp;
        x = 10;
        for (i = 0; i < x; i++){
            temp = durations[i].duration;
            sum += temp
            if (min == -1 || temp < min){
                min = temp;
            }
            if (max == -1 || temp > max){
                max = temp;
            }
        }
        // Update sum to not inlcude outlier values
        sum-=(min+max);
        average = sum/(x-2);
    }
    return average;
}

