'use strict';
require('dotenv').config();
const express = require('express'),
    caseRouter = express.Router();

const caseController = require("../../controller/case");
const roomService = require("../../services/rpc/room");

caseRouter.use("/placeholder", require("./placeholderCase"));
caseRouter.use("/requested", require("./requestedCase"));
caseRouter.use("/scheduled", require("./scheduledCase"));
caseRouter.use("/template", require("../cnf/templateCase"));
caseRouter.use("/extend", require("../cnf/extendedCase"));

/**************************
 *   General Case Routes  *
 **************************/

/**
 GET All Cases
 **/
caseRouter.route("/")
    .get(function(req, res){
        // caseService.findAllCases(req.decoded, function(err, result){
        caseController.searchCases(req.decoded, {}, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

/**
 POST Search for cases
 **/
caseRouter.route("/search")
    .post(function(req, res){
        caseController.searchCases(req.decoded, req.body, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

/**
 * POST Create a new case
 */
caseRouter.route("/")
    .post(function(req, res){
        caseController.createCase(req.decoded, req.body, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

/**
 * Edit an existing case
 */
caseRouter.route("/:id")
    .put(function(req, res){
        caseController.editCase(req.decoded, req.params.id, req.body, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

/************************
 *   Other Case Routes  *
 ************************/
caseRouter.route("/:caseId/concurrentCase/:concurrentCaseId")
    .put(function(req, res){
        caseController.associateConcurrentCase(req.decoded, req.params.caseId, req.params.concurrentCaseId, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

    caseRouter.route("/concurrentCase/:parentId")
    .put(function(req, res){
        caseController.createConcurrentCase(req.decoded, req.params.parentId, req.body, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

/**
 * Convert a case to a requested case
 */
caseRouter.route("/convert/requested")
    .put(function(req, res){
        if (req.body.caseType === "PLACEHOLDER"){
            caseController.convertPlaceholderToRequested(req.decoded, req.body, function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);}
            });
        }
        else if (req.body.caseType === "SCHEDULED"){
            caseController.convertScheduledToRequested(req.decoded, req.body, function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);}
            });
        }
        else{
            res.json("Invalid case type for this method: " + req.body.caseType);
        }
    });

/**
 * Convert a case to a scheduled case
 */
caseRouter.route("/convert/scheduled")
    .put(function(req, res){
        if (req.body.caseType === "PLACEHOLDER"){
            caseController.convertPlaceholderToScheduled(req.decoded, req.body, function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);}
            });
        }
        else if (req.body.caseType === "REQUESTED"){
            caseController.convertRequestedToScheduled(req.decoded, req.body, function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);}
            });
        }
        else{
            res.json("Invalid case type for this method: " + req.body.caseType);
        }
    });

/**
 * POST Cancel Case by Case ID
 */
caseRouter.route("/cancel/:caseId")
    .put (function(req, res){
        let temp = req.body;
        caseController.cancelCase(req.decoded, req.params.caseId, req.body, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });

/**
 * POST Reschedule Case by Case ID
 */
caseRouter.route("/reschedule/:caseId")
    .put (function(req, res){
        caseController.rescheduleCase(req.decoded, req.params.caseId, req.body, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });


/**
 GET Cases by Case ID
 **/
caseRouter.route("/:caseId")
    .get(function(req, res){
        caseController.getCaseById(req.decoded, req.params.caseId, function(err, result){
        // caseController.searchCases(req.decoded, {"ien": req.params.caseId}, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });




/**
GET Cases for Provider by day
**/
caseRouter.route("/provider/:providerid/:date")
    .get(function(req, res){
       caseController.searchCases(req.decoded,
            {
                "date": req.params.date,
                "staffMember": req.params.providerid
                /*
                "primarySurgeon": req.params.providerid,
                "firstAsst": req.params.providerid,
                "secondAsst": req.params.providerid,
                "attendSurgeon": req.params.providerid
                */
            }, 
            function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);
            }
        }); 
    });

/**
GET Cases for Date Range
**/
caseRouter.route("/provider/:providerid/:start_date/:end_date")
    .get(function(req, res){
       caseController.searchCases(req.decoded,
            {
                "dateRange": {"start": req.params.start_date, "end": req.params.end_date },
                "staffMember": req.params.providerid
                /*
                 "primarySurgeon": req.params.providerid,
                 "firstAsst": req.params.providerid,
                 "secondAsst": req.params.providerid,
                 "attendSurgeon": req.params.providerid
                 */
            }, 
            function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);
            }
        }); 
    });


/******************
 *   Room Routes  *
 ******************/

// get cases by room id
caseRouter.route("/room/:roomid/:date")
    .get(function(req, res){
        caseController.searchCases(req.decoded,
            {
                "date": req.params.date,
                "orRoom": req.params.roomid
            }, 
            function(err, result){
                if(err){ res.status(err.statusCode || 500).json(err);}
                else{res.json(result);
            }
        }); 
    });

caseRouter.route("/room/:caseId")
    .get(function(req, res){
        roomService.getRoomDetailsByCaseId(req.decoded, req.params.caseId, function(err, result){
            if(err){ res.status(err.statusCode || 500).json(err);}
            else{res.json(result);}
        });
    });





module.exports = caseRouter;