'use strict';
const caseService = require("../services/rpc/case");
const vista = require("../../lib/rpcvista/vista_surgery_services");
const extendedCaseService = require("../services/cnf/extendedCase");
const Case = require("../model/case");
const FullCase = require("../model/fullCase");
const _ = require("lodash");
const errorHandler = require("../common/errorHandler");

/************************************
 *        Case CRUD Methods         *
 ************************************/
function createCase(loginOptions, data, callback){
    let v = validateCase(data);
    if (!v.valid){
        callback(errorHandler.validationError(v), null);
    }
    else {
        caseService.createNewCase(loginOptions, data, function (err, result) {
            if (err) {
                callback(err, null);
            }
            else if (result === undefined || result === null) {
                callback(errorHandler.serverCreationIssue("Error: Creating case - null value returned."), null);
            }
            else {
                let id = result.ien;
                getCaseById(loginOptions, id, callback);
            }
        })
    }
}

/**
 * Retrieves a case by IEN
 * @param loginOptions
 * @param id
 * @param callback
 */
function getCaseById(loginOptions, id, callback){
    caseService.findCaseById(loginOptions, id, function(err, vistaCase){
       if (err){
           callback(err, null);
       }
       else if (vistaCase === undefined || vistaCase === null){
           callback(errorHandler.notFound("Error: Error while trying to retrieve case by IEN." + id), null);
       }
       else{
           callback(null, vistaCase);
       }
    });
}

/**
 * Retrieves a case for search criteria
 * @param loginOptions
 * @param criteria
 * @param callback
 */
function searchCases(loginOptions, criteria, callback){
    caseService.searchCases(loginOptions, criteria, function(err, resultList){
        if (err){callback(errorHandler.serverIssue(err), null);}
        else {callback(null, resultList);}
    })
};

/**
 * Edits an existing case
 * @param loginOptions
 * @param id
 * @param data
 * @param callback
 */
function editCase(loginOptions, id, data, callback) {
    // Get Case from Vista

    caseService.findCaseById(loginOptions, id, function (err, existingCase) {
        if (err) {
            callback(err, null);
        }
        else if (existingCase === undefined || existingCase === null) {
            callback(errorHandler.notFound("Error: Case does not exist for IEN: " + id), null);
        }
        else {
            // Update field values
            updateCaseData(existingCase, data, function (updatedCase) {
                if (updatedCase instanceof Error){
                    let error = updatedCase;
                    errorHandler.statusCode = errorHandler.statusCode || 500;
                    callback(error, null);
                }
                else {
                    let v = validateCase(updatedCase);
                    if (!v.valid) {
                        callback(errorHandler.validationError(v), null);
                    }
                    else {
                        caseService.editExistingCase(loginOptions, id, updatedCase, function (err, vistaCase) {
                            if (err) {
                                callback(err, null);
                            }
                            else if (vistaCase === undefined || vistaCase === null) {
                                callback(errorHandler.notFound("Error: Error while trying to retrieve case by IEN." + id));
                            }
                            else {
                                getCaseById(loginOptions, id, callback);
                            }
                        });
                    }
                }
            });
        }
    });
}

/**
 * Updates the existing case with new data
 * @param existingCase
 * @param data
 * @param callback
 */
function updateCaseData(existingCase, data, callback){
    let property;
    for (property in existingCase) {

        if (property === "ien"){
            continue;
        }
        // New and Old Value
        let newVal = data[property],
            oldVal = existingCase[property];

        if (newVal === ""){
            console.log("Nothing provided for field: " + property);
        }

        if (newVal === undefined){
            continue;
        }

        // Check deleted values
        if (oldVal !== undefined && oldVal !== null && oldVal !== "" && newVal === ""){
            // Add MUMPS RPC - DELETE CHARACTER
            existingCase[property] = "@";
        }
        else {
            // Update Value
            existingCase[property] = newVal;
        }
    }
    callback(existingCase);
};



/******************************************
 *        Concurrent Case Methods         *
 ******************************************/
/**
 * Creates a new case and adds it to the specified case as a concurrent case
 * @param loginOptions
 * @param parentCaseId
 * @param caseData
 * @param callback
 */
exports.createConcurrentCase = function (loginOptions, parentCaseId, caseData, callback) {

    let v = validateCase(caseData)
    if (!v.valid){
        callback(v, null);
    }
    else {
        // Update concurrent reference
        if (caseData.concurrentCase === undefined || caseData.concurrentCase === "") {
            caseData.concurrentCase = parentCaseId;
        }
        // Create concurrent case
        caseService.createNewCase(loginOptions, caseData, function (createErr, concurrentIEN) {

            if (createErr) {
                console.log("Error: " + createErr);
                callback(createErr, null);
            }
            else if (concurrentIEN === undefined || concurrentIEN === null || concurrentIEN === ""
                || concurrentIEN.ien === undefined || concurrentIEN.ien === null || concurrentIEN === "") {
                callback("Error creating concurrent case: concurrent case was not created, can't update initial case", null);
            }
            else {
                let newIEN = concurrentIEN.ien;
                // Get parent case
                caseService.findCaseById(loginOptions, parentCaseId, function (searchErr, searchResult) {

                    if (searchErr) {
                        console.log("Error creating concurrent case: problem with initial case - " + searchErr);
                        callback(searchErr, null);
                    }
                    else if (searchResult === undefined || searchResult === null || searchResult === "") {
                        callback("Error creating concurrent case: inital case for IEN " + parentCaseId + "does not exist.", null);
                    }
                    else {
                        // Update parent case
                        if (Object.prototype.toString.call(searchResult) == '[object Array]' && searchResult.length !== 0) {
                            // Handle Array
                            if (searchResult.length > 1) {
                                callback("Error creating concurrent case: More than one case with the IEN " + parentCaseId + ". Can not create case.", null);
                            }
                            else {
                                let temp = {};
                                temp["concurrentCase"] = newIEN;
                                temp["caseCreateDate"] = searchResult[0].caseCreateDate;
                                caseService.editExistingCase(loginOptions, parentCaseId, temp, function (editErr, editResult) {
                                    if (editErr) {
                                        callback(editErr, null);
                                    }
                                    else {
                                        callback(null, {"ien": newIEN, "parentIen": editResult.ien});
                                    }
                                });
                            }
                        }
                        else {
                            searchResult.concurrentCase = newIEN;
                            caseService.editExistingCase(loginOptions, parentCaseId, searchResult, function (editErr, editResult) {
                                if (editErr) {
                                    callback(editErr, null);
                                }
                                else {
                                    callback(null, {"ien": newIEN, "parentIen": editResult});
                                }
                            });
                        }
                    }
                });
            }

        });
    }
}

/**
 * Associates two existing cases as concurrent cases
 * @param loginOptions
 * @param parentCaseId
 * @param childCaseId
 * @param callback
 */
exports.associateConcurrentCase = function (loginOptions, parentCaseId, childCaseId, callback) {

    // Get parent case
    getCaseById(loginOptions, parentCaseId, function (searchErr, parentCase) {
        if (searchErr) {
            console.log("Error creating concurrent case: problem with initial case - " + searchErr);
            callback(searchErr, null);
        }
        else {
            // Get child case
            getCaseById(loginOptions, childCaseId, function (searchErr, childCase) {
                if (searchErr) {
                    console.log("Error creating concurrent case: problem with initial case - " + searchErr);
                    callback(searchErr, null);
                }
                else {
                    // Update parent case
                    parentCase.concurrentCase = childCaseId;
                    editCase(loginOptions, parentCaseId, parentCase, function (editErr, editResult) {
                        if (editErr) {
                            callback(editErr, null);
                        }
                        else {
                            childCase.concurrentCase = parentCaseId;
                            editCase(loginOptions, childCaseId, childCase, function(editErr, editResult){
                                if (editErr){
                                    callback(editErr, null);
                                }
                                else {
                                    callback(null, "Update Successful");
                                }
                            });
                        }
                    });
                }
            });
        }
    });
}


/*************************************************
 *        Case Cancel/Reschedule Methods         *
 *************************************************/
/**
 * Cancels an existing case
 * @param loginOptions
 * @param id
 * @param data
 * @param callback
 */
exports.cancelCase = function (loginOptions, id, data, callback) {
    console.log("Entering cancelCase ...");
    // Extract data
    let cancelReason = data["cancelCode"] || "",
        comments = data["cancelComments"] || "",
        cancelDate = new Date(),
        user = loginOptions["accessCode"] || data["cancelledBy"] || "",
        cancellationTimeframe;

    // Get case
    getCaseById(loginOptions, id, function (err, vistaCase) {
        if (err) {
            console.log(err);
            callback(err, null);
        }
        else {
            // Update Case with cancel fields
            let startDateTime = vistaCase.scheduledStartTime;
            if (startDateTime === null || startDateTime === undefined) {
                callback("Error: Cannot cancel a case that does not have scheduled start time.", null);
            }
            else {
                let timeframe = (startDateTime - cancelDate) / 1000 / 60 / 60;
                cancellationTimeframe = (timeframe > 48 ? caseService.CANCEL_TIMEFRAME_MORE_THAN_48 : caseService.CANCEL_TIMEFRAME_LESS_THAN_48);
                // Convert date
                vista.convertToVistaDate(cancelDate);

                // Update Case
                vistaCase.cancelTimeframe = cancellationTimeframe;
                vistaCase.cancelDate = cancelDate;
                vistaCase.cancelComments = comments;
                vistaCase.primaryCancelReason = cancelReason;
                vistaCase.caseStatus = caseService.CANCELLED;

                caseService.editExistingCase(loginOptions, id, vistaCase, function(error, result){
                    if (!error) {
                        console.log(result);
                        callback(null, result);
                    }
                    else {
                        console.log("Error when cancelling case: " + error);
                        callback(error, null);
                    }
                });
            }
        }
    })
};

/**
 * Reschedules a case
 * @param loginOptions
 * @param id
 * @param data
 * @param callback
 */
exports.rescheduleCase = function (loginOptions, id, data, callback) {
    console.log("Entering rescheduleCase ...");
    // Extract data
    console.log(data);
    let currentProcDate = data["procedureDate"];
    let reschDate = data["reschDate"];
    let comment = data["cancelComments"];
    // TODO - Take in time parameter too

    console.log(id);
    console.log(currentProcDate);
    console.log(reschDate);
    console.log(comment);

    // Check for cancel comments
    if (comment === undefined) {
        // Reschedule the case
        // TODO - implement logic
    }
    else {
        // Create new case(requested? scheduled?) with reference to old case data
        // TODO - implement logic

        // Cancel old case and add reference to new case
        // TODO - implement logic

    }

    callback(null, "");

    /****************************************************************
     * TODO - This blocked comment is how we will have to implement
     * the code if we don't go with the checking of the cancelComments
     //Check delta of current date/time and case scheduled date/time
     // If delta < 48 hours cancel case and create new
     // If delta > 48 hours modify case data
     *****************************************************************/
};


/******************************************
 *        Case Conversion Methods         *
 ******************************************/

/**
 * Converts a placheolder case to a requested case
 * @param loginOptions
 * @param caseData
 * @param callback
 */
exports.convertPlaceholderToRequested = function (loginOptions, caseData, callback) {
    console.log("Entering convertPlaceholderToRequested ...");
    // TODO - Implement method
    console.log(caseData);
    callback(null, 'MOCKED METHOD WAITING FOR RPC');
};

/**
 * Converts a placeholder case to scheduled case
 * @param loginOptions
 * @param caseData
 * @param callback
 */
exports.convertPlaceholderToScheduled = function (loginOptions, caseData, callback) {
    console.log("Entering convertPlaceholderToScheduled ...");
    // TODO - Implement method
    console.log(caseData);
    callback(null, 'MOCKED METHOD WAITING FOR RPC');
};

/**
 * Converts a requested case to a scheduled case
 * @param loginOptions
 * @param caseData
 * @param callback
 */
exports.convertRequestedToScheduled = function (loginOptions, caseData, callback) {
    console.log("Entering convertRequestedToScheduled ...");
    // TODO - Implement method
    console.log(caseData);
    callback(null, 'MOCKED METHOD WAITING FOR RPC');
};

/**
 * Converts a scheduled case back to a requested case
 * @param loginOptions
 * @param caseData
 * @param callback
 */
exports.convertScheduledToRequested = function (loginOptions, caseData, callback) {
    console.log("Entering convertScheduledToRequested ...");
    // TODO - Implement method
    console.log(caseData);
    callback(null, 'MOCKED METHOD WAITING FOR RPC');
};


/**************************************
 *        Extend Case Methods         *
 **************************************/

/**
 * Extends case endtime and finds impacted cases
 * @param loginOptions
 * @param extendedCase
 * @param callback
 */
exports.extendCaseEndTime = function (loginOptions, extendedCase, callback) {
    let caseId = extendedCase.vistaId || undefined,
        minutes = extendedCase.estimatedOverTime || undefined;

    // Find Case
    caseService.findCaseById(loginOptions, caseId, function (error, result) {
        if (error) {
            callback(error, null);
        }
        else if (result === undefined || result === null) {
            callback("ERROR: Case does not exist for case ID: " + caseId, null);
        }
        else {
            // Case exists in VistA - check end time
            let endTime = result.scheduledEndTime || result.procedureEndTime || undefined;
            if (endTime === undefined) {
                callback("Error: The current case does not have an end time to extend.", null);
            }
            else {
                // Create Mongo extendedCase object with caseId and extended minutes
                extendedCaseService.createExtendedCase({
                    vistaId: caseId,
                    estimatedOverTime: minutes
                }, function (error, data) {
                    if (error) {
                        callback(error, null);
                    }
                    else {
                        console.log("made it this far");
                        // Get all cases impacted
                        findAllImpactedCases(loginOptions, result, minutes, function (err, res) {
                            if (err) {
                                console.log("didn't find impacted cases");
                            }
                            else {
                                console.log("found some impacted cases");
                                if (res !== undefined && res.length > 0) {
                                    let i, x, impactedCases = [];
                                    for (i = 0, x = res.length; i < x; i++) {
                                        impactedCases.push(res[i].ien);
                                    }

                                    // Update Mongo with affected cases
                                    extendedCaseService.editExtendedCase(caseId, {
                                        vistaId: caseId,
                                        estimatedOverTime: minutes,
                                        affectedCases: impactedCases
                                    }, function (err, updatedExtendedCase) {
                                        if (err) {
                                            console.log(err);
                                        }
                                        else {
                                            console.log("Updated extended case: " + updatedExtendedCase);
                                        }
                                    })
                                }
                            }
                        });
                        // callback(null, {"vistaId": data.vistaId, "estimatedOverTime":data.estimatedOverTime});
                        callback(null, data);
                    }
                });
            }
        }
    });
}

/**
 * Checks VistA for all cases that are impacted by the specified case going over the scheduled time.
 * @param loginOptions
 * @param extendedCase
 * @param minutes
 * @param resultArray
 * @param checkArray
 * @param callback
 */
function findAllImpactedCases(loginOptions, extendedCase, minutes, resultArray, checkArray, callback) {
    if (arguments.length === 4) {
        callback = resultArray;
        resultArray = [];
        checkArray = [];
    }

    // Get immediate impacted cases for caseId
    caseService.getImpactedCases(loginOptions, minutes, extendedCase, function (error, result) {
        if (error) {
            console.log("ERROR: Error occurred when checking for impacted cases. " + error);
            // TODO - figure out if we can handle the error or return
        }
        else {
            let tempCase;
            resultArray = _.unionBy(resultArray, result, 'ien');
            checkArray = checkArray.concat(result);
            if (checkArray.length === 0) {
                callback(null, resultArray);
            }
            else{
                tempCase = checkArray.shift();
                findAllImpactedCases(loginOptions, tempCase, minutes, resultArray, checkArray, callback);
            }
        }
    });

}

/**
 * Calls the extended case service to retrieve object by VistA ID
 * @param vistaId
 * @param callback
 */
exports.findExtendedCaseByVistaId = function(vistaId, callback){
    extendedCaseService.findById(vistaId, function(err, result){
        if (err){
            callback(err, null);
        }
        else if (result === undefined || result == null){
            callback("ERROR: Record does not exist for case id: " + vistaId, null);
        }
        else{
            callback(null, result);
        }
    })
}

/**
 * Calls the extended case service to retrieve all extended cases in Mongo
 * @param callback
 */
exports.getAllExtendedCases = function(callback){
    extendedCaseService.getAllExtendedCases(function(err, result){
        if (err){
            callback(err, null);
        }
        else if (result === undefined || result == null){
            callback("ERROR: No extended cases", null);
        }
        else{
            callback(null, result);
        }
    })
}

/**
 * Calls the extended case service to edit object by VistA ID
 * @param vistaId
 * @param data
 * @param callback
 */
exports.editExtendedCase = function(vistaId, data, callback){
    extendedCaseService.editExtendedCase(vistaId, data, function(err, result){
        if (err){
            callback(err, null);
        }
        else if (result === undefined || result == null){
            callback("ERROR: Somehow record is null or undefined: " + result, null);
        }
        else{
            callback(null, result);
        }
    })
}

/**
 * Calls extended case service to remove object by VistA ID
 * @param vistaId
 * @param callback
 */
exports.removeExtendedCase = function(vistaId, callback){
    extendedCaseService.removeExtendedCase(vistaId, data, function(err, result){
        if (err){
            callback(err, null);
        }
        else if (result === undefined || result == null){
            console.log("ERROR: Record does not exist for id: " + id + ".");
        }
        else{
            callback(null, result);
        }
    });
}

function validateCase(data){
    let fullCase = new FullCase();
    fullCase.setData(data);
    let validation = fullCase.validate();

    if (validation.errors !== undefined && validation.errors.length > 0){
        // Return validation errors
        return {"valid":false, "message": "Validation Error", "errors":validation.errors};
    }
    else {
        return {"valid": true};
    }
}

// Exports
exports.findAllImpactedCases = findAllImpactedCases;

exports.getCaseById = getCaseById;
exports.editCase = editCase;
exports.updateCaseData = updateCaseData;
exports.createCase = createCase;
exports.searchCases = searchCases;