const Marionette           = require("backbone.marionette");
const Radio                = require("backbone.radio");
const FacilitiesCollection = require("../../entities/facilities/collection");
const parseForm            = require("../../utils/parseForm");
const storage              = require("../../utils/storage");
const config               = require("../../env.json");
const authHeaders          = require("../../utils/authHeaders");
const facDetailsCollection = require("../../entities/facilities/details/collection");
const FacDetailsModel      = require("../../entities/facilities/details/model");
				

module.exports = Marionette.StateView.extend({
	template: require("./template.hbs"),

	ui: {
		"form": ".js-login-form"
	},

	triggers: {
		"submit @ui.form": "form:submit"
	},

	defaultState: {
		"loadingFacilities": true,
		"loadingFacilitiesDetails": false
	},

	stateEvents: {
		"change": "render"
	},

	events: {		// Listen for enter key press on login form and submit if pressed.
		"submit form": "submit"
	},


	templateContext() {
		const data = {};

		if (this.getOption("facilities")) {
			data.facilities = this.getOption("facilities").toJSON();
		}

		if ("" !== config.NODE_ACCESS) {
			console.log("Setting autofocus for AccessCode field");
			data.accessCode = config.NODE_ACCESS;
			data.inputFieldType ="xxtype";
			data.autofocus = "autofocus";

		}
		else {
			console.log("Clearing autofocus for AccessCode field");
			data.inputFieldType ="type";
			data.autofocus = "";
		}

		if ("" !== config.NODE_VERIFY) {
			data.verifyCode = config.NODE_VERIFY;
		}

		return data;
	},

	onAttach() {
		this.loadFacilities();
	},

	loadFacilities() {
		const facilities = new FacilitiesCollection();
		this.state.set("loadingFacilities", true);

		return facilities.fetch()
			.then(() => {
				this.facilities = facilities;
				this.state.set("loadingFacilities", false);
			});
	},

	loadFacilitiesDetails(stationID) {
		this.state.set("loadingFacilitiesDetails", true);
		const facDetails = new FacDetailsModel({"stationID": stationID});
		return facDetails.fetch(authHeaders())
			.then(() => {
				facDetails.saveToLocalStorage();
				Radio.request("store", "set", "activeFacility", facDetails);
				this.state.set("loadingFacilitiesDetails", false);
			});
	},

	onFormSubmit() {
		const { facility, accessCode, verifyCode } = parseForm(this.ui.form);
		const selectedFacility = this.facilities.findWhere({ stationID: facility });

		// Ensure everything has been filled out
		if (!accessCode || !verifyCode) {
			return this.state.set({
				"alertType": "danger",
				"alertMessage": "Please enter data into all fields"
			});
		}

		this.state.set({
			"alertType": undefined,
			"alertMessage": undefined
		});

		return Radio.request("auth", "login", {
			"access": accessCode,
			"verify": verifyCode,
			"facilityName": selectedFacility.get("facilityName"),
			stationID: facility
		})
			.then(() => {
				Radio.request("store", "set", "activeFacility", selectedFacility);
				return this.loadFacilitiesDetails(facility);
			})
			.then(() => {
				Radio.request("root", "close:modal");
				Radio.request("global", "navigate", this.getOption("returnUrl"), { trigger: true, replace: true });

			})
			.catch(() => this.state.set({
				"alertType": "danger",
				"alertMessage": "Invalid credentials"
			}));
	}
});
