﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager;

namespace CRS_EFR
{
    public class LabResultsManager : BaseBO
    {
        // TODO!!!
        private static int REGISTRY_ID = 1;

        private int LabResultCount;
        public LabResultsManager()
        {
            _defaultSortField = "CaseId desc";
        }
        private void SetLoadwithForLabResults(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL); //To filter the results for EFR
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs);
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.ANALYSIS_DETAILs);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadwithForLabResultsCreateNewCopy(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs);
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<SPECIMEN_ANALYSI>(e => e.ANALYSIS_DETAILs);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForFragments(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.WKF_CASE_ACTIVITY);
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYSISMETHOD);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYTETYPE);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);

            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            db.LoadOptions = lo;

        }
        /// <summary>
        /// Lab results with status - New are available for Bio and Fragments.
        /// </summary>
        /// <returns></returns>
        public IEnumerable<LabResult> GetNewLabResults(string searchValue, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort)) sort = _defaultSortField;
            using (_db = GetDataContext())
            {
                SetLoadwithForLabResults(_db);

                var bios = (
                        from t in _db.WKF_CASE_ACTIVITies.Where(act => act.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                            && act.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                            && act.ARCHIVED_FLAG == false
                            && act.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.New)).PatientLike("LAST_NAME", searchValue)
                        join sa in _db.SPECIMEN_ANALYSIs on t.WKF_CASE_ACTIVITY_ID equals sa.WKF_CASE_ACTIVITY_ID
                        select new LabResult
                        {
                            CaseId = t.WKF_CASE_ID,
                            ActivityId = t.WKF_CASE_ACTIVITY_ID,
                            AcceptedDate = sa.RESULT_ACKNOWLEDGE_DATE,
                            ReceivedDate = sa.CREATED,
                            //ReceivedDate = sa.CREATED != null ? String.Format("{0:d}", sa.CREATED) : String.Empty,
                            PatientName = t.WKF_CASE.PATIENT.FullName,
                            CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                            LabContact = t.CONTACT_NAME
                            ,
                            Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER
                            ,
                            SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME
                        }).ToList();

                var frags = (from t in _db.WKF_CASE_ACTIVITies.Where(act => act.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                                     && act.ARCHIVED_FLAG == false
                                     && act.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                                     && act.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.New)
                                     && !(bios.Select(t => t.ActivityId).ToArray().Contains(act.WKF_CASE_ACTIVITY_ID))).PatientLike("LAST_NAME", searchValue)
                             select new LabResult
                             {
                                 CaseId = t.WKF_CASE_ID,
                                 ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                 PatientName = t.WKF_CASE.PATIENT.FullName,
                                 CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                 LabContact = t.CONTACT_NAME,
                                 Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                 SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME

                             }).ToList();

                IEnumerable<LabResult> LabResults = frags.Union(bios).ToList().OrderBy(sort);
                LabResultCount = LabResults.Count();
                return LabResults.Skip(startRow).Take(maxRows);
            }
        }

        public int GetNewLabResultsCount(string searchValue)
        {
            return LabResultCount;
        }
        /// <summary>
        /// Lab results with status -In process are available only for fragment analysis
        /// V.S. - this is an old assumption, and in-process lab results now show both bio and frag analysis 
        /// </summary>
        /// <returns></returns>
        public IEnumerable<LabResult> GetInProcessLabResults(string searchValue, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort)) sort = _defaultSortField;
            using (_db = GetDataContext())
            {
                SetLoadwithForLabResults(_db);
                //IEnumerable<LabResult> LabResults = 

                var frags = (from t in _db.WKF_CASE_ACTIVITies.PatientLike("LAST_NAME", searchValue)
                             where t.FRAGMENT_ANALYSIs.Count > 0
                                && t.ARCHIVED_FLAG == false
                             where t.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                                     && t.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                                     && t.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.InProcess)

                             let o = t.WKF_CASE.WKF_CASE_ACTIVITies.Where(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabOrder)).First()
                             select new LabResult
                             {
                                 CaseId = t.WKF_CASE_ID,
                                 ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                 LabReportDate = o.LAB_ORDERs.First().LAB_REPORT_DATE,
                                 AcceptedDate = o.LAB_ORDERs.First().RESULT_ACKNOWLEDGE_DATE,
                                 ReceivedDate = t.FRAGMENT_ANALYSIs.Max(fs => fs.CREATED),
                                 PatientName = t.WKF_CASE.PATIENT.FullName,
                                 CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                 LabContact = t.CONTACT_NAME,
                                 Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                 SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME

                             }).ToList().OrderBy(sort);

                var bios = (from t in _db.WKF_CASE_ACTIVITies.PatientLike("LAST_NAME", searchValue)
                            join sa in _db.SPECIMEN_ANALYSIs on t.WKF_CASE_ACTIVITY_ID equals sa.WKF_CASE_ACTIVITY_ID
                            where t.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                                    && t.ARCHIVED_FLAG == false
                                    && t.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                                   && t.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.InProcess)
                            select new LabResult
                            {
                                CaseId = t.WKF_CASE_ID,
                                ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                LabReportDate = sa.LAB_REPORT_DATE,
                                ReceivedDate = sa.CREATED,
                                PatientName = t.WKF_CASE.PATIENT.FullName,
                                CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                LabContact = t.CONTACT_NAME,
                                Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME
                            }).ToList().OrderBy(sort);

                IEnumerable<LabResult> LabResults = frags.Union(bios).ToList().OrderBy(sort);
                LabResultCount = LabResults.Count();
                return LabResults.Skip(startRow).Take(maxRows);
            }
        }
        /// <summary>
        /// Lab results are received electronically. New entries are created when the results are received. 
        /// </summary>
        /// <returns></returns>
        public IEnumerable<LabResult> GetCompletedLabResults(string searchValue, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort)) sort = _defaultSortField;
            using (_db = GetDataContext())
            {
                SetLoadwithForLabResults(_db);

                var frags = (from t in _db.WKF_CASE_ACTIVITies.Where(act => act.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                                     && act.ARCHIVED_FLAG == false
                                     && act.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                                     && act.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted)).PatientLike("LAST_NAME", searchValue)
                             where t.FRAGMENT_ANALYSIs.Count > 0
                             let o = t.WKF_CASE.WKF_CASE_ACTIVITies.Where(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabOrder)).First()
                             select new LabResult
                             {
                                 CaseId = t.WKF_CASE_ID,
                                 ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                 LabReportDate = o.LAB_ORDERs.First().LAB_REPORT_DATE,
                                 AcceptedDate = o.LAB_ORDERs.First().RESULT_ACKNOWLEDGE_DATE,
                                 ReceivedDate = t.FRAGMENT_ANALYSIs.Max(fs => fs.CREATED),
                                 PatientName = t.WKF_CASE.PATIENT.FullName,
                                 CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                 LabContact = t.CONTACT_NAME,
                                 Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                 SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME

                             }).ToList();

                var bios = (
                        from t in _db.WKF_CASE_ACTIVITies.Where(act => act.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                            && act.ARCHIVED_FLAG == false
                            && act.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                            && act.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted)).PatientLike("LAST_NAME", searchValue)
                        join sa in _db.SPECIMEN_ANALYSIs on t.WKF_CASE_ACTIVITY_ID equals sa.WKF_CASE_ACTIVITY_ID
                        select new LabResult
                        {
                            CaseId = t.WKF_CASE_ID,
                            ActivityId = t.WKF_CASE_ACTIVITY_ID,
                            AcceptedDate = sa.RESULT_ACKNOWLEDGE_DATE,
                            ReceivedDate = sa.CREATED,
                            //ReceivedDate = sa.CREATED != null ? String.Format("{0:d}", sa.CREATED) : String.Empty,
                            LabReportDate = sa.LAB_REPORT_DATE,
                            PatientName = t.WKF_CASE.PATIENT.FullName,
                            CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                            LabContact = t.CONTACT_NAME,
                            Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                            SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME

                        }).ToList();

                //return frags.Union(bios).OrderByDescending(d => d.CaseId).ToList();
                IEnumerable<LabResult> LabResults = frags.Union(bios).ToList().OrderBy(sort);
                LabResultCount = LabResults.Count();
                return LabResults.Skip(startRow).Take(maxRows);
            }
        }

        public IEnumerable<LabResult> GetVoidedLabResults(string searchValue, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort)) sort = _defaultSortField;
            using (_db = GetDataContext())
            {
                SetLoadwithForLabResults(_db);

                var frags = (from t in _db.WKF_CASE_ACTIVITies.PatientLike("LAST_NAME", searchValue)
                             where t.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                                    && t.ARCHIVED_FLAG == false
                                          && t.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                                          && t.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Voided)
                             where t.FRAGMENT_ANALYSIs.Count > 0
                             let o = t.WKF_CASE.WKF_CASE_ACTIVITies.Where(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabOrder)).First()
                             select new LabResult
                             {
                                 CaseId = t.WKF_CASE_ID,
                                 ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                 LabReportDate = o.LAB_ORDERs.First().LAB_REPORT_DATE,
                                 AcceptedDate = o.LAB_ORDERs.First().RESULT_ACKNOWLEDGE_DATE,
                                 ReceivedDate = t.FRAGMENT_ANALYSIs.Max(fs => fs.CREATED),
                                 PatientName = t.WKF_CASE.PATIENT.FullName,
                                 CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                 LabContact = t.CONTACT_NAME,
                                 Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                 SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME

                             }).ToList();

                var bios = (from t in _db.WKF_CASE_ACTIVITies.PatientLike("LAST_NAME", searchValue)
                            join sa in _db.SPECIMEN_ANALYSIs on t.WKF_CASE_ACTIVITY_ID equals sa.WKF_CASE_ACTIVITY_ID
                            where t.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                               && t.ARCHIVED_FLAG == false
                               && t.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                               && t.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Voided)

                            select new LabResult
                            {
                                CaseId = t.WKF_CASE_ID,
                                ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                LabReportDate = sa.LAB_REPORT_DATE,
                                ReceivedDate = sa.CREATED,
                                PatientName = t.WKF_CASE.PATIENT.FullName,
                                CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                LabContact = t.CONTACT_NAME,
                                Site = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                SiteName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME

                            }).ToList();

                //JH 8/11/2010 : addding frag results to voided queue because bug 611 adds voided functionality to fragment lab results 
                IEnumerable<LabResult> LabResults = frags.Union(bios).OrderBy(sort);
                LabResultCount = LabResults.Count();
                return LabResults.Skip(startRow).Take(maxRows);
            }
        }


        private IQueryable<FRAGMENT_ANALYSI> LinqAll()
        {
            //populate LinqAll
            IQueryable<FRAGMENT_ANALYSI> tempLinq = (from e in _db.FRAGMENT_ANALYSIs
                                                     select e);

            //TODO - add all business filtering rules 

            return tempLinq;

        }

        private IQueryable<FRAGMENT_ANALYSI> SelectByActivityIdLinqFilter(int id)
        {
            var linqFilter = LinqAll();

            linqFilter = from t in linqFilter
                         where t.WKF_CASE_ACTIVITY_ID == id
                         select t;

            return linqFilter;

        }

        public IEnumerable<Fragment> SelectFragmentsByActivityId(int id, string sort, int maxRows, int startRow)
        {

            if (string.IsNullOrEmpty(sort))
            {
                sort = "FRAGMENT_ANALYSIS_ID desc";
            }
            using (_db = GetDataContext())
            {

                SetLoadwithForLabResults(_db);

                var entities = SelectByActivityIdLinqFilter(id).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities.Select(e => new Fragment
                {
                    FragmentId = e.FRAGMENT_ANALYSIS_ID,
                    FragmentDescription = e.FRAGMENT_DESCR,
                    FragmentEnteredId = e.FRAGMENT_ID_NUMBER

                });

            }
        }

        //gets specific fragment by frag id
        public FRAGMENT_ANALYSI getFragByFragmentID(int id)
        {
            FRAGMENT_ANALYSI entity;
            using (_db = GetDataContext())
            {
                //SetLoadwithForLabResults(_db);
                SetLoadWithForFragments(_db);
                entity = (from fa in _db.FRAGMENT_ANALYSIs where fa.FRAGMENT_ANALYSIS_ID == id select fa).FirstOrDefault();
            }

            return entity;
        }

        /// <summary>
        /// Create a new copy of lab results and insert to the database
        /// This is for archive the current copy of the lab results.
        /// </summary>
        /// <param name="activityId"></param>
        /// <returns></returns>
        public int CreateNewCopyOfCurrentLabResults(int activityId)
        {
            int returnStatus = 0;

            WKF_CASE_ACTIVITY labResults = null;
            using (_db = GetDataContext())
            {
                SetLoadwithForLabResultsCreateNewCopy(_db);
                labResults = (from c in _db.WKF_CASE_ACTIVITies
                              where c.WKF_CASE_ACTIVITY_ID == activityId
                              select c).FirstOrDefault();
                labResults.SetAsChangeTrackingRoot();
            }

            WKF_CASE_ACTIVITY newLab = labResults.Clone();
            newLab.SetAsChangeTrackingRoot();
            newLab.SetAsInsertOnSubmit();
            newLab.STD_WKFACTIVITYSTS_ID = (int)WorkflowCaseActivityStatus.InProcess;

            foreach (SPECIMEN_ANALYSI specimen in newLab.SPECIMEN_ANALYSIs)
            {
                foreach (ANALYSIS_DETAIL detail in specimen.ANALYSIS_DETAILs)
                {
                    detail.SetAsInsertOnSubmit();
                }
                specimen.SetAsInsertOnSubmit();
            }

            foreach (FRAGMENT_ANALYSI fragment in newLab.FRAGMENT_ANALYSIs)
            {
                foreach (ANALYSIS_DETAIL detail in fragment.ANALYSIS_DETAILs)
                {
                    detail.SetAsInsertOnSubmit();
                }
                fragment.SetAsInsertOnSubmit();
            }

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                newLab.SynchroniseWithDataContext(_db, true);

                returnStatus = -1;

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = 0;

                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = 0;
                }
            }

            return returnStatus;
        }

        public int SelectFragmentsByActivityIdCount(int id)
        {
            using (_db = GetDataContext())
            {
                return SelectByActivityIdLinqFilter(id).Count();
            }
        }

        private void SetLoadwithForLabResultDetails(RegistriesDataAccess db, int activityId)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL); //To filter the results for EFR
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<PATIENT>(e => e.STD_GENDER);
            lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
            lo.AssociateWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies.Where(a => a.WKF_CASE_ACTIVITY_ID == activityId));
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs);
            lo.AssociateWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs.Where(s => s.WKF_CASE_ACTIVITY_ID == activityId));
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<SPECIMEN_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYSISMETHOD);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_UNIT);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_UNIT1);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_UNIT2);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }
        /// <summary>
        /// Gets lab result details for a selected activity Id
        /// </summary>
        /// <param name="activityId"></param>
        /// <returns></returns>
        public WKF_CASE GetLabResultDetails(int activityId)
        {
            WorkflowManager Wfm = new WorkflowManager();
            int WfType = Wfm.GetWorkflowType(activityId);

            //Use WKF_CASE to initiate next activity
            using (_db = GetDataContext())
            {
                if (WfType == Convert.ToInt16(WorkFlowTypes.Biomonitoring))
                    SetLoadwithForLabResultDetails(_db, activityId);
                if (WfType == Convert.ToInt16(WorkFlowTypes.FragmentAnalysis))
                    SetLoadwithForFragLabResultDetails(_db, activityId);

                return (from c in _db.WKF_CASEs where c.WKF_CASE_ACTIVITies.Any(t => t.WKF_CASE_ACTIVITY_ID == activityId) select c).FirstOrDefault();
            }
        }

        private void SetLoadwithForFragLabResultDetails(RegistriesDataAccess db, int activityId)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL); //To filter the results for EFR
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<PATIENT>(e => e.STD_GENDER);
            lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
            lo.AssociateWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies.Where(a => a.WKF_CASE_ACTIVITY_ID == activityId));
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.LAB_ORDERs);
            lo.AssociateWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs.Where(s => s.WKF_CASE_ACTIVITY_ID == activityId));
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYSISMETHOD);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYTETYPE);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        public void UpdateAnalyte(ANALYSIS_DETAIL detail)
        {
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                detail.SynchroniseWithDataContext(_db, true);
                try
                {
                    ChangeSet changeSet = _db.GetChangeSet();


                    _db.SubmitChanges();

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = detail.ANALYSIS_DETAIL_ID;

                    RaiseSaveEvent(this, eventArgs);
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }
            }
        }

        public void UpdateResultActivity(WKF_CASE wCase)
        {
            WorkflowManager wfManager = new WorkflowManager();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                wCase.SynchroniseWithDataContext(_db, true);
                try
                {
                    ChangeSet changeSet = _db.GetChangeSet();

                    foreach (WKF_CASE_ACTIVITY activity in
                    changeSet.Updates.OfType<WKF_CASE_ACTIVITY>()
                    .Where(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)))
                    {
                        if (wfManager.AnyArchivedRecords(wCase.WKF_CASE_ID))
                        {
                            //ModifiedMemberInfo[] mmi = this.WKF_CASE_ACTIVITies.GetModifiedMembers(activity);
                            ProcessLabResultUpdate(activity, _db);
                        }
                    }

                    foreach (SPECIMEN_ANALYSI analyte in changeSet.Updates.OfType<SPECIMEN_ANALYSI>()
                        .Where(a => a.WKF_CASE_ACTIVITY.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)))
                    {
                        if (wfManager.AnyArchivedRecords(wCase.WKF_CASE_ID))
                        {
                            ProcessSpecimenAnalysisUpdate(analyte, _db, wCase.WKF_CASE_ID);

                        }
                    }

                    foreach (ANALYSIS_DETAIL detail in changeSet.Updates.OfType<ANALYSIS_DETAIL>().
                        Where(a => a.SPECIMEN_ANALYSI.WKF_CASE_ACTIVITY.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)))
                    {
                        if (wfManager.AnyArchivedRecords(wCase.WKF_CASE_ID))
                        {
                            ProcessAnalysisDetailUpdate(detail, _db, wCase.WKF_CASE_ID);

                        }
                    }
                    foreach (ANALYSIS_DETAIL detail in changeSet.Inserts.OfType<ANALYSIS_DETAIL>()
                        .Where(a => a.SPECIMEN_ANALYSI.WKF_CASE_ACTIVITY.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)))
                    {
                        if (wfManager.AnyArchivedRecords(wCase.WKF_CASE_ID))
                        {

                            ProcessAnalysisDetailInsert(detail, _db, wCase.WKF_CASE_ID);

                        }
                    }

                    foreach (ANALYSIS_DETAIL detail in changeSet.Deletes.OfType<ANALYSIS_DETAIL>()
                        .Where(a => a.SPECIMEN_ANALYSI.WKF_CASE_ACTIVITY.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)))
                    {
                        if (wfManager.AnyArchivedRecords(wCase.WKF_CASE_ID))
                        {

                            ProcessAnalysisDetailDelete(detail, _db, wCase.WKF_CASE_ID);
                        }
                    }

                    _db.SubmitChanges();

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = wCase.WKF_CASE_ID;

                    RaiseSaveEvent(this, eventArgs);
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }
            }
        }

        private void ProcessLabResultUpdate(WKF_CASE_ACTIVITY activity, RegistriesLinqDataContext context)
        {
            ModifiedMemberInfo[] mmi = context.WKF_CASE_ACTIVITies.GetModifiedMembers(activity);

            foreach (ModifiedMemberInfo info in mmi)
            {

                switch (info.Member.Name)
                {
                    case "CONTACT_NAME":

                        string field = "Lab Contact";
                        CreateHistoryLogEntry(field, info, context, activity.WKF_CASE_ID, "UpdateField");
                        break;
                }
            }
        }

        private void ProcessSpecimenAnalysisUpdate(SPECIMEN_ANALYSI analyte, RegistriesLinqDataContext context, int caseId)
        {
            ModifiedMemberInfo[] mmi = context.SPECIMEN_ANALYSIs.GetModifiedMembers(analyte);
            string field = String.Empty;

            foreach (ModifiedMemberInfo info in mmi)
            {

                switch (info.Member.Name)
                {
                    case "LAB_UID_NUMBER":

                        field = "Accession [UID]";

                        break;

                    case "LAB_REPORT_DATE":

                        field = "Lab Report Date";

                        break;

                    case "SPECIMEN_COLLECTION_DATE":

                        field = "Specimen Collection Date";

                        break;

                    case "TIME_ELAPSED_NUMBER":

                        field = "Elapsed Time";

                        break;

                    case "TOTAL_VOLUME_NUMBER":

                        field = "Total Volume";

                        break;

                    case "ANALYTE_VOLUME_NUMBER":

                        field = "Volume of Analyte Test";

                        break;
                }
                CreateHistoryLogEntry(field, info, context, caseId, "UpdateField");
            }
        }

        private void ProcessAnalysisDetailDelete(ANALYSIS_DETAIL details, RegistriesLinqDataContext context, int caseId)
        {

            string field = "Analyte";
            ModifiedMemberInfo info = new ModifiedMemberInfo();

            CreateHistoryLogEntry(field, context, details, caseId, "deleteAnalyte", info);

        }
        private void ProcessAnalysisDetailInsert(ANALYSIS_DETAIL details, RegistriesLinqDataContext context, int caseId)
        {
            string field = "Analyte";
            CreateHistoryLogEntry(field, context, details, caseId, "insertAnalyte", new ModifiedMemberInfo());
        }

        private void ProcessAnalysisDetailUpdate(ANALYSIS_DETAIL detail, RegistriesLinqDataContext context, int caseId)
        {
            ModifiedMemberInfo[] mmi = context.ANALYSIS_DETAILs.GetModifiedMembers(detail);
            string field = String.Empty;
            string oldValue, newValue;
            int originalValue = 0;

            UnitTestReferenceRangeManager manager = new UnitTestReferenceRangeManager();

            if (mmi.Where(s => s.Member.Name == "TEST_NAME" || s.Member.Name == "ANALYTE_NAME").Count() > 0)
            {

                var testNames = new Dictionary<string, string>()
                {
                    { "old", mmi.Where(s => s.Member.Name == "TEST_NAME").Count() > 0 ?
                        mmi.Where(s => s.Member.Name == "TEST_NAME").FirstOrDefault().OriginalValue.ToString() : detail.TEST_NAME},
                    {"new", mmi.Where(s => s.Member.Name == "TEST_NAME").Count() > 0 ?
                        mmi.Where(s => s.Member.Name == "TEST_NAME").FirstOrDefault().CurrentValue.ToString() : detail.TEST_NAME}

                };

                var analyteNames = new Dictionary<string, string>()
                {
                    { "old", mmi.Where(s => s.Member.Name == "ANALYTE_NAME").Count() > 0 ?
                        mmi.Where(s => s.Member.Name == "ANALYTE_NAME").FirstOrDefault().OriginalValue.ToString() : detail.TEST_NAME},
                    {"new", mmi.Where(s => s.Member.Name == "ANALYTE_NAME").Count() > 0 ?
                        mmi.Where(s => s.Member.Name == "ANALYTE_NAME").FirstOrDefault().CurrentValue.ToString() : detail.TEST_NAME}

                };

                CreateHistoryLogEntryForAnalyteChange(field, context, caseId, testNames, analyteNames);
            }
            foreach (ModifiedMemberInfo info in mmi)
            {
                switch (info.Member.Name)
                {
                    case "RESULT_TEXT":
                        field = "Result";
                        CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);

                        break;

                    case "ANALYSIS_METHOD_NAME":
                        field = "Analysis Method";
                        CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);

                        break;

                    case "RESULT_UNIT_ID":
                        field = "Unit";

                        if (info.OriginalValue == null)
                        {
                            originalValue = getDefaultStdUnitId();
                        }
                        else
                        {
                            originalValue = (int)info.OriginalValue;
                        }

                        oldValue = manager.GetTestUnitNameById(originalValue);
                        newValue = manager.GetTestUnitNameById((int)info.CurrentValue);

                        CreateHistoryLogEntryForDropdownData(field, context, detail, caseId, "updateAnalyteField", oldValue, newValue);

                        break;

                    case "LAB_REFERENCE_RANGE_UNIT_ID":
                        field = "Reference Range Unit";

                        if (info.OriginalValue == null)
                        {
                            originalValue = getDefaultStdUnitId();
                        }
                        else
                        {
                            originalValue = (int)info.OriginalValue;
                        }

                        oldValue = manager.GetTestUnitNameById(originalValue);
                        newValue = manager.GetTestUnitNameById((int)info.CurrentValue);

                        CreateHistoryLogEntryForDropdownData(field, context, detail, caseId, "updateAnalyteField", oldValue, newValue);

                        break;

                    case "LAB_REFERENCE_RANGE":
                        field = "Lab Reference Range";
                        CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);

                        break;

                    case "LAB_ABOVE_REFERENCE_RANGE":
                        field = "Above Reference Range";
                        CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);

                        break;

                    case "EVAL_COMMENT_TEXT":
                        field = "Evaluation Note";
                        CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);

                        break;

                    case "OTHER_COMMENT_TEXT":
                        field = "Comment";
                        CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);

                        break;

                }
            }
        }

        private int getDefaultStdUnitId()
        {
            int retValue = 4; // Shot in the dark if we don't find it for some reason ... this should be the right response 

            var stdUnitId = SqlProvider.ExecuteScalar(
                        "RegistryConnectionString", "dbo.GetEFRDefaultStdUnitId", new object[] { });

            if (stdUnitId != DBNull.Value && stdUnitId != null)
            {
                retValue = Convert.ToInt32(stdUnitId);
            }

            return retValue;
        }

        #region create logs


        private void CreateHistoryLogEntryForAnalyteChange(string field, RegistriesLinqDataContext context, int id, Dictionary<string, string> tests, Dictionary<string, string> analytes)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            tracking.CHANGE_FIELD = "Analyte";

            tracking.CHANGE_DESCRIPTION = @"Analyte """ + tests["old"] + @"/" + analytes["old"]
                + @""" changed to """ + tests["new"] + @"/" + analytes["new"] + @""".";

            tracking.SetAsInsertOnSubmit();
            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);

        }

        private void CreateHistoryLogEntryForDropdownData(string field, RegistriesLinqDataContext context, ANALYSIS_DETAIL detail, int id, string action, string oldValue,
            string newValue)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            switch (action)
            {
                case "updateAnalyteField":
                    tracking.CHANGE_FIELD = "Analyte";
                    tracking.CHANGE_DESCRIPTION = @"Analyte """ + detail.TEST_NAME + @"/" + detail.ANALYTE_NAME + @""" " + field
                        + @" changed from """ + oldValue + @""" to """ + newValue + @""".";
                    break;

                case "changeAnalyteType":

                    break;

            }

            tracking.SetAsInsertOnSubmit();
            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);


        }
        private void CreateHistoryLogEntry(string field, RegistriesLinqDataContext context, ANALYSIS_DETAIL detail, int id, string action, ModifiedMemberInfo info)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            switch (action)
            {
                case "insertAnalyte":

                    tracking.CHANGE_FIELD = field;
                    //tracking.CHANGE_DESCRIPTION = "Analyte " + "'" + detail.TestName + "/";//+ detail.STD_ANALYTETYPE.NAME + "' added."; 
                    tracking.CHANGE_DESCRIPTION = @"Analyte """ + detail.TEST_NAME + @"/" + detail.ANALYTE_NAME + @""" added.";
                    break;

                case "updateAnalyteField":

                    tracking.CHANGE_FIELD = "Analyte";
                    tracking.CHANGE_DESCRIPTION = @"Analyte """ + detail.TEST_NAME + @"/" + detail.ANALYTE_NAME + @""" " + field
                        + @" changed from """ + info.OriginalValue + @""" to """ + info.CurrentValue + @""".";
                    break;

                case "deleteAnalyte":

                    tracking.CHANGE_FIELD = field;
                    tracking.CHANGE_DESCRIPTION = @"Analyte """ + detail.TEST_NAME + @"/" + detail.ANALYTE_NAME + @""" was deleted.";
                    break;


            }

            tracking.SetAsInsertOnSubmit();
            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);

        }

        private void CreateHistoryLogEntry(string field, ModifiedMemberInfo info, RegistriesLinqDataContext context, int id, string action)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            string originalValue = null, newValue = null;

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            var typeName = info.OriginalValue == null ? info.CurrentValue.GetType().Name : info.OriginalValue.GetType().Name;

            switch (typeName)
            {
                case "DateTime":

                    originalValue = (info.OriginalValue as DateTime? == null ? String.Empty : (info.OriginalValue as DateTime?).Value.ToShortDateString());
                    newValue = (info.CurrentValue as DateTime? == null ? String.Empty : (info.CurrentValue as DateTime?).Value.ToShortDateString());
                    break;

                case "String":

                    originalValue = info.OriginalValue.ToString();
                    newValue = info.CurrentValue.ToString();
                    break;

            }

            switch (action)
            {

                case "UpdateField":

                    tracking.CHANGE_DESCRIPTION = @"""" + originalValue + @""" was changed to """ + newValue + @""".";
                    tracking.CHANGE_FIELD = field;
                    break;

                case "UpdateAnalyteField":

                    tracking.CHANGE_FIELD = field;
                    tracking.CHANGE_DESCRIPTION = @"""" + originalValue + @""" was changed to """ + newValue + @""".";

                    break;

            }
            //tracking.CHANGE_DESCRIPTION = "'" + originalValue + "'" + " was changed to " + "'" +  newValue + "'";
            tracking.SetAsInsertOnSubmit();

            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);
        }


        #endregion

        public void UpdateFragment(FRAGMENT_ANALYSI fragment)
        {
            WorkflowManager wfManager = new WorkflowManager();
            WorkflowCaseManager caseManager = new WorkflowCaseManager();
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                fragment.SynchroniseWithDataContext(_db, true);
                try
                {
                    ChangeSet changeSet = _db.GetChangeSet();

                    foreach (FRAGMENT_ANALYSI frag in changeSet.Updates.OfType<FRAGMENT_ANALYSI>())
                    {
                        var activity = caseManager.getActivityByID(frag.WKF_CASE_ACTIVITY_ID);

                        if (wfManager.AnyArchivedRecords(activity.WKF_CASE_ID))
                        {
                            ProcessFragmentAnalysisUpdate(frag, _db, activity.WKF_CASE_ID);
                        }
                    }

                    foreach (FRAGMENT_ANALYSI frag in changeSet.Inserts.OfType<FRAGMENT_ANALYSI>())
                    {
                        var activity = caseManager.getActivityByID(frag.WKF_CASE_ACTIVITY_ID);

                        if (wfManager.AnyArchivedRecords(activity.WKF_CASE_ID))
                        {
                            ProcessFragmentAnalysisInsert(frag, _db, activity.WKF_CASE_ID);
                        }
                    }
                    foreach (ANALYSIS_DETAIL detail in changeSet.Updates.OfType<ANALYSIS_DETAIL>())
                    {
                        var activity = caseManager.getActivityByID(detail.FRAGMENT_ANALYSI.WKF_CASE_ACTIVITY_ID);

                        if (wfManager.AnyArchivedRecords(activity.WKF_CASE_ID))
                        {
                            //ProcessFragmentAnalysisInsert(frag, _db, activity.WKF_CASE_ID);
                            ProcessFragmentAnalysisDetailUpdate(detail, _db, activity.WKF_CASE_ID);
                        }
                    }

                    foreach (ANALYSIS_DETAIL detail in changeSet.Deletes.OfType<ANALYSIS_DETAIL>())
                    {
                        var activity = caseManager.getActivityByID(detail.FRAGMENT_ANALYSI.WKF_CASE_ACTIVITY_ID);

                        if (wfManager.AnyArchivedRecords(activity.WKF_CASE_ID))
                        {
                            //ProcessFragmentAnalysisInsert(frag, _db, activity.WKF_CASE_ID);
                            ProcessFragmentAnalysisDetailDelete(detail, _db, activity.WKF_CASE_ID);

                        }
                    }

                    foreach (ANALYSIS_DETAIL detail in changeSet.Inserts.OfType<ANALYSIS_DETAIL>())
                    {
                        var activity = caseManager.getActivityByID(detail.FRAGMENT_ANALYSI.WKF_CASE_ACTIVITY_ID);

                        if (wfManager.AnyArchivedRecords(activity.WKF_CASE_ID))
                        {
                            //ProcessFragmentAnalysisInsert(frag, _db, activity.WKF_CASE_ID);
                            ProcessFragmentAnalysisDetailInsert(detail, _db, activity.WKF_CASE_ID);
                        }
                    }

                    _db.SubmitChanges();

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = fragment.FRAGMENT_ANALYSIS_ID;

                    //WKF_CASE_ACTIVITY activity = caseManager.getActivityByID(fragment.WKF_CASE_ACTIVITY_ID)  ;

                    //activity.SetAsUpdateOnSubmit(); 
                    //activity.STD_WKFACTIVITYSTS_ID = Convert.ToInt16(WorkflowCaseActivityStatus.InProcess);

                    //activity.SynchroniseWithDataContext(_db, true);

                    //_db.SubmitChanges();

                    RaiseSaveEvent(this, eventArgs);
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }
            }

        }

        private void ProcessFragmentAnalysisInsert(FRAGMENT_ANALYSI analyte, RegistriesLinqDataContext context, int caseId)
        {
            string field = "Fragment Details";
            CreateFragDetailsHistoryLogEntry(field, analyte, new ModifiedMemberInfo(), context, caseId, "insertFragDetails", null);

        }



        private void ProcessFragmentAnalysisUpdate(FRAGMENT_ANALYSI analyte, RegistriesLinqDataContext context, int caseId)
        {
            ModifiedMemberInfo[] mmi = context.FRAGMENT_ANALYSIs.GetModifiedMembers(analyte);
            string field = String.Empty;

            foreach (ModifiedMemberInfo info in mmi)
            {
                switch (info.Member.Name)
                {
                    case "FRAGMENT_ID_NUMBER":
                        field = "Fragment ID";

                        break;

                    case "FRAGMENT_DESCR":
                        field = "Description";


                        break;

                    case "FRAGMENT_LENGTH_MEASURE":
                        field = "Length";

                        break;

                    case "FRAGMENT_WIDTH_MEASURE":
                        field = "Width";

                        break;

                    case "FRAGMENT_HEIGHT_MEASURE":
                        field = "Height";

                        break;

                    case "FRAGMENT_MASS_MEASURE":
                        field = "Mass";

                        break;

                    case "FRAGMENT_RADIOACTIVE_TEST_IND":
                        field = "Radioactive";

                        break;

                    case "FRAGMENT_RADIOACTIVE_RESULT":
                        field = "Radioactive Result";
                        break;

                    case "COMMENT_TEXT":
                        field = "Comments";
                        break;

                    case "TISSUE_SENT_FLAG":
                        field = "Tissue Sent";
                        break;

                    case "TISSUE_RESULT_TEXT":
                        field = "Tissue Associated with Fragment Details";
                        break;

                }
                //UpdateFragDetailsField
                CreateFragDetailsHistoryLogEntry(field, analyte, info, context, caseId, "UpdateFragDetailsField", null);
            }
        }

        private void ProcessFragmentAnalysisDetailUpdate(ANALYSIS_DETAIL detail, RegistriesLinqDataContext context, int caseId)
        {
            ModifiedMemberInfo[] mmi = context.ANALYSIS_DETAILs.GetModifiedMembers(detail);
            string field = String.Empty;
            string oldValue, newValue;
            string analyte = String.Empty;
            UnitTestReferenceRangeManager manager = new UnitTestReferenceRangeManager();

            if (mmi.Where(s => s.Member.Name == "STD_ANALYTETYPE_ID").Count() > 0)
            {

                var oldAnalyte = manager.GetAnalyteTypeById(int.Parse(mmi.Where(s => s.Member.Name == "STD_ANALYTETYPE_ID").FirstOrDefault().OriginalValue.ToString()));
                var newAnalyte = manager.GetAnalyteTypeById(int.Parse(mmi.Where(s => s.Member.Name == "STD_ANALYTETYPE_ID").FirstOrDefault().CurrentValue.ToString()));

                var analyteNames = new Dictionary<string, string>()
                {
                    { "old",
                            oldAnalyte },
                    { "new",
                            newAnalyte }
                };

                CreateFragHistoryLogEntryForAnalyteChange(field, context, caseId, analyteNames, detail.FRAGMENT_ANALYSI.FRAGMENT_ID_NUMBER);
            }
            foreach (ModifiedMemberInfo info in mmi)
            {

                switch (info.Member.Name)
                {
                    case "RESULT_TEXT":
                        field = "Result";

                        analyte = manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value);
                        CreateFragDetailsHistoryLogEntry(field, detail.FRAGMENT_ANALYSI, info, context, caseId, "updateAnalyteField", analyte);

                        break;

                    case "STD_ANALYSISMETHOD_ID":
                        field = "Analysis Method";

                        oldValue = manager.GetAnalysisMethodById((int)info.OriginalValue);
                        newValue = manager.GetAnalysisMethodById((int)info.CurrentValue);
                        analyte = manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value);

                        CreateFragHistoryLogEntryForDropdownData(field, context, detail, caseId, "updateAnalyteField", oldValue, newValue, analyte);

                        break;

                    case "STD_FRAGMENT_ANALYSIS_TYPE_ID":
                        field = "Analysis Type";

                        oldValue = manager.GetFragmentAnalysisTypeById((int)info.OriginalValue);
                        newValue = manager.GetFragmentAnalysisTypeById((int)info.CurrentValue);
                        analyte = manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value);

                        CreateFragHistoryLogEntryForDropdownData(field, context, detail, caseId, "updateAnalyteField", oldValue, newValue, analyte);

                        break;

                    case "CAS_NUMBER":
                        field = "CAS Number";
                        //CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);
                        analyte = manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value);

                        CreateFragDetailsHistoryLogEntry(field, detail.FRAGMENT_ANALYSI, info, context, caseId, "updateAnalyteField", analyte);

                        break;

                    case "EVAL_COMMENT_TEXT":
                        field = "Comments";
                        //CreateHistoryLogEntry(field, context, detail, caseId, "updateAnalyteField", info);
                        analyte = manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value);

                        CreateFragDetailsHistoryLogEntry(field, detail.FRAGMENT_ANALYSI, info, context, caseId, "updateAnalyteField", analyte);

                        break;
                }
            }
        }

        private void ProcessFragmentAnalysisDetailDelete(ANALYSIS_DETAIL details, RegistriesLinqDataContext context, int caseId)
        {
            string field = "Analyte Details";
            CreateFragHistoryLogEntry(field, context, details, caseId, "deleteAnalyte", new ModifiedMemberInfo());
        }

        private void ProcessFragmentAnalysisDetailInsert(ANALYSIS_DETAIL details, RegistriesLinqDataContext context, int caseId)
        {
            string field = "Analyte Details";
            CreateFragHistoryLogEntry(field, context, details, caseId, "insertAnalyte", new ModifiedMemberInfo());

        }
        private void CreateFragHistoryLogEntry(string field, RegistriesLinqDataContext context, ANALYSIS_DETAIL detail, int id, string action, ModifiedMemberInfo info)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();
            UnitTestReferenceRangeManager manager = new UnitTestReferenceRangeManager();

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            switch (action)
            {
                case "insertAnalyte":


                    tracking.CHANGE_FIELD = field;
                    tracking.CHANGE_DESCRIPTION = @"Fragment #" + detail.FRAGMENT_ANALYSI.FRAGMENT_ID_NUMBER + @" Analyte """ +
                        manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value) + @""" was added.";
                    break;

                case "updateAnalyteField":

                    tracking.CHANGE_FIELD = "Analyte";
                    tracking.CHANGE_DESCRIPTION = @"Analyte """ + detail.TEST_NAME + "/" + detail.ANALYTE_NAME
                        + @""" " + field + @" changed from """ + info.OriginalValue + @""" to """ + info.CurrentValue + @""".";
                    break;

                case "deleteAnalyte":

                    tracking.CHANGE_FIELD = field;
                    tracking.CHANGE_DESCRIPTION = @"Fragment #" + detail.FRAGMENT_ANALYSI.FRAGMENT_ID_NUMBER + @" Analyte """
                        + manager.GetAnalyteTypeById(detail.STD_ANALYTETYPE_ID.Value) + @""" was deleted.";

                    break;


            }

            tracking.SetAsInsertOnSubmit();
            LabResultsTrackingManager trmanager = new LabResultsTrackingManager();

            trmanager.Update(tracking);

        }

        private void CreateFragHistoryLogEntryForAnalyteChange(string field, RegistriesLinqDataContext context, int caseId, Dictionary<string, string> analyteNames, string fragmentId)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = caseId;
            tracking.CHANGE_DATE = context.GetSystemDate();

            tracking.CHANGE_FIELD = "Analyte Data";

            tracking.CHANGE_DESCRIPTION = @"Fragment #" + fragmentId + @" Analyte """ + analyteNames["old"]
                + @"""  changed to """ + analyteNames["new"] + @""".";

            tracking.SetAsInsertOnSubmit();
            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);
        }
        private void CreateFragHistoryLogEntryForDropdownData(string field, RegistriesLinqDataContext context, ANALYSIS_DETAIL detail, int id, string action, string oldValue,
                    string newValue, string analyteName)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            switch (action)
            {
                case "updateAnalyteField":
                    tracking.CHANGE_FIELD = "Analyte Data";
                    tracking.CHANGE_DESCRIPTION = @"Fragment #" + detail.FRAGMENT_ANALYSI.FRAGMENT_ID_NUMBER + @" Analyte """ + analyteName
                        + @""" field """ + field + @""" changed from """ + oldValue + @""" to """ + newValue + @""".";
                    break;

                case "changeAnalyteType":

                    break;

            }

            tracking.SetAsInsertOnSubmit();
            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);


        }
        private void CreateFragDetailsHistoryLogEntry(string field, FRAGMENT_ANALYSI fragment, ModifiedMemberInfo info, RegistriesLinqDataContext context, int id, string action, string analyteName)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            string originalValue = null, newValue = null;

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();
            //this is to avoid null reference exception
            switch (action)
            {
                case "insertFragDetails":

                    tracking.CHANGE_FIELD = field;
                    tracking.CHANGE_DESCRIPTION = @"Fragment #" + fragment.FRAGMENT_ID_NUMBER + @" added.";

                    break;

                case "UpdateFragDetailsField":
                    var typeName = info.OriginalValue == null ? info.CurrentValue.GetType().Name : info.OriginalValue.GetType().Name;

                    switch (typeName)
                    {
                        case "DateTime":

                            originalValue = (info.OriginalValue as DateTime? == null ? String.Empty : (info.OriginalValue as DateTime?).Value.ToShortDateString());
                            newValue = (info.CurrentValue as DateTime? == null ? String.Empty : (info.CurrentValue as DateTime?).Value.ToShortDateString());
                            break;

                        case "String":

                            originalValue = info.OriginalValue.ToString();
                            newValue = info.CurrentValue.ToString();
                            break;

                        case "Boolean":

                            originalValue = (bool)info.OriginalValue ? "True" : "False";
                            newValue = (bool)info.CurrentValue ? "True" : "False";
                            break;

                        case "Char":
                            originalValue = ((char?)info.OriginalValue).HasValue ? (char?)info.OriginalValue == 'Y' ? "True" : "False" : "Not Tested";
                            newValue = ((char?)info.CurrentValue).HasValue ? (char?)info.CurrentValue == 'Y' ? "True" : "False" : "Not Tested";
                            break;
                    }
                    tracking.CHANGE_DESCRIPTION = @"Fragment #" + fragment.FRAGMENT_ID_NUMBER + @" field """ + field + @""" changed from """ + originalValue
                        + @""" to """ + newValue + @""".";
                    tracking.CHANGE_FIELD = "Fragment Details";
                    break;

                case "updateAnalyteField":

                    originalValue = info.OriginalValue.ToString();
                    newValue = info.CurrentValue.ToString();

                    tracking.CHANGE_FIELD = "Analyte Data";
                    tracking.CHANGE_DESCRIPTION = @"Fragment #" + fragment.FRAGMENT_ID_NUMBER + @" Analyte """ + analyteName
                        + @""" field """ + field + @" """ + originalValue + @""" was changed to """ + newValue + @""".";

                    break;

            }
            //tracking.CHANGE_DESCRIPTION = "'" + originalValue + "'" + " was changed to " + "'" +  newValue + "'";
            tracking.SetAsInsertOnSubmit();

            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);

        }
        /// <summary>
        /// Gets the lab order id for a lab results work flow activity Id
        /// </summary>
        /// <param name="ActivityId"></param>
        /// <returns></returns>
        public int GetLabOrderId(int ActivityId)
        {
            using (_db = GetDataContext())
            {
                var order = //(from lo in _db.LAB_ORDERs
                             (from wc in _db.WKF_CASEs
                              where wc.WKF_CASE_ACTIVITies.Any(c => c.WKF_CASE_ACTIVITY_ID == ActivityId)
                              let c = wc.WKF_CASE_ACTIVITies.Where(c => c.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabOrder)).SingleOrDefault()
                              select c.LAB_ORDERs.FirstOrDefault().LAB_ORDER_ID).FirstOrDefault();
                return (int)order;
            }
        }


        /// <summary>
        /// Checks if there are any duplicate lab results exists for the Case Id
        /// </summary>
        /// <param name="wcaseId"></param>
        /// <returns></returns>
        public bool DuplicateLabResultExists(int wcaseId)
        {
            using (_db = GetDataContext())
            {
                return _db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ID == wcaseId && e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult) && e.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted)).Any();
            }
        }

        public WKF_CASE_ACTIVITY CreateNewInterpretationActivity(int caseId)
        {
            WKF_CASE_ACTIVITY Activity = new WKF_CASE_ACTIVITY();
            Activity.WKF_CASE_ID = caseId;
            Activity.STD_WKFACTIVITYSTS_ID = Convert.ToInt16(WorkflowCaseActivityStatus.New);
            Activity.STD_WKFACTIVITYTYPE_ID = Convert.ToInt16(WorkFlowActivityTypes.Interpretation);
            Activity.REPRESENTATIVE_NAME = "";
            return Activity;
        }


        public class Fragment
        {
            public int FragmentId { get; set; }
            public string FragmentEnteredId { get; set; }
            public string FragmentDescription { get; set; }


        }

        public class LabResult
        {
            public DateTime? LabReportDate { get; set; }
            public DateTime? AcceptedDate { get; set; }
            public DateTime? ReceivedDate { get; set; }
            public int CaseId { get; set; }
            public string CaseType { get; set; }
            public int ActivityId { get; set; }
            public string PatientName { get; set; }
            public string LabContact { get; set; }
            public string Site { get; set; }
            public string SiteName { get; set; }
        }
    }
}
