﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager;

namespace CRS_EFR
{
    public class LabOrderManager : BaseBO
    {
        public LabOrderManager()
        {
            _defaultSortField = "Workflow.WKF_CASE_ID desc";

        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
            lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
            lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
            lo.LoadWith<WKF_CASE>(e => e.PATIENT);
            lo.LoadWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs);
            lo.LoadWith<PATIENT>(e => e.STD_INSTITUTION);
            lo.LoadWith<PATIENT>(e => e.STD_SERVICEBRANCH);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_STATE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.LAB_ORDERs);
            //lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.QUES_FRAGMENT_SPECIMEN_COLLECTION_KIT_FORMs);
            lo.LoadWith<LAB_ORDER>(e => e.LAB_ORDER_DETAILs);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private IQueryable<WKF_CASE_ACTIVITY> LinqAll()
        {
            //populate LinqAll
            IQueryable<WKF_CASE_ACTIVITY> tempLinq = (from e in _db.WKF_CASE_ACTIVITies
                                                      where e.ARCHIVED_FLAG == false
                                                      select e);

            //TODO - add all business filtering rules 

            return tempLinq;

        }

        private IQueryable<WKF_CASE_ACTIVITY> SelectByStatusLinqFilter(string valueType, string value, int id)
        {
            var linqFilter = LinqAll();

            linqFilter = from t in linqFilter
                         where t.STD_WKFACTIVITYTYPE_ID == 3 && t.STD_WKFACTIVITYSTS_ID == id
                         select t;

            //string fieldSearch = valueType; // "LAST_NAME";"BSO_NUMBER";


            if (valueType == "LAST_NAME")
            {

                Expression<Func<WKF_CASE_ACTIVITY, bool>> lambda;
                if (String.IsNullOrEmpty(value))
                {
                    lambda = x => true;
                }
                else
                {
                    lambda = s => s.WKF_CASE.PATIENT.LAST_NAME.Contains(value);
                }

                linqFilter = linqFilter.Where(lambda);

            }

            else if (valueType == "BSO_NUMBER")
            {
                Expression<Func<WKF_CASE_ACTIVITY, bool>> lambda;
                if (String.IsNullOrEmpty(value))
                {
                    lambda = x => true;
                }
                else
                {
                    lambda = s => s.LAB_ORDERs.Any(e => e.BSO_NUMBER == value);
                }

                linqFilter = linqFilter.Where(lambda);

            }

            else if (valueType == "LAB_REQUEST_NUMBER" && !String.IsNullOrEmpty(value))
            {
                int orderId = 0;
                Expression<Func<WKF_CASE_ACTIVITY, bool>> lambda = s => s.LAB_ORDERs.Any(e => e.LAB_ORDER_ID.Equals(-1));
                if (int.TryParse(value, out orderId))
                {
                    lambda = s => s.LAB_ORDERs.Any(e => e.LAB_ORDER_ID == orderId);
                }
                linqFilter = linqFilter.Where(lambda);
            }
            return linqFilter;

        }

        public IEnumerable<LabOrder> SelectByStatus(string searchCriteria, string searchValue, int id, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }
            using (_db = GetDataContext())
            {

                SetLoadWith(_db);

                var entities = (from t in SelectByStatusLinqFilter(searchCriteria, searchValue, id) select new { Workflow = t, AssociatedOrder = t.LAB_ORDERs.FirstOrDefault() }).OrderBy(sort).Select(p => p.Workflow).Skip(startRow).Take(maxRows).ToList();
                return entities.Select(e => new LabOrder
                {

                    LabOrderId = e.WKF_CASE_ACTIVITY_ID,
                    CaseId = e.WKF_CASE_ID,
                    LabRequestId = e.LAB_ORDERs.Count() != 0 ?

                    (e.LAB_ORDERs.First().LAB_SEND_DATE.HasValue ?
                        e.LAB_ORDERs.First().LAB_SEND_DATE.Value.ToString("yy") :
                            DateTime.Today.ToString("yy"))
                         + "-" + "TEF" + e.LAB_ORDERs.First().LAB_ORDER_ID.ToString().PadLeft(5, '0')
                            : String.Empty,

                    PatientName = e.WKF_CASE.PATIENT.LAST_NAME + ", " + e.WKF_CASE.PATIENT.FIRST_NAME + " " + e.WKF_CASE.PATIENT.MIDDLE_NAME,
                    StationNumber = e.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                    Institution = e.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME,
                    RequestDate = e.LAB_ORDERs.Count() != 0 ? String.Format("{0:d}", e.LAB_ORDERs.First().LAB_SEND_DATE) : String.Empty,
                    ReportDate = e.LAB_ORDERs.Count() != 0 ? String.Format("{0:d}", e.LAB_ORDERs.First().LAB_RETURN_DATE) : String.Empty,
                    ReceivedByMetalsLabDate = e.LAB_ORDERs.Count() != 0 ? String.Format("{0:d}", e.LAB_ORDERs.First().LAB_RECEIVED_DATE) : String.Empty,
                    LabOrderStatus = e.STD_WKFACTIVITYST.NAME,
                    LabOrderType = e.WKF_CASE.STD_WKFCASETYPE.Name,

                    BSONumber = new Func<WKF_CASE_ACTIVITY, string>
                    (
                        item =>
                        {
                            if (item.LAB_ORDERs.Count() != 0)
                            {
                                if (item.WKF_CASE.STD_WKFCASETYPE_ID == 3)
                                {
                                    return (item.LAB_ORDERs.First().BSO_NUMBER);

                                }
                                else
                                {
                                    return "N/A";
                                }
                            }
                            else
                            {
                                return String.Empty;

                            }

                        }

                    )(e)
                });
            }
        }

        public int SelectByStatusCount(string searchCriteria, string searchValue, int id)
        {
            using (_db = GetDataContext())
            {
                //SetLoadWith(_db);
                return SelectByStatusLinqFilter(searchCriteria, searchValue, id).Count();
            }
        }

        public object getOrderStatusByActivityID(int id)
        {
            //STD_WKFACTIVITYST entity;
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                var entity = (from wfs in _db.WKF_CASE_ACTIVITies.Where(r => r.WKF_CASE_ACTIVITY_ID == id)
                              select new LabOrderStatus
                              {
                                  CaseId = wfs.WKF_CASE_ID,
                                  Name = wfs.STD_WKFACTIVITYST.NAME
                              }).SingleOrDefault();

                return entity;

            }


        }

        public STD_WKFACTIVITYST GetNameByStatusId(int id)
        {
            STD_WKFACTIVITYST wst;
            using (_db = GetDataContext())
            {

                wst = (from st in _db.STD_WKFACTIVITYSTs where st.ID == id select st).FirstOrDefault();

            }

            return wst;
        }


        public LAB_ORDER_STATIC GetLabOrderStaticInfo()
        {
            LAB_ORDER_STATIC los;
            using (_db = GetDataContext())
            {

                los = (from lo in _db.LAB_ORDER_STATICs select lo).FirstOrDefault();

            }

            return los;

        }
        public LabOrderPatient GetPatientByActivityId(int id)
        {

            using (_db = GetDataContext())
            {

                return (from acts in _db.WKF_CASE_ACTIVITies
                        where acts.WKF_CASE_ACTIVITY_ID == id
                        select
                            new LabOrderPatient
                            {
                                PatientId = acts.WKF_CASE.PATIENT.PATIENT_ID,
                                PatientName = acts.WKF_CASE.PATIENT.FullName,
                                PatientServiceBranch = acts.WKF_CASE.PATIENT.STD_SERVICEBRANCH.NAME,
                                PatientSSN = acts.WKF_CASE.PATIENT.Snum,
                                PatientIcn = acts.WKF_CASE.PATIENT.PATIENT_ICN,
                                PatientServiceStatus = acts.WKF_CASE.PATIENT.SERVICE_STATUS,
                                PatientZip = acts.WKF_CASE.PATIENT.POSTAL_CODE

                            }).FirstOrDefault();

            }

        }

        public WKF_CASE_ACTIVITY getLabOrderByOrderID(int id)
        {
            WKF_CASE_ACTIVITY entity;
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                entity = (from srv in _db.WKF_CASE_ACTIVITies where srv.WKF_CASE_ACTIVITY_ID == id select srv).FirstOrDefault();
            }

            return entity;
        }

        public void Update(WKF_CASE_ACTIVITY caseActivity)
        {
            WorkflowManager wfManager = new WorkflowManager();
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                caseActivity.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    ChangeSet changeSet = _db.GetChangeSet();
                    foreach (LAB_ORDER order in
                    changeSet.Updates.OfType<LAB_ORDER>())
                    {
                        if (wfManager.AnyArchivedRecords(caseActivity.WKF_CASE_ID))
                        {
                            //ModifiedMemberInfo[] mmi = this.WKF_CASE_ACTIVITies.GetModifiedMembers(activity);
                            ProcessLabOrderUpdate(order, _db, caseActivity.WKF_CASE_ID);
                        }
                    }

                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = caseActivity.WKF_CASE_ACTIVITY_ID;

                    RaiseSaveEvent(this, eventArgs);

                }
                catch
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }

            }
        }

        private void ProcessLabOrderUpdate(LAB_ORDER order, RegistriesLinqDataContext context, int caseId)
        {
            ModifiedMemberInfo[] mmi = context.LAB_ORDERs.GetModifiedMembers(order);
            string field = String.Empty;

            foreach (ModifiedMemberInfo info in mmi)
            {
                switch (info.Member.Name)
                {
                    case "BSO_NUMBER":
                        field = "Accession #";
                        break;

                    case "LAB_REPORT_DATE":
                        field = "Metals Lab Report Date";
                        break;

                    case "LAB_ID_NUMBER":
                        field = "Lab ID";
                        break;

                    case "METALS_LAB_CODE":
                        field = "Lab Code";
                        break;

                    case "FRAGMENT_PROD_EVENT_DATE":
                        field = "Producing Event Date";
                        break;

                }
                CreateHistoryLogEntry(field, info, context, caseId, "UpdateField");
            }

        }


        private void CreateHistoryLogEntry(string field, ModifiedMemberInfo info, RegistriesLinqDataContext context, int id, string action)
        {
            LABRESULT_CHANGE_TRACKING tracking = new LABRESULT_CHANGE_TRACKING();

            string originalValue = null, newValue = null;

            tracking.SetAsChangeTrackingRoot();
            tracking.WKF_CASE_ID = id;
            tracking.CHANGE_DATE = context.GetSystemDate();

            switch (info.OriginalValue.GetType().Name)
            {
                case "DateTime":

                    originalValue = ((DateTime)info.OriginalValue).ToShortDateString();
                    newValue = ((DateTime)info.CurrentValue).ToShortDateString();
                    break;

                case "String":

                    originalValue = info.OriginalValue.ToString();
                    newValue = info.CurrentValue.ToString();
                    break;

            }

            switch (action)
            {

                case "UpdateField":

                    tracking.CHANGE_DESCRIPTION = "'" + originalValue + "'" + " was changed to " + "'" + newValue + "'";
                    tracking.CHANGE_FIELD = field;
                    break;

                case "UpdateAnalyteField":

                    tracking.CHANGE_FIELD = field;
                    tracking.CHANGE_DESCRIPTION = "'" + originalValue + "'" + " was changed to " + "'" + newValue + "'";

                    break;

            }
            //tracking.CHANGE_DESCRIPTION = "'" + originalValue + "'" + " was changed to " + "'" +  newValue + "'";
            tracking.SetAsInsertOnSubmit();

            LabResultsTrackingManager manager = new LabResultsTrackingManager();

            manager.Update(tracking);


        }



        public bool ActivityExists(int wkfId, WorkFlowActivityTypes type)
        {
            using (_db = GetDataContext())
            {
                return (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ID == wkfId
                    && e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(type)
                    && e.ARCHIVED_FLAG == false
                    ).Select(t => t.WKF_CASE_ACTIVITY_ID).Any());
            }
        }
        public WKF_CASE_ACTIVITY GetLabOrderByCaseId(int caseId)
        {
            WKF_CASE_ACTIVITY activity;
            using (_db = GetDataContext())
            {
                activity = (from acts in _db.WKF_CASE_ACTIVITies
                            where acts.WKF_CASE_ID == caseId
                            && acts.STD_WKFACTIVITYTYPE_ID == (int)WorkFlowActivityTypes.LabOrder
                            && acts.ARCHIVED_FLAG == false
                            select acts).FirstOrDefault();
                activity.SetAsChangeTrackingRoot();



            }

            return activity;

        }



        public bool CheckWkfCaseStatusForReOpen(int wkfId, WorkFlowActivityTypes type)
        {
            bool isWorkflowValidForReOpen = false;
            List<int> statuscode = new List<int>() { 1, 6 };
            using (_db = GetDataContext())
            {
                isWorkflowValidForReOpen = _db.WKF_CASE_ACTIVITies.Any(s => s.WKF_CASE_ID == wkfId
                    && s.STD_WKFACTIVITYTYPE_ID == (int)type
                    && statuscode.Contains(s.STD_WKFACTIVITYSTS_ID)
                    && s.ARCHIVED_FLAG == false);
            }
            return isWorkflowValidForReOpen;
        }


        public class LabOrder
        {
            public int LabOrderId { get; set; }
            public int CaseId { get; set; }
            public string LabRequestId { get; set; }
            public string PatientName { get; set; }
            public string RequestDate { get; set; }
            public string ReportDate { get; set; }
            public string ReceivedByMetalsLabDate { get; set; }
            public string LabOrderStatus { get; set; }
            public string LabOrderType { get; set; }
            public string BSONumber { get; set; }
            public string StationNumber { get; set; }
            public string Institution { get; set; }


        }

        public class LabOrderPatient
        {
            public int PatientId { get; set; }
            public string PatientName { get; set; }
            public string PatientServiceBranch { get; set; }
            public string PatientSSN { get; set; }
            public string PatientIcn { get; set; }
            public string PatientServiceStatus { get; set; }
            public string PatientZip { get; set; }

        }

        public class LabOrderStatus
        {
            public string Name { get; set; }
            public int CaseId { get; set; }

        }

    }
}
