﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions; 
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager;

namespace CRS_EFR
{
    public enum StdShipmentStatus
    {
        KitLost=2,
        PatientRefused=3,
        Other =6
    }

    public class LabKitOrderManager : BaseBO
    {
        // TODO!!!
        private static int REGISTRY_ID = 1;
        private static string USER_NAME = "DNS      \\DNS   ";

        private IEnumerable<WKF_CASE_ACTIVITY> CompletedKitOrders;
        private int KitOrdersCount;
        private int NewKitOrdersCount;
        private string QueueDefaultSort;

       public LabKitOrderManager() 
        {
            _defaultSortField = "Facility";
            QueueDefaultSort = "WkfCaseId desc";
        }

       #region Kit orders group

        /// <summary>
        /// Gets patient list for new kit orders
        /// </summary>
        /// <returns></returns>
       public List<LabKitOrderGroup> GetPatientsForNewKitOrders(string lastName, string facilityId, int startRow, string sort, int maxRows)
        {
            if (string.IsNullOrEmpty(sort)) sort = _defaultSortField;
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);



                string fieldSearch = "LAST_NAME";

                var param = Expression.Parameter(typeof(WKF_CASE), "PATIENT");

                // e => e.WKF_CASE.PATIENT.LAST_NAME.Contains()
                var name = Expression.PropertyOrField(Expression.PropertyOrField(param, "PATIENT"), fieldSearch);
                var search = Expression.Constant(lastName, typeof(string));

                var body = Expression.Call(name, "Contains", null, search);

                Expression<Func<WKF_CASE, bool>> lambda;
                if (String.IsNullOrEmpty(lastName))
                {
                    lambda = x => true;
                }
                else
                {
                    lambda = Expression.Lambda<Func<WKF_CASE, bool>>(body, param);
                }

                List<LabKitOrderGroup> KitOrders = new List<LabKitOrderGroup>();
                //Get New orders. New orders do not have lab kit order activity in activity table. But display patients who bypassed kit orders and went on with the later steps? (look at let..) 
                IEnumerable<WKF_CASE> Cases = (from t in _db.WKF_CASEs.Where(lambda)
                              let ordersPlaced = (from r in _db.WKF_CASE_ACTIVITies where  r.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabKit) select r.WKF_CASE_ID).Distinct().ToArray()
                              where !(ordersPlaced.Contains(t.WKF_CASE_ID))
                              && t.STD_WKFCASESTS_ID == Convert.ToInt16(WorkFlowCaseStatus.InProcess) // TODO - If status is new, there should not be any activity in wkfactivity. So for ordersPlaced list, add InProcess
                              && t.REFERRAL.STD_REFERRALSTS_ID == Convert.ToInt16(ReferralStatus.InProcess)
                              && (t.STD_WKFCASETYPE_ID == Convert.ToInt16(WorkFlowTypes.Biomonitoring) || t.STD_WKFCASETYPE_ID == Convert.ToInt16(WorkFlowTypes.FragmentAnalysis)) select t).ToList();

                if (!string.IsNullOrEmpty(facilityId))
                {
                    int institutionId = 0;
                    if (int.TryParse(facilityId, out institutionId))
                    {
                        if (institutionId > 0)
                        {
                            Cases = (from c in Cases
                                     where c.REFERRAL.STD_INSTITUTION_ID == institutionId
                                     select c);
                        }
                    }
                }

                Lookup<string, string> FirstLookUps = (Lookup<string, string>)Cases.ToLookup(p => p.REFERRAL.STD_INSTITUTION_ID + "%" + p.REFERRAL.STD_INSTITUTION.NAME + "%" + p.STD_WKFCASETYPE.ID + "%" + p.STD_WKFCASETYPE.CODE + "%" + p.REFERRAL.STD_INSTITUTION.STATIONNUMBER, p => p.REFERRAL.PATIENT.FullName + " - Workflow ID: " + p.WKF_CASE_ID);

                foreach (IGrouping<string, string> FirstLookUp in FirstLookUps)
                {
                    string[] Groupings = FirstLookUp.Key.Split(new Char[] { '%' });
                    LabKitOrderGroup SingleOrder = new LabKitOrderGroup { InstitutionId = Convert.ToInt32(Groupings[0]), StationNumber = Groupings[4] ,Facility = Groupings[1], KitTypeID = Convert.ToInt32(Groupings[2]), KitTypeCd = Groupings[3], OrderType = 0 };
                    SingleOrder.Patient = FirstLookUp.Select(e => e).ToList();
                    KitOrders.Add(SingleOrder);
                    SingleOrder.Count = SingleOrder.Patient.Count();
                }
                IEnumerable<LabKitOrderGroup> OrderByList = KitOrders;
                KitOrdersCount = OrderByList.Count();
                return OrderByList.OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            }
        }
       public int GetKitsCount(string lastName, string facilityId)
       {
           return KitOrdersCount;
       }

       public int GetPatientsForNewKitOrders(string lastName, string facilityId)
       {
           return KitOrdersCount;
       }

        #endregion

       #region Updates

       public void InsertActivities(List<WKF_CASE_ACTIVITY> activities)
        {
            using(_db=GetDataContext())
            {
                _db.WKF_CASE_ACTIVITies.InsertAllOnSubmit<WKF_CASE_ACTIVITY>(activities);
                _db.SubmitChanges();
            }
        }

       public void UpdateActivities(List<WKF_CASE_ACTIVITY> activities)
       {
           using (_db = GetDataContext())
           {
               _db.WKF_CASE_ACTIVITies.AttachAll(activities,true);
               try
               {
                   _db.SubmitChanges(ConflictMode.ContinueOnConflict);
               }
               catch
               {
                   _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
               }
           }
       }
       
       public void UpdateKitActivity(WKF_CASE wCase, Func<bool> checkDups)
       {
           //Initiate New activity for Questionnaire
           if (wCase.WKF_CASE_ACTIVITies[0].SHIPPING_DETAILs != null)
               if (wCase.WKF_CASE_ACTIVITies[0].SHIPPING_DETAILs[0].SURVEY_RECEIVED_FLAG == true && !ActivityExists(wCase.WKF_CASE_ID, WorkFlowActivityTypes.Questionnaire) && checkDups())
                   InitiateActivity(wCase, WorkFlowActivityTypes.Questionnaire);
           //Initiate New activity for Lab order
           if (wCase.WKF_CASE_ACTIVITies[0].SHIPPING_DETAILs != null)
               if (wCase.WKF_CASE_ACTIVITies[0].SHIPPING_DETAILs[0].RECEIVED_DATE != null && !ActivityExists(wCase.WKF_CASE_ID, WorkFlowActivityTypes.LabOrder) && checkDups())
                   InitiateActivity(wCase, WorkFlowActivityTypes.LabOrder);
           
           using (_db = GetDataContext())
           {
               _db.DeferredLoadingEnabled = false;
               wCase.SynchroniseWithDataContext(_db, true);
               try
               {
                   _db.SubmitChanges();

                   //create an instance of the custom eventArgs in order to populate the id selected
                   BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                   eventArgs.SavedItemId = wCase.WKF_CASE_ID;

                   RaiseSaveEvent(this, eventArgs);
               }
               catch (ChangeConflictException)
               {
                   _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
               }
           }
       }

       private void InitiateActivity(WKF_CASE wcase, WorkFlowActivityTypes type)
       {
           WKF_CASE_ACTIVITY Activity = wcase.WKF_CASE_ACTIVITies[0];
           WKF_CASE_ACTIVITY NewActivity = new WKF_CASE_ACTIVITY();
           //NewActivity.WKF_CASE_ID = Activity.WKF_CASE_ID;
           NewActivity.WKF_CASE_ID = wcase.WKF_CASE_ID; 
           NewActivity.STD_WKFACTIVITYTYPE_ID = Convert.ToInt16(type);
           NewActivity.STD_WKFACTIVITYSTS_ID = Convert.ToInt16(WorkflowCaseActivityStatus.New);
           NewActivity.REPRESENTATIVE_NAME = Activity.REPRESENTATIVE_NAME;
           NewActivity.CONTACT_NAME = Activity.CONTACT_NAME;
           wcase.WKF_CASE_ACTIVITies.Add(NewActivity);
           NewActivity.SetAsChangeTrackingRoot(EntityState.New, true);
           NewActivity.SetAsInsertOnSubmit();
       }

       #endregion

       #region Retrieve Kit orders and details
       private void SetLoadWith(RegistriesDataAccess db)
       {
           DataLoadOptions lo = new DataLoadOptions();
           lo.LoadWith<REFERRAL>(e => e.PATIENT);
           lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
           lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
           //lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
           lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
           db.LoadOptions = lo;
           db.DeferredLoadingEnabled = false;

       }

       private void SetLoadWithActivities(RegistriesDataAccess db, bool addShipping)
       {
           DataLoadOptions lo = new DataLoadOptions();
           lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
           lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
           lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
           lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
           lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
           lo.LoadWith<REFERRAL>(e => e.PATIENT);
           lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
           if (addShipping)
           {
               lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SHIPPING_DETAILs);
               lo.LoadWith<SHIPPING_DETAIL>(e => e.STD_SHIPMENTST);
           }
           db.LoadOptions = lo;
           db.DeferredLoadingEnabled = false;

       }

       public List<LabKitOrderPatientsBO> GetNewLabKitOrders(int instId, int kitTypeId, int startRow, string sort, int maxRows)
       {
           if (string.IsNullOrEmpty(sort)) sort = "WkfCaseId desc";

           CityStateManager addressManager = new CityStateManager();

           using (_db = GetDataContext())
           {
               SetLoadWith(_db);

               List<WKF_CASE> cases = (from t in _db.WKF_CASEs
                                       let ordersPlaced = (from r in _db.WKF_CASE_ACTIVITies
                                                           where r.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabKit)
                                                           select r.WKF_CASE_ID).Distinct().ToArray()
                                       where !(ordersPlaced.Contains(t.WKF_CASE_ID))
                                           && t.STD_WKFCASESTS_ID == Convert.ToInt16(WorkFlowCaseStatus.InProcess)
                                           && t.REFERRAL.STD_INSTITUTION_ID == instId
                                           && t.REFERRAL.STD_REFERRALSTS_ID == Convert.ToInt16(ReferralStatus.InProcess)
                                           && t.STD_WKFCASETYPE_ID == kitTypeId
                                       select t).ToList();

               List<LabKitOrderPatientsBO> NewLabKitOrders = new List<LabKitOrderPatientsBO>();
               foreach (WKF_CASE item in cases)
               {
                   LabKitOrderPatientsBO labOrderKit = new LabKitOrderPatientsBO();

                   labOrderKit.WkfCaseId = item.WKF_CASE_ID;
                   labOrderKit.County = item.REFERRAL.COUNTY;
                   labOrderKit.VAMCContactNm = item.REFERRAL.VAMC_CONTACT_NAME;
                   labOrderKit.VAMCContactPh = item.REFERRAL.VAMC_CONTACT_PHONE;
                   labOrderKit.KitOrderedDate = DateTime.Today;
                   labOrderKit.PatientName = item.REFERRAL.PATIENT.FullName;
                   labOrderKit.ZipPlus4 = item.REFERRAL.ZIP_PLUS_4;
                   labOrderKit.TEFSCRep = USER_NAME;
                   labOrderKit.KitType = item.STD_WKFCASETYPE.CODE;
                   labOrderKit.InstName = item.REFERRAL.STD_INSTITUTION.NAME;

                   if (!string.IsNullOrEmpty(item.REFERRAL.ADDRESS_LINE1))
                   {
                       labOrderKit.AddressLine1 = item.REFERRAL.ADDRESS_LINE1;
                       labOrderKit.AddressLine2 = item.REFERRAL.ADDRESS_LINE2;
                       labOrderKit.AddressLine3 = item.REFERRAL.ADDRESS_LINE3;
                       labOrderKit.City = item.REFERRAL.CITY;
                       labOrderKit.State = item.REFERRAL.STATE;
                       labOrderKit.PostalCode = item.REFERRAL.POSTAL_CODE;
                       labOrderKit.Country = item.REFERRAL.COUNTRY;
                   }
                   else if (item.REFERRAL.STD_INSTITUTION != null)
                   {

                       if (string.IsNullOrEmpty(item.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE1))
                       {
                           labOrderKit.AddressLine1 = item.REFERRAL.STD_INSTITUTION.STREETADDRESSLINE1;
                           labOrderKit.AddressLine2 = item.REFERRAL.STD_INSTITUTION.STREETADDRESSLINE2;
                           labOrderKit.AddressLine3 = item.REFERRAL.STD_INSTITUTION.STREETADDRESSLINE3;
                           labOrderKit.City = item.REFERRAL.STD_INSTITUTION.STREETCITY;
                           labOrderKit.PostalCode = item.REFERRAL.STD_INSTITUTION.STREETPOSTALCODE;
                           if (item.REFERRAL.STD_INSTITUTION.STREETSTATE_ID.HasValue)
                           {
                               labOrderKit.State = addressManager.GetStateByID(
                                   (int)item.REFERRAL.STD_INSTITUTION.STREETSTATE_ID);
                           }
                           if (item.REFERRAL.STD_INSTITUTION.STREETCOUNTRY_ID.HasValue)
                           {
                               labOrderKit.Country = addressManager.GetCountryByID(
                                   (int)item.REFERRAL.STD_INSTITUTION.STREETCOUNTRY_ID);
                           }
                       }
                       else
                       {
                           labOrderKit.AddressLine1 = item.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE1;
                           labOrderKit.AddressLine2 = item.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE2;
                           labOrderKit.AddressLine3 = item.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE3;
                           labOrderKit.City = item.REFERRAL.STD_INSTITUTION.MAILINGCITY;
                           labOrderKit.PostalCode = item.REFERRAL.STD_INSTITUTION.MAILINGPOSTALCODE;
                           if (item.REFERRAL.STD_INSTITUTION.MAILINGSTATE_ID.HasValue)
                           {
                               labOrderKit.State = addressManager.GetStateByID(
                                   (int)item.REFERRAL.STD_INSTITUTION.MAILINGSTATE_ID);
                           }
                           if (item.REFERRAL.STD_INSTITUTION.MAILINGCOUNTRY_ID.HasValue)
                           {
                               labOrderKit.Country = addressManager.GetCountryByID(
                                   (int)item.REFERRAL.STD_INSTITUTION.MAILINGCOUNTRY_ID);
                           }
                       }
                   }

                   NewLabKitOrders.Add(labOrderKit);
               }

               NewKitOrdersCount = NewLabKitOrders.Count();
               return NewLabKitOrders.Skip(startRow).Take(maxRows).ToList();
           }
       }

       public int GetNewLabKitOrdersCount(int instId, int kitTypeId)
       {
           return NewKitOrdersCount;
       }
       public IEnumerable<WKF_CASE_ACTIVITY> GetCompletedLabKitOrdersGraph()
       {
           return CompletedKitOrders;
       }

       public List<LabKitOrderPatientsBO> GetCompletedLabKitOrders(int instId, int kitTypeId)
       {
           using (_db = GetDataContext())
           {
               SetLoadWith(_db);
               IQueryable<WKF_CASE_ACTIVITY> LinqActivities = (from a in _db.WKF_CASE_ACTIVITies
                                                               where a.WKF_CASE.REFERRAL.STD_INSTITUTION_ID == instId
                                                               && a.WKF_CASE.STD_WKFCASETYPE_ID == kitTypeId //Below filters applied in the list screen and does not change - TODO
                                                               //&& a.WKF_CASE.STD_WKFCASESTS_ID == Convert.ToInt16(WorkFlowCaseStatus.InProcess)
                                                               //&& a.WKF_CASE.REFERRAL.STD_REFERRALSTS_ID == Convert.ToInt16(ReferralStatus.InProcess)
                                                               select a);
               CompletedKitOrders = LinqActivities.ToList();

               CityStateManager addressManager = new CityStateManager();
               foreach (WKF_CASE_ACTIVITY activity in LinqActivities)
               {
                   if (string.IsNullOrEmpty(activity.ADDRESS_LINE1))
                   {
                       if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION !=null)
                       {
                           if (string.IsNullOrEmpty(activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE1))
                           {
                               activity.ADDRESS_LINE1 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETADDRESSLINE1;
                               activity.ADDRESS_LINE2 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETADDRESSLINE2;
                               activity.ADDRESS_LINE3 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETADDRESSLINE3;
                               activity.CITY = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETCITY;
                               activity.ZIP_PLUS_4 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETPOSTALCODE;
                               if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETSTATE_ID.HasValue)
                                    activity.STATE = addressManager.GetStateByID((int)activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETSTATE_ID);
                               if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETCOUNTRY_ID.HasValue)
                                    activity.COUNTRY = addressManager.GetCountryByID((int)activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETCOUNTRY_ID);
                               if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETCOUNTY_ID.HasValue)
                                    activity.COUNTY = addressManager.GetCountyByID((int)activity.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETCOUNTY_ID);
                           }
                           else
                           {
                               activity.ADDRESS_LINE1 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE1;
                               activity.ADDRESS_LINE2 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE2;
                               activity.ADDRESS_LINE3 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGADDRESSLINE3;
                               activity.CITY = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCITY;
                               activity.ZIP_PLUS_4 = activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGPOSTALCODE;
                               if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGSTATE_ID.HasValue)
                                    activity.STATE = addressManager.GetStateByID((int)activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGSTATE_ID);
                               if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCOUNTRY_ID.HasValue)
                                    activity.COUNTRY = addressManager.GetCountryByID((int)activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCOUNTRY_ID);
                               if (activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCOUNTY_ID.HasValue)
                                    activity.COUNTY = addressManager.GetCountyByID((int)activity.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCOUNTY_ID);
                          }
                       }
                   }
               }

               return (from a in LinqActivities
                       select new LabKitOrderPatientsBO
                       {
                           WkfCaseId = a.WKF_CASE_ID,
                           AddressLine1 = a.ADDRESS_LINE1,
                           AddressLine2 = a.ADDRESS_LINE2,
                           AddressLine3 = a.ADDRESS_LINE3,
                           City = a.CITY,
                           Country = a.COUNTRY,
                           County = a.COUNTY,
                           VAMCContactNm = a.CONTACT_NAME,
                           //KitOrderdDate = a.KIT_ORDER_DATE,
                           KitOrderedDate =  a.KIT_ORDER_DATE,                           
                           PatientName = a.WKF_CASE.REFERRAL.PATIENT.FullName,
                           PostalCode = a.POSTAL_CODE,
                           State = a.STATE,
                           ZipPlus4 = a.ZIP_PLUS_4,
                           TEFSCRep = a.REPRESENTATIVE_NAME,
                           VAMCContactPh = a.CONTACT_PHONE,
                           KitType = a.WKF_CASE.STD_WKFCASETYPE.CODE,
                           InstName = a.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME
                       }).ToList();

           }
       }

       private IQueryable<WKF_CASE_ACTIVITY> GetLinqInProcessKitOrders(string lastName, string facilityId, RegistriesDataAccess db, bool includeShipDetails)
       {
           SetLoadWithActivities(db, includeShipDetails);

           string fieldSearch = "LAST_NAME";

           var param = Expression.Parameter(typeof(WKF_CASE_ACTIVITY), "PATIENT");

           // e => e.WKF_CASE.PATIENT.LAST_NAME.Contains()
           var name = Expression.PropertyOrField(Expression.PropertyOrField(Expression.PropertyOrField(param, "WKF_CASE"), "PATIENT"), fieldSearch);
           var search = Expression.Constant(lastName, typeof(string));

           var body = Expression.Call(name, "Contains", null, search);

           Expression<Func<WKF_CASE_ACTIVITY, bool>> lambda;
           if (String.IsNullOrEmpty(lastName))
           {
               lambda = x => true;
           }
           else
           {
               lambda = Expression.Lambda<Func<WKF_CASE_ACTIVITY, bool>>(body, param);
           }

           IQueryable<WKF_CASE_ACTIVITY> kitOrders = (from a in _db.WKF_CASE_ACTIVITies.Where(lambda)
                   where (a.WKF_CASE.STD_WKFCASETYPE_ID == Convert.ToInt16(WorkFlowTypes.Biomonitoring)
                   || a.WKF_CASE.STD_WKFCASETYPE_ID == Convert.ToInt16(WorkFlowTypes.FragmentAnalysis))
                   && a.WKF_CASE.REFERRAL.STD_REGISTRY_ID == REGISTRY_ID
                   && a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabKit)
                   select a);

           if (!string.IsNullOrEmpty(facilityId))
           {
               int institutionId = 0;
               if (int.TryParse(facilityId, out institutionId))
               {
                   if (institutionId > 0)
                   {
                       kitOrders = (from c in kitOrders
                                where c.WKF_CASE.REFERRAL.STD_INSTITUTION_ID == institutionId
                                select c);
                   }
               }
           }
           
           return kitOrders;
       }

        /// <summary>
        /// Get all the completed kit orders (Kit orders sent and waiting for them to be returned to TEFSC.)
        /// </summary>
       /// <returns>WKF_CASE_ACTIVITY</returns>
       public IEnumerable<LabInProcessKitDetails> GetOrderedKitOrders(string lastName, string facilityId, int maxRows, int startRow, string sort)
       {
           if (string.IsNullOrEmpty(sort)) sort = QueueDefaultSort;
           using (_db = GetDataContext())
           {
               IEnumerable<LabInProcessKitDetails>
                   OrderedKitOrders = (from t in GetLinqInProcessKitOrders(lastName, facilityId, _db, false)
                                       where t.STD_WKFACTIVITYSTS_ID == 4
                                       select new LabInProcessKitDetails
                                       {
                                           ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                           WkfCaseId = t.WKF_CASE_ID,
                                           FromStock = t.KIT_USE_STOCK_FLAG != null ? (bool)t.KIT_USE_STOCK_FLAG : false,
                                           PatientName = t.WKF_CASE.REFERRAL.PATIENT.FullName,
                                           InstName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME,
                                           StationNumber = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                           //City = t.CITY,
                                           City = t.CITY != null ? t.CITY : (t.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCITY != null ? t.WKF_CASE.REFERRAL.STD_INSTITUTION.MAILINGCITY : t.WKF_CASE.REFERRAL.STD_INSTITUTION.STREETCITY),
                                           VAMCContactNm = t.CONTACT_NAME,
                                           KitType = t.WKF_CASE.STD_WKFCASETYPE.CODE,
                                           //KitOrderdDate = t.CREATED,
                                           KitOrderedDate = t.KIT_ORDER_DATE, 
                                           ReceivedDate = t.RECEIVED_DATE,
                                           Status = t.STD_WKFACTIVITYST.NAME
                                       }).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();



               return OrderedKitOrders.Distinct(new WorkflowCaseActivityComparer());
           }
       }

       public int GetOrderedKitOrdersCount(string lastName, string facilityId)
       {
           using (_db = GetDataContext())
           {
               return  (from t in GetLinqInProcessKitOrders(lastName, facilityId, _db, false)
                                 where t.STD_WKFACTIVITYSTS_ID == 4
                                 select t).Count();
           }
       }
       public IEnumerable<WKF_CASE_ACTIVITY> GetOrderedKitOrders(string lastName, string facilityId)
       {
           using (_db = GetDataContext())
           {
               return (from t in GetLinqInProcessKitOrders(lastName, facilityId,_db, false)
                       let ships = (from s in _db.SHIPPING_DETAILs where (s.RECEIVED_DATE != null || s.STD_SHIPMENTSTS_ID != null) select s.WKF_CASE_ACTIVITY_ID).Distinct().ToArray()
                       where !(ships.Contains(t.WKF_CASE_ACTIVITY_ID))
                       //orderby t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME, t.CITY, t.CONTACT_NAME, t.WKF_CASE.STD_WKFCASETYPE.CODE descending
                       //orderby t.WKF_CASE.WKF_CASE_ID descending
                       select t).OrderBy("WKF_CASE_ID").ToList();
           }
       }

       private IQueryable<WKF_CASE_ACTIVITY> GetLinqLabKitsReturned(string lastName, string facilityId, RegistriesDataAccess db)
       {
               IQueryable<WKF_CASE_ACTIVITY> LinqQry = GetLinqInProcessKitOrders(lastName, facilityId, db, true);
               return (from a in LinqQry 
                       where a.STD_WKFACTIVITYSTS_ID == 5
                       select a);
       }

       private IQueryable<WKF_CASE_ACTIVITY> GetLinqLabKitsCanceled(string lastName, string facilityId, RegistriesDataAccess db)
       {
           IQueryable<WKF_CASE_ACTIVITY> LinqQry = GetLinqInProcessKitOrders(lastName, facilityId, db, true);
           return  (from a in LinqQry
                   where a.STD_WKFACTIVITYSTS_ID == 3
                   select a);

       }
        /// <summary>
        /// Gets all the lab kits returned from patients.
        /// </summary>
       /// <returns>WKF_CASE_ACTIVITY</returns>
       public IEnumerable<LabInProcessKitDetails> GetLabKitsReturned(string lastName, string facilityId, string sort, int maxRows, int startRow)
       {
           if (string.IsNullOrEmpty(sort)) sort = QueueDefaultSort;
           using (_db = GetDataContext())
           {
               IEnumerable<LabInProcessKitDetails> LabKitsReturned = 
                   (from t in GetLinqLabKitsReturned(lastName, facilityId, _db)
                        select new LabInProcessKitDetails
                        {
                            ActivityId = t.WKF_CASE_ACTIVITY_ID,
                            WkfCaseId = t.WKF_CASE_ID,
                            FromStock = t.KIT_USE_STOCK_FLAG != null ? (bool)t.KIT_USE_STOCK_FLAG : false,
                            PatientName = t.WKF_CASE.REFERRAL.PATIENT.FullName,
                            InstName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME,
                            StationNumber = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                            City = t.CITY,
                            VAMCContactNm = t.CONTACT_NAME,
                            KitType = t.WKF_CASE.STD_WKFCASETYPE.CODE,
                            //KitOrderdDate = t.CREATED,
                            KitOrderedDate = t.KIT_ORDER_DATE, 
                            ReceivedDate = t.RECEIVED_DATE,
                            Status = t.STD_WKFACTIVITYST.NAME
                        }).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

               return LabKitsReturned; 
           }
       }

       public int GetLabKitsReturnedCount(string lastName, string facilityId)
       {
           using (_db = GetDataContext())
           {
               return (from t in GetLinqLabKitsReturned(lastName, facilityId, _db)
                                 select t).Count();
           }
       }

       /// <summary>
       /// Gets all the lab kits cncelled by patients.
       /// </summary>
       /// <returns>WKF_CASE_ACTIVITY</returns>
       public IEnumerable<LabInProcessKitDetails> GetLabKitsCanceled(string lastName, string facilityId, string sort, int maxRows, int startRow)
       {
           if (string.IsNullOrEmpty(sort)) sort = QueueDefaultSort;
           using (_db = GetDataContext())
           {
               IEnumerable<LabInProcessKitDetails> LabKitsCanceled = 
                   (from t in GetLinqLabKitsCanceled(lastName, facilityId, _db)
                         select new LabInProcessKitDetails
                         {
                             ActivityId = t.WKF_CASE_ACTIVITY_ID,
                             WkfCaseId = t.WKF_CASE_ID,
                             FromStock = t.KIT_USE_STOCK_FLAG != null ? (bool)t.KIT_USE_STOCK_FLAG : false,
                             PatientName = t.WKF_CASE.REFERRAL.PATIENT.FullName,
                             InstName = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME,
                             StationNumber = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                             City = t.CITY,
                             VAMCContactNm = t.CONTACT_NAME,
                             KitType = t.WKF_CASE.STD_WKFCASETYPE.CODE,
                             //KitOrderdDate = t.CREATED,
                             KitOrderedDate = t.KIT_ORDER_DATE, 
                             ReceivedDate = t.RECEIVED_DATE,
                             Status = t.STD_WKFACTIVITYST.NAME
                         }).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

               return LabKitsCanceled; 
           }
       }

       public int GetLabKitsCanceledCount(string lastName, string facilityId)
       {
           using (_db = GetDataContext())
           {
               return (from t in GetLinqLabKitsCanceled(lastName, facilityId, _db)
                                 select t).Count();
           }
       }

       private void SetLoadWithForInitializeOtherActivities(RegistriesDataAccess db, int activityId)
       {
           DataLoadOptions lo = new DataLoadOptions();

           lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
           lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
           lo.LoadWith<REFERRAL>(e => e.PATIENT);
           lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
           lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
           lo.AssociateWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies.Where(a => a.WKF_CASE_ACTIVITY_ID == activityId));
           lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
           lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
           lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SHIPPING_DETAILs);
           lo.AssociateWith<WKF_CASE_ACTIVITY>(e => e.SHIPPING_DETAILs.Where(s=>s.WKF_CASE_ACTIVITY_ID==activityId));
           lo.LoadWith<SHIPPING_DETAIL>(e => e.STD_SHIPMENTST);
           db.LoadOptions = lo;
           db.DeferredLoadingEnabled = false;

       }
        
        /// <summary>
        /// Gets the Kit order and the obtain (returned) kit details (if any)
        /// </summary>
        /// <param name="activityId"></param>
       /// <returns>WKF_CASE_ACTIVITY</returns>
       public WKF_CASE KitOrderDetails(int activityId)
       {
           using (_db = GetDataContext())
           {
               SetLoadWithForInitializeOtherActivities(_db, activityId);
               //return _db.WKF_CASEs.FirstOrDefault();
               //Adding where criteria in the first query    
               return     (from t in _db.WKF_CASEs join activity in _db.WKF_CASE_ACTIVITies on t.WKF_CASE_ID equals activity.WKF_CASE_ID
                                               where activity.WKF_CASE_ACTIVITY_ID == activityId select t).FirstOrDefault();
           }
       }

       #endregion

       #region Select by Id
       public STD_WKFACTIVITYST GetWorkflowStatus(int workFlowId)
       {
           using (_db = GetDataContext())
           {
               DataLoadOptions lo = new DataLoadOptions();
               lo.LoadWith<WKF_CASE_ACTIVITY>(e=>e.STD_WKFACTIVITYST);
               _db.LoadOptions = lo;
               return (_db.WKF_CASE_ACTIVITies.Where(e=>e.WKF_CASE_ACTIVITY_ID == workFlowId).Select(t=>t.STD_WKFACTIVITYST).FirstOrDefault());
           }
       }

       public bool ActivityExists(int wkfId, WorkFlowActivityTypes type)
       {
           using (_db = GetDataContext())
           {
               //return (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ID == wkfId && e.STD_WKFACTIVITYTYPE_ID== Convert.ToInt16(type)).Select(t => t.WKF_CASE_ACTIVITY_ID).Any());
               return (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ID == wkfId && e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(type)).Count() > 0);
           }
       }

       public void SimulateLabResults(int caseId)
       {
           using (_db = GetDataContext())
           {
               _db.sp_LoadBioLabResults(caseId);
           }

       }
       #endregion
    }

    public class WorkflowCaseActivityComparer : IEqualityComparer<LabInProcessKitDetails >
    {
        public bool Equals(LabInProcessKitDetails x, LabInProcessKitDetails y)
        {
            return (x.ActivityId == y.ActivityId);

        }

        public int GetHashCode(LabInProcessKitDetails obj)
        {

            return obj.ActivityId.GetHashCode();

        }
    } 

    #region Kit order objects
    public class LabKitOrderGroup
    {
        public int? InstitutionId { get; set; }
        public string Facility { get; set; }
        public string StationNumber { get; set; }
        public string KitTypeCd { get; set; }
        public int OrderType { get; set; } //0 - for New, 1 - for Completed
        public int KitTypeID { get; set; }
        public int Count { get; set; }
        public List<string> Patient { get; set; }
        //public IEnumerable<int> WkfCases { get; set; }
    }
    public class LabKitOrderPatientsBO
    {
        public int WkfCaseId { get; set; }
        public bool FromStock { get; set; }
        public bool Ordered { get; set; }
        public string KitType { get; set; }
        public string PatientName{ get; set; }
        public string AddressLine1{ get; set; }
        public string AddressLine2{ get; set; }
        public string AddressLine3 { get; set; }
        public string City { get; set; }
        public string County { get; set; }
        public string State { get; set; }
        public string Country { get; set; }
        public string PostalCode { get; set; }
        public string ZipPlus4 { get; set; }
        public string TEFSCRep { get; set; }
        //public DateTime? KitOrderdDate { get; set; }
        public DateTime? KitOrderedDate { get; set; }
        public string VAMCContactNm { get; set; }
        public string VAMCContactPh { get; set; }
        public string InstName { get; set; }
        public string StationNumber { get; set; }
    }

    /// <summary>
    /// Used to store ordered/received kits information
    /// </summary>
    public class LabKitOrderedAndReceivedBO : LabKitOrderPatientsBO
    {
        public string ReceivingFacility { get; set; }
        public string Status { get; set; }
        public DateTime? ReceivedDate { get; set; }
    }

    public class LabInProcessKitDetails : LabKitOrderedAndReceivedBO
    {
        public int ActivityId { get; set; }
    }
    #endregion
}
