﻿using System.Collections.Specialized;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;

namespace CRS_EFR
{
    internal static class WebRequestor
    {
        static WebRequestor()
        {
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Ssl3
                | SecurityProtocolType.Tls
                | SecurityProtocolType.Tls11
                | SecurityProtocolType.Tls12;
        }

        public static string Get(
            string requestUrl,
            string userId,
            string accept = WebConstants.JsonMimeType)
        {
            var request = WebRequestorCore.CreateRequest(requestUrl);

            request.KeepAlive = true;
            request.Method = "GET";
            request.Accept = accept;

            return WebRequestorCore.GetResponse(request, userId);
        }

        public static string Get(
            string requestUrl,
            X509Certificate2 certificate,
            string userId,
            NameValueCollection headersCollection = null,
            string accept = WebConstants.JsonMimeType)
        {
            var request = WebRequestorCore.CreateRequest(requestUrl);

            request.KeepAlive = true;
            request.Method = "GET";
            request.Accept = accept;
            request.ServerCertificateValidationCallback += (sender, cert, chain, sslPolicyErrors) =>
            {
                return sslPolicyErrors == SslPolicyErrors.None;
            };

            if (certificate != null)
                request.ClientCertificates.Add(certificate);

            if (headersCollection != null && headersCollection.HasKeys())
                request.Headers.Add(headersCollection);

            return WebRequestorCore.GetResponse(request, userId);
        }

        public static string Post(
            string requestUrl,
            byte[] data,
            string userId,
            NameValueCollection headersCollection = null,
            string contentType = WebConstants.JsonMimeType)
        {
            return WriteRequest(
                requestUrl,
                HttpRequestType.Post,
                data,
                contentType,
                userId,
                headersCollection);
        }

        public static string Put(
            string requestUrl,
            byte[] data,
            string userId,
            NameValueCollection headersCollection = null,
            string contentType = WebConstants.JsonMimeType)
        {
            return WriteRequest(
                requestUrl,
                HttpRequestType.Put,
                data,
                contentType,
                userId,
                headersCollection);
        }

        public static string Delete(
            string requestUrl,
            string userId,
            NameValueCollection headersCollection = null)
        {
            return WriteRequest(
                requestUrl,
                HttpRequestType.Delete,
                null,
                null,
                userId,
                headersCollection);
        }

        private static string WriteRequest(
            string requestUrl,
            HttpRequestType type,
            byte[] data,
            string contentType,
            string userId,
            NameValueCollection headersCollection = null)
        {
            var request = WebRequestorCore.CreateRequest(requestUrl);

            request.KeepAlive = true;
            request.Method = type.ToString();

            if (contentType != null)
                request.ContentType = contentType;

            if (headersCollection != null && headersCollection.HasKeys())
                request.Headers.Add(headersCollection);

            if (data != null)
            {
                request.ContentLength = data.Length;

                using (var dataStream = request.GetRequestStream())
                {
                    dataStream.Write(data, 0, data.Length);
                }

                return WebRequestorCore.GetResponse(request, userId, data);
            }

            return WebRequestorCore.GetResponse(request, userId);
        }
    }
}