﻿using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading;
using ThreadingTasks = System.Threading.Tasks;

using Newtonsoft.Json;

namespace CRS_EFR
{
    public class StudyService
    {
        private AuthService _authService;
        private AuthService Auth
        {
            get
            {
                if (_authService == null)
                    _authService = new AuthService();

                return _authService;
            }
        }

        private PatientImageRepository _patientImageRepository;
        private PatientImageRepository ImageRepository
        {
            get
            {
                if (_patientImageRepository == null)
                    _patientImageRepository = new PatientImageRepository();

                return _patientImageRepository;
            }
        }

        private VistaImagingConfigurationService _vistaImagingConfigurationService;
        private VistaImagingConfigurationService ConfigurationService
        {
            get
            {
                if (_vistaImagingConfigurationService == null)
                    _vistaImagingConfigurationService = new VistaImagingConfigurationService();

                return _vistaImagingConfigurationService;
            }
        }

        public StudyService()
        {
            _patientImageRepository = new PatientImageRepository();
        }

        public StudyQueryResponse GetStudiesForPatient(string patientIcn, string userId)
        {
            StudyQueryResponse validatedStudyQueryResponse = null;

            var accessToken = Auth.GetToken(userId);

            var headers = AuthService.GetAuthRequestHeaders(accessToken.Value);

            var requestBytes = GetRequestAsBytes(patientIcn);

            try
            {
                var response = WebRequestor.Post(
                    ConfigurationService.GetConfiguration().StudyQueryEndpoint,
                    requestBytes,
                    userId,
                    headers);

                validatedStudyQueryResponse = JsonSchemaValidator.DeserializeAndValidate<StudyQueryResponse>(response);
                if (validatedStudyQueryResponse != null)
                    validatedStudyQueryResponse.studies = new ImageStudies(validatedStudyQueryResponse.studies).EncodedImageStudies;
            }
            catch (WebException)
            {
                #region Note

                // NOTE:    As of this release VistA Imaging will throw an exception (500 error) if a patient is not found.
                //          They should return a 404.  Because this is a common occurance, we don't log the details here or 
                //          re-throw.  If the situation changes in the future, revise accordingly.
                //          HttpWebRequestExtensions.TryGetResponse will log any WebExceptions.

                #endregion
            }

            return validatedStudyQueryResponse;
        }

        public IEnumerable<PatientStudyCount> CheckPatientsForImages(
                IEnumerable<string> patientIcns,
                string userId)
        {
            var accessToken = Auth.GetToken(userId);

            using (var headers = new ThreadLocal<NameValueCollection>(() => AuthService.GetAuthRequestHeaders(accessToken.Value)))
            {
                var stack = new ConcurrentStack<PatientStudyCount>();

                ThreadingTasks.Parallel.ForEach(patientIcns, (patientIcn) =>
                {
                    try
                    {
                        var request = new StudyQueryRequest { patientICN = patientIcn };
                        var json = JsonConvert.SerializeObject(request);
                        var requestBytes = Encoding.UTF8.GetBytes(json);

                        var response = WebRequestor.Post(
                            ConfigurationService.GetConfiguration().StudyQueryEndpoint,
                            requestBytes,
                            userId,
                            headers.Value);

                        var validatedStudyQueryResponse = JsonSchemaValidator.DeserializeAndValidate<StudyQueryResponse>(response);

                        var result = new PatientStudyCount
                        {
                            PatientIcn = patientIcn,
                            StudyCount = new ImageStudies(validatedStudyQueryResponse.studies).ImageStudyCount
                        };

                        stack.Push(result);
                    }
                    catch (WebException)
                    {
                    #region Note

                    // NOTE:    As of this release VistA Imaging will throw an exception (500 error) if a patient is not found.
                    //          They should return a 404.  Because this is a common occurance, we don't log the details here or 
                    //          re-throw.  If the situation changes in the future, revise accordingly.
                    //          HttpWebRequestExtensions.TryGetResponse will log any WebExceptions.

                    #endregion
                }
                });

                return stack.ToList();
            }
        }

        private byte[] GetRequestAsBytes(string patientIcn)
        {
            var request = new StudyQueryRequest { patientICN = patientIcn };

            var json = JsonConvert.SerializeObject(request);

            return Encoding.UTF8.GetBytes(json);
        }

        public void InsertUpdateStudies(IEnumerable<Study> studies)
        {
            ImageRepository.InsertUpdateStudies(studies);
        }

    }
}
