﻿using System;
using System.Configuration;
using System.Security.Cryptography.X509Certificates;

namespace CRS_EFR
{
    internal class CertificateService
    {
        private const string APP_SETTINGS_CERTIFICATE_STORE = "CertificateStore";
        private const string APP_SETTINGS_CERTIFICATE_STORE_FOLDER = "CertificateStoreFolder";
        private const string APP_SETTINGS_CERTIFICATE_THUMBPRINT_SOURCE = "CertificateThumbprintSource";
        private const string APP_SETTINGS_CERTIFICATE_THUMBPRINT = "CertificateThumbprint";

        private const string CERTIFICATE_STORE_FOLDER = "Folder";
        private const string CERTIFICATE_STORE_LOCAL_MACHINE = "LocalMachine";

        private const string THUMBPRINT_SOURCE_DB = "Database";
        private const string THUMBPRINT_SOURCE_CONFIG = "Web.Config";

        private readonly object _locker = new object();

        private VistaImagingConfigurationService _vistaImagingConfigurationService;
        private VistaImagingConfigurationService ConfigurationService
        {
            get
            {
                if (_vistaImagingConfigurationService == null)
                    _vistaImagingConfigurationService = new VistaImagingConfigurationService();

                return _vistaImagingConfigurationService;
            }
        }

        private X509Certificate2 _certificate;
        internal X509Certificate2 Certificate
        {
            get
            {
                if (_certificate == null)
                {
                    lock (_locker)
                    {
                        _certificate = GetCertificateFromStore();
                    }
                }

                return _certificate;
            }
        }

        private X509Certificate2 GetCertificateFromStore()
        {
            X509Certificate2 result = null;

            var certificateStore = ConfigurationManager.AppSettings[APP_SETTINGS_CERTIFICATE_STORE];
            if (!string.IsNullOrEmpty(certificateStore))
            {
                if (certificateStore.Equals(CERTIFICATE_STORE_LOCAL_MACHINE, StringComparison.OrdinalIgnoreCase))
                    result = GetCertificateFromLocalMachine();

                else if (certificateStore.Equals(CERTIFICATE_STORE_FOLDER, StringComparison.OrdinalIgnoreCase))
                    result = GetCertificateFromFolder();
            }

            if (result == null)
                throw new Exception("Cannot get Certificate: Unknown Store or Not Found.");

            return result;
        }

        private X509Certificate2 GetCertificateFromLocalMachine()
        {
            X509Certificate2 result = null;

            var store = new X509Store(StoreLocation.LocalMachine);
            try
            {
                store.Open(OpenFlags.ReadOnly);

                X509Certificate2Collection certCollection = store.Certificates;

                var certificateThumbprint = GetCertificateThumbprint();

                var tryFindCert = certCollection.Find(X509FindType.FindByThumbprint, ConfigurationService.GetConfiguration().CertificateThumbprint, true);
                if (tryFindCert.Count > 0)
                    result = tryFindCert[0];

            }
            finally
            {
                store.Close();
            }

            return result;
        }

        private X509Certificate2 GetCertificateFromFolder()
        {
            X509Certificate2 result = null;

            var certificateStoreFolder = ConfigurationManager.AppSettings[APP_SETTINGS_CERTIFICATE_STORE_FOLDER];
            if (!string.IsNullOrEmpty(certificateStoreFolder))
                result = new X509Certificate2(certificateStoreFolder);

            return result;
        }

        private string GetCertificateThumbprint()
        {
            string result = null;

            var thumbprintSource = ConfigurationManager.AppSettings[APP_SETTINGS_CERTIFICATE_THUMBPRINT_SOURCE];
            if (!string.IsNullOrEmpty(thumbprintSource))
            {
                if (thumbprintSource.Equals(THUMBPRINT_SOURCE_DB, StringComparison.OrdinalIgnoreCase))
                    result = ConfigurationService.GetConfiguration().CertificateThumbprint;

                else if (thumbprintSource.Equals(THUMBPRINT_SOURCE_CONFIG, StringComparison.OrdinalIgnoreCase))
                    result = ConfigurationManager.AppSettings[APP_SETTINGS_CERTIFICATE_THUMBPRINT];
            }

            if (string.IsNullOrEmpty(result))
                throw new Exception("Cannot get Certificate Thumbprint: Unknown Source.");

            return result;
        }
    }
}