﻿using System;
using System.Collections.Specialized;
using VeteransAffairs.Registries.Business.Utilities;

namespace CRS_EFR
{
    internal class AuthService
    {
        private const int CACHE_TTL_HOURS = 4;

        private const string CACHE_KEY_NAMESPACE = "VeteransAffairs.Registries.BusinessManager";
        private const string CACHE_KEY_CLASS = "AuthService";
        private const string ALL_COLLECTIONS_KEY = "AllAccessTokens";

        private static readonly string _allCollectionsKey;

        private VistaImagingConfigurationService _vistaImagingConfigurationService;
        private VistaImagingConfigurationService ConfigurationService
        {
            get
            {
                if (_vistaImagingConfigurationService == null)
                    _vistaImagingConfigurationService = new VistaImagingConfigurationService();

                return _vistaImagingConfigurationService;
            }
        }

        private CertificateService _certificateService;
        private CertificateService CertService
        {
            get
            {
                if (_certificateService == null)
                    _certificateService = new CertificateService();

                return _certificateService;
            }
        }

        private ApiTokenValidator _tokenValidator;
        private ApiTokenValidator TokenValidator
        {
            get
            {
                if (_tokenValidator == null)
                    _tokenValidator = new ApiTokenValidator();

                return _tokenValidator;
            }
        }

        static AuthService()
        {
            _allCollectionsKey = new CacheKey(CACHE_KEY_NAMESPACE, CACHE_KEY_CLASS, ALL_COLLECTIONS_KEY).Build();

            RemoveAllCacheKeys();
        }

        internal AccessToken GetToken(string userId)
        {
            var cacheKey = GetCacheKey(userId);
            var cachedValue = RefreshToken(cacheKey, userId);
            if (DateTime.UtcNow >= cachedValue.CacheDateTime.AddHours(CACHE_TTL_HOURS)
                || DateTime.UtcNow.Day != cachedValue.CacheDateTime.Day)
            {
                CacheHelper.Clear(cacheKey);
                cachedValue = RefreshToken(cacheKey, userId);
            }

            return cachedValue;
        }

        private AccessToken RefreshToken(
            string cacheKey,
            string userId)
        {
            return CacheHelper.GetAdd(cacheKey, () =>
            {
                try
                {
                    var tokenResponse = WebRequestor.Get(
                            ConfigurationService.GetConfiguration().TokenEndpoint,
                            CertService.Certificate,
                            userId,
                            GetTokenRequestHeaders(userId));

                    var token = TokenValidator.ValidateAndReturnToken(tokenResponse);

                    CacheHelper.AddKeyToCollection(_allCollectionsKey, userId);

                    return new AccessToken(token);
                }
                catch (Exception e)
                {
                    Logging.WriteLog(
                        userId,
                        "AuthService.RefreshToken",
                        e);

                    throw e;
                }
            });
        }

        internal static NameValueCollection GetAuthRequestHeaders(string accessToken)
        {
            return new NameValueCollection
            {
                { "apiToken", accessToken }
            };
        }

        private static NameValueCollection GetTokenRequestHeaders(string userId)
        {
            string userHeaderValue = null;

            if (!string.IsNullOrEmpty(userId))
            {
                var userIdComponents = userId.Split(new char[] { '\\' });
                if (userIdComponents.Length == 2)
                    userHeaderValue = userIdComponents[1];
                else
                    userHeaderValue = userId;
            }

            return new NameValueCollection
        {
            { "userId", userHeaderValue }
        };
        }

        private static string GetCacheKey(string userId)
        {
            return new CacheKey(CACHE_KEY_NAMESPACE, CACHE_KEY_CLASS, "GetToken")
                .AddPart(userId)
                .Build();
        }

        private static void RemoveAllCacheKeys()
        {
            CacheHelper.RemoveCollection(_allCollectionsKey);
        }
    }

}

