﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Text;

namespace CRS_EFR
{
    internal class PatientImageRepository
    {
        private const int TOKEN_REFRESH_TIME_HOURS = 4;

        private const string CONNECT_STRING_NAME = "RegistryConnectionString";

        private const string SP_GET_IMAGE_ANALYSIS = "EFR.usp_GetImageAnalysisForPatient";
        private const string SP_GET_IMAGES_FOR_PATIENT = "EFR.usp_GetImagesForPatient";
        private const string SP_GET_PATIENT_IMAGE_BODY_PARTS = "EFR.usp_GetPatientImageBodyParts";
        private const string SP_GET_PATIENT_IMAGE_SOURCE_REFRESH_DATE = "EFR.usp_GetPatientImageSourceRefreshDate";
        private const string SP_GET_PATIENTS_IMAGE_STUDIES_FLAGS = "EFR.usp_GetPatientsImageStudiesFlags";
        private const string SP_GET_STANDARD_IMAGE_FRAGMENT_SIZES = "EFR.usp_GetStandardImageFragmentSizes";

        private const string SP_INSERT_PATIENT_IMAGES = "EFR.usp_InsertPatientImages";
        private const string SP_INSERT_UPDATE_IMAGE_ANALYSIS = "EFR.usp_InsertUpdateImageAnalysisForPatient";
        private const string SP_INSERT_UPDATE_IMAGE_ANALYSIS_PATIENT_STUDIES_FLAG = "EFR.usp_InsertUpdateImageAnalysisPatientStudiesFlags";

        private const string SP_IMAGE_ANALYSIS_HISTORY_EXISTS = "EFR.usp_ImageAnalysisHistoryExists";

        private const string SP_DELETE_IMAGE_ANALYSIS_FOR_PATIENT = "EFR.usp_DeleteImageAnalysisForPatient";

        internal static IEnumerable<string> LoadReferenceImageFragmentSizes()
        {
            return ImagingDataRepository.LoadReferenceValuesNameWithDescription(SP_GET_STANDARD_IMAGE_FRAGMENT_SIZES);
        }

        internal PatientImageAnalysis GetPatientImageAnalysis(string patientIcn)
        {
            using (var imageAnalysisTable = SqlProvider.ExecuteSPDataTable(
                CONNECT_STRING_NAME,
                SP_GET_IMAGE_ANALYSIS,
                new[] { patientIcn }))
            {
                if (imageAnalysisTable != null
                    && imageAnalysisTable.Rows != null
                    && imageAnalysisTable.Rows.Count > 0)
                {
                    var imageAnalysisRow = imageAnalysisTable.Rows[0];

                    var result = new PatientImageAnalysis
                    (
                        imageAnalysisRow.Field<bool?>("IMAGES_IN_CPRS_FLAG"),
                        imageAnalysisRow.Field<bool?>("FRAGMENT_VERIFIED_FLAG"),
                        imageAnalysisRow.Field<bool?>("BONE_FRACTURE_DOCUMENTED_FLAG"),
                        imageAnalysisRow.Field<bool?>("FRAGMENTS_IN_JOIN_SPACE_FLAG"),
                        imageAnalysisRow.Field<int?>("STD_IMAGE_FRAGMENT_SIZE_ID"),
                        imageAnalysisRow.Field<string>("JOINT_SPACE"),
                        imageAnalysisRow.Field<string>("OTHER_OBSERVATIONS"),
                        imageAnalysisRow.Field<DateTime?>("DATE_REVIEWED")
                    );

                    return result;
                }
            }

            return null;
        }

        internal string GetBodyPartsImaged(int patientImageId)
        {
            using (var imagedBodyPartsTable = SqlProvider.ExecuteSPDataTable(
                CONNECT_STRING_NAME,
                SP_GET_PATIENT_IMAGE_BODY_PARTS,
                new[] { string.Format("{0}", patientImageId) }))
            {
                if (imagedBodyPartsTable != null
                    && imagedBodyPartsTable.Rows != null
                    && imagedBodyPartsTable.Rows.Count > 0)
                {
                    StringBuilder builder = new StringBuilder();

                    foreach (DataRow row in imagedBodyPartsTable.Rows)
                    {
                        if (builder.Length > 0) builder.Append(", ");

                        builder.Append(row.Field<string>("DESCRIPTION_TEXT"));
                    }

                    return builder.ToString();
                }
            }

            return null;
        }

        internal void InsertUpdateStudies(IEnumerable<Study> studies)
        {
            if (studies != null && studies.Any())
            {
                var connectionString = ConfigurationManager.ConnectionStrings[CONNECT_STRING_NAME].ConnectionString;

                using (SqlConnection connection = new SqlConnection(connectionString))
                using (SqlCommand updateCommand = new SqlCommand(SP_INSERT_PATIENT_IMAGES, connection))
                {
                    connection.Open();

                    updateCommand.CommandType = CommandType.StoredProcedure;

                    SqlParameter tvpStudies = updateCommand.Parameters.AddWithValue("@Studies",
                        GetInsertUpdateStudiesParameterTable(studies));
                    tvpStudies.SqlDbType = SqlDbType.Structured;

                    updateCommand.ExecuteNonQuery();
                }
            }
        }

        internal static DataTable GetInsertUpdateStudiesParameterTable(IEnumerable<Study> collection)
        {
            var parameters = new DataTable();
            parameters.Columns.Add("STUDY_ID");
            parameters.Columns.Add("PATIENT_ICN");
            parameters.Columns.Add("IMAGE_COUNT");
            parameters.Columns.Add("IS_SENSITIVE");
            parameters.Columns.Add("STUDY_CLASS");
            parameters.Columns.Add("STUDY_TYPE");
            parameters.Columns.Add("STUDY_DESCRIPTION");
            parameters.Columns.Add("PROCEDURE_DESCRIPTION");
            parameters.Columns.Add("SPECIALTY_DESCRIPTION");
            parameters.Columns.Add("DETAILS_URL");
            parameters.Columns.Add("REPORT_URL");
            parameters.Columns.Add("MANAGE_URL");
            parameters.Columns.Add("VIEWER_URL");
            parameters.Columns.Add("THUMBNAIL_URL");
            parameters.Columns.Add("STUDY_DATE");
            parameters.Columns.Add("ACQUISITION_DATE");

            foreach (var study in collection)
            {
                DataRow row = parameters.Rows.Add();
                row.SetField("STUDY_ID", study.studyId);
                row.SetField("PATIENT_ICN", study.patientICN);
                row.SetField("IMAGE_COUNT", study.imageCount);
                row.SetField("IS_SENSITIVE", study.isSensitive ? 1 : 0);
                row.SetField("STUDY_CLASS", study.studyClass);
                row.SetField("STUDY_TYPE", study.studyType);
                row.SetField("STUDY_DESCRIPTION", study.studyDescription);
                row.SetField("PROCEDURE_DESCRIPTION", study.procedureDescription);
                row.SetField("SPECIALTY_DESCRIPTION", study.specialtyDescription);
                row.SetField("DETAILS_URL", study.detailsUrl);
                row.SetField("REPORT_URL", study.reportUrl);
                row.SetField("MANAGE_URL", study.manageUrl);
                row.SetField("VIEWER_URL", study.viewerUrl);
                row.SetField("THUMBNAIL_URL", study.thumbnailUrl);
                row.SetField("STUDY_DATE", study.studyDate != null ? DateTime.Parse(study.studyDate) : (DateTime?)null);
                row.SetField("ACQUISITION_DATE", study.acquisitionDate != null ? DateTime.Parse(study.acquisitionDate) : (DateTime?)null);
            }

            return parameters;
        }

        internal int DeleteImageAnalysis(string patientIcn)
        {
            return SqlProvider.PerformUpdate(
                CONNECT_STRING_NAME,
                SP_DELETE_IMAGE_ANALYSIS_FOR_PATIENT,
                new[] { string.Format("{0}", patientIcn) });
        }

        internal int InsertUpdateImageAnalysis(object[] parameters)
        {
            return SqlProvider.PerformUpdate(
                CONNECT_STRING_NAME,
                SP_INSERT_UPDATE_IMAGE_ANALYSIS,
                parameters);
        }

        internal bool ImageAnalysisHistoryExists(string patientIcn)
        {
            using (var checkTable = SqlProvider.ExecuteSPDataTable(
                CONNECT_STRING_NAME,
                SP_IMAGE_ANALYSIS_HISTORY_EXISTS,
                new[] { string.Format("{0}", patientIcn) }))
            {
                if (checkTable != null
                    && checkTable.Rows != null
                    && checkTable.Rows.Count > 0)
                {
                    return checkTable.Rows[0].Field<bool>("ExistsFlag");
                }
            }

            return false;
        }

        internal DataTable GetPatientImagesDataTable(string patientIcn)
        {
            var result = SqlProvider.ExecuteSPDataTable(
                CONNECT_STRING_NAME,
                SP_GET_IMAGES_FOR_PATIENT,
                new[] { string.Format("{0}", patientIcn) });

            return result;
        }

        internal bool ShouldRefreshImageData(int patientId)
        {
            DateTime? refreshDate = null;

            using (var checkTable = SqlProvider.ExecuteSPDataTable(
                CONNECT_STRING_NAME,
                SP_GET_PATIENT_IMAGE_SOURCE_REFRESH_DATE,
                new[] { string.Format("{0}", patientId) }))
            {
                if (checkTable != null
                    && checkTable.Rows != null
                    && checkTable.Rows.Count > 0)
                {
                    refreshDate = checkTable.Rows[0].Field<DateTime?>("SourceRefreshDate");
                }
            }

            return !refreshDate.HasValue
                || refreshDate.Value.Date != DateTime.Now.Date
                || refreshDate.Value <= DateTime.Now.AddHours(-TOKEN_REFRESH_TIME_HOURS);
        }

        internal Dictionary<string, bool?> GetImagesValidatedPatientIcns(IEnumerable<string> patientIcns)
        {
            Dictionary<string, bool?> result = new Dictionary<string, bool?>();

            var connectionString = ConfigurationManager.ConnectionStrings[CONNECT_STRING_NAME].ConnectionString;

            using (SqlConnection connection = new SqlConnection(connectionString))
            using (SqlCommand cmd = new SqlCommand(SP_GET_PATIENTS_IMAGE_STUDIES_FLAGS, connection))
            {
                connection.Open();

                cmd.CommandType = CommandType.StoredProcedure;

                if (patientIcns != null && patientIcns.Any())
                {
                    SqlParameter tvpPatientIcns = cmd.Parameters.AddWithValue("@PatientIcns", CreateStringTable(patientIcns));
                    tvpPatientIcns.SqlDbType = SqlDbType.Structured;
                    tvpPatientIcns.TypeName = "dbo.StringTable";
                }

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var isNull = reader.IsDBNull(1);
                        bool? hasImages = isNull ? (bool?)null : reader.GetFieldValue<bool?>(1);
                        result.Add(reader.GetFieldValue<string>(0), hasImages);
                    }

                }
            }

            return result;
        }

        private static DataTable CreateStringTable(IEnumerable<string> values)
        {
            var dt = new DataTable("Items");
            dt.Columns.Add("value", typeof(string));

            foreach (var value in values)
                dt.Rows.Add(value.ToString());

            return dt;
        }

        internal void InsertUpdateImageAnalysisPatientStudiesFlags(IEnumerable<PatientStudyCount> patientStudyCounts)
        {
            if (patientStudyCounts != null && patientStudyCounts.Any())
            {
                var connectionString = ConfigurationManager.ConnectionStrings[CONNECT_STRING_NAME].ConnectionString;

                using (SqlConnection connection = new SqlConnection(connectionString))
                using (SqlCommand updateCommand = new SqlCommand(SP_INSERT_UPDATE_IMAGE_ANALYSIS_PATIENT_STUDIES_FLAG, connection))
                {
                    connection.Open();

                    updateCommand.CommandType = CommandType.StoredProcedure;

                    SqlParameter tvpStudies = updateCommand.Parameters.AddWithValue("@PatientStudiesFlags",
                        GetInsertUpdateImageAnalysisPatientStudiesFlagsParameterTable(patientStudyCounts));
                    tvpStudies.SqlDbType = SqlDbType.Structured;

                    updateCommand.ExecuteNonQuery();
                }
            }
        }

        internal static DataTable GetInsertUpdateImageAnalysisPatientStudiesFlagsParameterTable(
            IEnumerable<PatientStudyCount> patientStudyCounts)
        {
            var dt = new DataTable("Items");
            dt.Columns.Add("key", typeof(string));
            dt.Columns.Add("value", typeof(bool));

            foreach (var patientStudyCount in patientStudyCounts)
            {
                DataRow row = dt.Rows.Add();
                row.SetField("key", patientStudyCount.PatientIcn);
                row.SetField("value", patientStudyCount.StudyCount.HasValue && patientStudyCount.StudyCount.Value > 0);
            }

            return dt;
        }
    }

}

