﻿using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;

namespace CRS_EFR
{
    public class PatientImageDetailRepository
    {
        private const string CONNECT_STRING_NAME = "RegistryConnectionString";
        private const string SELECT_VALUE_MESSAGE = "(Select a Value)";

        private const string SP_GET_PATIENT_IMAGE_DETAIL = "EFR.usp_GetPatientImageDetail";
        private const string SP_GET_PATIENT_IMAGE_BODY_PARTS = "EFR.usp_GetPatientImageBodyParts";
        private const string SP_GET_PATIENT_IMAGING_REASONS = "EFR.usp_GetPatientImagingReasons";

        private const string SP_GET_STD_IMAGE_BODY_PARTS = "EFR.usp_GetStandardImageBodyParts";
        private const string SP_GET_STD_IMAGE_FRAGMENT_COUNT = "EFR.usp_GetStandardFragmentCount";
        private const string SP_GET_STD_IMAGE_REASONS = "EFR.usp_GetStandardImageReasons";
        private const string SP_GET_STD_IMAGE_TYPES = "EFR.usp_GetStandardImageTypes";

        private const string SP_INSERT_UPDATE_PATIENT_IMAGE_REVIEW = "EFR.usp_InsertUpdatePatientImageReview";

        internal static IEnumerable<string> LoadReferenceImageBodyParts()
        {
            return ImagingDataRepository.LoadReferenceValuesDescription(SP_GET_STD_IMAGE_BODY_PARTS);
        }

        internal static IEnumerable<string> LoadReferenceImageFragmentCounts()
        {
            return ImagingDataRepository.LoadReferenceValuesDescription(SP_GET_STD_IMAGE_FRAGMENT_COUNT, SELECT_VALUE_MESSAGE);
        }

        internal static IEnumerable<string> LoadReferenceImageReasons()
        {
            return ImagingDataRepository.LoadReferenceValuesDescription(SP_GET_STD_IMAGE_REASONS);
        }

        internal static IEnumerable<string> LoadReferenceImageTypes()
        {
            return ImagingDataRepository.LoadReferenceValuesDescription(SP_GET_STD_IMAGE_TYPES, SELECT_VALUE_MESSAGE);
        }

        internal PatientImageDetail GetPatientImageDetails(int patientImageId)
        {
            using (var patientImageTable = SqlProvider.ExecuteSPDataTable(
                    CONNECT_STRING_NAME,
                    SP_GET_PATIENT_IMAGE_DETAIL,
                    new[] { string.Format("{0}", patientImageId) }))
            {
                if (patientImageTable != null && patientImageTable.Rows != null && patientImageTable.Rows.Count > 0)
                    return BuildPatientImageDetailFromDataRow(patientImageTable.Rows[0]);
            }

            return null;
        }

        private PatientImageDetail BuildPatientImageDetailFromDataRow(DataRow dr)
        {
            var id = dr.Field<int>("PATIENT_IMAGE_ID");
            var patientId = dr.Field<int>("PATIENT_ID");
            var imageFragmentCountId = dr.Field<int?>("STD_IMAGE_FRAGMENT_COUNT_ID");
            var imageTypeId = dr.Field<int?>("STD_IMAGE_TYPE_ID");

            var largestFragmentLength = dr.Field<decimal?>("LARGEST_FRAGMENT_LENGTH");
            var largestFragmentWidth = dr.Field<decimal?>("LARGEST_FRAGMENT_WIDTH");
            var smallestFragmentLength = dr.Field<decimal?>("SMALLEST_FRAGMENT_LENGTH");
            var smallestFragmentWidth = dr.Field<decimal?>("SMALLEST_FRAGMENT_WIDTH");

            var fragmentVerified = dr.Field<bool?>("FRAGMENT_VERIFIED_FLAG");
            var systemIdentified = dr.Field<bool?>("SYSTEM_IDENTIFIED_FRAGMENT_FLAG");
            var reviewed = dr.Field<bool?>("REVIEWED_FLAG");
            var includeInReport = dr.Field<bool?>("INCLUDE_IN_REPORT_FLAG");

            var studyId = dr.Field<string>("STUDY_ID");
            var studyDescription = dr.Field<string>("STUDY_DESCRIPTION");
            var comments = dr.Field<string>("COMMENTS");
            var imageReasonOtherText = dr.Field<string>("IMAGE_REASON_OTHER_TEXT");
            var imageTypeOtherText = dr.Field<string>("IMAGE_TYPE_OTHER_TEXT");

            return new PatientImageDetail(
                id,
                patientId,
                imageFragmentCountId,
                imageTypeId,
                largestFragmentLength,
                largestFragmentWidth,
                smallestFragmentLength,
                smallestFragmentWidth,
                fragmentVerified,
                systemIdentified,
                reviewed,
                includeInReport,
                studyId,
                studyDescription,
                comments,
                imageReasonOtherText,
                imageTypeOtherText);
        }

        internal void InsertUpdatePatientImageReview(
                object[] parameters,
                IReadOnlyList<int> imageBodyParts,
                IReadOnlyList<int> imagingReasons,
                string userId)
        {
            if (!AuthorizedUser(userId))
                return;

            var connectionString = ConfigurationManager.ConnectionStrings[CONNECT_STRING_NAME].ConnectionString;

            using (SqlConnection connection = new SqlConnection(connectionString))
            using (SqlCommand updateCommand = new SqlCommand(SP_INSERT_UPDATE_PATIENT_IMAGE_REVIEW, connection))
            {
                connection.Open();

                updateCommand.CommandType = CommandType.StoredProcedure;

                updateCommand.Parameters.AddWithValue("@PatientId", parameters[0]);
                updateCommand.Parameters.AddWithValue("@PatientImageId", parameters[1]);
                updateCommand.Parameters.AddWithValue("@StdImageFragmentCountId", parameters[2]);
                updateCommand.Parameters.AddWithValue("@StdImageTypeId", parameters[3]);
                updateCommand.Parameters.AddWithValue("@FragmentVerifiedFlag", parameters[4]);
                updateCommand.Parameters.AddWithValue("@ReviewedFlag", parameters[5]);
                updateCommand.Parameters.AddWithValue("@IncludeInReportFlag", parameters[6]);
                updateCommand.Parameters.AddWithValue("@LargestFragmentLength", parameters[7]);
                updateCommand.Parameters.AddWithValue("@LargestFragmentWidth", parameters[8]);
                updateCommand.Parameters.AddWithValue("@SmallestFragmentLength", parameters[9]);
                updateCommand.Parameters.AddWithValue("@SmallestFragmentWidth", parameters[10]);
                updateCommand.Parameters.AddWithValue("@Comments", parameters[11]);
                updateCommand.Parameters.AddWithValue("@ImageReasonOtherText", parameters[12]);
                updateCommand.Parameters.AddWithValue("@ImageTypeOtherText", parameters[13]);
                updateCommand.Parameters.AddWithValue("@User", parameters[14]);

                if (imageBodyParts != null && imageBodyParts.Any())
                {
                    SqlParameter tvpImageBodyParts = updateCommand.Parameters.AddWithValue("@ImageBodyParts", GetParameterTable(imageBodyParts));
                    tvpImageBodyParts.SqlDbType = SqlDbType.Structured;
                }

                if (imagingReasons != null && imagingReasons.Any())
                {
                    SqlParameter tvpReasons = updateCommand.Parameters.AddWithValue("@ImageReasons", GetParameterTable(imagingReasons));
                    tvpReasons.SqlDbType = SqlDbType.Structured;
                }

                updateCommand.ExecuteNonQuery();
            }
        }

        private DataTable GetParameterTable(IReadOnlyList<int> collection)
        {
            var parameters = new DataTable();
            parameters.Columns.Add("id", typeof(int));

            foreach (var id in collection)
            {
                DataRow row = parameters.Rows.Add();
                row.SetField("id", id + 1);
            }

            return parameters;
        }

        internal PatientImageReviewMap[] GetImageBodyParts(int patientImageId)
        {
            return GetImageMap(patientImageId, SP_GET_PATIENT_IMAGE_BODY_PARTS);
        }

        internal PatientImageReviewMap[] GetImageReasons(int patientImageId)
        {
            return GetImageMap(patientImageId, SP_GET_PATIENT_IMAGING_REASONS);
        }

        internal PatientImageReviewMap[] GetImageMap(int patientImageId, string sproc)
        {
            var map = new List<PatientImageReviewMap>();

            using (var patientImageTable = SqlProvider.ExecuteSPDataTable(
                                CONNECT_STRING_NAME,
                                sproc,
                                new[] { string.Format("{0}", patientImageId) }))
            {
                if (patientImageTable != null && patientImageTable.Rows != null && patientImageTable.Rows.Count > 0)
                {
                    foreach (DataRow row in patientImageTable.Rows)
                        map.Add(GetPatientImageMap(row));
                }
            }

            return map.ToArray();
        }

        private PatientImageReviewMap GetPatientImageMap(DataRow dr)
        {
            var patientImageId = dr.Field<int>("PATIENT_IMAGE_REVIEW_ID");
            var id = dr.Field<int>("ID");
            var description = dr.Field<string>("DESCRIPTION_TEXT");

            return new PatientImageReviewMap(
                patientImageId,
                id,
                description);
        }

        private bool AuthorizedUser(string userId)
        {
            string registry = Helpers.GetConfigEntry("Registry");
            var userAccountManager = new UserAccountManager(userId, registry);

            var roles = userAccountManager.GetRoles();

            if (roles == null || !roles.Any())
                return false;

            return
                roles.Contains((int)RoleCode.EFR_DATA_ENTRY_PERSONNEL)
                ||
                roles.Contains((int)RoleCode.EFR_TEFSC_ADMINISTRATOR)
                ||
                roles.Contains((int)RoleCode.EFR_TEFSC_COORDINATOR)
                ||
                roles.Contains((int)RoleCode.EFR_TEFSC_NURSE)
                ||
                roles.Contains((int)RoleCode.EFR_TEFSC_PROVIDER);
        }
    }
}