﻿using System.Collections.Generic;
using System.Data;
using System.Linq;
using VeteransAffairs.Registries.Business.Utilities;
using VeteransAffairs.Registries.BusinessManager;

namespace CRS_EFR
{
    public class PatientImagingManager : BaseBO
    {
        private const string CACHE_KEY_NAMESPACE = "VeteransAffairs.Registries.BusinessManager";
        private const string CACHE_KEY_CLASS = "PatientImagingManager";
        private const string IMAGING_SERVICES_NOT_AVAILABLE = "VistA Imaging Services are not currently available.  Please try again later.";

        private readonly PatientImageRepository _patientImageRepository;
        private readonly StudyService _studyService;

        private readonly object _locker;

        static PatientImagingManager()
        {
            CacheHelper.Clear(GetCheckForImagesCacheKey());
        }

        public PatientImagingManager()
        {
            _patientImageRepository = new PatientImageRepository();
            _studyService = new StudyService();

            _locker = new object();
        }

        public static string ImagingServicesNotAvailableMessage
        {
            get
            {
                return IMAGING_SERVICES_NOT_AVAILABLE;
            }
        }

        public static IEnumerable<string> LoadReferenceImageFragmentSizes()
        {
            return PatientImageRepository.LoadReferenceImageFragmentSizes();
        }

        public PatientImageAnalysis GetPatientImageAnalysis(string patientIcn)
        {
            return _patientImageRepository.GetPatientImageAnalysis(patientIcn);
        }

        public string GetBodyPartsImaged(int patientImageId)
        {
            return _patientImageRepository.GetBodyPartsImaged(patientImageId);
        }

        public int DeleteImageAnalysis(string patientIcn)
        {
            var eventArgs = new BOSaveSuccessEventArgs { SaveStatusArg = SaveStatus.SaveFail };

            int result = _patientImageRepository.DeleteImageAnalysis(patientIcn);

            eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;

            RaiseSaveEvent(this, eventArgs);

            return result;
        }

        public void InsertUpdateImageAnalysis(object[] parameters)
        {
            var eventArgs = new BOSaveSuccessEventArgs { SaveStatusArg = SaveStatus.SaveFail };

            _patientImageRepository.InsertUpdateImageAnalysis(parameters);

            eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;

            RaiseSaveEvent(this, eventArgs);
        }

        public bool ImageAnalysisHistoryExists(string patientIcn)
        {
            return _patientImageRepository.ImageAnalysisHistoryExists(patientIcn);
        }

        private void ClearCachedImagesForPatient(string patientIcn)
        {
            var key = GetPatientImagesCacheKey(patientIcn);

            CacheHelper.Clear(key);
        }

        private string GetPatientImagesCacheKey(string patientIcn)
        {
            return new CacheKey(CACHE_KEY_NAMESPACE, CACHE_KEY_CLASS, "GetPatientImagesDataTable")
                .AddPart(patientIcn)
                .Build();
        }

        public DataTable GetPatientImagesDataTable(
            string patientIcn,
            int patientId,
            string context)
        {
            lock (_locker)
            {
                var shouldRefresh = _patientImageRepository.ShouldRefreshImageData(patientId);
                if (shouldRefresh)
                    RefreshImageData(patientIcn, context);

                return _patientImageRepository.GetPatientImagesDataTable(patientIcn);
            }
        }

        private void RefreshImageData(string patientIcn, string context)
        {
            var response = _studyService.GetStudiesForPatient(patientIcn, context);
            if (response != null && response.studies != null && response.studies.Any())
                _studyService.InsertUpdateStudies(response.studies);

            ClearCachedImagesForPatient(patientIcn);
        }

        public void CheckForImages(
            IEnumerable<string> patientIcns, 
            string userId)
        {
            var icnsWithImages = CacheHelper.Get<Dictionary<string, bool?>>(GetCheckForImagesCacheKey());

            if (icnsWithImages == null 
                || !LinqUtils.ContainsAllItems(icnsWithImages.Keys.ToList(), patientIcns.ToList()))
                    CheckDatabaseForImages(patientIcns, userId);
        }

        private void CheckDatabaseForImages(
            IEnumerable<string> patientIcns,
            string userId)
        {
            Dictionary<string, bool?> existingPatients = _patientImageRepository.GetImagesValidatedPatientIcns(patientIcns);

            var notCheckedIcns = existingPatients.Keys.Where(k => !existingPatients[k].HasValue);
            if (notCheckedIcns.Any())
            {
                CheckVistAForImages(patientIcns, notCheckedIcns, userId);
                existingPatients = _patientImageRepository.GetImagesValidatedPatientIcns(patientIcns);
            }

            if (existingPatients != null && existingPatients.Any())
                UpdateCheckImagesCache(existingPatients);
        }

        private void CheckVistAForImages(
            IEnumerable<string> patientIcns,
            IEnumerable<string> notCheckedIcns, 
            string userId)
        {
            IEnumerable<PatientStudyCount> patientStudyCounts = _studyService.CheckPatientsForImages(notCheckedIcns, userId);
            if (patientStudyCounts != null && patientStudyCounts.Any())
                _patientImageRepository.InsertUpdateImageAnalysisPatientStudiesFlags(patientStudyCounts);
        }

        private void UpdateCheckImagesCache(Dictionary<string, bool?> existingPatients)
        {
            var cacheKey = GetCheckForImagesCacheKey();
     
            Dictionary<string, bool?> icnsWithImages = CacheHelper.Get<Dictionary<string, bool?>>(GetCheckForImagesCacheKey());
            if (icnsWithImages == null)
                icnsWithImages = new Dictionary<string, bool?>();

            icnsWithImages = icnsWithImages
                .Union(existingPatients)
                .ToDictionary(d => d.Key, d => d.Value);

            CacheHelper.Add(cacheKey, icnsWithImages);
        }

        public bool CheckForImages(string patientIcn)
        {
            var icnsWithImages = CacheHelper.Get<Dictionary<string, bool?>>(GetCheckForImagesCacheKey());

            if (icnsWithImages != null && icnsWithImages.Keys.Contains(patientIcn))
            {
                var patientImages = icnsWithImages[patientIcn];
                return patientImages.HasValue && patientImages.Value;
            }

            return false;
        }

        public string CheckForImagesToolTip(string patientIcn)
        {
            const string IMAGING_SERVICES_ERROR = "VistA imaging services are not available.";

            var icnsWithImages = CacheHelper.Get<Dictionary<string, bool?>>(GetCheckForImagesCacheKey());
            if (icnsWithImages == null)
                return IMAGING_SERVICES_ERROR;

            string result = "Patient was not found in VistA Imaging.";

            if (icnsWithImages.Any() && icnsWithImages.Keys.Contains(patientIcn))
            {
                var patientImages = icnsWithImages[patientIcn];
                if (patientImages.HasValue)
                    result = patientImages.Value ? "Patient has image studies." : "No image studies were found for patient.";
            }

            return result;
        }

        private static string GetCheckForImagesCacheKey()
        {
            return new CacheKey(
                CACHE_KEY_NAMESPACE, 
                CACHE_KEY_CLASS, 
                "CheckForImages").Build();
        }
    }
}