﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager;

namespace CRS_EFR
{
    public class DoDFragmentManager:BaseBO 
    {
        // TODO!!!
        private static string USER_NAME = "DNS      \\DNS   ASKEWD";

        private int _selectFragmentsByPatientIdCount = 0;
        private int _selectDoDFragmentDetailsCount = 0;
        private int _selectDoDFragmentsRawDataByPatientIdCount = 0;

        /// <summary>
        /// Select DoD_FRAGMENT_DATA_RECEIVELOG by patient SSN
        /// </summary>
        /// <param name="ssn"></param>
        /// <returns></returns>
        public DoD_FRAGMENT_DATA_RECEIVELOG SelectBySSN(string ssn)
        {
            using (_db = GetDataContext())
            {
                DoD_FRAGMENT_DATA_RECEIVELOG fragment = SelectBySSNLinqFilter(ssn).FirstOrDefault();

                return fragment;
            }

        }

        private IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> SelectBySSNLinqFilter(string ssn)
        {
            IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> linqFilter = (from e in _db.DoD_FRAGMENT_DATA_RECEIVELOGs
             where e.PATIENTSNUM.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
             select e);
            return linqFilter;
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT_LAB>(e => e.DOD_FRAGMENTs);
            lo.LoadWith<DOD_FRAGMENT>(e => e.DOD_FRAGMENT_DETAILs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private void SetLoadWithAnalytes(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT>(e => e.DOD_FRAGMENT_DETAILs);
            lo.LoadWith<DOD_FRAGMENT_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private void SetLoadWithFragment(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT>(e => e.DOD_FRAGMENT_DETAILs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }
        private void SetLoadWithFragmentDetail(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_FRAGMENT_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private void SetLoadWithDoDRawData(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DoD_FRAGMENT_DATA_RECEIVELOG>(e => e.DOD_FRAGMENT_RAW_DATA);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        /// <summary>
        /// Get the last extract file date for each DoD lab
        /// </summary>
        /// <returns></returns>
        public List<DoDFragmentDataExtractByLab> GetLastExtractDateForAllLabs()
        {
           using (_db = GetDataContext())
           {
               return (from e in _db.DOD_FRAGMENT_RAW_DATAs 
                                   group e by e.SOURCE_LAB into g
                                   select new DoDFragmentDataExtractByLab 
                                   {
                                       LabName = g.Key,
                                       LastExtractReceivedDate = g.Max( e => e.CREATED)
                                   }).ToList();
           }
        }

        /// <summary>
        /// Get the last DoD Fragment Data Extract Date
        /// </summary>
        /// <returns></returns>
        public System.Nullable<DateTime> GetLastExtractDate(string ssn)
        {
            using (_db = GetDataContext())
            {
                System.Nullable<DateTime> extractDate = null;
                int logCount = (from e in _db.DoD_FRAGMENT_DATA_RECEIVELOGs
                                where e.PATIENTSNUM.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
                                select e).Count();
                if (logCount > 0)
                {
                    extractDate = (from e in _db.DoD_FRAGMENT_DATA_RECEIVELOGs
                                   where e.PATIENTSNUM.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
                                   select e.ORIGINALINPUTDATE).Min();
                }

                return extractDate;
            }

        }

        /// <summary>
        /// Select DoD Fragment Raw Data by Patient ID
        /// </summary>
        /// <param name="patientId"></param>
        /// <param name="sort"></param>
        /// <param name="maxRows"></param>
        /// <param name="startRow"></param>
        /// <returns></returns>
        public IEnumerable<DoD_FRAGMENT_DATA_RECEIVELOG> SelectDoDFragmentsRawDataByPatientId(int patientId, string sort, int maxRows, int startRow)
        {
            IEnumerable<DoD_FRAGMENT_DATA_RECEIVELOG> results = null;

            if (patientId > 0)
            {
                PatientManager patientManager = new PatientManager();
                PATIENT patient = patientManager.SelectByPatientID(patientId);

                if (patient != null)
                {
                    if (string.IsNullOrEmpty(sort))
                    {
                        sort = "DOD_FRAGMENT_RAW_DATA_ID desc";
                    }

                    using (_db = GetDataContext())
                    {
                        SetLoadWithDoDRawData(_db);

                        results = SelectDoDFragmentsRawDataByPatientIdLinqFilter(patient.Snum).OrderBy(sort).ToList();
                        _selectDoDFragmentsRawDataByPatientIdCount = results.Count();
                    }
                }
            }
            return results;
        }

        /// <summary>
        /// Count for Select DoD Fragment Raw Data by Patient ID
        /// </summary>
        /// <param name="patientId"></param>
        /// <returns></returns>
        public int SelectDoDFragmentsRawDataByPatientIdCount(int patientId)
        {
            return _selectDoDFragmentsRawDataByPatientIdCount;
        }

        /// <summary>
        /// Select a record from DOD_FRAGMENT_RAW_DATA table by Raw Data ID
        /// </summary>
        /// <param name="rawDataId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_RAW_DATA SelectByRawDataId(int rawDataId)
        {
            using (_db = GetDataContext())
            {
                DOD_FRAGMENT_RAW_DATA rawData = (from e in _db.DOD_FRAGMENT_RAW_DATAs
                        where e.DOD_FRAGMENT_RAW_DATA_ID == rawDataId 
                        select e).FirstOrDefault();

                return rawData;
            }
        }
        
        /// <summary>
        /// Get all fragments for a patient
        /// </summary>
        /// <param name="patientId"></param>
        /// <param name="sort"></param>
        /// <param name="maxRows"></param>
        /// <param name="startRow"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT_LAB> SelectFragmentsByPatientId(int patientId)
        {
            IEnumerable<DOD_FRAGMENT_LAB> results = null;

            if (patientId > 0)
            {
                using (_db = GetDataContext())
                {
                    results = SelectFragmentsByPatientIdLinqFilter(patientId).OrderBy("LAB_NAME").ToList();
                    _selectFragmentsByPatientIdCount = results.Count();
                }
            }
            return results;
        }

        /// <summary>
        /// Select fragments by lab ID
        /// </summary>
        /// <param name="labId"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT> SelectFragmentsByLabId(int labId)
        {
            IEnumerable<DOD_FRAGMENT> results = null;

            if (labId > 0)
            {
                using (_db = GetDataContext())
                {
                    results = SelectFragmentsByLabIdLinqFilter(labId).OrderBy("DOD_FRAGMENT_ID").ToList();
                }
            }
            return results;
        }

        /// <summary>
        /// Select fragment analytes by fragment ID
        /// </summary>
        /// <param name="fragmentId"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT_DETAIL> SelectFragmentAnalytesByFragmentId(int fragmentId)
        {
            IEnumerable<DOD_FRAGMENT_DETAIL> results = null;

            if (fragmentId > 0)
            {
                using (_db = GetDataContext())
                {
                    SetLoadWithAnalytes(_db);
                    results = SelectFragmentAnalytesByFragmentIdLinqFilter(fragmentId).OrderBy("DOD_FRAGMENT_DETAIL_ID").ToList();
                }
            }
            return results;
        }

        private IQueryable<DOD_FRAGMENT_LAB> SelectFragmentsByPatientIdLinqFilter(int patientId)
        {
            return (from t in _db.DOD_FRAGMENT_LABs
                    where t.PATIENT_ID == patientId
                    select t);
        }

        private IQueryable<DOD_FRAGMENT> SelectFragmentsByLabIdLinqFilter(int labId)
        {
            return (from t in _db.DOD_FRAGMENTs
                    where t.DOD_FRAGMENT_LAB_ID == labId
                    select t);
        }

        private IQueryable<DOD_FRAGMENT_DETAIL> SelectFragmentAnalytesByFragmentIdLinqFilter(int fragmentId)
        {
            return (from t in _db.DOD_FRAGMENT_DETAILs
                    where t.DOD_FRAGMENT_ID == fragmentId
                    select t);
        }

        private IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> SelectDoDFragmentsRawDataByPatientIdLinqFilter(string ssn)
        {
            IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> fullSet = null;
            IQueryable<DoD_FRAGMENT_DATA_RECEIVELOG> groupedSet = null;

            //Select all records for current patient
            fullSet = from t in _db.DoD_FRAGMENT_DATA_RECEIVELOGs
                      where t.PATIENTSNUM.Replace(" ", "").Replace("-", "").Equals(ssn.Replace(" ", "").Replace("-", ""))
                      select t;

            if (fullSet != null)
            {
                //Group by raw data file
                groupedSet = from n in fullSet
                             group n by n.DOD_FRAGMENT_RAW_DATA_ID into g
                             select g.OrderByDescending(t => t.LAB_REPORT_DATE).FirstOrDefault();
            }

            return groupedSet;

        }

        /// <summary>
        /// Get the count all fragments for a patient
        /// </summary>
        /// <param name="patientId"></param>
        public int SelectFragmentsByPatientIdCount(int patientId)
        {
            return _selectFragmentsByPatientIdCount;
        }

        /// <summary>
        /// Select DoD_Fragment_DETAIL by DoD Fragment ID
        /// </summary>
        /// <param name="dodFragmentId"></param>
        /// <param name="sort"></param>
        /// <param name="maxRows"></param>
        /// <param name="startRow"></param>
        /// <returns></returns>
        public IEnumerable<DOD_FRAGMENT_DETAIL> SelectDoDFragmentDetails(int dodFragmentId, string sort, int maxRows, int startRow)
        {
            IEnumerable<DOD_FRAGMENT_DETAIL> results = null;

            using (_db = GetDataContext())
            {
                if (string.IsNullOrEmpty(sort))
                {
                    sort = "DOD_FRAGMENT_DETAIL_ID desc";
                }
                if (maxRows == 0)
                {
                    maxRows = 9999;
                }

                results = SelectDoDFragmentDetailsLinqFilter(dodFragmentId).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
                _selectDoDFragmentDetailsCount = results.Count();
            }
            
            return results;
        }

        private IQueryable<DOD_FRAGMENT_DETAIL> SelectDoDFragmentDetailsLinqFilter(int dodFragmentId)
        {
            var linqFilter = from t in _db.DOD_FRAGMENT_DETAILs
                             where t.DOD_FRAGMENT_ID == dodFragmentId
                             select t;

            return linqFilter;
        }
        
        public int SelectDoDFragmentDetailsCount(int dodFragmentId)
        {
            return _selectDoDFragmentDetailsCount;
        }


        /// <summary>
        /// Get DoD Fragment record by DoD fragment ID
        /// </summary>
        /// <param name="dodFragmentId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT GetDoDFragmentById(int dodFragmentId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithFragment(_db);

                return (from f in _db.DOD_FRAGMENTs
                        where f.DOD_FRAGMENT_ID == dodFragmentId
                        select f).FirstOrDefault();
            }
            
        }


        /// <summary>
        /// Get a DoD fragment analyte by analyte ID
        /// </summary>
        /// <param name="analyteId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_DETAIL GetFragmentAnalyteById(int analyteId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithFragmentDetail(_db);

                return (from f in _db.DOD_FRAGMENT_DETAILs
                        where f.DOD_FRAGMENT_DETAIL_ID == analyteId
                        select f).FirstOrDefault();
            }

        }


        /// <summary>
        /// Get DoD fragment lab by ID
        /// </summary>
        /// <param name="labId"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_LAB GetDoDFragmentLabById(int labId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return (from f in _db.DOD_FRAGMENT_LABs
                        where f.DOD_FRAGMENT_LAB_ID == labId
                        select f).FirstOrDefault();
            }
        }

        /// <summary>
        /// Update DOD_FRAGMENT_LAB record
        /// </summary>
        /// <param name="lab"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_LAB Update(DOD_FRAGMENT_LAB lab)
        {
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                lab.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }

            }

            return lab;
        }


        /// <summary>
        /// Update DOD_FRAGMENT record
        /// </summary>
        /// <param name="lab"></param>
        /// <returns></returns>
        public DOD_FRAGMENT Update(DOD_FRAGMENT fragment)
        {
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                fragment.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }

            }

            return fragment;
        }

        /// <summary>
        /// Update DOD_FRAGMENT_DETAIL record
        /// </summary>
        /// <param name="fragmentAnalyte"></param>
        /// <returns></returns>
        public DOD_FRAGMENT_DETAIL Update(DOD_FRAGMENT_DETAIL fragmentAnalyte)
        {
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                fragmentAnalyte.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }

            }

            return fragmentAnalyte;
        }
    }



    public class DoDFragmentDataExtractByLab
    {
        public string LabName { get; set; }
        public DateTime LastExtractReceivedDate { get; set; }
    }      

}
