﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;  
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;  

namespace VeteransAffairs.Registries.BusinessManager
{
    //TODO - Update as per db values
    public enum WorkFlowActivityTypes
    {
        LabKit = 1,
        Questionnaire = 2,
        LabOrder = 3,
        LabResult = 4,
        Diagnosis = 5,
        LetterSent=13,
        Interpretation =22
    }

    public enum WorkFlowTypes
    {
        Biomonitoring=3,
        FragmentAnalysis=4,
        Consultation=5
    }

    public enum WorkFlowCaseStatus
    {
        New=1,
        InProcess=2,
        Completed =3,
        Canceled=4
    }

    //Use Sortorder values here
    public enum ReferralStatus
    {
        New=4,
        InProcess=5,
        Completed=6,
        Ineligible=9

    }
    public class WorkflowManager:BaseBO
    {
        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            //lo.LoadWith<WKF_CASE>(i => i.PATIENT);
            lo.LoadWith<WKF_CASE>(i => i.REFERRAL);
            lo.LoadWith<REFERRAL>(i => i.PATIENT);   
            lo.LoadWith<WKF_CASE>(i => i.STD_WKFCASEST);
            lo.LoadWith<WKF_CASE>(i => i.STD_WKFCASETYPE); 
            lo.LoadWith<REFERRAL>(i => i.REFERRAL_DETAILs);
            lo.LoadWith<REFERRAL_DETAIL>(i => i.STD_HEALTHFACTOR);
            lo.LoadWith<PATIENT>(i => i.STD_INSTITUTION);
            db.LoadOptions = lo;
            db.ObjectTrackingEnabled = false;
        }

        private void SetLoadWithReferralCases(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE>( i => i.STD_WKFCASEST );
            lo.LoadWith<WKF_CASE>( i => i.STD_WKFCASETYPE );
            db.LoadOptions = lo;
        }
        public WorkflowManager()
        {
             _defaultSortField = "WKF_CASE_ID";


        }

        private IQueryable<WKF_CASE> LinqAll()
        {
            //populate LinqAll
            

                IQueryable<WKF_CASE> tempLinq = (from e in _db.WKF_CASEs
                                                 select e);

                return tempLinq;
            
        }

        public int GetWorkflowsByReferralIdCount(int id)
        {
            int objReturn = 0;

            _db = GetDataContext();
            //SetLoadWith(_db);
            objReturn = SelectLinqFilter(id).Count();
            _db.Dispose();

            return objReturn;
        }

        private IQueryable<WKF_CASE> SelectLinqFilter(int id)
        {
            IQueryable<WKF_CASE> linqFilter = LinqAll().Where(e => e.REFERRAL_ID == id);
            return linqFilter;
        }

        public IEnumerable<WorkflowsGetByReferralResult> GetWorkflowsByReferralId(int id, string sort, int maxRows, int startRow)
        {
            IEnumerable<WorkflowsGetByReferralResult> objReturn = null;

            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            //IEnumerable<WKF_CASE> entities;
            
            _db = GetDataContext();

            SetLoadWith(_db);
            var entities = SelectLinqFilter(id).OrderBy(sort).Skip(startRow).Take(maxRows);
            objReturn = entities.Select(wf => new WorkflowsGetByReferralResult
            {
                wfStatus = wf.STD_WKFCASEST.NAME,
                wfURL = "",
                wfID = wf.WKF_CASE_ID.ToString(),
                wfType = wf.STD_WKFCASETYPE.Name,
                wfUpdated = String.Format("{0:d}", wf.UPDATED)
            }).ToList();

            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Gets workflows by referral id 
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public IEnumerable<ReferralCase> GetCasesByReferralId(int referralId)
        {
            _db = GetDataContext();
            SetLoadWithReferralCases(_db);

            List<ReferralCase> refCases = SelectLinqFilter(referralId).Select(e => new ReferralCase
            {
                CaseId = e.WKF_CASE_ID,
                CaseStatus = e.STD_WKFCASEST.NAME,
                CaseStatusId = e.STD_WKFCASESTS_ID, 
                CaseType = e.STD_WKFCASETYPE.Name,
                CaseTypeId = e.STD_WKFCASETYPE_ID 
  
            }).ToList();

            List<ReferralCase> refOrderedCases = refCases.OrderBy(t => t.CaseStatusId).ThenBy(s => s.CaseTypeId).ThenByDescending(t => t.CaseId).ToList();
            _db.Dispose();

            return refOrderedCases;
            
        }
        /// <summary>
        /// Gets Workflow case for a case id.
        /// </summary>
        /// <param name="workFlowCaseId"></param>
        /// <returns></returns>
        public WKF_CASE GetWorkFlowCase(int workFlowCaseId)
        {
            WKF_CASE objReturn = null;

            _db = GetDataContext();
            SetLoadWith(_db);
            //TODO - Apply institution check
            objReturn = (from t in LinqAll() where t.WKF_CASE_ID == workFlowCaseId select t).FirstOrDefault();
            
            _db.Dispose();

            return objReturn;
        }
        /// <summary>
        /// Gets all case activities for a case id.
        /// </summary>
        /// <param name="workFlowCaseId"></param>
        /// <returns></returns>
        public IEnumerable<WKF_CASE_ACTIVITY> GetCaseActivities(int workFlowCaseId)
        {
            IEnumerable<WKF_CASE_ACTIVITY> objReturn = null;

            _db = GetDataContext();
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(i => i.SHIPPING_DETAILs);
            _db.LoadOptions = lo;
            _db.ObjectTrackingEnabled = false;
            objReturn = (from t in _db.WKF_CASE_ACTIVITies where t.WKF_CASE_ID == workFlowCaseId select t).ToList();
            _db.Dispose();

            return objReturn;
        }

        public bool AnyArchivedRecords(int caseId)
        {
            bool objReturn = false;

            _db = GetDataContext();
            objReturn = (from a in _db.WKF_CASE_ACTIVITies where a.WKF_CASE_ID == caseId select a)
                .Where(a => a.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.InProcess || a.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.Accepted  )
                    .Any(c => c.ARCHIVED_FLAG == true);

            _db.Dispose();

            return objReturn;
        }


        private void SetLoadWithWFC(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE>(i => i.PATIENT);
            lo.LoadWith<WKF_CASE>(i => i.STD_WKFCASEST);
            lo.LoadWith<WKF_CASE>(i => i.STD_WKFCASETYPE);
            db.LoadOptions = lo;
            db.ObjectTrackingEnabled = false;
        }
        /// <summary>
        /// Gets all workflow cases
        /// </summary>
        /// <returns></returns>
        //TODO - For a referral /Patient?
        public IEnumerable<WKF_CASE> GetWorkFlowCases()
        {
            IEnumerable<WKF_CASE> objReturn = null;

            _db = GetDataContext();
            SetLoadWithWFC(_db);
            objReturn = _db.WKF_CASEs.ToList();

            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Gets all workflow cases for a case type.
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        public IEnumerable<WKF_CASE> GetWorkFlowCases(WorkFlowTypes type)
        {
            IEnumerable<WKF_CASE> objReturn = null;

            _db = GetDataContext();
            SetLoadWithWFC(_db);
            objReturn = (from t in _db.WKF_CASEs where t.STD_WKFCASETYPE.ID == Convert.ToInt16(type) select t).ToList();
            _db.Dispose();

            return objReturn;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        public int GetWorkFlowCasesCount(WorkFlowTypes type)
        {
            int objReturn = 0;

            _db = GetDataContext();
            SetLoadWithWFC(_db);
            objReturn = (from t in _db.WKF_CASEs where t.STD_WKFCASETYPE.ID == Convert.ToInt16(type) select t).Count();
            _db.Dispose();

            return objReturn;
        }
        /// <summary>
        /// Gets workflow case type for an activity
        /// </summary>
        /// <param name="workFlowActivityId"></param>
        /// <returns></returns>
        public int GetWorkflowType(int workFlowActivityId)
        {
            int objReturn = 0;

            _db = GetDataContext();
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
            _db.LoadOptions = lo;
            objReturn = _db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ACTIVITY_ID == workFlowActivityId).Select(t => t.WKF_CASE.STD_WKFCASETYPE.ID).FirstOrDefault();
            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Gets workflow case type for a case
        /// </summary>
        /// <param name="workFlowActivityId"></param>
        /// <returns></returns>
        public WorkFlowTypes GetWorkflowCaseType(int workFlowCaseId)
        {
            WorkFlowTypes objReturn;

            _db = GetDataContext();
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
            _db.LoadOptions = lo;
            objReturn = (WorkFlowTypes)_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ID == workFlowCaseId).Select(t => t.WKF_CASE.STD_WKFCASETYPE.ID).FirstOrDefault();
            _db.Dispose();

            return objReturn;
        }

        public class ReferralCase
        {
            public int CaseId { get; set; }
            public string CaseStatus{ get; set; }
            public int CaseStatusId { get; set; }
            public string CaseType { get; set; }
            public int CaseTypeId { get; set; }

        }
        public class WorkflowsGetByReferralResult
        {
            public string wfStatus { get; set; }
            public string wfType { get; set; }
            public string wfURL { get; set; }
            public string wfID { get; set; }
            public string wfUpdated { get; set; }
        }
    }
}
