﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Linq.Dynamic;
using System.Linq.Expressions;  
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;

namespace VeteransAffairs.Registries.BusinessManager
{
    public class WorkflowCaseManager : BaseBO 
    {
        public enum WorkFlowCaseActivityType
        {
            LabKit = 1, 
            Questionnaire = 2, 
            LabOrder = 3, 
            LabResult = 4, 
            Diagnosis = 5, 
            TBI1Scheduled = 8, 
            TBI2Scheduled = 9, 
            TBI3Scheduled = 14, 
            CommunicationContact = 20, 
            InterpretationDiagnosis = 22, 
            DVEIREditedReferral = 23, 
            DVEIRMarkedReadyForReview = 25, 
            DVEIRMarkedReadyToSendToDVEIR = 26, 
            DVEIRSentBackToAbstractor = 27, 
            DVEIRSentToDVEIR = 28
        }
        
        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);   
            db.LoadOptions = lo;

        }

        private void SetLoadActivityTypeWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SHIPPING_DETAILs);
            lo.AssociateWith<WKF_CASE_ACTIVITY>( e => e.SHIPPING_DETAILs.Take(1));  
               
            db.LoadOptions = lo;

        }

        private IQueryable<WKF_CASE_ACTIVITY> LinqAll()
        {
            //populate LinqAll
            IQueryable<WKF_CASE_ACTIVITY> tempLinq = (from e in _db.WKF_CASE_ACTIVITies
                                                      select e);

            //TODO - add all business filtering rules 

            return tempLinq;

        }

        private IQueryable<WKF_CASE_ACTIVITY> SelectByCaseLinqFilter(int caseId)
        {
            var linqFilter = LinqAll();

            linqFilter = from t in linqFilter 
                         where t.WKF_CASE_ID == caseId 
                            && t.ARCHIVED_FLAG == false
                         select t;

            return linqFilter;

        }

        private IQueryable<WKF_CASE_ACTIVITY> SelectByCaseAndActivityTypeLinqFilter(int caseId, int activityTypeId)
        {
            var linqFilter = LinqAll();

            linqFilter = from t in linqFilter
                         where t.WKF_CASE_ID == caseId
                            && t.STD_WKFACTIVITYTYPE_ID == activityTypeId
                            && t.ARCHIVED_FLAG == false
                         select t;

            return linqFilter;

        }

        /// <summary>
        /// Update work flow case activity record with status and archived flag
        /// </summary>
        /// <param name="activityId"></param>
        /// <param name="statusId"></param>
        /// <param name="archived"></param>
        /// <returns></returns>
        public int UpdateActivityStatus(int activityId, int statusId, bool archived)
        {
            int returnStatus = 0;
            if (activityId > 0 && statusId > 0)
            {
                _db = GetDataContext();
                WKF_CASE_ACTIVITY activity = (from e in _db.WKF_CASE_ACTIVITies
                                                where e.WKF_CASE_ACTIVITY_ID == activityId
                                                select e).FirstOrDefault();

                activity.SetAsChangeTrackingRoot();
                activity.STD_WKFACTIVITYSTS_ID = statusId;
                activity.ARCHIVED_FLAG = archived;
                activity.SetAsUpdateOnSubmit();

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = 0;

                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = 0;
                }
                catch
                {
                    returnStatus = -1;
                }
                _db.Dispose();
            }
            return returnStatus;
        }


        /// <summary>
        /// Delete a work case activity record by work flow case activity ID
        /// </summary>
        /// <param name="workflowActivityId"></param>
        /// <returns></returns>
        public int DeleteWorkCaseActivity(int workflowActivityId)
        {
            int returnStatus = 0;
            if (workflowActivityId > 0)
            {
                _db = GetDataContext();
                WKF_CASE_ACTIVITY activity = (from e in _db.WKF_CASE_ACTIVITies
                                                where e.WKF_CASE_ACTIVITY_ID == workflowActivityId
                                                select e).FirstOrDefault();
                _db.WKF_CASE_ACTIVITies.DeleteOnSubmit(activity);

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = 0;

                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = 0;
                }
                catch
                {
                    returnStatus = -1;
                }
                _db.Dispose();
            }
            return returnStatus;
        }
        
        public WKF_CASE_ACTIVITY SelectByActivityType(int caseId, string activityType)
        {
            WKF_CASE_ACTIVITY objReturn = null;

            _db = GetDataContext();
            SetLoadActivityTypeWith(_db); 

            WorkFlowActivityTypes type = (WorkFlowActivityTypes)Enum.Parse(typeof(WorkFlowActivityTypes), activityType);

            objReturn = SelectByCaseLinqFilter(caseId).Where(e => e.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(type)).FirstOrDefault();
            _db.Dispose();

            return objReturn;
        }
        public IEnumerable<WorkflowCaseActivity> SelectByCaseId(int caseId, int caseTypeId)
        {
            _db = GetDataContext();
            SetLoadWith(_db);

            var activities = SelectByCaseLinqFilter(caseId);

            List<WorkflowCaseActivity> caseActivities = activities.Select(e => new WorkflowCaseActivity
            {
                CaseTypeId = e.WKF_CASE.STD_WKFCASETYPE_ID,  
                CaseActivityExists = true,
                CaseActivityId = e.WKF_CASE_ACTIVITY_ID,
                CaseActivityType = e.STD_WKFACTIVITYTYPE.NAME,
                CaseActivityTypeId = e.STD_WKFACTIVITYTYPE_ID, 
                CaseActivityStatus = e.STD_WKFACTIVITYST.NAME,
                UpdatedDate = String.Format("{0:d}", e.UPDATED),
                CaseActivityArchived = e.ARCHIVED_FLAG,
                CaseActivityStatusId = e.STD_WKFACTIVITYSTS_ID
            }).ToList();

            var results = CreateCompleteActivitiesList(caseActivities, caseTypeId).ToList(); 
 
            results.Where(t => t.CaseActivityTypeId == 2 && t.CaseTypeId == 4 ).ToList().ForEach( item => item.CaseActivityType = "Form");
            _db.Dispose();

            return results.OrderBy( e => e.CaseActivityTypeId ) ;
            
        }

        /// <summary>
        /// Select by Case ID and Activity Type ID
        /// </summary>
        /// <param name="caseId"></param>
        /// <param name="activityTypeId"></param>
        /// <returns></returns>
        public WKF_CASE_ACTIVITY SelectByCaseIdAndActivityTypeId(int caseId, int activityTypeId)
        {
            WKF_CASE_ACTIVITY objReturn = null;

            _db = GetDataContext();
            SetLoadWith(_db);

            objReturn = SelectByCaseAndActivityTypeLinqFilter(caseId, activityTypeId).FirstOrDefault();
            _db.Dispose();

            return objReturn;
        }
        
        private IQueryable<WKF_CASE_ACTIVITY> SelectBioFollowUpsByReferralLinqFilter(int refId)
        {
            var linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.WKF_CASE.REFERRAL_ID == refId 
                             && t.FOLLOWUP_REQUIRED_FLAG == true && t.FOLLOWUP_DECLINED_DATE == null && t.FOLLOWUP_WKF_CASE_ID == null select t;


            return linqFilter;

        }

        public IEnumerable<BioFollowUpWorkflowCaseActivity> SelectBioFollowUpsByReferralId(int id)
        {
            _db = GetDataContext();
            SetLoadWith(_db);

            var activities = SelectBioFollowUpsByReferralLinqFilter(id).ToList();

            List<BioFollowUpWorkflowCaseActivity> bioFollowupActivities = activities.Select(e => new BioFollowUpWorkflowCaseActivity
            {
                CaseActivityId = e.WKF_CASE_ACTIVITY_ID,
                DueDate = e.FOLLOWUP_PLAN_DATE.HasValue ? e.FOLLOWUP_PLAN_DATE.Value.ToString("d") : String.Empty     
                   

            }).ToList();

            _db.Dispose();

            return bioFollowupActivities;
        }

        public WKF_CASE_ACTIVITY getActivityByID(int activityId)
        {
            WKF_CASE_ACTIVITY entity;
            _db = GetDataContext();
            SetLoadWith(_db);
            entity = (from act in _db.WKF_CASE_ACTIVITies where act.WKF_CASE_ACTIVITY_ID == activityId select act).FirstOrDefault();
            entity.SetAsChangeTrackingRoot();
            _db.Dispose();

            return entity;
        }



        private IEnumerable<WorkflowCaseActivity> CreateCompleteActivitiesList(IEnumerable<WorkflowCaseActivity> activities, int typeId) 
        {
            var caseTypes = new Dictionary<int, string>{ {1, "Lab Kit"}, {2, "Questionnaire"},
                {3, "Lab Order"}, {4, "Lab Result"}, {22, "Interpretation & Diagnosis"}}; 

            foreach (var activity in caseTypes)
            {
                if (activity.Key.In(activities.Select(t => t.CaseActivityTypeId)))
                {

                    yield return activities.First(t => t.CaseActivityTypeId == activity.Key);
                }
                else
                {
                    yield return new WorkflowCaseActivity
                    {
                        CaseActivityTypeId = activity.Key,
                        CaseActivityExists = false,
                        CaseActivityId = 0,
                        CaseTypeId = typeId,                         
                        CaseActivityStatus = "N/A",
                        UpdatedDate = "N/A",
                        CaseActivityType = activity.Value
                    };
                }
            }
             
        }

        public void Update(WKF_CASE_ACTIVITY caseActivity)
        {


            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;


            caseActivity.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

            try
            {
                _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                //create an instance of the custom eventArgs in order to populate the id selected
                BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                eventArgs.SavedItemId = caseActivity.WKF_CASE_ACTIVITY_ID;

                RaiseSaveEvent(this, eventArgs);

            }
            catch
            {
                _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
            }

            _db.Dispose();
        }
        /// <summary>
        /// A special update for reminders functionality
        /// </summary>
        /// <param name="followUp"></param>
        /// <param name="activity"></param>
        /// <returns></returns>
        public WKF_CASE Update(WKF_CASE followUp, WKF_CASE_ACTIVITY activity)
        {
            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;

            followUp.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

            try
            {
                _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                //create an instance of the custom eventArgs in order to populate the id selected
                BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                eventArgs.SavedItemId = followUp.WKF_CASE_ID;

                RaiseSaveEvent(this, eventArgs);

                //activity.FOLLOWUP_REQUIRED_FLAG = false;
                activity.FOLLOWUP_WKF_CASE_ID = followUp.WKF_CASE_ID;

                activity.SynchroniseWithDataContext(_db);

                _db.SubmitChanges(ConflictMode.ContinueOnConflict);

            }
            catch (ChangeConflictException)
            {
                _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
            }
            _db.Dispose();
            return followUp;
        }

        public class WorkflowCaseActivity
        {            
            public int CaseActivityId { get; set; }
            public string CaseActivityType { get; set; }
            public int CaseActivityTypeId { get; set; }
            public string CaseActivityStatus { get; set; }
            public int CaseActivityStatusId { get; set; }
            public string UpdatedDate { get; set; }
            public bool CaseActivityExists { get; set; }
            public int CaseTypeId { get; set; }
            public bool CaseActivityArchived { get; set; }
        }

        public class BioFollowUpWorkflowCaseActivity
        {
            public int CaseActivityId { get; set; }
            public string DueDate { get; set; }

        }

        public class WorkflowCaseComparer : IEqualityComparer<WKF_CASE>
        {
            public bool Equals(WKF_CASE x, WKF_CASE y)
            {
                if (x == null || y == null)
                {
                    return false;
                }

                return x.WKF_CASE_ID == y.WKF_CASE_ID;
            }

            public int GetHashCode(WKF_CASE obj) { return obj.WKF_CASE_ID.GetHashCode(); }
        } 


    }
}
