﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.VITAS
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class VitasRecallManager : BaseBO
    {
        public const int DefaultValueForStdTargetDate = 60;

        public VitasRecallManager() 
        {
            _defaultSortField = "RECALL_ID DESC";

        }

        private IQueryable<IM_RECALL> LinqAll()
        {
            //populate LinqAll
            IQueryable<IM_RECALL> tempLinq = (from e in _db.IM_RECALLs
                                            select e);
            return tempLinq;
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #region Method for Check Existence

        public bool RecallExists(int id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.IM_RECALLs where e.RECALL_ID == id select e).Count();

                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        public bool SourceRecallExists(string sourceID)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.IM_RECALLs where e.SOURCE_RECALL_ID == sourceID select e).Count();

                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        #endregion

        #region Methods for Select

        public IEnumerable<IM_RECALL> SelectRecalls(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                List<IM_RECALL> entities;
                entities = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;

            }
        }

        public int SelectRecallsCount()
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return SelectLinqFilter().Count();
            }
        }

        public int SelectRecallHitsCount(string manufacturer, string model, string serial, string lot)
        {
            using (_db = GetDataContext())
            {
                int? count = -1;
                int ret;

                ret = _db.GetRecalledImplantCount(manufacturer, model, serial, lot, ref count);

                if (count == null)
                    return -1;
                else
                    return (int)count;
            }
        }

        private IQueryable<IM_RECALL> SelectLinqFilter()
        {
            IQueryable<IM_RECALL> linqFilter = LinqAll();
            return linqFilter;
        }

        #endregion

        #region Methods for SelectByID

        public IM_RECALL SelectByID(int id)
        {
            IM_RECALL recall;
            if (id == 0)
            {
                recall = new IM_RECALL();

                recall.SetAsChangeTrackingRoot(EntityState.New, true);
                recall.SetAsInsertOnSubmit();
            }
            else
            {
                using (_db = GetDataContext())
                {
                    SetLoadWith(_db);
                    recall = SelectByIDLinqFilter(id).SingleOrDefault();

                    if (recall != null)
                    {
                        recall.SetAsChangeTrackingRoot(true);
                        recall.SetAsUpdateOnSubmit();
                    }
                }
            }
            return recall;
        }

        private IQueryable<IM_RECALL> SelectByIDLinqFilter(int id)
        {
            IQueryable<IM_RECALL> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.RECALL_ID == id select t;

            return linqFilter;
        }

        #endregion

        #region Methods for Update

        /// <summary>
        /// Save Recall data to database.
        /// </summary>
        /// <param name="actionTaken"></param>
        /// <returns></returns>
        private int UpdateRecall(IM_RECALL recall)
        {
            recall.SetAsUpdateOnSubmit();
            return Save(recall);
        }

        public int Update(IM_RECALL recall)
        {
            if (recall.RECALL_ID > 0)
                return UpdateRecall(recall);
            else
                return CreateNewRecall(recall);
        }

        public int SaveTargetDate(int recallID, string targetDate)
        {
            using (_db = GetDataContext())
            {
                int? errCode = 0;
                int ret;

                ret = _db.SetRecallTargetDate(recallID, targetDate, ref errCode);

                if (errCode == null)
                    return -1;
                else
                    return (int)errCode;
            }
        }

        public int Delete(IM_RECALL recall)
        {
            int returnStatus = 0;

            recall.SetAsChangeTrackingRoot(EntityState.Deleted, true);
            recall.SetAsDeleteOnSubmit();

            IM_RECALL recallCopy = recall.Clone();
            returnStatus = Save(recallCopy);

            return returnStatus;
        }

        #endregion

        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref IM_RECALL recall)
        {

            return 0;
        }
        #endregion

        #region Private Methods

        private int Save(IM_RECALL recall)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                recall.SynchroniseWithDataContext(_db);

                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                            eventArgs.SavedItemId = recall.RECALL_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                            eventArgs.SavedItemId = recall.RECALL_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch (Exception ex)
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                        Console.Write(ex.Message);
                    }
                }

            }

            return returnStatus;
        }

        /// <summary>
        /// Assign "NULL VALUE" fore NOT NULL item if not set
        /// </summary>
        /// <param name="actionTaken"></param>
        /// <returns></returns>

        private int CreateNewRecall(IM_RECALL recall)
        {
            int returnStatus = 0;

            recall.SetAsChangeTrackingRoot(EntityState.New, true);
            recall.SetAsInsertOnSubmit();

            IM_RECALL recallCopy = recall.Clone();
            returnStatus = Save(recallCopy);

            return returnStatus;
        }

        /// <summary>
        /// Insert a new recall to IM_RECALL table
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        private int InsertRecall(IM_RECALL recall)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                recall.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                //Check if any actual changes will occur
                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                        eventArgs.SavedItemId = recall.RECALL_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                        eventArgs.SavedItemId = recall.RECALL_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }
                return returnStatus;
            }
        }

        #endregion

    }
}
