﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.VITAS
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class VitasPatientManager : BaseBO
    {
        private UserAccountManager _user = (UserAccountManager)System.Threading.Thread.CurrentPrincipal;

        public VitasPatientManager() 
        {
            _defaultSortField = "PATIENT_ID";

        }

        private IQueryable<PATIENT> LinqAll()
        {
            //populate LinqAll
            IQueryable<PATIENT> tempLinq = (from e in _db.PATIENTs
                                                 select e);

            //tempLinq = (from p in tempLinq
            //            where p.REFERRALs.OfType<VITAS_REFERRAL>().Any()
            //            select p).Distinct();
        
            return tempLinq;

        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<PATIENT>(e => e.STD_INSTITUTION);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_INSTITUTION_PARENT);
            lo.LoadWith<PATIENT>(e => e.STD_COMBATLOCATION);
            lo.LoadWith<PATIENT>(e => e.STD_GENDER);
            lo.LoadWith<PATIENT>(e => e.STD_MARITALSTATUS);
            lo.LoadWith<PATIENT>(e => e.STD_RACE);
            lo.LoadWith<PATIENT>(e => e.STD_ETHNICITY);
            lo.LoadWith<PATIENT>(e => e.STD_SERVICEBRANCH);
            lo.LoadWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs);
            lo.AssociateWith<PATIENT>(p => p.REGISTRY_DEMOGRAPHICs.Where(rd => rd.STD_REGISTRY.ID == _user.RegistryId));
            lo.LoadWith<REGISTRY_DEMOGRAPHIC>(e => e.STD_ADDRESSTYPE);
            lo.LoadWith<PATIENT>(e => e.PRIMARY_CARE_PHYSICIANs);
            lo.LoadWith<PRIMARY_CARE_PHYSICIAN>(e => e.PROVIDER);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForListViews(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<PATIENT>(e => e.STD_INSTITUTION);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_INSTITUTION_PARENT);
            lo.LoadWith<PATIENT>(e => e.STD_SERVICEBRANCH);
            lo.LoadWith<PATIENT>(e => e.STD_GENDER);
            lo.AssociateWith<PATIENT>(p => p.REGISTRY_DEMOGRAPHICs.Where(rd => rd.STD_REGISTRY.ID == _user.RegistryId));
            lo.LoadWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs);
            lo.LoadWith<PATIENT>(e => e.PRIMARY_CARE_PHYSICIANs);
            lo.LoadWith<PRIMARY_CARE_PHYSICIAN>(e => e.PROVIDER);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        #region Method for Check Existence
        
        public bool PatientExists(int id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.PATIENTs where e.PATIENT_ID == id select e).Count();
                
                if (count > 0)
                {
                    return true;
                }
                else 
                {
                    return false;
                }
            }
        }

        #endregion

        #region Methods for Select

        public IEnumerable<PATIENT> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                List<PATIENT> entities;
                entities = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
                 
            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectLinqFilter().Count();
            }
        }

        private IQueryable<PATIENT> SelectLinqFilter()
        {
            IQueryable<PATIENT> linqFilter = LinqAll();
            return linqFilter;
        }

        #endregion

        #region Methods for SelectByID

        public PATIENT SelectByID(int id)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                PATIENT patient = SelectByIDLinqFilter(id).SingleOrDefault();

                return patient;

            }

        }

        private IQueryable<PATIENT> SelectByIDLinqFilter(int id)
        {
            IQueryable<PATIENT> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.PATIENT_ID == id select t;

            return linqFilter;
        }

        #endregion


        #region Methods for SelectByCprsSearchCriteria

        /// <summary>
        /// Select a patient by the patient info returned from CPRS service
        /// </summary>
        /// <param name="cprsWithPatientInfo"></param>
        /// <returns></returns>
        public PatientICNMaster SelectByCprsSearchCriteria(string patientIEN, string siteCode)
        {
            using (_db = GetDataContext())
            {
                return SelectByCprsSearchCriteriaLinqFilter(patientIEN, siteCode).FirstOrDefault();
            }
        }


        private IQueryable<PatientICNMaster> SelectByCprsSearchCriteriaLinqFilter(string patientIEN, string siteCode)
        {
            return ( from t in _db.PatientICNMasters
                     where t.PatientIEN.Trim().Equals(patientIEN.Trim())
                           && t.Sta3n.ToString().Equals(siteCode.Trim())
                     select t);
        }

        #endregion


        #region Methods for SelectBySearchCriteria


        public IEnumerable<PATIENT> SelectBySearchCriteria(string searchStringFirstName,
            string searchStringLastName, string searchStringMiddleName, 
            string searchStringSSN, string searchStringGender,
            string searchStringDateOfBirth,
            string searchStringAddress, string searchStringCity,
            string searchStringState, string searchStringZip,
            string searchStringHomePhone, string searchStringWorkPhone,
            string searchStringCellPhone, string searchStringAltPhone,
            string searchStringStateDescription,
            string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(searchStringFirstName)
                && string.IsNullOrEmpty(searchStringLastName)
                && string.IsNullOrEmpty(searchStringMiddleName)
                && string.IsNullOrEmpty(searchStringSSN)
                && string.IsNullOrEmpty(searchStringGender)
                && string.IsNullOrEmpty(searchStringDateOfBirth)
                && string.IsNullOrEmpty(searchStringAddress)
                && string.IsNullOrEmpty(searchStringCity)
                && string.IsNullOrEmpty(searchStringState)
                && string.IsNullOrEmpty(searchStringZip)
                && string.IsNullOrEmpty(searchStringHomePhone)
                && string.IsNullOrEmpty(searchStringWorkPhone)
                && string.IsNullOrEmpty(searchStringCellPhone)
                && string.IsNullOrEmpty(searchStringAltPhone))
            {
                return null;
            }

            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                IEnumerable<PATIENT> entities;
                entities = SelectBySearchCriteriaLinqFilter(searchStringFirstName,
                        searchStringLastName, searchStringMiddleName,
                        searchStringSSN, searchStringGender,
                        searchStringDateOfBirth,
                        searchStringAddress, searchStringCity,
                        searchStringState, searchStringZip,
                        searchStringHomePhone, searchStringWorkPhone,
                        searchStringCellPhone, searchStringAltPhone, searchStringStateDescription).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        public int SelectBySearchCriteriaCount(string searchStringFirstName,
            string searchStringLastName, string searchStringMiddleName,
            string searchStringSSN, string searchStringGender,
            string searchStringDateOfBirth,
            string searchStringAddress, string searchStringCity,
            string searchStringState, string searchStringZip,
            string searchStringHomePhone, string searchStringWorkPhone,
            string searchStringCellPhone, string searchStringAltPhone,
            string searchStringStateDescription,
            string sort, int startRow, int maxRows)
        {

            using (_db = GetDataContext())
            {
                return SelectBySearchCriteriaLinqFilter(searchStringFirstName,
                        searchStringLastName, searchStringMiddleName,
                        searchStringSSN, searchStringGender,
                        searchStringDateOfBirth,
                        searchStringAddress, searchStringCity,
                        searchStringState, searchStringZip,
                        searchStringHomePhone, searchStringWorkPhone,
                        searchStringCellPhone, searchStringAltPhone,
                        searchStringStateDescription).Count();
            }
        }

        private IQueryable<PATIENT> SelectBySearchCriteriaLinqFilter(string searchStringFirstName,
            string searchStringLastName, string searchStringMiddleName, 
            string searchStringSSN, string searchStringGender,
            string searchStringDateOfBirth,
            string searchStringAddress, string searchStringCity,
            string searchStringState, string searchStringZip,
            string searchStringHomePhone, string searchStringWorkPhone,
            string searchStringCellPhone, string searchStringAltPhone,
            string searchStringStateDescription)
        {
            IQueryable<PATIENT> linqFilter = LinqAll();

            if (!String.IsNullOrEmpty(searchStringFirstName))
            {
                linqFilter = from t in linqFilter where t.FIRST_NAME.Replace(" ", "").Equals(searchStringFirstName.Trim().Replace(" ", "")) select t;
            }

            if (!String.IsNullOrEmpty(searchStringLastName))
            {
                linqFilter = from t in linqFilter where t.LAST_NAME.Replace(" ", "").Equals(searchStringLastName.Trim().Replace(" ", "")) select t;
            }

            if (!String.IsNullOrEmpty(searchStringMiddleName))
            {
                linqFilter = from t in linqFilter where t.MIDDLE_NAME.Equals(searchStringMiddleName.Trim()) select t;
            }

            if (!String.IsNullOrEmpty(searchStringSSN))
            {
                string formattedSSN = searchStringSSN.Trim().Replace("-", "");
                linqFilter = from t in linqFilter where t.SSN.Replace("-","").Equals(formattedSSN) select t;
            }

            if (!String.IsNullOrEmpty(searchStringGender))
            {
                linqFilter = from t in linqFilter where t.STD_GENDER_ID.Equals(searchStringGender.Trim()) select t;
            }

            if (!String.IsNullOrEmpty(searchStringDateOfBirth))
            {
                linqFilter = from t in linqFilter
                             where ((DateTime)t.BIRTH_DATE).Date.Equals((Convert.ToDateTime(searchStringDateOfBirth)).Date)
                             select t;
            }

            if (!String.IsNullOrEmpty(searchStringAddress))
            {
                linqFilter = from t in linqFilter where t.ADDRESS_LINE1.Equals(searchStringAddress.Trim()) select t;
            }

            if (!String.IsNullOrEmpty(searchStringCity))
            {
                linqFilter = from t in linqFilter where t.CITY.Equals(searchStringCity.Trim()) select t;
            }

            if (!String.IsNullOrEmpty(searchStringState))
            {
                linqFilter = from t in linqFilter
                             where t.STATE.ToUpper().Equals(searchStringState.Trim().ToUpper())
                             || t.STATE.ToUpper().Equals(searchStringStateDescription.Trim().ToUpper())
                             select t;
            }

            if (!String.IsNullOrEmpty(searchStringZip))
            {
                linqFilter = from t in linqFilter where t.POSTAL_CODE.Equals(searchStringZip.Trim()) select t;
            }
            if (!String.IsNullOrEmpty(searchStringHomePhone))
            {
                linqFilter = from t in linqFilter where t.HOME_PHONE.Equals(searchStringHomePhone.Trim()) select t;
            }
            if (!String.IsNullOrEmpty(searchStringWorkPhone))
            {
                linqFilter = from t in linqFilter where t.WORK_PHONE.Equals(searchStringWorkPhone.Trim()) select t;
            }
            if (!String.IsNullOrEmpty(searchStringCellPhone))
            {
                linqFilter = from t in linqFilter where t.CELL_PHONE.Equals(searchStringCellPhone.Trim()) select t;
            }
            if (!String.IsNullOrEmpty(searchStringAltPhone))
            {
                linqFilter = from t in linqFilter where t.PRIMARY_PHONE.Equals(searchStringAltPhone.Trim()) select t;
            }

            return linqFilter;
        }


        #endregion

        #region Methods for Update

        public int Update(PATIENT patient)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                int status = patient.EnforceBusinessRules(true, _db);
                patient.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                //Check if any actual changes will occur
                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                        eventArgs.SavedItemId = patient.PATIENT_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                        eventArgs.SavedItemId = patient.PATIENT_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    
                    catch
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }

            }

            return returnStatus;
        }

        #endregion

        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref PATIENT patient)
        {
            using (_db = GetDataContext())
            {
                int status = patient.EnforceBusinessRules(false, _db);
            }

            return 0;
        }


        #endregion
        #region Methods to change foreign key references
        #endregion
    }
}
