﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.VITAS
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class VitasNotificationManager : BaseBO
    {
        public VitasNotificationManager() 
        {
            _defaultSortField = "notification.ID";

        }

        public const int NullValueForStdNotificationTypeId = 4;

        private IQueryable<IM_NOTIFICATION_LETTER> LinqAll()
        {
            //populate LinqAll
            IQueryable<IM_NOTIFICATION_LETTER> tempLinq = (from e in _db.IM_NOTIFICATION_LETTERs select e);
            return tempLinq;
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<IM_NOTIFICATION_LETTER>(e => e.IM_IMPLANT_ITEM);
            lo.LoadWith<IM_NOTIFICATION_LETTER>(e => e.IM_RECALL);
            lo.LoadWith<IM_NOTIFICATION_LETTER>(e => e.STD_IM_NOTIFICATION_STATUS);
            lo.LoadWith<IM_NOTIFICATION_LETTER>(e => e.STD_IM_NOTIFICATION_TYPE);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_PATIENT_SURGERY);
            lo.LoadWith<IM_PATIENT_SURGERY>(e => e.PATIENT);
            lo.LoadWith<IM_PATIENT_SURGERY>(e => e.STD_INSTITUTION);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #region Method for Check Existence


        #endregion


        #region Methods for Select

        public IEnumerable<IM_NOTIFICATION_LETTER> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                List<IM_NOTIFICATION_LETTER> entities;
                entities = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;

            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return SelectLinqFilter().Count();
            }
        }

        private IQueryable<IM_NOTIFICATION_LETTER> SelectLinqFilter()
        {
            IQueryable<IM_NOTIFICATION_LETTER> linqFilter = LinqAll();
            return linqFilter;
        }

        public IEnumerable<IM_NOTIFICATION_LETTER> GetNotifications(string nameSearchValue, string recallSearchValue, NotificationStatus status, string ntUserName, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = "notification.IM_RECALL.SOURCE_RECALL_ID";

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<IM_NOTIFICATION_LETTER> entities;
                entities = (GetNotificationsLinqFilter(nameSearchValue, recallSearchValue, status, ntUserName)).Select(c => new { notification = c }).OrderBy(sort).Select(c => c.notification).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        public int GetNotificationsCount(string nameSearchValue, string recallSearchValue, NotificationStatus status, string ntUserName)
        {
            using (_db = GetDataContext())
            {
                return GetNotificationsLinqFilter(nameSearchValue, recallSearchValue, status, ntUserName).Count();
            }
        }

        private IQueryable<IM_NOTIFICATION_LETTER> GetNotificationsLinqFilter(string nameSearchValue, string recallSearchValue, NotificationStatus status, string ntUserName)
        {
            IQueryable<IM_NOTIFICATION_LETTER> linqFilter = (from a in _db.IM_NOTIFICATION_LETTERs
                                                             where a.ID.Equals(-1)
                                                             select a);  

            List<int> patientIds = (from p in _db.PRIMARY_CARE_PHYSICIANs
                                    where p.PROVIDER.NTUSERNAME.Equals(ntUserName)
                                    select p.PATIENT_ID).ToList();

            List<int> patientIds2 = (from p in _db.IM_NOTIFICATION_LETTERs
                                    where p.PROVIDER.NTUSERNAME.Equals(ntUserName)
                                    select p.IM_IMPLANT_ITEM.IM_PATIENT_SURGERY.PATIENT_ID).ToList();

            if (patientIds2.Count > 0)
            {
                patientIds.AddRange(patientIds2);
            }

            if (patientIds.Count > 0)
            {

                linqFilter = (from a in _db.IM_NOTIFICATION_LETTERs
                              where a.STD_IM_NOTIFICATION_STATUS_ID.Equals(status)
                              && a.PROVIDER_ID != null
                              && patientIds.Contains(a.IM_IMPLANT_ITEM.IM_PATIENT_SURGERY.PATIENT_ID)
                              select a);

                if (!String.IsNullOrEmpty(nameSearchValue))
                {
                    linqFilter = (from a in linqFilter
                                  where a.IM_IMPLANT_ITEM.IM_PATIENT_SURGERY.PATIENT.LAST_NAME.Equals(nameSearchValue)
                                  select a);
                }

                if (!String.IsNullOrEmpty(recallSearchValue))
                {
                    linqFilter = (from a in linqFilter
                                  where a.IM_RECALL.SOURCE_RECALL_ID.Equals(recallSearchValue)
                                  select a);
                }
            }
            return linqFilter;
        }

        #endregion

        #region Methods for SelectByID

        public IM_NOTIFICATION_LETTER SelectByID(int id)
        {
            IM_NOTIFICATION_LETTER notificationLetter;
            if (id == 0)
            {
                notificationLetter = new IM_NOTIFICATION_LETTER();
                notificationLetter.SetAsChangeTrackingRoot(EntityState.New, true);
                notificationLetter.SetAsInsertOnSubmit();
            }
            else
            {
                using (_db = GetDataContext())
                {
                    SetLoadWith(_db);
                    notificationLetter = SelectByIDLinqFilter(id).SingleOrDefault();

                    if (notificationLetter != null)
                    {
                        notificationLetter.SetAsChangeTrackingRoot(true);
                        notificationLetter.SetAsUpdateOnSubmit();
                    }
                }
            }
            return notificationLetter;
        }

        private IQueryable<IM_NOTIFICATION_LETTER> SelectByIDLinqFilter(int id)
        {
            IQueryable<IM_NOTIFICATION_LETTER> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.ID == id select t;

            return linqFilter;
        }

        public IM_NOTIFICATION_LETTER GetPendingFollowupNotification(int parentID)
        {
            IM_NOTIFICATION_LETTER notificationLetter;

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                notificationLetter = SelectPendingFollowupLinqFilter(parentID).SingleOrDefault();
            }

            if (notificationLetter != null)
            {
                notificationLetter.SetAsChangeTrackingRoot(true);
                notificationLetter.SetAsUpdateOnSubmit();
            }
            else
            {
                notificationLetter = new IM_NOTIFICATION_LETTER();
                notificationLetter.SetAsChangeTrackingRoot(EntityState.New, true);
                notificationLetter.SetAsInsertOnSubmit();
            }

            return notificationLetter;
        }

        private IQueryable<IM_NOTIFICATION_LETTER> SelectPendingFollowupLinqFilter(int parentID)
        {
            IQueryable<IM_NOTIFICATION_LETTER> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.PARENT_NOTIFICATION_LETTER_ID == parentID
                         && t.STD_IM_NOTIFICATION_STATUS_ID != Convert.ToInt16(NotificationStatus.Completed)
                         select t;

            return linqFilter;
        }

        #endregion

        #region Methods for Update

        /// <summary>
        /// Save Notification Letter data to database.
        /// </summary>
        /// <param name="notificationLetter"></param>
        /// <returns></returns>
        public int Update(IM_NOTIFICATION_LETTER notificationLetter)
        {
            if (notificationLetter.ID > 0)
                return UpdateNotificationLetter(notificationLetter);
            else
                return CreateNewNotificationLetter(notificationLetter);
        }

        #endregion

        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref IM_NOTIFICATION_LETTER notificationLetter)
        {
            return 0;
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Assign "NULL VALUE" fore NOT NULL item if not set
        /// </summary>
        /// <param name="notificationLetter"></param>
        /// <returns></returns>
        private IM_NOTIFICATION_LETTER SetDefaultValues(IM_NOTIFICATION_LETTER notificationLetter)
        {
            if (notificationLetter.STD_IM_NOTIFICATION_TYPE_ID == 0)
            {
                notificationLetter.STD_IM_NOTIFICATION_TYPE_ID = NullValueForStdNotificationTypeId;
            }

            return notificationLetter;
        }

        private int UpdateNotificationLetter(IM_NOTIFICATION_LETTER notificationLetter)
        {
            notificationLetter.SetAsUpdateOnSubmit();

            notificationLetter = SetDefaultValues(notificationLetter);

            return Save(notificationLetter);
        }

        private int CreateNewNotificationLetter(IM_NOTIFICATION_LETTER notificationLetter)
        {
            int returnStatus = 0;

            notificationLetter.SetAsChangeTrackingRoot(EntityState.New, true);
            notificationLetter.SetAsInsertOnSubmit();

            notificationLetter = SetDefaultValues(notificationLetter);

            IM_NOTIFICATION_LETTER notificationLetterCopy = notificationLetter.Clone();
            returnStatus = Save(notificationLetterCopy);

            return returnStatus;
        }

        private int Save(IM_NOTIFICATION_LETTER notificationLetter)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                notificationLetter.SynchroniseWithDataContext(_db);

                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present, submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                            eventArgs.SavedItemId = notificationLetter.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                            eventArgs.SavedItemId = notificationLetter.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch (Exception ex)
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                        Console.Write(ex.Message);
                    }
                }

            }

            return returnStatus;
        }

        #endregion

    }
}
