﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.VITAS
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class VitasImplantManager : BaseBO
    {
        public VitasImplantManager() 
        {
            _defaultSortField = "implant.ID";

        }

        public const int NullValueForStdImplantTypeId = 76;
        public const int NullValueForStdImplantSpecialtyAreaId = 20;
        public const int NullValueForStdImplantProviderId = 2;
        public const int NullValueForStdImplantFacilityId = 1;
        public const int NullValueForStdImplantAnatomicalLocationId = 23;
        public const int NullValueForStdImplantInformationSourceId = 9;
        public const int NullValueForStdImplantNotificationStatusId = 4;

        public const int NullValueForStdImplantDeviceStatusId = 7;

        public const int DefaultImplantStatusId = 1;

        private IQueryable<IM_IMPLANT_ITEM> LinqAll()
        {
            //populate LinqAll
            IQueryable<IM_IMPLANT_ITEM> tempLinq = (from e in _db.IM_IMPLANT_ITEMs
                                            select e);
            return tempLinq;
        }

        private IQueryable<IM_IMPLANT_ITEM> LinqAllByPatientId(int patientId)
        {
            IQueryable<IM_IMPLANT_ITEM> tempLinq = (from a in _db.IM_IMPLANT_ITEMs
                                                    where a.IM_PATIENT_SURGERY.PATIENT_ID.Equals(patientId)
                                                    select a);
            return tempLinq;
        }
        
        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.STD_IM_SPECIALTY_AREA);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.STD_IM_IMPLANT_TYPE);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_IMPLANT_ITEM_MATERIAL_TYPEs);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_IMPLANT_DEVICE_STATUS);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_RECALL);
            lo.LoadWith<IM_IMPLANT_ITEM_MATERIAL_TYPE>(e => e.STD_IM_MATERIAL_TYPE);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_PATIENT_SURGERY);
            lo.LoadWith<IM_PATIENT_SURGERY>(e => e.PATIENT);
            lo.LoadWith<IM_PATIENT_SURGERY>(e => e.STD_INSTITUTION);
            lo.LoadWith<IM_PATIENT_SURGERY>(e => e.STD_IM_INFORMATIN_SOURCE);
            lo.LoadWith<IM_IMPLANT_DEVICE_STATUS>(e => e.STD_IM_DEVICE_STATUS);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForListViews(RegistriesDataAccess db, int patientId)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.STD_IM_SPECIALTY_AREA);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.STD_IM_IMPLANT_TYPE);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_PATIENT_SURGERY);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_RECALL);
            lo.LoadWith<IM_IMPLANT_ITEM>(e => e.IM_IMPLANT_DEVICE_STATUS);
            lo.LoadWith<IM_IMPLANT_DEVICE_STATUS>(e => e.STD_IM_DEVICE_STATUS);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }
        #region Method for Check Existence

        public bool ImplantExists(int id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.IM_IMPLANT_ITEMs where e.ID == id select e).Count();

                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        #endregion


        public STD_IM_SPECIALTY_AREA_IMPLANT_TYPE GetSpecialtyAreaImplantTypeByIds(int implantTypeId, int specialtyid)
        {
            STD_IM_SPECIALTY_AREA_IMPLANT_TYPE entity;

            using (_db = GetDataContext())
            {
                entity = (from sp in _db.STD_IM_SPECIALTY_AREA_IMPLANT_TYPEs
                          where sp.STD_IM_SPECIALTY_AREA_ID == specialtyid
                              && sp.STD_IMPLANT_TYPE_ID == implantTypeId
                          select sp).FirstOrDefault();

            }

            return entity;
        }

        public STD_IM_SPECIALTY_AREA GetSpecialtyAreaTypeById(int specialtyid)
        {
            STD_IM_SPECIALTY_AREA entity;

            using (_db = GetDataContext())
            {
                entity = (from sa in _db.STD_IM_SPECIALTY_AREAs
                           where sa.ID == specialtyid
                          select sa).FirstOrDefault();

            }

            return entity;
        }

        #region Methods for Select

        public IEnumerable<IM_IMPLANT_ITEM> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                List<IM_IMPLANT_ITEM> entities;
                entities = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;

            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return SelectLinqFilter().Count();
            }
        }

        private IQueryable<IM_IMPLANT_ITEM> SelectLinqFilter()
        {
            IQueryable<IM_IMPLANT_ITEM> linqFilter = LinqAll();
            return linqFilter;
        }

        public IEnumerable<IM_IMPLANT_ITEM> GetNotifications(string searchValue, NotificationStatus status, string email, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = "implant.IM_RECALL.SOURCE_RECALL_ID";

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<IM_IMPLANT_ITEM> entities;
                entities = (GetNotificationsLinqFilter(searchValue, status, email)).Select(c => new { implant = c, status = c.IM_IMPLANT_DEVICE_STATUS.First() }).OrderBy(sort).Select(c => c.implant).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        public int GetNotificationsCount(string searchValue, NotificationStatus status, string email)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return GetNotificationsLinqFilter(searchValue, status, email).Count();
            }
        }

        private IQueryable<IM_IMPLANT_ITEM> GetNotificationsLinqFilter(string searchValue, NotificationStatus status, string email)
        {
            IQueryable<IM_IMPLANT_ITEM> linqFilter;

            if (email == String.Empty)
            {
                linqFilter = (from a in _db.IM_IMPLANT_ITEMs
                              where a.ID.Equals(-1)
                              select a);
            }
            else
            {
                if ((searchValue == null) || (searchValue == ""))
                {
                    linqFilter = (from a in _db.IM_IMPLANT_ITEMs
                                  where a.STD_IM_NOTIFICATION_STATUS_ID.Equals(status)
                                  && a.PROVIDER.EMAIL_ADDRESS.Equals(email)
                                  select a);
                }
                else
                {
                    linqFilter = (from a in _db.IM_IMPLANT_ITEMs
                                  where a.IM_PATIENT_SURGERY.PATIENT.LAST_NAME.Equals(searchValue)
                                  && a.STD_IM_NOTIFICATION_STATUS_ID.Equals(status)
                                  && a.PROVIDER.EMAIL_ADDRESS.Equals(email)
                                  select a);
                }
            }
            
            return linqFilter;
        }

        #endregion

        #region Methods for SelectByID

        public IM_IMPLANT_ITEM SelectByID(int id)
        {
            IM_IMPLANT_ITEM implantItem;
            if (id == 0)
            {
                implantItem = new IM_IMPLANT_ITEM();
                //implantItem.IM_PATIENT_SURGERY = new IM_PATIENT_SURGERY();
                //implantItem.STD_IM_SPECIALTY_AREA_IMPLANT_TYPE.STD_IM_SPECIALTY_AREA = new STD_IM_SPECIALTY_AREA();
                //implantItem.STD_IM_SPECIALTY_AREA_IMPLANT_TYPE.STD_IM_IMPLANT_TYPE = new STD_IM_IMPLANT_TYPE();
                //implantItem.STD_IM_MANUFACTURE = new STD_IM_MANUFACTURE();
                //implantItem.IM_IMPLANT_ITEM_MATERIAL_TYPEs = new EntitySet<IM_IMPLANT_ITEM_MATERIAL_TYPE>();
                //implantItem.IM_PATIENT_SURGERY.STD_IM_INFORMATIN_SOURCE = new STD_IM_INFORMATIN_SOURCE();

                implantItem.SetAsChangeTrackingRoot(EntityState.New, true);
                implantItem.SetAsInsertOnSubmit();
            }
            else
            {
                using (_db = GetDataContext())
                {
                    SetLoadWith(_db);
                    implantItem = SelectByIDLinqFilter(id).SingleOrDefault();

                    if (implantItem != null)
                    {
                        implantItem.SetAsChangeTrackingRoot(true);
                        implantItem.SetAsUpdateOnSubmit();

                        int status = implantItem.EnforceBusinessRules(true, _db);
                    }
                }
            }
            return implantItem;
        }


     

        private IQueryable<IM_IMPLANT_ITEM> SelectByIDLinqFilter(int id)
        {
            IQueryable<IM_IMPLANT_ITEM> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.ID == id select t;

            return linqFilter;
        }

        public IEnumerable<IM_IMPLANT_ITEM> SelectByPatientID(int patientId, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                if (string.IsNullOrEmpty(sort))
                    sort = _defaultSortField;

                SetLoadWithForListViews(_db, patientId);
                IEnumerable<IM_IMPLANT_ITEM> entities;
                entities = (LinqAllByPatientId(patientId)).Select( c => new {implant = c, status = c.IM_IMPLANT_DEVICE_STATUS.First()}).OrderBy(sort).Select( c => c.implant).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }

        }

        public int SelectByPatientIDCount(int patientId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db, patientId);
                return (LinqAllByPatientId(patientId)).Count();
            }
        }

        #endregion

        #region Methods for Update

        /// <summary>
        /// Save Implant Item data to database.
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        public int Update(IM_IMPLANT_ITEM implantItem, int statusId)
        {
            if (implantItem.ID > 0)
                return UpdateImplantItem(implantItem, statusId);
            else
                return CreateNewImplantItem(implantItem, statusId);
        }

        #endregion

        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref IM_IMPLANT_ITEM  implantItem)
        {
            using (_db = GetDataContext())
            {
                int status = implantItem.EnforceBusinessRules(false, _db);
            }

            return 0;
        }
        #endregion

        #region Private Methods

        /// <summary>
        /// Update IM_PATIENT_SURGERY table
        /// </summary>
        /// <param name="patientSurgeryObject"></param>
        /// <param name="patientSurgeryId"></param>
        /// <returns></returns>
        private int Update(IM_PATIENT_SURGERY patientSurgery, IM_IMPLANT_ITEM item)
        //private int Update(IM_IMPLANT_ITEM item)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                item.SynchroniseWithDataContext(_db);
                patientSurgery.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.
                //item.SynchroniseWithDataContext(_db);
                //Check if any actual changes will occur
                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                            eventArgs.SavedItemId = patientSurgery.ID;
                        //eventArgs.SavedItemId = item.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                            eventArgs.SavedItemId = patientSurgery.ID;
                        //eventArgs.SavedItemId = item.ID; 
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch (Exception ex)
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                        Console.Write(ex.Message);
                    }
                }

            }

            return returnStatus;
        }


        private int InsertSurgery(IM_PATIENT_SURGERY patientSurgery)
        //private int Update(IM_IMPLANT_ITEM item)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                
                patientSurgery.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.
                //item.SynchroniseWithDataContext(_db);
                //Check if any actual changes will occur
                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                            eventArgs.SavedItemId = patientSurgery.ID;
                        //eventArgs.SavedItemId = item.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                            eventArgs.SavedItemId = patientSurgery.ID;
                        //eventArgs.SavedItemId = item.ID; 
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch (Exception ex)
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                        Console.Write(ex.Message);
                    }
                }

            }

            return returnStatus;
        }


        /// <summary>
        /// Insert a new Device Status for new Implant Item.
        /// </summary>
        /// <param name="patientSurgeryId"></param>
        /// <returns></returns>
        private void InsertDeviceStatus(IM_IMPLANT_ITEM implantItem, int statusId)
        {
            IM_IMPLANT_DEVICE_STATUS newDeviceStatus = new IM_IMPLANT_DEVICE_STATUS();
            newDeviceStatus.STD_IM_DEVICE_STATUS_ID = statusId;
            newDeviceStatus.IM_IMPLANT_ITEM_ID = implantItem.ID;
            newDeviceStatus.STATUS_DATE = DateTime.Now;
            implantItem.IM_IMPLANT_DEVICE_STATUS.Add(newDeviceStatus);
        }

        /// <summary>
        /// Assign "NULL VALUE" fore NOT NULL item if not set
        /// </summary>
        /// <param name="patientSurgery"></param>
        /// <returns></returns>
        private IM_PATIENT_SURGERY SetDefaultValues(IM_PATIENT_SURGERY patientSurgery)
        {
            if (patientSurgery.STD_IMPLANT_FACILITY_ID == 0)
                patientSurgery.STD_IMPLANT_FACILITY_ID = null;

            if (patientSurgery.STD_INFORMATION_SOURCE_ID == 0)
                patientSurgery.STD_INFORMATION_SOURCE_ID = NullValueForStdImplantInformationSourceId;

            return patientSurgery;
        }

        /// <summary>
        /// Assign "NULL VALUE" fore NOT NULL item if not set
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        private IM_IMPLANT_ITEM SetDefaultValues(IM_IMPLANT_ITEM implantItem)
        {
            if (implantItem.STD_IM_SPECIALTY_AREA_ID == 0)
                implantItem.STD_IM_SPECIALTY_AREA_ID = NullValueForStdImplantSpecialtyAreaId;

            if (implantItem.STD_IM_IMPLANT_TYPE_ID == 0)
                implantItem.STD_IM_IMPLANT_TYPE_ID = NullValueForStdImplantTypeId;

            if (implantItem.STD_IM_NOTIFICATION_STATUS_ID == 0)
                implantItem.STD_IM_NOTIFICATION_STATUS_ID = NullValueForStdImplantNotificationStatusId;

            return implantItem;
        }

        private IM_IMPLANT_ITEM MaintainAlertFlag(IM_IMPLANT_ITEM implantItem, int deviceStatusID)
        {
            if (deviceStatusID == 5) //Explanted
            {
                implantItem.IMPLANT_ALERT_FLAG = false;
            }

            return implantItem;
        }

        /// <summary>
        /// Update an Implant Item is actually updating the root patient surgery object.
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        private int UpdateImplantItem(IM_IMPLANT_ITEM implantItem, int statusId)
        {
            EntitySet<IM_IMPLANT_ITEM_MATERIAL_TYPE> materialTypes = implantItem.IM_IMPLANT_ITEM_MATERIAL_TYPEs.Clone();
            IM_PATIENT_SURGERY patientSurgery = implantItem.IM_PATIENT_SURGERY;

            patientSurgery.SetAsChangeTrackingRoot(true);

            patientSurgery = SetDefaultValues(patientSurgery);

            patientSurgery.SetAsUpdateOnSubmit();
            implantItem.SetAsUpdateOnSubmit();

            // Keep the entity state for material types.
            foreach (IM_IMPLANT_ITEM_MATERIAL_TYPE item in materialTypes)
            {
                if (item.LINQEntityState == EntityState.New)
                    implantItem.IM_IMPLANT_ITEM_MATERIAL_TYPEs[materialTypes.IndexOf(item)].SetAsInsertOnSubmit();
                else if (item.LINQEntityState == EntityState.Deleted)
                    implantItem.IM_IMPLANT_ITEM_MATERIAL_TYPEs[materialTypes.IndexOf(item)].SetAsDeleteOnSubmit();
            }

            if (implantItem.IM_IMPLANT_DEVICE_STATUS.Count() == 0)
            {
                InsertDeviceStatus(implantItem, statusId);
            }
            else
            {
                implantItem.IM_IMPLANT_DEVICE_STATUS.First().STATUS_DATE = DateTime.Now;
                implantItem.IM_IMPLANT_DEVICE_STATUS.First().STD_IM_DEVICE_STATUS = GetStatusById(statusId);
                implantItem.IM_IMPLANT_DEVICE_STATUS.First().STD_IM_DEVICE_STATUS_ID = statusId;
            }

            implantItem = SetDefaultValues(implantItem);
            implantItem = MaintainAlertFlag(implantItem, statusId);

            return Update(patientSurgery, implantItem);
        }

        /// <summary>
        /// Create a new implant item
        ///    Step 1:  Insert a new record to IM_PATIENT_SURGERY table
        ///             and get the newly created IM_PATIENT_SURGERY_ID
        ///    Step 2:  Insert a new record to IM_IMPLANT_DEVICE_STATUS table
        ///    Step 3:  Insert a new record to IM_IMPLANT_ITEM table 
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        private int CreateNewImplantItem(IM_IMPLANT_ITEM implantItem, int deviceStatusId)
        {
            int returnStatus = 0;

            // Creat a new IM_PATIENT_SURGERY record and get the IM_PATIENT_SURGERY_ID
            IM_PATIENT_SURGERY patientSurgery = implantItem.IM_PATIENT_SURGERY.Clone();
            //patientSurgery.STD_IM_INFORMATIN_SOURCE = implantItem.IM_PATIENT_SURGERY.STD_IM_INFORMATIN_SOURCE.Clone();
            patientSurgery.IM_IMPLANT_ITEMs.Clear();
            patientSurgery = SetDefaultValues(patientSurgery);
            patientSurgery.SetAsChangeTrackingRoot(EntityState.New, true);
            patientSurgery.SetAsInsertOnSubmit();

            //IM_IMPLANT_ITEM copyItem = implantItem.Clone();

            //implantItem.SetAsInsertOnSubmit();
            if (InsertSurgery(patientSurgery) > 0)
            {
                // Insert a new record for Device Status
                InsertDeviceStatus(implantItem, deviceStatusId);

                // Insert a new record for implant item
                implantItem.IM_PATIENT_SURGERY = patientSurgery;
                implantItem.IM_PATIENT_SURGERY_ID = patientSurgery.ID;
                implantItem = SetDefaultValues(implantItem);
                implantItem.SetAsChangeTrackingRoot(EntityState.New, true);
                implantItem.SetAsInsertOnSubmit();

                IM_IMPLANT_ITEM implantCopy = implantItem.Clone();
                returnStatus = InsertImplantItem(implantCopy);
            }

            return returnStatus;
        }


        private STD_IM_DEVICE_STATUS GetStatusById(int id)
        {
            STD_IM_DEVICE_STATUS idst;
            using (_db = GetDataContext())
            {

                idst = (from st in _db.STD_IM_DEVICE_STATUS where st.ID == id select st).FirstOrDefault();

            }

            return idst;

        }
        /// <summary>
        /// Insert a new implant item to IM_IMPLANT_ITEM table
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        private int InsertImplantItem(IM_IMPLANT_ITEM implantItem)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                implantItem.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                //Check if any actual changes will occur
                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                        eventArgs.SavedItemId = implantItem.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                        eventArgs.SavedItemId = implantItem.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }
                return returnStatus;
            }
        }

        #endregion

    }
}
