﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.VITAS
{
    public enum ActionTakenType
    {
        None = 0,
        NullValue = 1,
        Other = 2,
        ResponsetoImplantAlert = 3,
        Explanted = 4,
        Abandoned = 5
    }

    [Serializable()]
    [System.ComponentModel.DataObject]
    public class VitasActionManager : BaseBO
    {
        public VitasActionManager() 
        {
            _defaultSortField = "action.ID";

        }

        public const int NullValueForStdActionType = 1;
        public const int NullValueForStdNoActionTakenId = 9;
        public const int NullValueForStdExplantedReasonId = 17;
        public const int NullValueForStdAbandonedReasonId = 1;

        private IQueryable<IM_RECORD_ACTION_TAKEN> LinqAll()
        {
            //populate LinqAll
            IQueryable<IM_RECORD_ACTION_TAKEN> tempLinq = (from e in _db.IM_RECORD_ACTION_TAKENs
                                            select e);
            return tempLinq;
        }

        private IQueryable<IM_RECORD_ACTION_TAKEN> LinqAllActionsByImplantId(int implantId)
        {
            IQueryable<IM_RECORD_ACTION_TAKEN> tempLinq = (from a in _db.IM_RECORD_ACTION_TAKENs
                                                           where a.IM_IMPLANT_ITEM.ID.Equals(implantId)
                                                    select a);
            return tempLinq;
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<IM_RECORD_ACTION_TAKEN>(e => e.STD_IM_ACTION_TYPE);
            lo.LoadWith<IM_RECORD_ACTION_TAKEN>(e => e.STD_IM_NO_ACTION_TAKEN);
            lo.LoadWith<IM_RECORD_ACTION_TAKEN>(e => e.STD_IM_EXPLANTED_REASON);
            lo.LoadWith<IM_RECORD_ACTION_TAKEN>(e => e.STD_IM_ABANDONED_REASON);
            lo.LoadWith<IM_RECORD_ACTION_TAKEN>(e => e.IM_IMPLANT_ITEM);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForListViewsActions(RegistriesDataAccess db, int implantId)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<IM_RECORD_ACTION_TAKEN>(e => e.STD_IM_ACTION_TYPE);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #region Method for Check Existence

        public bool ActionExists(int id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.IM_RECORD_ACTION_TAKENs where e.ID == id select e).Count();

                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        #endregion


        public STD_IM_ACTION_TYPE GetActionTypeById(int actionTypeID)
        {
            STD_IM_ACTION_TYPE entity;

            using (_db = GetDataContext())
            {
                entity = (from sa in _db.STD_IM_ACTION_TYPEs
                          where sa.ID == actionTypeID
                          select sa).FirstOrDefault();

            }

            return entity;
        }

        #region Methods for Select

        public IEnumerable<IM_RECORD_ACTION_TAKEN> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                List<IM_RECORD_ACTION_TAKEN> entities;
                entities = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;

            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                return SelectLinqFilter().Count();
            }
        }

        private IQueryable<IM_RECORD_ACTION_TAKEN> SelectLinqFilter()
        {
            IQueryable<IM_RECORD_ACTION_TAKEN> linqFilter = LinqAll();
            return linqFilter;
        }

        #endregion

        #region Methods for SelectByID

        public IM_RECORD_ACTION_TAKEN SelectByID(int id)
        {
            IM_RECORD_ACTION_TAKEN actionTaken;
            if (id == 0)
            {
                actionTaken = new IM_RECORD_ACTION_TAKEN();

                actionTaken.SetAsChangeTrackingRoot(EntityState.New, true);
                actionTaken.SetAsInsertOnSubmit();
            }
            else
            {
                using (_db = GetDataContext())
                {
                    SetLoadWith(_db);
                    actionTaken = SelectByIDLinqFilter(id).SingleOrDefault();

                    if (actionTaken != null)
                    {
                        actionTaken.SetAsChangeTrackingRoot(true);
                        actionTaken.SetAsUpdateOnSubmit();

                        int status = actionTaken.EnforceBusinessRules(true, _db);
                    }
                }
            }
            return actionTaken;
        }

        private IQueryable<IM_RECORD_ACTION_TAKEN> SelectByIDLinqFilter(int id)
        {
            IQueryable<IM_RECORD_ACTION_TAKEN> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.ID == id select t;

            return linqFilter;
        }

        public IEnumerable<IM_RECORD_ACTION_TAKEN> SelectActionsByImplantID(int implantId, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                if (string.IsNullOrEmpty(sort))
                    sort = _defaultSortField;

                SetLoadWithForListViewsActions(_db, implantId);
                IEnumerable<IM_RECORD_ACTION_TAKEN> entities;
                entities = (LinqAllActionsByImplantId(implantId)).Select(c => new { action = c }).OrderBy(sort).Select(c => c.action).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }

        }

        public int SelectActionsByImplantIDCount(int implantId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViewsActions(_db, implantId);
                return (LinqAllActionsByImplantId(implantId)).Count();
            }
        }

        #endregion

        #region Methods for Update

        /// <summary>
        /// Save Action Taken data to database.
        /// </summary>
        /// <param name="actionTaken"></param>
        /// <returns></returns>
        private int UpdateAction(IM_RECORD_ACTION_TAKEN actionTaken)
        {
            actionTaken.SetAsUpdateOnSubmit();
            return Save(actionTaken);
        }

        public int Update(IM_RECORD_ACTION_TAKEN actionTaken)
        {
            if (actionTaken.ID > 0)
                return UpdateAction(actionTaken);
            else
                return CreateNewAction(actionTaken);
        }

        #endregion

        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref IM_RECORD_ACTION_TAKEN actionTaken)
        {
            using (_db = GetDataContext())
            {
                int status = actionTaken.EnforceBusinessRules(false, _db);
            }

            return 0;
        }
        #endregion

        #region Private Methods

        private int Save(IM_RECORD_ACTION_TAKEN actionTaken)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                actionTaken.SynchroniseWithDataContext(_db);

                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                            eventArgs.SavedItemId = actionTaken.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                            eventArgs.SavedItemId = actionTaken.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch (Exception ex)
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                        Console.Write(ex.Message);
                    }
                }

            }

            return returnStatus;
        }

        /// <summary>
        /// Assign "NULL VALUE" fore NOT NULL item if not set
        /// </summary>
        /// <param name="actionTaken"></param>
        /// <returns></returns>
        private IM_RECORD_ACTION_TAKEN SetDefaultValues(IM_RECORD_ACTION_TAKEN actionTaken)
        {
            if (actionTaken.STD_IM_ABANDONED_REASON_ID == 0)
                actionTaken.STD_IM_ABANDONED_REASON_ID = NullValueForStdAbandonedReasonId;

            if (actionTaken.STD_IM_EXPLANTED_REASON_ID == 0)
                actionTaken.STD_IM_EXPLANTED_REASON_ID = NullValueForStdExplantedReasonId;

            if (actionTaken.STD_IM_NO_ACTION_TAKEN_ID == 0)
                actionTaken.STD_IM_NO_ACTION_TAKEN_ID = NullValueForStdNoActionTakenId;

            return actionTaken;
        }

        private int CreateNewAction(IM_RECORD_ACTION_TAKEN actionTaken)
        {
            int returnStatus = 0;

            actionTaken.SetAsChangeTrackingRoot(EntityState.New, true);
            actionTaken.SetAsInsertOnSubmit();

            IM_RECORD_ACTION_TAKEN actionCopy = actionTaken.Clone();
            returnStatus = Save(actionCopy);

            return returnStatus;
        }

        /// <summary>
        /// Insert a new action to IM_RECORD_ACTION_TAKEN table
        /// </summary>
        /// <param name="implantItem"></param>
        /// <returns></returns>
        private int InsertAction(IM_RECORD_ACTION_TAKEN actionTaken)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                actionTaken.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                //Check if any actual changes will occur
                ChangeSet changeSet = _db.GetChangeSet();

                if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                || changeSet.Updates.Count > 0)
                {
                    //if changes present submit changes
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                        eventArgs.SavedItemId = actionTaken.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        returnStatus =
                        eventArgs.SavedItemId = actionTaken.ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }

                    catch
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }
                return returnStatus;
            }
        }

        #endregion

    }
}
