﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Text;
using System.Data;
using System.Data.Linq;
using System.Security.Principal;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.Business.Utilities;
using System.Web.Configuration;
using System.Web.Security;

namespace VeteransAffairs.Registries.BusinessManager
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class UsersManager : BaseBO
    {
        private UserAccountManager _user = (UserAccountManager)System.Threading.Thread.CurrentPrincipal;
        private RegistriesCommonManager _commonManager = new RegistriesCommonManager();

        #region Constructor

        public UsersManager()
        {
            _defaultSortField = "USER_ID";
        }

        #endregion


        /// <summary>
        /// Returns iqueryable of all users in system.  This is used when need to find all usersnames in system.
        /// </summary>
        /// <param name="RegistryCode"></param>
        /// <returns></returns>
        private List<USER> LinqAll()
        {
            //populate LinqAll
            List<USER> users = (from e in _commonManager.GetAllUsers()
                                select e).ToList();
            return users;
        }

        /// <summary>
        /// Returns iqueryable of active users in specific registry
        /// </summary>
        /// <param name="RegistryCode"></param>
        /// <returns></returns>
        //TLB.Fortify
        //private List<USER> LinqAllActiveUsers(string RegistryCode)
        //{
        //    //populate LinqAll
        //    List<USER> users = (from e in _commonManager.GetAllUsers()
        //                        where e.USER_REGISTRies.Any(s => s.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode))
        //                        && e.INACTIVE_FLAG == false
        //                        select e).ToList();
        //    return users;
        //}

        /// <summary>
        /// Returns iqueryable of active users in specific registry
        /// </summary>
        /// <param name="RegistryCode"></param>
        /// <returns></returns>
        private IQueryable<USER_ROLE> LinqAllActiveRoles(string RegistryCode)
        {
            return (from r in _db.USER_ROLEs
                    where r.STD_ROLE.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode)
                        && r.STD_ROLE.INACTIVE_FLAG == false
                    && r.INACTIVE_FLAG == false
                    select r);
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER>(i => i.USER_ROLEs);
            lo.AssociateWith<USER>(i => i.USER_ROLEs.Where(e => e.INACTIVE_FLAG == false));
            lo.LoadWith<STD_ROLE>(i => i.ROLE_PERMISSIONs);
            lo.LoadWith<ROLE_PERMISSION>(i => i.STD_APPPAGE);
            lo.LoadWith<USER>(i => i.USER_REGISTRies);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForListViews(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER_ROLE>(i => i.STD_ROLE);
            lo.LoadWith<USER_ROLE>(i => i.USER);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForListVISNs(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER_ROLE>(i => i.STD_ROLE);
            lo.LoadWith<USER_ROLE>(i => i.USER);
            lo.LoadWith<USER_ROLE>(i => i.STD_INSTITUTION);
            lo.LoadWith<STD_INSTITUTION>(i => i.STD_INSTITUTION_PARENT);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        //private void RemoveAdHocReportingRole(string userName, string userDomain)
        //{
        //    string adminDomain = string.Empty;

        //    VeteransAffairs.Registries.Web.Membership.RegistryMembershipProvider _customProvider = (VeteransAffairs.Registries.Web.Membership.RegistryMembershipProvider)Membership.Provider;

        //    string adminUserName = _customProvider.DecryptString(GetConfigEntry("AdhocReportsAdminUserName"));
        //    string adminPwd = _customProvider.DecryptString(GetConfigEntry("AdhocReportsAdminUserPwd"));
        //    string adhocReportsFolder = GetConfigEntry("AdhocReportsFolder");

        //    string[] userNameInfo = adminUserName.Split(new char[] { '\\' });
        //    if (userNameInfo.Length == 2)
        //    {
        //        adminDomain = userNameInfo[0];
        //        adminUserName = userNameInfo[1];
        //    }
        //    else
        //    {
        //        adminUserName = string.Empty;
        //    }

        //    VeteransAffairs.Registries.BusinessManager.BCCCR.ReportsUserManager reportingManager = new VeteransAffairs.Registries.BusinessManager.BCCCR.ReportsUserManager(adminUserName, adminPwd, adminDomain);
        //    reportingManager.RemoveUserFromReportBuilder(userName, adhocReportsFolder);
        //    reportingManager.Dispose();
        //}

        //TLB.Fortify
        //private static string GetConfigEntry(string key)
        //{
        //    string Val = string.Empty;
        //    if (WebConfigurationManager.AppSettings.Count > 0)
        //        Val = WebConfigurationManager.AppSettings.Get(key);

        //    return Val;
        //}

        #region Methods for Select All
        /// <summary>
        /// Return a list of all users in system.  Both active and inactive
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public IEnumerable<USER> Select(string RegistryCode, string sort, int startRow, int maxRows)
        {
            IEnumerable<USER> objReturn = null;

            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectLinqFilter(RegistryCode).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            _db.Dispose();

            return objReturn;
        }

        public IEnumerable<USER> SelectActive(string RegistryCode, string sort, int startRow, int maxRows)
        {
            IEnumerable<USER> objReturn = null;

            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectListRegistryUsersLinqFilter(RegistryCode, null).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            _db.Dispose();

            return objReturn;
        }

        public int SelectCount(string RegistryCode, string sort, int startRow, int maxRows)
        {
            int objReturn = 0;

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectLinqFilter(RegistryCode).Count();
            _db.Dispose();

            return objReturn;
        }

        private IQueryable<USER> SelectLinqFilter(string RegistryCode)
        {
            IQueryable<USER> users = (from e in _db.USERs
                                      where e.USER_REGISTRies.Any(s => s.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode))
                                      && e.INACTIVE_FLAG == false
                                      && e.USER_ROLEs.Any(r => r.STD_ROLE.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode))
                                      select e);
            return users;
        }

        #endregion

        #region Methods for Select Search
        /// <summary>
        /// Return a list of all users in specific registry searched by last name and login.  Both active and inactive
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public IEnumerable<USER> SelectListRegistryUsers(string RegistryCode, string searchCriteria, string sort, int startRow, int maxRows)
        {
            IEnumerable<USER> objReturn = null;

            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectListRegistryUsersLinqFilter(RegistryCode, searchCriteria).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            _db.Dispose();

            return objReturn;
        }

        public int SelectListRegistryUsersCount(string RegistryCode, string searchCriteria, string sort, int startRow, int maxRows)
        {
            int objReturn = 0;

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectListRegistryUsersLinqFilter(RegistryCode, searchCriteria).Count();
            _db.Dispose();

            return objReturn;
        }

        private IQueryable<USER> SelectListRegistryUsersLinqFilter(string RegistryCode, string searchCriteria)
        {
            IQueryable<USER> linqFilter = from e in _db.USERs
                                          where e.USER_REGISTRies.Any(s => s.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode))
                                          && e.INACTIVE_FLAG == false
                                          select e;

            if (searchCriteria != null)
            {
                linqFilter = (from t in linqFilter
                              where t.USERNAME.ToLower().Contains(searchCriteria.Trim().ToLower())
                             ||
                             t.FULL_NAME.ToLower().Contains(searchCriteria.Trim().ToLower())
                             ||
                             t.FIRST_NAME.ToLower().Contains(searchCriteria.Trim().ToLower())
                             ||
                             t.LAST_NAME.ToLower().Contains(searchCriteria.Trim().ToLower())
                              select t);
            }

            return linqFilter;
        }

        #endregion

        #region Methods for Select Search By Username
        /// <summary>
        /// Return a list of all users in system searched by username.  Both active and inactive.
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public USER SelectByUsername(string Username)
        {
            USER objReturn = null;

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectByUsernameLinqFilter(Username).SingleOrDefault();
            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Search provider by first name, last name or station number
        /// </summary>
        /// <param name="searchStringFirstName"></param>
        /// <param name="searchStringLastName"></param>
        /// <param name="stationNumber"></param>
        /// <returns></returns>
        public IEnumerable<PROVIDER> SelectProviderBySearchCriteria(string searchStringFirstName, string searchStringLastName, string stationNumber)
        {
            _db = GetDataContext();
            IEnumerable<PROVIDER> entities;

            entities = SelectProviderBySearchCriteriaLinqFilter(searchStringFirstName, searchStringLastName, stationNumber).ToList();
                
            _db.Dispose();

            return entities;
        }

        private IQueryable<PROVIDER> SelectProviderBySearchCriteriaLinqFilter(string searchStringFirstName, string searchStringLastName, string stationNumber)
        {
            IQueryable<PROVIDER> linqFilter = (from p in _db.PROVIDERs
                                               select p);

            if (!string.IsNullOrEmpty(searchStringFirstName))
            {
                linqFilter = from t in linqFilter where t.FIRST_NAME.ToLower().Contains(searchStringFirstName.Trim().ToLower()) select t;
            }

            if (!string.IsNullOrEmpty(searchStringLastName))
            {
                linqFilter = from t in linqFilter where t.LAST_NAME.ToLower().Contains(searchStringLastName.Trim().ToLower()) select t;
            }

            if (!string.IsNullOrEmpty(stationNumber))
            {
                linqFilter = from t in linqFilter where t.STA3N == Helpers.ExtractSta3n(stationNumber) select t;
            }

            if (string.IsNullOrEmpty(searchStringFirstName)
                && string.IsNullOrEmpty(searchStringLastName)
                && string.IsNullOrEmpty(stationNumber))
            {
                linqFilter = from t in linqFilter where t.STA3N == 9999 select t;
            }

            return linqFilter;
        }

        /// <summary>
        /// Return a PROVIDER that matches the NTUSERNAME.
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public List<PROVIDER> SelectProviderByUsername(string Username)
        {
            List<PROVIDER> objReturn = null;

            _db = GetDataContext();
            objReturn = SelectProviderByUsernameLinqFilter(Username).ToList();
            _db.Dispose();

            return objReturn;
        }

        //TLB.Fortify
        //private PROVIDER SelectProviderByID(int id)
        //{
        //    PROVIDER objReturn = null;

        //    _db = GetDataContext();
        //    objReturn = (from e in _db.PROVIDERs where e.PROVIDER_ID == id select e).SingleOrDefault();
        //    _db.Dispose();

        //    return objReturn;
        //}


        /// <summary>
        /// Get Provider ID by Staff IEN and STA3N
        /// </summary>
        /// <param name="staffIen"></param>
        /// <param name="sta3n"></param>
        /// <returns></returns>
        public System.Nullable<int> GetProviderIdByStaffIenAndSta3n(string staffIen, int sta3n)
        {
            System.Nullable<int> providerId = null;

            if (!string.IsNullOrEmpty(staffIen) && sta3n > Helpers.InvalidSta3n)
            {
                _db = GetDataContext();
                PROVIDER provider = (from e in _db.PROVIDERs where e.STAFFIEN == staffIen && e.STA3N == sta3n select e).FirstOrDefault();
                if (provider != null)
                {
                    providerId = provider.PROVIDER_ID;
                }
                _db.Dispose();
            }

            return providerId;
        }

        private IQueryable<PROVIDER> SelectProviderByUsernameLinqFilter(string Username)
        {
            IQueryable<PROVIDER> linqFilter = (from t in _db.PROVIDERs
                                               where t.NTUSERNAME.ToLower().Equals(Username.ToLower())
                                               select t);

            return linqFilter;
        }

        private List<USER> SelectByUsernameLinqFilter(string Username)
        {
            List<USER> linqFilter = LinqAll();

            linqFilter = (from t in linqFilter
                          where t.USERNAME.Equals(Username, StringComparison.OrdinalIgnoreCase)
                          select t).ToList();

            return linqFilter;
        }


        #endregion

        #region Methods for SelectByID

        /// <summary>
        /// Return a single user by id
        /// </summary>
        /// <param name="RegistryCode"></param>
        /// <param name="id"></param>
        /// <returns></returns>
        public USER SelectByID(string RegistryCode, int id)
        {
            _db = GetDataContext();
            SetLoadWith(_db);
            USER user;

            if (id == 0)
            {
                user = new USER();
                _db.USERs.Attach(user);

                user.SetAsChangeTrackingRoot(EntityState.New, true);

                USER_REGISTRy userRegistry = new USER_REGISTRy();
                userRegistry.STD_REGISTRY_ID = _commonManager.GetRegistryId(RegistryCode);

                user.USER_REGISTRies.Add(userRegistry);

                user.SetAsInsertOnSubmit();
                userRegistry.SetAsInsertOnSubmit();
            }
            else
            {
                if (RegistriesGlobal.CurrentUser == null)
                {
                    RegistriesGlobal.CurrentUser = SelectByIDLinqFilter(RegistryCode, id).FirstOrDefault();
                }
                else if (RegistriesGlobal.CurrentUser.USER_ID != id)
                {
                    RegistriesGlobal.CurrentUser = SelectByIDLinqFilter(RegistryCode, id).FirstOrDefault();
                }

                user = RegistriesGlobal.CurrentUser;

                if (user != null)
                {
                    user.SetAsChangeTrackingRoot(true);
                }

            }

            if (user != null)
            {
                int status = user.EnforceBusinessRules(false, _db);
            }

            _db.Dispose();

            return user;

        }

        private IQueryable<USER> SelectByIDLinqFilter(string RegistryCode, int id)
        {
            IQueryable<USER> linqFilter = (from t in _db.USERs
                                           where t.USER_ID == id
                                                && t.USER_REGISTRies.Any(s => s.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode))
                                           select t);
            return linqFilter;
        }



        #endregion

        #region Methods for Select List Users Roles
        /// <summary>
        /// Return a list of all active users with all active roles in specific registry.
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public IEnumerable<USER_ROLE> SelectListUsersRoles(string RegistryCode, string searchCriteria, string sort, int startRow, int maxRows)
        {
            IEnumerable<USER_ROLE> objReturn = null;

            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            _db = GetDataContext();
            SetLoadWithForListViews(_db);
            objReturn = SelectListUsersRolesLinqFilter(RegistryCode, searchCriteria).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            _db.Dispose();

            return objReturn;
        }

        public int SelectListUsersRolesCount(string RegistryCode, string searchCriteria, string sort, int startRow, int maxRows)
        {
            int objReturn = 0;

            _db = GetDataContext();
            SetLoadWithForListViews(_db);
            objReturn = SelectListUsersRolesLinqFilter(RegistryCode, searchCriteria).Count();
            _db.Dispose();

            return objReturn;
        }

        private IQueryable<USER_ROLE> SelectListUsersRolesLinqFilter(string RegistryCode, string searchCriteria)
        {
            IQueryable<USER_ROLE> linqFilter = LinqAllActiveRoles(RegistryCode);

            if (searchCriteria != null)
            {
                linqFilter = from t in linqFilter
                             where t.USER.USERNAME.Contains(searchCriteria)
                             ||
                             t.USER.FULL_NAME.Contains(searchCriteria)
                             ||
                             t.USER.FIRST_NAME.Contains(searchCriteria)
                             select t;
            }

            return linqFilter;
        }

        public class UserRolesListResult
        {
            public int USER_ID { get; set; }
            public string USERNAME { get; set; }
            public string FULL_NAME { get; set; }
            public string LAST_NAME { get; set; }
            public string ROLE { get; set; }
            public int ROLE_ID { get; set; }
        }

        #endregion

        #region Methods for Select List Users VISNs
        /// <summary>
        /// Return a list of all active users with all active VISNs in specific registry.
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public IEnumerable<USER_ROLE> SelectListUsersVISNs(string RegistryCode, string searchCriteria, string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            IEnumerable<USER_ROLE> objReturn = null;

            _db = GetDataContext();
            SetLoadWithForListVISNs(_db);
            objReturn = SelectListUsersVISNsLinqFilter(RegistryCode, searchCriteria).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            _db.Dispose();

            return objReturn;
        }

        public int SelectListUsersVISNsCount(string RegistryCode, string searchCriteria, string sort, int startRow, int maxRows)
        {
            int objReturn = 0;

            _db = GetDataContext();
            SetLoadWithForListVISNs(_db);
            objReturn = SelectListUsersVISNsLinqFilter(RegistryCode, searchCriteria).Count();
            _db.Dispose();

            return objReturn;
        }

        private IQueryable<USER_ROLE> SelectListUsersVISNsLinqFilter(string RegistryCode, string searchCriteria)
        {
            IQueryable<USER_ROLE> linqFilter = LinqAllActiveRoles(RegistryCode);

            if (searchCriteria != null)
            {
                linqFilter = from t in linqFilter
                             where t.USER.USERNAME.Contains(searchCriteria)
                             ||
                             t.USER.FULL_NAME.Contains(searchCriteria)
                             ||
                             t.USER.FIRST_NAME.Contains(searchCriteria)
                             select t;
            }

            return linqFilter;
        }

        #endregion


        #region Check for Existence

        /// <summary>
        /// Returns false if user does not exist
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public bool UserIDExists(int id)
        {
            _db = GetDataContext();
            int count = (from e in LinqAll() where e.USER_ID == id select e).Count();
            _db.Dispose();

            if (count > 0)
            {
                return true;
            }
            else
            {
                return false;
            }
            
        }

        /// <summary>
        /// Returns false if username does not exist
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public bool UsernameExists(string nameToCheck)
        {
            _db = GetDataContext();
            int count = (from e in LinqAll() where e.USERNAME == nameToCheck select e).Count();
            _db.Dispose();

            if (count > 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Returns true if username is in specified registry
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public bool IsInRegistry(string Username, string RegistryCode)
        {
            _db = GetDataContext();
            USER_REGISTRy userRegistry = _commonManager.GetUserRegistry(_commonManager.GetRegistryId(RegistryCode), Username);
            _db.Dispose();

            if (userRegistry != null)
            {
                return true;
            }
            else
            {
                return false;
            }
            
        }


        #endregion


        #region Methods for Update

        /// <summary>
        /// Update Method.  Returns an int representing the ID of the item if the update/insert was sucessful.
        /// </summary>
        public int Update(USER user)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;

            int status = user.EnforceBusinessRules(true, _db);

            if (user.BrokenRules.Count == 0)
            {
                user.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                    // reset the global variable to null 
                    // so that the next time the variables will be retrieved again
                    RegistriesGlobal.UserRoleAll = null;
                    RegistriesGlobal.UserAll = null;
                    RegistriesGlobal.UserRegistryAll = null;
                    RegistriesGlobal.CurrentUser = null;

                    returnStatus = eventArgs.SavedItemId = user.USER_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    RaiseSaveEvent(this, eventArgs);

                }
                catch (System.Data.SqlClient.SqlException)
                {

                    eventArgs.SavedItemId = -1;
                    eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                    RaiseSaveEvent(this, eventArgs);

                    returnStatus = -1;
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _db.SubmitChanges(ConflictMode.FailOnFirstConflict);
                        RegistriesGlobal.UserAll = null;

                        eventArgs.SavedItemId = user.USER_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);

                        returnStatus = 1;
                    }
                    catch
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }
                catch
                {
                    returnStatus = 0;
                    eventArgs.SavedItemId = 0;
                    eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                    RaiseSaveEvent(this, eventArgs);
                }
            }

            _db.Dispose();

            return returnStatus;
            //TO DO : set a error message that udpate did not occur
        }


        /// <summary>
        /// Update Provider record with NTUSERNAME.  Returns an int representing the ID of the item if the update/insert was sucessful.
        /// </summary>
        public int LinkProviderWithUser(int providerId, int userId)
        {

            int returnStatus = 0;

            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
            PROVIDER itemToUpdate = null;

            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;

            try
            {
                itemToUpdate =
                    (from a in _db.PROVIDERs where a.PROVIDER_ID == providerId select a).SingleOrDefault();

                if (itemToUpdate != null)
                {
                    itemToUpdate.SetAsChangeTrackingRoot();

                    string userName = (from u in _db.USERs
                                        where u.USER_ID == userId
                                        select u.USERNAME).SingleOrDefault();
                    itemToUpdate.NTUSERNAME = userName;
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                    returnStatus =
                    eventArgs.SavedItemId = itemToUpdate.PROVIDER_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    RaiseSaveEvent(this, eventArgs);
                }
            }
            catch (ChangeConflictException)
            {
                _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                returnStatus =
                eventArgs.SavedItemId = itemToUpdate.PROVIDER_ID;
                eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                RaiseSaveEvent(this, eventArgs);
            }
            catch
            {
                returnStatus =
                eventArgs.SavedItemId = -1;
                eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                RaiseSaveEvent(this, eventArgs);
            }

            _db.Dispose();
            return returnStatus;

        }
        /// <summary>
        /// Update Provider record with null NTUSERNAME.  Returns an int representing the ID of the item if the update/insert was sucessful.
        /// </summary>
        public int UninkProviderFromUser(int providerId)
        {

            int returnStatus = 0;

            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
            PROVIDER itemToUpdate = null;

            _db = GetDataContext();
                _db.DeferredLoadingEnabled = false;

                try
                {
                    itemToUpdate =
                        (from a in _db.PROVIDERs where a.PROVIDER_ID == providerId select a).SingleOrDefault();

                    if (itemToUpdate != null)
                    {
                        itemToUpdate.SetAsChangeTrackingRoot();

                        itemToUpdate.NTUSERNAME = null;
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus =
                        eventArgs.SavedItemId = itemToUpdate.PROVIDER_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                    returnStatus =
                    eventArgs.SavedItemId = itemToUpdate.PROVIDER_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    RaiseSaveEvent(this, eventArgs);
                }
                catch
                {
                    returnStatus =
                    eventArgs.SavedItemId = -1;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    RaiseSaveEvent(this, eventArgs);
                }

            _db.Dispose();
            return returnStatus;

        }

        public int RemoveUserFromRegistry(int USER_ID, string RegistryCode)
        {
            int returnStatus = 0;

            if (USER_ID == 0 || RegistryCode == "")
            {
                return returnStatus;
            }

            //get the user
            USER user = SelectByID(RegistryCode, USER_ID);

            //if user is not null then the user is in the registry, go ahead and remove from that registry
            if (user != null)
            {
                USER_REGISTRy userRegistry = (from e in user.USER_REGISTRies
                                              where e.STD_REGISTRY_ID == _commonManager.GetRegistryId(RegistryCode)
                                              select e).FirstOrDefault();

                userRegistry.SetAsDeleteOnSubmit();

                //remove roles associated with the registry
                RoleManager roleManager = new RoleManager();
                IEnumerable<STD_ROLE> roles = roleManager.SelectActiveByRegistryCode(RegistryCode);
                foreach (var role in roles)
                    user.RemoveRoleFromUser(user, role.ID);

                //if the user does not exist in any other registry then mark as inactive
                //if the count of registries for the use was 1, then mark as inactive since we just deleted the last one
                if (user.USER_REGISTRies.Count == 1)
                {
                    user.INACTIVE_FLAG = true;
                }

                returnStatus = Update(user);

            }

            return returnStatus;
        }

        /// <summary>
        /// Returns 1 if user was added to registry
        /// </summary>
        /// <param name="USER">USER</param>
        /// <param name="RegistryCode">RegistryCode</param>
        /// <returns>int</returns>
        public int AddUserToRegistry(USER user, string RegistryCode)
        {
            int returnStatus = 0;
            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

            if (user == null || RegistryCode == "")
            {
                return returnStatus;
            }

            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;
            int registryId = _commonManager.GetRegistryId(RegistryCode);

            //check if user is in registry
            USER_REGISTRy userRegistry = _commonManager.GetUserRegistry(registryId, user.USER_ID);

            if (userRegistry == null)
            {

                userRegistry = new USER_REGISTRy();

                userRegistry.USER_ID = user.USER_ID;
                userRegistry.STD_REGISTRY_ID = registryId;

                _db.USER_REGISTRies.InsertOnSubmit(userRegistry);

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                    RegistriesGlobal.UserRegistryAll = null;

                    eventArgs.SavedItemId = user.USER_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    RaiseSaveEvent(this, eventArgs);

                    returnStatus = 1;
                }
                catch (System.Data.SqlClient.SqlException)
                {

                    eventArgs.SavedItemId = -1;
                    eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                    RaiseSaveEvent(this, eventArgs);

                    returnStatus = -1;
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _db.SubmitChanges(ConflictMode.FailOnFirstConflict);
                        RegistriesGlobal.UserRegistryAll = null;

                        eventArgs.SavedItemId = user.USER_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        RaiseSaveEvent(this, eventArgs);

                        returnStatus = 1;
                    }
                    catch
                    {
                        returnStatus = -1;
                        eventArgs.SavedItemId = -1;
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                        RaiseSaveEvent(this, eventArgs);
                    }
                }
                catch
                {
                    returnStatus = 0;
                    eventArgs.SavedItemId = 0;
                    eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                    RaiseSaveEvent(this, eventArgs);
                }

            }
            else
            {
                returnStatus = 0;
            }
            _db.Dispose();

            //if user was successfully added to the registry then set to active if not active
            if (returnStatus == 1 && user.INACTIVE_FLAG == true)
            {
                user.INACTIVE_FLAG = false;
                user.INACTIVE_DATE = null;

                Update(user);
            }

            return returnStatus;
        }


        #endregion


        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref USER user)
        {
            _db = GetDataContext();
                int status = user.EnforceBusinessRules(false, _db);
            _db.Dispose();

            return 0;
        }


        #endregion
    }
}
