﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Linq.Dynamic;
using System.Data;
using System.Data.Linq;
using System.Security.Principal;
using System.Web;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.Business.Utilities;

namespace VeteransAffairs.Registries.BusinessManager
{
    public class UserAccountManager : BaseBO, IPrincipal
    {
        private const string CACHE_KEY_NAMESPACE = "VeteransAffairs.Registries.BusinessManager";
        private const string CACHE_KEY_CLASS = "UserAccountManager";

        private int _userId;
        private int _registryId;
        private CustomIdentity identity;
        private string _NTId;
        private bool _isSuperUser;
        private RegistriesCommonManager _commonManager = new RegistriesCommonManager();

        public UserAccountManager(string userId, string registry)
        {
            _defaultSortField = "USER_ID";
            _NTId = userId;
            int ind = userId.IndexOf("\\") + 1;
            userId = userId.Substring(ind, userId.Length - ind);
            _userId = SelectIdByNTId(userId);
            
            _registryId = _commonManager.GetRegistryId(registry);
            _isSuperUser = GetSuperUserStatus(_userId);
            identity = new CustomIdentity(_NTId);
        }

        #region Methods for Select

        private List<USER> LinqAll()
        {
            //populate LinqAll
            return (from e in _commonManager.GetAllUsers()
                    where e.INACTIVE_FLAG == false
                    select e).ToList();
            //add all business filtering rules 
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER>(i => i.USER_ROLEs.Where((j => ((j.INACTIVE_FLAG == false)))));

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

       public IEnumerable<USER> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            IEnumerable<USER> objReturn = null;

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            _db.Dispose();

            return objReturn;
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            int objReturn = 0;

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectLinqFilter().Count();
            _db.Dispose();

            return objReturn;
        }

        private IQueryable<USER> SelectLinqFilter()
        {
            return (from e in _db.USERs
                    where e.INACTIVE_FLAG == false
                    select e);
        }

        #endregion

        #region Methods for SelectByID

        public IEnumerable<USER> SelectByID(int userId)
        {
            IEnumerable<USER> objReturn = null;

            _db = GetDataContext();
            SetLoadWith(_db);
            objReturn = SelectByIDLinqFilter(userId).ToList();
            _db.Dispose();

            return objReturn;
        }
        /// <summary>
        /// Get only if the user is active. Only active users will be given access to the system. Inactive users can be accessed 
        /// from admin screens.
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        private int SelectIdByNTId(string userId)
        {
            //Performance Enhancement




            //return ((from user in _commonManager.GetAllUsers() 
            var key = new CacheKey(CACHE_KEY_NAMESPACE, CACHE_KEY_CLASS, "SelectIdByNTId")
                .AddPart(userId)
                .Build();
            //         where user.USERNAME.ToUpper() == userId.ToUpper() 
            return CacheHelper.GetAddValue(key, () =>
            {
                var vUserId = SqlProvider.ExecuteScalar("RegistryConnectionString", "CRS_SP_GET_USERID_BY_USERNAME", new object[] { userId });
                int iUserId = 0;
            //            && user.INACTIVE_FLAG==false 
                int.TryParse(vUserId.ToString(), out iUserId);
            //         select user.USER_ID).FirstOrDefault());
                return iUserId;
            });
        }

        /// <summary>
        /// Return a bool to indicate if the user is a super user in the current registry
        /// from admin screens.
        /// </summary>
        /// <param name="userId"></param>
        /// <returns>bool</returns>
        private bool GetSuperUserStatus(int userId)
        {
            var key = new CacheKey(CACHE_KEY_NAMESPACE, CACHE_KEY_CLASS, "GetSuperUserStatus")
                .AddPart(userId)
                .Build();

            return CacheHelper.GetAddValue(key, () =>
            {
                int iNumRoles = 0;

                var numSuperUserRoles = SqlProvider.ExecuteScalar("RegistryConnectionString", "CRS_SP_GET_NUM_SUPERUSER_ROLES", new object[] { _registryId, userId });

                int.TryParse(numSuperUserRoles.ToString(), out iNumRoles);

                if (iNumRoles > 0)
                    return true;
                else
                    return false;
            });          

            //List<USER_ROLE> userRoles = LinqAccess(_registryId).ToList();

            //string isSuperUser = userRoles.Any(e => e.STD_ROLE.SUPER_USER_FLAG == true).ToString();
                    
            //bool tempBool;
            //bool result = bool.TryParse(isSuperUser, out tempBool);
            //if (result)
            //{
            //    return tempBool;
            //}
            //else
            //{
            //    return false;
            //}                    
        }

        private List<USER> SelectByIDLinqFilter(int userId)
        {
            List<USER> linqFilter = LinqAll();

            linqFilter = (from t in linqFilter where t.USER_ID == userId 
                         select  t).ToList();

            return linqFilter;
        }
        #endregion
      
        #region USER Roles

        /// <summary>
        /// Gets all the active roles for the user
        /// </summary>
        /// <returns></returns>
        private List<USER_ROLE> LinqAccess()
        {
            return (from userRole in _commonManager.GetAllUserRoles() 
                    where userRole.USER_ID == _userId 
                    select userRole).ToList();
        }

        /// <summary>
        /// Reusable query for retrieving user roles for a registry
        /// </summary>
        /// <param name="registryId"></param>
        /// <returns></returns>
        private List<USER_ROLE> LinqAccess(int registryId)
        {
            return (from t in LinqAccess()
                    join role in _commonManager.GetAllRoles() on t.STD_ROLE_ID equals role.ID
                    where role.STD_REGISTRY_ID == registryId
                    select t).ToList();
        }

        /// <summary>
        /// Returns user roles for the user id. 
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public int[] GetRoles()
        {
            //Performance Enhancement

            List<int> roleIds = new List<int>();
            int[] iRoleIds; 

            if (HttpContext.Current.Session["UserRoleIds"] == null)
            {
                SqlDataReader sReader = SqlProvider.ExecuteSPReader("RegistryConnectionString", "CRS_SP_GET_ROLEIDS_BY_REGISTRYID_USERID", new object[] {_registryId, _userId}); 
                roleIds = new List<int>();

                while (sReader.Read())
                {
                    roleIds.Add((int)(sReader["ID"]));
                }

                iRoleIds = new int[roleIds.Count];
                for (int i = 0; i < roleIds.Count; i++)
                {
                    iRoleIds[i] = roleIds[i];
                }

                HttpContext.Current.Session["UserRoleIds"] = iRoleIds;
            }
            else
            {
                iRoleIds = (int[])HttpContext.Current.Session["UserRoleIds"];
            }

            return iRoleIds;

            //int[] objReturn = null;

            //_db = GetDataContext();
            //objReturn = ((from t in LinqAccess(_registryId) select t.STD_ROLE_ID).Distinct()).ToArray();
            //_db.Dispose();

            //return objReturn;
        }

        /// <summary>
        /// Indicates whether the user is in a specified role
        /// </summary>
        /// <param name="roleId"></param>
        /// <returns>true/false</returns>
        public bool IsInRole(int roleId)
        {
            int[] roles = GetRoles();
            return (roles.Contains(roleId));
        }

        
        /// <summary>
        /// Checks the the access for the registry
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="registryId"></param>
        /// <returns>true/false</returns>
        public bool HasAccessToRegistry()
        {
            //Performance Enhancement
            int[] roles = GetRoles();
            
            if (roles.Count() > 0)
            {
                return true;
            }
            else
            {
                return false;
            }

           //return (LinqAccess(_registryId).Count() >= 1);
        }
        /// <summary>
        /// Checks access for the registry and the institution
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="registryId"></param>
        /// <param name="institutionId"></param>
        /// <returns>true/false</returns>
        public bool HasAccessToInstitution(int institutionId)
        {
            List<USER_ROLE> LinqRegAccess = LinqAccess(_registryId);
            return ((from t in LinqRegAccess where t.STD_INSTITUTION_ID == institutionId select t).Count() >= 1);
        }

        /// <summary>
        /// Get userid of current authenticated user
        /// </summary>
        /// <returns>int</returns>
        public int UserId
        {
            get
            {
                return _userId;
            }
        }
        
        /// <summary>
        /// Get full name of current authenticated user from database user table
        /// </summary>
        /// <returns>string</returns>
        public string FullName
        {
            get
            {
                return ((from user in _commonManager.GetAllUsers() 
                         where user.USER_ID == _userId select user.FULL_NAME).FirstOrDefault());
            }
        }

        public string EMail
        {
            get
            {
                return ((from user in _commonManager.GetAllUsers()
                         where user.USER_ID == _userId 
                         select user.EMAIL_ADDRESS).FirstOrDefault());
            }
        }

        /// <summary>
        /// Readonly property to indicate if user is a super user for the current registry
        /// </summary>
        /// <returns>bool</returns>
        public bool IsSuperUser
        {
            get
            {
                return _isSuperUser;
            }
        }

        /// <summary>
        /// Readonly property to return the current Registry ID 
        /// </summary>
        /// <returns></returns>
        public int RegistryId 
        {
            get 
            {
                return _registryId;
            }
        }

        /// <summary>
        /// Link to Role
        /// </summary>
        /// <returns></returns>
        private List<ROLE_PERMISSION> LinqRoleFilter()
        {
            return (from t in LinqAccess(_registryId)
                    join rolePerm in _db.ROLE_PERMISSIONs on t.STD_ROLE_ID equals rolePerm.STD_ROLE_ID select rolePerm).ToList();
        }

        /// <summary>
        /// Link to Role
        /// </summary>
        /// <returns></returns>
        private List<ROLE_PERMISSION> LinqRoleFilter(int InstitutionId)
        {
            return (from t in LinqAccess(_registryId)
                    join rolePerm in _db.ROLE_PERMISSIONs on t.STD_ROLE_ID equals rolePerm.STD_ROLE_ID
                    where t.STD_INSTITUTION_ID == InstitutionId
                    select rolePerm).ToList();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>

        private List<STD_APPPAGE> LinqPermFilter()
        {
            return (from t in LinqRoleFilter()
                    join perm in _commonManager.GetAllAppPages() on t.STD_APPPAGE_ID equals perm.ID
                    where perm.INACTIVE_FLAG == false select perm).ToList();
        }

        /// <summary>
        /// Returns all accessible page IDs for the user
        /// </summary>
        /// <returns>Integer array of page IDs</returns>
        public int[] GetPagesById()
        {
            int[] objReturn = null;

            _db = GetDataContext();
            objReturn = ((from t in LinqPermFilter() select t.ID).Distinct()).ToArray();
            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Returns all accessible page codes for the user
        /// </summary>
        /// <returns>String array of page codes</returns>
        public string[] GetPagesByCode()
        {
            string[] objReturn = null;

            int[] ids = GetPagesById();
            _db = GetDataContext();
            objReturn = ((from t in _commonManager.GetAllAppPages() where ids.Contains(t.ID) select t.CODE)).ToArray();
            _db.Dispose();

            return objReturn;
        }
        /// <summary>
        /// Gets access level for an Object Id(Screen Id)
        /// </summary>
        /// <param name="objId"></param>
        /// <returns></returns>
        public Permissions GetPermissions(int objId)
        {
            Permissions objReturn;

            if (_isSuperUser)
            {
                objReturn = new Permissions(true, true, true, false, false, objId);
            }
            
            //Performance Enhancement
            
            string key = "Apppage_" + objId.ToString();

            if (HttpContext.Current.Session[key] == null)
            {
                SqlDataReader sReader = SqlProvider.ExecuteSPReader("RegistryConnectionString", "CRS_SP_GET_APPPAGE_PERMS", new object[] { _registryId, _userId, objId });
                bool I = false, D = false, U = false;

                while (sReader.Read())
                {
                    if ((bool)sReader["INSERT_FLAG"]) I = true;
                    if ((bool)sReader["DELETE_FLAG"]) D = true;
                    if ((bool)sReader["UPDATE_FLAG"]) U = true;
                }

                objReturn = LoadPermissions(I, D, U, objId);

                HttpContext.Current.Session[key] = objReturn;
            }
            else
            {
                objReturn = (Permissions)HttpContext.Current.Session[key];
            }

            //_db = GetDataContext();

            //var vPerm = (from rolePerm in LinqRoleFilter() where rolePerm.STD_APPPAGE_ID == objId select new { UpdateFl = rolePerm.UPDATE_FLAG, DeleteFl = rolePerm.DELETE_FLAG, InsertFl = rolePerm.INSERT_FLAG, RoleId=rolePerm.STD_ROLE_ID}).ToList();
                
            //if (vPerm == null || vPerm.Count == 0)
            //    objReturn = new Permissions(false, false, false, false, true, objId);
            //else
            //{
            //    bool I = false, D = false, U = false; 
            //    foreach (var v in vPerm)
            //    {
            //        if (v.InsertFl == true) I = true;
            //        if (v.DeleteFl == true) D = true;
            //        if(v.UpdateFl == true) U = true;
            //    }
                    
            //    objReturn = LoadPermissions(I, D, U, objId);
            //}

            //_db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Gets access level for an Object Id(Screen Id)
        /// </summary>
        /// <param name="objId"></param>
        /// <param name="institutionId"></param>
        /// <returns></returns>
        public Permissions GetPermissions(int objId, int institutionId)
        {
            Permissions objReturn;

            if (_isSuperUser)
            {
                objReturn = new Permissions(true, true, true, false, false, objId);
            }

            _db = GetDataContext();
                
            var vPerm = (from rolePerm in LinqRoleFilter(institutionId) 
                            where rolePerm.STD_APPPAGE_ID == objId 
                             
                            select new { UpdateFl = rolePerm.UPDATE_FLAG, DeleteFl = rolePerm.DELETE_FLAG, InsertFl = rolePerm.INSERT_FLAG, RoleId = rolePerm.STD_ROLE_ID }).ToList();

            if (vPerm == null || vPerm.Count == 0)
                objReturn = new Permissions(false, false, false, false, true, objId);
            else
            {
                bool I = false, D = false, U = false;
                foreach (var v in vPerm)
                {
                    if (v.InsertFl == true) I = true;
                    if (v.DeleteFl == true) D = true;
                    if (v.UpdateFl == true) U = true;
                }

                objReturn = LoadPermissions(I, D, U, objId);
            }

            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Gets access level for an Object code (Screen Name) - Code is not reusable in db. 
        /// </summary>
        /// <param name="ObjCd"></param>
        /// <returns></returns>
        /// //TODO - Readonly Attributes
        public Permissions GetPermissions(string ObjCd)
        {
            //Performance Enhancement
            string key = "ApppageName_" + ObjCd;
            int objId = 0;

            if (HttpContext.Current.Session[key] == null)
            {
                var vId = VeteransAffairs.Registries.Business.Utilities.SqlProvider.ExecuteScalar("RegistryConnectionString", "CRS_SP_GET_STDAPPPAGEID_BY_CODE", new object[] { ObjCd });
                objId = 0;

                int.TryParse(vId.ToString(), out objId);

                HttpContext.Current.Session[key] = objId;
            }
            else
            {
                objId = (int)HttpContext.Current.Session[key];
            }

            //int objId;
            //_db = GetDataContext();
            //objId = (from t in _commonManager.GetAllAppPages() where t.CODE.ToUpper() == ObjCd.ToUpper() select t.ID).FirstOrDefault();
            //_db.Dispose();

            if (_isSuperUser)
            {
                return new Permissions(true, true, true, false, false, objId);
            }

            return GetPermissions(objId);

            //TO DO : Does not make page non visible if user/role does not have access.
                //base page should redirect to warning.
        }

        /// <summary>
        /// Gets access level for an Object code (Screen Name) - Code is not reusable in db. 
        /// </summary>
        /// <param name="ObjCd"></param>
        /// <returns></returns>
        /// //TODO - Readonly Attributes
        public Permissions GetPermissions(string ObjCd, int InstitutionId)
        {
            int objId;
            _db = GetDataContext();
                objId = (from t in _commonManager.GetAllAppPages() where t.CODE.ToUpper() == ObjCd.ToUpper() select t.ID).FirstOrDefault();
            _db.Dispose();
            
            if (_isSuperUser)
            {
                return new Permissions(true, true, true, false, false, objId);
            }

            return GetPermissions(objId, InstitutionId);

            //TO DO : Does not make page non visible if user/role does not have access.
            //base page should redirect to warning.
        }

        
        /// <summary>
        /// Get role name for an Id
        /// </summary>
        /// <param name="roleId"></param>
        /// <returns></returns>
        public string GetRoleName(int roleId)
        {
            return ((from role in _commonManager.GetAllRoles()
                     where role.ID == roleId && role.INACTIVE_FLAG == false 
                     select role.NAME).FirstOrDefault());
        }

        /// <summary>
        /// Returns users roles in comma separated list
        /// </summary>
        public string RoleName
        {
            get {

                //Performance Enhancement
                string key = "RoleName_" + _NTId;
                string RoleName = "";

                if (HttpContext.Current.Session[key] == null)
                {
                    SqlDataReader sReader = SqlProvider.ExecuteSPReader("RegistryConnectionString", "CRS_SP_GET_ROLENAMES_BY_REGISTRYID_USERID", new object[] { _registryId, _userId });
                    RoleName = "";

                    while (sReader.Read())
                    {
                        if (RoleName.Length > 0)
                        {
                            RoleName += ", " + sReader["NAME"];
                        }
                        else
                        {
                            RoleName += sReader["NAME"];
                        }
                    }

                    HttpContext.Current.Session[key] = RoleName;
                }
                else
                {
                    RoleName = (string)HttpContext.Current.Session[key];
                }

                return RoleName;

                //int[] roles = GetRoles();
                //return String.Join(", ", (from t in _commonManager.GetAllRoles()
                //                          where roles.Contains(t.ID)
                //                          select t.NAME).ToArray());
            }
        }

        private Permissions LoadPermissions(bool insFl, bool delFl, bool updtFl, int objId)
        {
            bool Readonly=false;
            if (insFl == false && updtFl == false && delFl == false)
                Readonly = true;
            return new Permissions(insFl,delFl,updtFl,Readonly,false,objId);
        }


        public int[] GetInstitutions()
        {
            int[] objReturn = null;

            _db = GetDataContext();
            objReturn = (((from t in LinqAccess(_registryId) select t.STD_INSTITUTION_ID).Distinct()).ToArray());
            _db.Dispose();

            return objReturn;
        }

        #endregion

        #region IPrincipal Members

        public IIdentity Identity
        {
            get {return identity;}
        }

        public bool IsInRole(string role)
        {
            //TODO - implement by role name
           throw(new NotImplementedException("IsInRole not implemented for a role name."));
        }

        #endregion

        public int SessionLog(int objectId, int activityTypeId, string activityDescription, string webUrl)
        {
            USER_LOGIN_DETAIL userLogin = new USER_LOGIN_DETAIL();
            int returnStatus = 0;

            userLogin.OBJECT_NUMBER = (objectId == 0)? null:objectId.ToString();
            userLogin.ACTIVITY_TYPE_ID = activityTypeId;
            userLogin.ACTIVITY_DESC = activityDescription;
            userLogin.WEB_URL = webUrl;
            userLogin.STD_REGISTRY_ID = RegistryId;
            userLogin.USER_ID = UserId;

            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;
            _db.USER_LOGIN_DETAILs.InsertOnSubmit(userLogin);
                
            try
            {
                _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                returnStatus = userLogin.USER_LOGIN_ID;
            }
            catch (ChangeConflictException)
            {
                _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                returnStatus = userLogin.USER_LOGIN_ID;
            }
            catch
            {
                returnStatus = -1;
            }
            _db.Dispose();
            return returnStatus;
        }
    }
}
