﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Linq.Dynamic;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.Business.Utilities;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using System.IO; 

namespace VeteransAffairs.Registries.BusinessManager
{
    public class RegistriesCommonManager : BaseBO
    {

        /// <summary>
        /// Get the database name
        /// </summary>
        /// <returns></returns>
        public string GetDatabaseName()
        {
            string objReturn = string.Empty;

            _db = GetDataContext();
            objReturn = _db.Connection.Database;
            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Get all roles in the database
        /// </summary>
        /// <returns></returns>
        public List<STD_ROLE> GetAllRoles()
        {
            if (RegistriesGlobal.RoleAll == null)
            {
                _db = GetDataContext();

                RegistriesGlobal.RoleAll = (from e in _db.STD_ROLEs
                                            orderby e.SORT_ORDER 
                                            select e).ToList();
                _db.Dispose();
            }

            return RegistriesGlobal.RoleAll;
        }


        /// <summary>
        /// Get all STD_GUI_CONTROL records in the database
        /// </summary>
        /// <returns></returns>
        public List<STD_GUI_CONTROL> GetAllStdGuiControls()
        {
            if (RegistriesGlobal.StdGuiControlAll == null)
            {
                _db = GetDataContext();

                RegistriesGlobal.StdGuiControlAll = (from e in _db.STD_GUI_CONTROLs
                                            orderby e.SORT_ORDER
                                            select e).ToList();
                _db.Dispose();
            }

            return RegistriesGlobal.StdGuiControlAll;
        }
        
        /// <summary>
        /// Get all user roles in USER_ROLE table
        /// </summary>
        /// <returns></returns>
        public List<USER_ROLE> GetAllUserRoles()
        {
            var key = "ALL_USER_ROLES";

            return CacheHelper.GetAdd(key, () =>
            {
                if (RegistriesGlobal.UserRoleAll == null)
                {
                    _db = GetDataContext();
                    SetLoadWithUserRole(_db);
                    RegistriesGlobal.UserRoleAll = (from e in _db.USER_ROLEs
                                                    select e).ToList();
                    _db.Dispose();
                }

                return RegistriesGlobal.UserRoleAll;
            });
        }

        /// <summary>
        /// Get all users in USER table
        /// </summary>
        /// <returns></returns>
        public List<USER> GetAllUsers()
        {
            var key = "ALL_USERS";

            return CacheHelper.GetAdd(key, () =>
            {
                _db = GetDataContext();
                SetLoadWithUser(_db);
                RegistriesGlobal.UserAll = (from e in _db.USERs
                                            select e).ToList();
                _db.Dispose();

                return RegistriesGlobal.UserAll;
            });
        }

        /// <summary>
        /// Get all user roles for one user
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public List<USER_ROLE> GetUserRoles(int userId)
        {

            List<USER_ROLE> userRoles = (from e in GetAllUserRoles()
                                         where e.USER_ID == userId
                                               && e.INACTIVE_FLAG == false
                                         select e).ToList();

            return userRoles;
        }

        /// <summary>
        /// Get STD_GUI_CONTROL records by category and registry
        /// </summary>
        /// <param name="category"></param>
        /// <param name="registryId"></param>
        /// <returns></returns>
        public List<STD_GUI_CONTROL> GetStdGuiControlsByCategory(string category, int registryId)
        {
            return (from e in GetAllStdGuiControls()
                    where e.CATEGORY == category
                        && e.STD_REGISTRY_ID == registryId
                    select e).ToList();
        }


        /// <summary>
        /// Get all active user roles for a user
        /// </summary>
        /// <param name="regiestryId"></param>
        /// <param name="userId"></param>
        /// <returns></returns>
        public List<USER_ROLE> GetActiveUserRoles(int regiestryId, int userId)
        {
            List<USER_ROLE> activeUserRoles = (from e in GetAllUserRoles()
                                               where e.INACTIVE_FLAG == false
                                               && e.USER_ID == userId
                                               && e.STD_ROLE.STD_REGISTRY_ID == regiestryId
                                               && e.STD_ROLE.INACTIVE_FLAG == false
                                               select e).ToList();

            return activeUserRoles;
        }

        private void SetLoadWithUserRole(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER_ROLE>(i => i.STD_ROLE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithUser(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER>(i => i.USER_REGISTRies);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// Get the APPLICATION_STATUS ocject for a registry
        /// </summary>
        /// <param name="registryCode"></param>
        /// <returns></returns>
        public APPLICATION_STATUS GetApplicationStatus(string registryCode)
        {
            APPLICATION_STATUS appStatus;

            _db = GetDataContext();
            SetLoadWithApplicationStatus(_db);
            appStatus = (from e in _db.APPLICATION_STATUS 
                            where e.STD_REGISTRY.CODE.Equals(registryCode.Trim())
                            && e.STD_REGISTRY.INACTIVE_FLAG == false
                            select e).FirstOrDefault();
            _db.Dispose();

            return appStatus;
        }

        /// <summary>
        /// Get registry ID by registry code
        /// </summary>
        /// <param name="registryCode"></param>
        /// <returns></returns>
        public int GetRegistryId(string registryCode)
        {
            //Performance Enhancement
            string key = "RegistryId" + registryCode;
            int RegistryId = 0;

            if (!CacheHelper.Get(key, out RegistryId))
            {
                var vRegistryId = SqlProvider.ExecuteScalar("RegistryConnectionString", "CRS_SP_GET_REGISTRYID_BY_CODE", new object[] { registryCode });
                RegistryId = 0;

                int.TryParse(vRegistryId.ToString(), out RegistryId);

            //int registryId = 0;
            //_db = GetDataContext();
            //if (RegistriesGlobal.RegistryAll == null)
            //{
            //    RegistriesGlobal.RegistryAll = (from e in _db.STD_REGISTRies
            //                                select e).ToList();
            //}
            //STD_REGISTRY registry = (from e in RegistriesGlobal.RegistryAll
            //                         where e.CODE == registryCode
            //                            && e.INACTIVE_FLAG == false
            //                         select e).FirstOrDefault();
                CacheHelper.Add(RegistryId, key);
            }

            return RegistryId;
        }

        /// <summary>
        /// Get user registry by registry ID and user ID
        /// </summary>
        /// <param name="registryId"></param>
        /// <param name="userId"></param>
        /// <returns></returns>
        public USER_REGISTRy GetUserRegistry(int registryId, int userId)
        {
            return (from e in GetAllUserRegistries()
                    where e.STD_REGISTRY_ID == registryId
                          && e.USER_ID == userId
                    select e).FirstOrDefault();
        }

        /// <summary>
        /// Get user registry by registry ID and user name
        /// </summary>
        /// <param name="registryId"></param>
        /// <param name="userName"></param>
        /// <returns></returns>
        public USER_REGISTRy GetUserRegistry(int registryId, string userName)
        {
            return (from e in GetAllUserRegistries()
                    where e.STD_REGISTRY_ID == registryId
                          && e.USER.USERNAME.Equals(userName, StringComparison.OrdinalIgnoreCase)
                    select e).FirstOrDefault();
        }

        /// <summary>
        /// Get all user registries from USER_REGISTRIES table
        /// </summary>
        /// <returns></returns>
        public List<USER_REGISTRy> GetAllUserRegistries()
        {
            if (RegistriesGlobal.UserRegistryAll == null)
            {
                _db = GetDataContext();
                SetLoadWithUserRegistry(_db);
                RegistriesGlobal.UserRegistryAll = (from e in _db.USER_REGISTRies
                                                select e).ToList();
                _db.Dispose();
            }

            return RegistriesGlobal.UserRegistryAll;
        }

        private void SetLoadWithUserRegistry(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<USER_REGISTRy>(i => i.USER);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// Get all app pages in STD_APPPAGE table
        /// </summary>
        /// <returns></returns>
        public List<STD_APPPAGE> GetAllAppPages()
        {
            if (RegistriesGlobal.AppPageAll == null)
            {
                _db = GetDataContext();

                RegistriesGlobal.AppPageAll = (from e in _db.STD_APPPAGEs
                                            select e).ToList();
                _db.Dispose();
            }

            return RegistriesGlobal.AppPageAll;
        }

        /// <summary>
        /// Get all Institutions in STD_INSTITUTION table
        /// </summary>
        /// <returns></returns>
        public List<STD_INSTITUTION> GetAllInstitutions()
        {
            if (RegistriesGlobal.StdInstitutionAll == null)
            {
                _db = GetDataContext();
                SetLoadWithInstitution(_db);
                RegistriesGlobal.StdInstitutionAll = (from e in _db.STD_INSTITUTIONs
                                                        select e).ToList();
                _db.Dispose();
            }

            return RegistriesGlobal.StdInstitutionAll;
        }

        private void SetLoadWithInstitution(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<STD_INSTITUTION>(i => i.STD_STATE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithApplicationStatus(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<APPLICATION_STATUS>(i => i.STD_REGISTRY);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// Retrieve all data from DatabaseChangeLog table
        /// </summary>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public IEnumerable<DatabaseChangeLog> SelectDatabaseChangeLog(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = "DatabaseChangeID DESC";

            }

            List<DatabaseChangeLog> entities;

            _db = GetDataContext();
                
            entities = SelectDatabaseChangeLogLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

            _db.Dispose();

            return entities;
        }

        /// <summary>
        /// Count all records in DatabaseChangeLog table
        /// </summary>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public int SelectDatabaseChangeLogCount(string sort, int startRow, int maxRows)
        {
            int objReturn = 0;

            _db = GetDataContext();
            objReturn =  SelectDatabaseChangeLogLinqFilter().Count();
            _db.Dispose();

            return objReturn;
        }

        /// <summary>
        /// Get build version string from assembly info
        /// </summary>
        /// <returns></returns>
        public string GetBuildVersion()
        {
            string version, buildTime;
            GetBuildVersion(out version, out buildTime);
            return version;
        }

        public void GetBuildVersion(out string version, out string buildTime)
        {
            version = string.Empty;
            buildTime = string.Empty;
            Assembly assembly = Assembly.GetExecutingAssembly();
            string[] assemblyInfo = assembly.FullName.Split(",");
            foreach (string info in assemblyInfo)
            {
                if (info.Contains("Version="))
                {
                    version = info.Split("=")[1];
                    break;
                }
            }
            FileInfo fileInfo = new FileInfo(assembly.Location);
            buildTime = fileInfo.LastWriteTime.ToString();
        }

        private IQueryable<DatabaseChangeLog> SelectDatabaseChangeLogLinqFilter()
        {
            IQueryable<DatabaseChangeLog> linqFilter = (from e in _db.DatabaseChangeLogs select e);
            return linqFilter;
        }


        #region Methods for Update

        public int Update(APPLICATION_STATUS appStatus)
        {
            int returnStatus = 0;

            _db = GetDataContext();
            _db.DeferredLoadingEnabled = false;

            appStatus.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

            //Check if any actual changes will occur
            ChangeSet changeSet = _db.GetChangeSet();

            if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
            || changeSet.Updates.Count > 0)
            {
                //if changes present submit changes
                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = appStatus.STD_REGISTRY_ID;
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                    returnStatus = -1;
                }

                catch
                {
                    returnStatus = -1;
                }
            }

            _db.Dispose();

            return returnStatus;
        }

        #endregion
    }
}
