﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.MDWS;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using System.Threading;
using System.Web.Configuration;

namespace VeteransAffairs.Registries.BusinessManager
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class CprsManager : BaseBO
    {
        private EmrSvc _cprsService;
        private SiteTO _siteTO;
        private PersonsTO _personsTO;
        public CprsServiceTO CprsServiceTo { get; set; }

        public bool LoadCprsService(CprsServiceTO cprsServiceTo)
        {
            _cprsService = new EmrSvc();
            //_cprsService.Url = "";
            _cprsService.CookieContainer = new System.Net.CookieContainer();
            _siteTO = addDataSource(cprsServiceTo);
            if (_siteTO == null || _siteTO.fault != null)
            {
                CprsServiceTo = null;
            }

            _personsTO = CPRSLaunch(cprsServiceTo);

            if (CprsServiceTo == null)
            {
                return false;
            }
            else
            {
                return true;
            }
        }


        private PersonsTO CPRSLaunch(CprsServiceTO cprsServiceTo)
        {
            PersonsTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsLaunch";
            _mdwsLog.PARAMETERS = xmlParameters(new string[] { "CPRS Launch", cprsServiceTo.siteCode, cprsServiceTo.DUZ, cprsServiceTo.DFN });

            try
            {
                result = _cprsService.cprsLaunch("REGISTRY-PWD", cprsServiceTo.siteCode, cprsServiceTo.DUZ, cprsServiceTo.DFN);
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch 
            {
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }
        
        private SiteTO addDataSource(CprsServiceTO cprsServiceTo)
        {
            SiteTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "addDataSource";
            _mdwsLog.PARAMETERS = xmlParameters(new string[] { cprsServiceTo.siteCode, "TESTSITE", cprsServiceTo.ServerIP, cprsServiceTo.ServerPort, "HIS", "VISTA", "11" });

            try
            {
                result = _cprsService.addDataSource(cprsServiceTo.siteCode, "TESTSITE", cprsServiceTo.ServerIP, cprsServiceTo.ServerPort, "HIS", "VISTA", "11");
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch 
            {
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        public void SetMDWSLogDefaults(ref MDWS_LOG log)
        {
            RegistriesCommonManager _commonManager = new RegistriesCommonManager();

            string registryCode = WebConfigurationManager.AppSettings.Get("Registry");

            if (!string.IsNullOrEmpty(registryCode))
                log.STD_REGISTRY_ID = _commonManager.GetRegistryId(registryCode);

            log.SENT = System.DateTime.Now;
            log.ERROR_LEVEL = 0;
        }

        public string xmlParameters(string[] parameterArray)
        {
            string result = String.Empty;
            string elementTemplate = "<pNumber>Parameter</pNumber>";
            string element;

            if (parameterArray != null)
            {
                for (int i = 0; i < parameterArray.Length; i++)
                {
                    element = elementTemplate.Replace("Number", (i + 1).ToString()); //Make parameter number 1-based
                    element = element.Replace("Parameter", parameterArray[i]);
                    result = result + element;
                }

                if (!String.IsNullOrEmpty(result))
                    result = "<Parameters>" + result + "</Parameters>";
            }

            return result;
        }

        /// <summary>
        /// Write Note to CPRS
        /// </summary>
        /// <param name="cprsServiceTO"></param>
        /// <returns></returns>
        public bool WriteNote(CprsServiceTO cprsServiceTO)
        {
            bool result = false;
            NoteResultTO _writeNoteResult = null;

            if (LoadCprsService(cprsServiceTO))
            {
                string titleIEN = "";
                string encounterString = "1;" + FormatDateForMDWS(DateTime.Now) + ";E";
                string noteText = "";
                string authorDUZ = "";
                string cosignerDUZ = "";
                string consultIEN = "";
                string prfIEN = "";

                //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
                noteText = cprsServiceTO.NoteText.Replace("<br>", "|").Replace(Environment.NewLine, "|");
                authorDUZ = cprsServiceTO.DUZ;

                if (!string.IsNullOrEmpty(cprsServiceTO.SelectedNoteTitleId))
                {
                    titleIEN = cprsServiceTO.SelectedNoteTitleId;
                }

                MDWSLog mdwsLog = new MDWSLog();
                MDWS_LOG _mdwsLog = new MDWS_LOG();

                bool savedFlag = false;

                SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "writeNote";
                _mdwsLog.PARAMETERS = xmlParameters(new string[] { titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN });

                try
                {
                    _writeNoteResult = _cprsService.writeNote(titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN);
                    if (_writeNoteResult != null)
                    {
                        _mdwsLog.RETURNED_DATA = _writeNoteResult.Serialize();

                        if (!String.IsNullOrEmpty(_writeNoteResult.id))
                        {
                            result = true;
                            mdwsLog.LogMDWSCall(_mdwsLog);
                            savedFlag = true;

                            CloseNote(_writeNoteResult.id, consultIEN);
                            DisconnectSite();
                        }

                        if (_writeNoteResult.fault != null)
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                catch
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }

                if (!savedFlag)
                {
                    mdwsLog.LogMDWSCall(_mdwsLog);
                }
            }
            return result;
        }


        public TaggedTextArray DisconnectSite()
        {
            TaggedTextArray result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "disconnect";
            _mdwsLog.PARAMETERS = "";

            try
            {
                result = _cprsService.disconnect();
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        // log fault - 
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch 
            {
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

    
        private TextTO CloseNote(string noteIEN, string consultIEN)
        {
            TextTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "closeNote";
            _mdwsLog.PARAMETERS = xmlParameters(new string[] { noteIEN, consultIEN });

            try
            {
                result = _cprsService.closeNote(noteIEN, consultIEN);
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch //(Exception ex)
            {
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        private string FormatDateForMDWS(DateTime dateValue)
        {
            // CPRS is expecting date in the following format "20000101.000000";
            return String.Format("{0:yyyyMMdd.HHmmss}", dateValue);
        }
    }


    [Serializable()]
    public class CprsServiceTO
    {
        public string siteCode { get; set; }
        public string DUZ { get; set; }
        public string DFN { get; set; }
        public string ServerIP { get; set; }
        public string ServerPort { get; set; }

        public string UserName { get; set; }
        public string PatientName { get; set; }
        public string PatientDOB { get; set; }
        public string PatientGender { get; set; }
        public string PatientSSN { get; set; }
        public string PatientLocationId { get; set; }
        public string PatientICN { get; set; }
        public string PatientDFN { get; set; }
        public string STA6N { get; set; }
        public string PatientAdmintTimeStamp { get; set; }
        public string InstitutionName { get; set; }

        public string NoteText { get; set; }
        public string SelectedNoteTitleId { get; set; }
    }
}
