﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;

namespace VeteransAffairs.Registries.Business.Utilities
{
    public static class CacheHelper
    {
        private const int DEFAULT_CACHE_EXPIRATION_TIME_MINUTES = 60;

        private static readonly object _lock;

        static CacheHelper()
        {
            _lock = new object();
        }

        /// <summary>
        /// Insert value into the cache using
        /// appropriate name/value pairs
        /// </summary>
        /// <typeparam name="T">Type of cached item</typeparam>
        /// <param name="o">Item to be cached</param>
        /// <param name="key">Name of item</param>
        public static void Add<T>(T o, string key)
        {
            // NOTE: Apply expiration parameters as you see fit.
            // I typically pull from configuration file.

            // In this example, I want an absolute
            // timeout so changes will always be reflected
            // at that time. Hence, the NoSlidingExpiration.
            HttpRuntime.Cache.Insert(
                key,
                o,
                null,
                DateTime.Now.AddMinutes(1440),
                System.Web.Caching.Cache.NoSlidingExpiration);
        }

        /// <summary>
        /// Add to Cache
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="key"></param>
        /// <param name="value"></param>
        /// <param name="cacheExpirationInMinutes"></param>
        /// <returns></returns>
        public static void Add<T>(
            string key,
            T value,
            int cacheExpirationInMinutes = DEFAULT_CACHE_EXPIRATION_TIME_MINUTES)
            where T : class
        {
            lock (_lock)
            {
                HttpRuntime.Cache.Insert(
                    key,
                    value,
                    null,
                    DateTime.Now.AddMinutes(cacheExpirationInMinutes),
                    System.Web.Caching.Cache.NoSlidingExpiration);
            }
        }

        /// <summary>
        /// Get or Add cache object
        /// </summary>
        /// <typeparam name="T">return Type</typeparam>
        /// <param name="key">cache key</param>
        /// <param name="getT">Func to obtain object reference</param>
        /// <returns></returns>
        public static T GetAdd<T>(
            string key, 
            Func<T> getT, 
            int cacheExpirationInMinutes = DEFAULT_CACHE_EXPIRATION_TIME_MINUTES)
            where T : class
        {
            T result = HttpRuntime.Cache[key] as T;
            if (result == default(T))
            {
                lock (_lock)
                {
                    result = HttpRuntime.Cache[key] as T;
                    if (result == default(T))
                    {
                        result = getT();

                        HttpRuntime.Cache.Insert(
                            key,
                            result,
                            null,
                            DateTime.Now.AddMinutes(cacheExpirationInMinutes),
                            System.Web.Caching.Cache.NoSlidingExpiration);
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// Get or Add cache value
        /// </summary>
        /// <typeparam name="T">return Type</typeparam>
        /// <param name="key">cache key</param>
        /// <param name="getT">Func to obtain object reference</param>
        /// <returns></returns>
        public static T GetAddValue<T>(
            string key,
            Func<T> getT,
            int cacheExpirationInMinutes = DEFAULT_CACHE_EXPIRATION_TIME_MINUTES)
            where T : struct
        {
            var result = HttpRuntime.Cache[key];
            if (result == null)
            {
                lock (_lock)
                {
                    result = HttpRuntime.Cache[key];
                    if (result == null)
                    {
                        result = getT();

                        HttpRuntime.Cache.Insert(
                        key,
                        result,
                        null,
                        DateTime.Now.AddMinutes(cacheExpirationInMinutes),
                        System.Web.Caching.Cache.NoSlidingExpiration);
                    }
                }
            }

            return result != null ? (T)result : default(T);
        }

        /// <summary>
        /// Get or Add cache value
        /// </summary>
        /// <typeparam name="T">return Type</typeparam>
        /// <param name="key">cache key</param>
        /// <param name="getT">Func to obtain object reference</param>
        /// <returns></returns>
        public static T AddUpdate<T>(
            string key,
            Func<T> addT,
            Func<T, T> updateT,
            int cacheExpirationInMinutes = DEFAULT_CACHE_EXPIRATION_TIME_MINUTES)
            where T : class
        {
            lock (_lock)
            {
                T t = default(T);

                var cached = HttpRuntime.Cache[key] as T;
                if (cached == null)
                    t = addT();
                else
                    t = updateT(cached);

                // insert does add/update
                HttpRuntime.Cache.Insert(
                    key,
                    t,
                    null,
                    DateTime.Now.AddMinutes(cacheExpirationInMinutes),
                    System.Web.Caching.Cache.NoSlidingExpiration);
            }

            return HttpRuntime.Cache[key] as T;
        }

        /// <summary>
        /// Remove item from cache
        /// </summary>
        /// <param name="key">Name of cached item</param>
        public static void Clear(string key)
        {
            HttpRuntime.Cache.Remove(key);
        }

        /// <summary>
        /// Check for item in cache
        /// </summary>
        /// <param name="key">Name of cached item</param>
        /// <returns></returns>
        public static bool Exists(string key)
        {
            return HttpRuntime.Cache[key] != null;
        }

        /// <summary>
        /// Retrieve cached item
        /// </summary>
        /// <typeparam name="T">Type of cached item</typeparam>
        /// <param name="key">Name of cached item</param>
        /// <param name="value">Cached value. Default(T) if 
        /// item doesn't exist.</param>
        /// <returns>Cached item as type</returns>
        public static bool Get<T>(string key, out T value)
        {
            try
            {
                if (!Exists(key))
                {
                    value = default(T);
                    return false;
                }

                value = (T)HttpRuntime.Cache[key];
            }
            catch
            {
                value = default(T);
                return false;
            }

            return true;
        }

        /// <summary>
        /// Retrieve cached oject of type T
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="key"></param>
        /// <returns></returns>
        public static T Get<T>(string key)
            where T : class
        {
            T result = default(T);

            result = HttpRuntime.Cache[key] as T;

            return result;
        }

        /// <summary>
        /// Retrieve cached value
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="key"></param>
        /// <returns></returns>
        public static T GetValue<T>(string key)
            where T : struct
        {
            var result = HttpRuntime.Cache[key];

            return result != null ? (T)result : default(T);
        }

        /// <summary>
        /// Add a cache key to a collection of keys.
        /// </summary>
        /// <param name="collectionKey"></param>
        /// <param name="key"></param>
        public static void AddKeyToCollection(string collectionKey, string key)
        {
            lock (_lock)
            {
                AddUpdate(collectionKey,
                () =>
                {
                    return new List<string> { key };
                },
                (collection) =>
                {
                    collection.Add(key);
                    return collection;
                });
            }
        }

        /// <summary>
        /// Removes all items in collection and then the collection itself from cache.
        /// First, each key in the collection is used to remove the key's corresponding cached item.
        /// Finally, the collection is removed using the collection key.
        /// </summary>
        /// <param name="collectionKey"></param>
        public static void RemoveCollection(string collectionKey)
        {
            lock (_lock)
            {
                var collection = Get<List<string>>(collectionKey);
                if (collection != null)
                {
                    if (collectionKey.Any())
                        foreach (var key in collection)
                            Clear(key);

                    Clear(collectionKey);
                }
            }
        }
    }
}
