﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using VeteransAffairs.Registries.Validation;

namespace VeteransAffairs.Registries.Business
{
    [Serializable()]
    public partial class TBI_POLYTRAUMA
    {
        #region private properties

        private ValidationErrorsCollection _BrokenRules = new ValidationErrorsCollection();

        private bool _sNumChanged;

        #endregion

        #region Public Properties


        public string FullName
        {
            get { return _LAST_NAME + ", " + _FIRST_NAME + " " + _MIDDLE_NAME; }
        }
  
        /// <summary>
        /// Ref Property containing list of broken business rules.
        /// </summary>
        public ValidationErrorsCollection BrokenRules
        {
            get
            {
                return _BrokenRules;
            }
        }

        public bool SnumChanged
        {
            get
            {
                return _sNumChanged;
            }
        }


        /// <summary>
        /// Property containing Length of stay = Discharge date minus Admission date minus days off unit.
        /// </summary>
        public int TotalLengthOfStay
        {
            get
            {
                DateTime dt1;
                DateTime dt2;
                TimeSpan result;
                int numberOfDays;

                dt1 = this.REHAB_DISCHARGE_DATE.GetValueOrDefault();

                if (this.REHAB_ADMIT_DATE != null)
                {
                    dt2 = this.REHAB_ADMIT_DATE.GetValueOrDefault();

                    result = dt1.Subtract(dt2);

                    numberOfDays = result.Days;
                    
                    if (this.DAYS_OFF_UNIT_TOTAL != null)
                    {
                        numberOfDays = numberOfDays - this.DAYS_OFF_UNIT_TOTAL.GetValueOrDefault();
                    }

                    if (numberOfDays > 0)
                    {
                        return numberOfDays;
                    }
                    else
                    {
                        return 0;
                    }
                }
                else
                {
                    return 0;
                }
            }
        }

        /// <summary>
        /// Property containing Age at Admission Admit date minus date of birth.
        /// </summary>
        public int AgeAtAdmission
        {
            get
            {
                DateTime dt1;
                DateTime dt2;
                int age;

                dt1 = this.REHAB_ADMIT_DATE.GetValueOrDefault();

                if (this.REHAB_ADMIT_DATE != null && this.BIRTH_DATE != null)
                {
                    dt2 = this.BIRTH_DATE.GetValueOrDefault();

                    //result = dt1.Subtract(dt2);
                    
                    // get the difference in years
                    age = dt1.Year - dt2.Year;
                    // subtract another year if we're before the
                    // birth day in the current year
                    if (dt1.Month < dt2.Month || (dt1.Month == dt2.Month && dt1.Day < dt2.Day))
                        age--;
                    
                    if (age > 0)
                    {
                        return age;
                    }
                    else
                    {
                        return 0;
                    }
                }
                else
                {
                    return 0;
                }
            }
        }
 
        #endregion

        #region Entity Events

        partial void OnSnumChanged()
        {
 	        _sNumChanged = true;
        }

        #endregion

       #region Public Methods

 
        #endregion

        #region Business Rules

        public int EnforceBusinessRules(bool SavingData, RegistriesDataAccess db)
        {
            _BrokenRules.Clear();

            //
            // Business Rule to format SSN entered by user as xxx-xx-xxxx
            //
            if (!String.IsNullOrEmpty(this.Snum))
            {
                Match snumFmtMatch = System.Text.RegularExpressions.Regex.Match(this.Snum, @"\d{3}-\d{2}-\d{4}");

                if (!snumFmtMatch.Success)
                {
                    this.Snum = String.Format("{0}-{1}-{2}", this.Snum.Substring(0, 3), this.Snum.Substring(3, 2), this.Snum.Substring(5, 4));
                }

            }

            //  string ssnCorrectFmt = Regex.Replace           (ssnFmtMatch, @"^\s*\d{3}-?\d{2}-?\d{4}\s*$",
            //}
            

            // fiscal year (discharge date) validation
            // - new: Permit one entry per Fiscal Year for both PRC and PTRP (If discharge date is NOT NULL - allow entry)
            // gets all records for patient for this type
            //TLB - 20150414 - Added SSN check and exclude TBI_POLYTRAUMA_ID so the validation 
            //                  doesn't fail because the current record is returned by query
            var patientRecords = from p in db.TBI_POLYTRAUMAs
                                 where (p.PATIENT_ID == this.PATIENT_ID) 
                                    && p.ENTRY_TYPE_ID == this.ENTRY_TYPE_ID
                                    && p.TBI_POLYTRAUMA_ID != this.TBI_POLYTRAUMA_ID
                                 select p;
            if (this.Snum != null)
            {
                patientRecords = from p in db.TBI_POLYTRAUMAs
                                 where (p.PATIENT_ID == this.PATIENT_ID || p.Snum.Replace("-", "").Replace(" ", "") == this.Snum.Replace("-", "").Replace(" ", ""))
                                        && p.ENTRY_TYPE_ID == this.ENTRY_TYPE_ID
                                        && p.TBI_POLYTRAUMA_ID != this.TBI_POLYTRAUMA_ID
                                 select p;
            }

            // determine if any fall in this fiscal year.
            DateTime? entryFiscalYear = null;

            if (this.REHAB_ADMIT_DATE != null) {
                entryFiscalYear = (DateTime)this.REHAB_ADMIT_DATE;
                entryFiscalYear = getFiscalYearStartFromDate((DateTime)entryFiscalYear);
            }

            foreach (TBI_POLYTRAUMA poly in patientRecords.Where(P => (P.IS_FOLLOW_UP == null || P.IS_FOLLOW_UP == false))) {
                //DateTime currentFiscalYear = getFiscalYearStartFromDate(DateTime.Now);
                if (poly.REHAB_ADMIT_DATE != null)
                {
                DateTime fiscalYearStart = getFiscalYearStartFromDate((DateTime)poly.REHAB_ADMIT_DATE);

                // determine if entry is in this fiscal year ...
                    if (entryFiscalYear != null)
                    {
                        if (entryFiscalYear == fiscalYearStart)
                        {
                        // is the discharge date null?
                            if (poly.REHAB_DISCHARGE_DATE == null)
                            {
                            // does not have discharge date ... no entry allowed until discharge is entered.
                            _BrokenRules.Add(new ValidationError("REHAB_DISCHARGE_DATE", "Only one (1) entry per type is allowed per fiscal year, unless patient was discharged."));
                            break;
                        } // else {
                        // has discharge date ... allow entry
                        // }
                        }
                    }
                }
                    // else {
                    // not in current fiscal year ... allow entry
                // }
            }

            //if (patientRecords.Where(P => (P.IS_FOLLOW_UP == null || P.IS_FOLLOW_UP == false)).Count() > 0) {
            //}


            //required fields
            //---------------------------------------------------
            //POLYTRAUMA_CENTER_ID
            if (this.POLYTRAUMA_CENTER_ID == 0)
            {
                _BrokenRules.Add(new ValidationError("POLYTRAUMA_CENTER_ID", "\"PolyTrauma VAMC\" is required."));
            }
            //---------------------------------------------------

            //PATIENT_HOME_VAMC_ID
            if (this.PATIENT_HOME_VAMC_ID == 0)
            {
                _BrokenRules.Add(new ValidationError("POLYTRAUMA_CENTER_ID", "\"Home VAMC\" is required."));
            }
            //---------------------------------------------------

            //dependent fields
            //---------------------------------------------------
            //Number of days of Unit needs to be difference of discharge date and admit date minus 1
                //if admit date, rehab date, and days off unit are all filled in then check for a bad condition by 
                //verifying that TotalLengthOfStay is greater than zero (this is because total length of stay equals:
                //rehab date minus admit date minus days off unit.
            if (this.REHAB_ADMIT_DATE != null 
                && this.REHAB_DISCHARGE_DATE != null 
                && this.DAYS_OFF_UNIT_TOTAL != null 
                && this.TotalLengthOfStay == 0)
            {
               _BrokenRules.Add(new ValidationError("DAYS_OFF_UNIT_TOTAL", "Number of Days off Unit needs to be 1 day less than total length of stay."));
            }

            //---------------------------------------------------


            return 0;
        }

        
        #endregion

        #region Properties for Validator Controls
   

        #endregion


        /// <summary>
        /// Accepts a DateTime object and returns the DateTime of the start of its fiscal year.
        /// </summary>
        /// <param name="val"></param>
        /// <returns></returns>
        private static DateTime getFiscalYearStartFromDate(DateTime val) {
            DateTime returnDate = new DateTime();

            if (val.Month < 10) {
                returnDate = new DateTime(val.Year - 1, 10, 01);
            } else if (val.Month >= 10) {
                returnDate = new DateTime(val.Year, 10, 01);
            }

            return returnDate;
        }
    }

 
}