﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.ServiceModel;
using System.Linq;
using VA.FTP2.Plugins.Helpers;
using Xrm;
using Microsoft.Crm.Sdk.Messages;

namespace VA.FTP2.Plugins.Request
{
    class RequestPreCreateRunner : MCSPlugins.PluginRunner
    {
        #region Constructor
        /// <summary>
        /// Standard Constructor
        /// </summary>
        /// <param name="serviceProvider"></param>
        public RequestPreCreateRunner(IServiceProvider serviceProvider)
            : base(serviceProvider)
        {
        }
        #endregion


        internal void Execute(IServiceProvider serviceProvider)
        {
            if (!PluginExecutionContext.InputParameters.Contains("Target") ||
                    !(PluginExecutionContext.InputParameters["Target"] is Entity)) return;
            try
            {
                var target = GetPrimaryEntity();
                if (target.Contains("ftp_reasonforrequest")) SetSingleRequestPriority(target);
                RequestAutomation();
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(McsSettings.getUnexpectedErrorMessage);
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(McsSettings.getUnexpectedErrorMessage);
            }
        }

        ///<summary>
        /// Sets the priority based on the reason for request
        /// </summary>
        private Entity SetSingleRequestPriority(Entity target)
        {
            // get reason for request lookup
            EntityReference reasonForRequestField = (EntityReference)target["ftp_reasonforrequest"];

            //if rfr is not multiple, update priority
            if (reasonForRequestField.Id.ToString() != "156db634-ee0d-e511-8108-00155d14711e")  //maps to Multiple Reasons for Request
            {
                OptionSetValue priorityOptionValue = PluginHelper.GetPriority(reasonForRequestField, OrganizationService);
                //Set priority value based on reason for request priority 
                if (priorityOptionValue != null)
                {
                    target["ftp_priority"] = priorityOptionValue;
                }

                return target;
            }
            //if rfr is multiple, do nothing
            return target;
        }

        /// <summary>
        /// Retrive the local time from the UTC time.
        /// </summary>
        /// <param name="utcTime"></param>
        /// <param name="service"></param>
        private DateTime RetrieveLocalTimeFromUTCTime(DateTime utcTime)
        {
            int? timeZoneCode = RetrieveCurrentUsersSettings();

            if (!timeZoneCode.HasValue)
                throw new Exception("Can't find time zone code");

            var request = new LocalTimeFromUtcTimeRequest
            {
                TimeZoneCode = timeZoneCode.Value,
                UtcTime = utcTime.ToUniversalTime()
            };

            var response = (LocalTimeFromUtcTimeResponse)OrganizationService.Execute(request);

            return response.LocalTime;
        }

        /// <summary>
        /// Retrieves the current users timezone code and locale id
        /// </summary>
        private int? RetrieveCurrentUsersSettings()
        {
            var xrm = new XrmServiceContext(OrganizationService);
            var currentUserSettings = OrganizationService.RetrieveMultiple(
            new QueryExpression("usersettings")
            {
                ColumnSet = new ColumnSet("localeid", "timezonecode"),
                Criteria = new FilterExpression
                {
                    Conditions = { new ConditionExpression("systemuserid", ConditionOperator.EqualUserId) }
                }
            }).Entities[0];

            return (int)currentUserSettings["timezonecode"];
        }

        /// <summary>
        /// Populates the start date and routes configurables to Local or Global Automation based on the results of veteran site retrieval.
        /// </summary>
        private void RequestAutomation()
        {
            // Populate Start Date
            var request = GetPrimaryEntity();
            if (!request.Contains("ftp_start")) request["ftp_start"] = RetrieveLocalTimeFromUTCTime(DateTime.UtcNow);

            // Assemble configurables list
            var configurables = new List<string>();
            if (!request.Contains("ftp_target")) configurables.Add("ftp_requesttargetdateoffset");
            if (!request.Contains("ftp_savetovista")) configurables.Add("ftp_savetovista");

            // If we have configurables (Server side data insertion or convert to request paths)
            // Otherwise /Request/JScript/main.js handles the event.
            if (configurables.Count > 0)
            {
                // Attempt to retreive the veteran's site
                var vRef = (EntityReference)request["customerid"];
                var veteran = OrganizationService.Retrieve(vRef.LogicalName, vRef.Id, new ColumnSet("ftp_siteid"));

                // If the user has a site proceed with local automation, otherwise proceed with global automation for all configurables
                if (veteran.Contains("ftp_siteid"))
                {
                    LocalAutomation((EntityReference)veteran["ftp_siteid"], configurables, request);
                }
                else
                {
                    GlobalAutomation(configurables, request);
                }
            }
        }

        /// <summary>
        /// Apply Global Automation for specified configurables.  
        /// Global Automation is when fields are populated based on configuration field values on the Active Settings record of the mcs_setting entity.
        /// </summary>
        /// <param name="configurables"></param>
        /// <param name="request"></param>
        private void GlobalAutomation(List<string> configurables, Entity request)
        {
            var qe = new QueryExpression("mcs_setting")
            {
                ColumnSet = new ColumnSet(configurables.ToArray()),
                Criteria = new FilterExpression()
            };
            qe.Criteria.AddCondition("mcs_name", ConditionOperator.Equal, "Active Settings");
            var settings = OrganizationService.RetrieveMultiple(qe);
            if (settings.Entities.Count > 0)
            {
                foreach (var attribute in settings[0].Attributes)
                {
                    if (attribute.Key == "ftp_requesttargetdateoffset")
                    {
                        var offsetDate = DateTime.UtcNow.AddDays((int)attribute.Value);
                        var validDate = CheckForValidDate(offsetDate);
                        request["ftp_target"] = RetrieveLocalTimeFromUTCTime(validDate);
                    }
                    else if (attribute.Key != "mcs_settingid")
                    {
                        request[attribute.Key] = attribute.Value;
                    }
                }
            }
        }

        /// <summary>
        /// Increments the offset date until it is not a weekend or a holiday
        /// </summary>
        /// <param name="date"></param>
        /// <returns></returns>
        private DateTime CheckForValidDate(DateTime date)
        {
            var weekend = new List<DayOfWeek>();
            weekend.Add(DayOfWeek.Saturday);
            weekend.Add(DayOfWeek.Sunday);

            while (weekend.Contains(date.DayOfWeek) || isHoliday(date))
            {
                date = date.AddDays(1);
            }

            return date;
        }

        private bool isHoliday(DateTime date)
        {
            var qe = new QueryExpression("ftp_holiday")
            {
                ColumnSet = new ColumnSet(new string[] { "ftp_date" })
            };
            var holidays = OrganizationService.RetrieveMultiple(qe);
            return holidays.Entities.Where(e => ((DateTime)e["ftp_date"]).Date == date.Date).Any();
        }

        /// <summary>
        /// Attempt to retrieve and apply the configurable settings from the site's mcs_setting record (local automation)
        /// Route the configurables that are not retrieved to global automation.
        /// </summary>
        /// <param name="siteRef"></param>
        /// <param name="configurables"></param>
        /// <param name="request"></param>
        private void LocalAutomation(EntityReference siteRef, List<string> configurables, Entity request)
        {
            var qe = new QueryExpression(siteRef.LogicalName)
            {
                ColumnSet = new ColumnSet(),
                Criteria = new FilterExpression()
            };
            qe.Criteria.AddCondition("siteid", ConditionOperator.Equal, siteRef.Id);

            var linkEntitySettings = new LinkEntity()
            {
                LinkFromEntityName = siteRef.LogicalName,
                LinkFromAttributeName = "siteid",
                LinkToEntityName = "mcs_setting",
                LinkToAttributeName = "ftp_siteid",
                JoinOperator = JoinOperator.Inner,
                Columns = new ColumnSet(configurables.ToArray()),
                EntityAlias = "Settings"
            };
            qe.LinkEntities.Add(linkEntitySettings);

            var site = OrganizationService.RetrieveMultiple(qe);
            if (site.Entities.Count > 0)
            {
                if (site[0].Contains("Settings.ftp_savetovista"))
                {
                    var savetovista = (AliasedValue)site[0]["Settings.ftp_savetovista"];
                    request["ftp_savetovista"] = savetovista.Value;
                    configurables.Remove("ftp_savetovista");
                }
                if (site[0].Contains("Settings.ftp_requesttargetdateoffset"))
                {
                    var targetdateoffset = (AliasedValue)site[0]["Settings.ftp_requesttargetdateoffset"];
                    var validDate = CheckForValidDate(DateTime.UtcNow.AddDays((int)targetdateoffset.Value));
                    request["ftp_target"] = RetrieveLocalTimeFromUTCTime(validDate);
                    configurables.Remove("ftp_requesttargetdateoffset");
                }
            }

            if (configurables.Count > 0) GlobalAutomation(configurables, request);
        }

        public override string McsSettingsDebugField
        {
            get { return "ftp_requestplugins"; }
        }

        public override Entity GetPrimaryEntity()
        {
            return (Entity)PluginExecutionContext.InputParameters["Target"];
        }

        public override Entity GetSecondaryEntity()
        {
            return (Entity)PluginExecutionContext.InputParameters["Target"];
        }
    }
}
