﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Workflow;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Activities;
using System.Collections.Generic;

namespace VRM.VCCM.Workflow
{
    public class GenerateVeteranEntitlementsForRFR : CodeActivity
    {
        [Input("Facility (recommended; else runs for all facilities linked to this Reason for Request)")]
        [ReferenceTarget("ftp_facility")]
        public InArgument<EntityReference> inputFacility { get; set; }

        [Input("Veteran (recommended; else runs for all veterans linked to the facility/ies linked to this Reason for Request)")]
        [ReferenceTarget("contact")]
        public InArgument<EntityReference> inputContact { get; set; }

        protected override void Execute(CodeActivityContext executionContext)
        {
            ITracingService tracer = executionContext.GetExtension<ITracingService>();
            IWorkflowContext context = executionContext.GetExtension<IWorkflowContext>();
            IOrganizationService service = ((IOrganizationServiceFactory)executionContext.GetExtension<IOrganizationServiceFactory>()).CreateOrganizationService(context.UserId);

            try
            {
                Entity contextRFR = new Entity();
                contextRFR.LogicalName = context.PrimaryEntityName;
                contextRFR.Id = context.PrimaryEntityId;
                EntityReference filterFacility = inputFacility.Get<EntityReference>(executionContext);
                EntityReference filterVeteran = inputContact.Get<EntityReference>(executionContext);
                RetrieveRequest rfrRequest = new RetrieveRequest()
                {
                    Target = contextRFR.ToEntityReference(),
                    ColumnSet = new ColumnSet("ftp_reason"),
                    RelatedEntitiesQuery = new RelationshipQueryCollection()
                    {
                        new KeyValuePair<Relationship, QueryBase>(
                            new Relationship("ftp_entitlementtemplate_ftp_reasonforrequest"),
                            new QueryExpression("entitlementtemplate")
                            {
                                ColumnSet = new ColumnSet(true),
                                LinkEntities = 
                                {
                                    new LinkEntity()
                                    {
                                        EntityAlias = "SLAforET",
                                        LinkFromEntityName = "entitlementtemplate",
                                        LinkFromAttributeName = "slaid",
                                        LinkToEntityName = "sla",
                                        LinkToAttributeName = "slaid",
                                        Columns = new ColumnSet("statecode"),
                                        LinkCriteria = 
                                        {
                                            Conditions =
                                            {
                                                new ConditionExpression("statecode", ConditionOperator.Equal, "Active")
                                            }
                                        }
                                    }
                                }
                            }
                        )
                    }
                };
                QueryExpression relatedFacilityQuery = new QueryExpression("ftp_facility")
                {
                    ColumnSet = new ColumnSet("ftp_facilityid", "ftp_name")
                };
                if (filterFacility != null)
                {
                    relatedFacilityQuery.Criteria.AddCondition(new ConditionExpression("ftp_facilityid", ConditionOperator.Equal, filterFacility.Id));
                }
                rfrRequest.RelatedEntitiesQuery.Add(
                    new KeyValuePair<Relationship, QueryBase>(
                        new Relationship("ftp_ftp_reasonforrequest_ftp_facility_NN"),
                        relatedFacilityQuery
                    )
                );

                try
                {
                    RetrieveResponse rfrResponse = (RetrieveResponse)service.Execute(rfrRequest);
                    if (rfrResponse.Entity.RelatedEntities.Contains(new Relationship("ftp_ftp_reasonforrequest_ftp_facility_NN")) && ((EntityCollection)rfrResponse.Entity.RelatedEntities[new Relationship("ftp_ftp_reasonforrequest_ftp_facility_NN")]).Entities.Count > 0)
                    {
                        if (rfrResponse.Entity.RelatedEntities.Contains(new Relationship("ftp_entitlementtemplate_ftp_reasonforrequest")) && ((EntityCollection)rfrResponse.Entity.RelatedEntities[new Relationship("ftp_entitlementtemplate_ftp_reasonforrequest")]).Entities.Count == 1)
                        {
                            Entity thisEntitlementTemplate = ((EntityCollection)rfrResponse.Entity.RelatedEntities[new Relationship("ftp_entitlementtemplate_ftp_reasonforrequest")]).Entities[0];
                            EntityReference SLAForThisEntitlementTemplate = (EntityReference)(thisEntitlementTemplate["slaid"]);

                            foreach (Entity thisFacility in ((EntityCollection)rfrResponse.Entity.RelatedEntities[new Relationship("ftp_ftp_reasonforrequest_ftp_facility_NN")]).Entities)
                            {
                                //retrieve active veterans whose Home Facility is this facility, optionally filter by provided Veteran from workflow
                                QueryByAttribute vetQBA = new QueryByAttribute("contact");
                                vetQBA.ColumnSet = new ColumnSet("fullname", "statecode", "ftp_facilityid");
                                vetQBA.AddAttributeValue("statecode", "Active");
                                if (filterVeteran != null)
                                {
                                    vetQBA.AddAttributeValue("contactid", filterVeteran.Id);
                                }
                                vetQBA.AddAttributeValue("ftp_facilityid", (Guid)thisFacility["ftp_facilityid"]);

                                EntityCollection retrievedVeterans = service.RetrieveMultiple(vetQBA);
                                if (retrievedVeterans.Entities.Count > 0)
                                {
                                    foreach (Entity thisActiveVeteran in retrievedVeterans.Entities)
                                    {
                                        RetrieveRequest vetRequest = new RetrieveRequest()
                                        {
                                            Target = thisActiveVeteran.ToEntityReference(),
                                            ColumnSet = new ColumnSet("fullname"),
                                            RelatedEntitiesQuery = new RelationshipQueryCollection()
                                            {
                                                new KeyValuePair<Relationship, QueryBase>(
                                                    new Relationship("contact_entitlement_Customer"),
                                                    new QueryExpression("entitlement")
                                                    {
                                                        ColumnSet = new ColumnSet("entitlementtemplateid", "customerid", "statecode"),
                                                        Criteria = 
                                                        {
                                                            Conditions =
                                                            {
                                                                new ConditionExpression("entitlementtemplateid", ConditionOperator.Equal, thisEntitlementTemplate.Id),
                                                                new ConditionExpression("slaid", ConditionOperator.Equal, SLAForThisEntitlementTemplate.Id)
                                                            }
                                                        }
                                                    }
                                                )
                                            }
                                        };

                                        RetrieveResponse vetResponse = (RetrieveResponse)service.Execute(vetRequest);
                                        if (vetResponse.Entity.RelatedEntities.Contains(new Relationship("contact_entitlement_Customer")) && ((EntityCollection)vetResponse.Entity.RelatedEntities[new Relationship("contact_entitlement_Customer")]).Entities.Count == 0)
                                        {
                                            List<string> fields = new List<string>()
                                            {
                                                "name", "slaid", "startdate", "enddate", "restrictcasecreation", "description", "allocationtypecode", "decreaseremainingon", "totalterms"
                                            };

                                            Entity newEntitlement = new Entity("entitlement");
                                            foreach (string thisField in fields)
                                            {
                                                if (thisEntitlementTemplate.Contains(thisField))
                                                {
                                                    if (thisField == "name")
                                                    {
                                                        string entitlementName = (string)thisActiveVeteran["fullname"] + " " + (string)(thisEntitlementTemplate[thisField]);
                                                        if (entitlementName.Length > 100) entitlementName = entitlementName.Substring(0, 97) + "...";
                                                        newEntitlement.Attributes.Add(thisField, entitlementName);
                                                    }
                                                    else
                                                    {
                                                        newEntitlement.Attributes.Add(thisField, thisEntitlementTemplate[thisField]);
                                                    }
                                                }
                                            }
                                            newEntitlement.Attributes.Add("customerid", thisActiveVeteran.ToEntityReference());
                                            newEntitlement.Attributes.Add("entitlementtemplateid", thisEntitlementTemplate.ToEntityReference());
                                            try
                                            {
                                                newEntitlement.Id = service.Create(newEntitlement);
                                            }
                                            catch (Exception ex)
                                            {
                                                //error creating the new veteran entitlement
                                                throw ex;
                                            }

                                            SetStateRequest ssr = new SetStateRequest()
                                            {
                                                State = new OptionSetValue(1), /*Active*/
                                                Status = new OptionSetValue(1), /*Active*/
                                                EntityMoniker = newEntitlement.ToEntityReference()
                                            };
                                            try
                                            {
                                                SetStateResponse resp = (SetStateResponse)service.Execute(ssr);
                                                tracer.Trace("Created and activated new veteran entitlement: " + (string)newEntitlement["name"]);
                                            }
                                            catch (Exception ex)
                                            {
                                                //error activating the new veteran entitlement
                                                throw ex;
                                            }
                                        }
                                        else
                                        {
                                            tracer.Trace("Veteran already has an entitlement built from this Reason for Request's entitlement template");
                                        }
                                    }
                                }
                                else
                                {
                                    if (filterVeteran != null)
                                    {
                                        if (filterFacility != null)
                                        {
                                            throw new Exception("The specified facility is not the Home Facility of the specified veteran.");
                                        }
                                        else
                                        {
                                            throw new Exception("None of this Reason for Request's facilities are the Home Facility of the specified veteran.");
                                        }
                                    }
                                    else
                                    {
                                        if (filterFacility != null)
                                        {
                                            throw new Exception("The specified facility is not the Home Facility for any veterans.");
                                        }
                                        else
                                        {
                                            //this facility is not the Home facility of any veterans at all, but facilities connected to thisRFR may have veterans
                                        }
                                    }
                                }
                            }
                        }
                        else
                        {
                            //throw new Exception("This Reason for Request does not have an active Entitlement Template, or its Entitlement Template does not have an active SLA.");
                        }
                    }
                    else
                    {
                        if (filterFacility != null)
                        {
                            throw new Exception("Could not find the specified facility related to this Reason for Request.");
                        }
                        else
                        {
                            //throw new Exception("This Reasons for Request does not have any related facilities.");
                        }
                    }
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
