﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Workflow;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Activities;
using System.Collections.Generic;

namespace VRM.VCCM.Workflow
{
    public class GenerateEntitlementsForVeteran : CodeActivity
    {
        [Input("Facility (optional; defaults to Veteran's Home Facility)")]
        [ReferenceTarget("ftp_facility")]
        public InArgument<EntityReference> inputFacility { get; set; }

        protected override void Execute(CodeActivityContext executionContext)
        {
            ITracingService tracer = executionContext.GetExtension<ITracingService>();
            IWorkflowContext context = executionContext.GetExtension<IWorkflowContext>();
            IOrganizationService service = ((IOrganizationServiceFactory)executionContext.GetExtension<IOrganizationServiceFactory>()).CreateOrganizationService(context.UserId);

            //throw new InvalidPluginExecutionException("begin.");
            try
            {
                Entity contextContact = new Entity();
                contextContact.LogicalName = context.PrimaryEntityName;
                contextContact.Id = context.PrimaryEntityId;
                Entity retrievedContact = service.Retrieve("contact", contextContact.Id, new ColumnSet("fullname", "ftp_facilityid"));
                tracer.Trace("before reading inputFacility.");
                EntityReference filterFacility = inputFacility != null ? inputFacility.Get(executionContext) : new EntityReference();
                tracer.Trace("after reading inputFacility.");
                if (filterFacility == null && retrievedContact.Contains("ftp_facilityid"))
                {
                    tracer.Trace("inputFacility was empty. REading ftp_facilityid from Veteran.");
                    filterFacility = (EntityReference)retrievedContact["ftp_facilityid"];
                }
                //throw new InvalidPluginExecutionException("stopped to read trace.");

                if (filterFacility != null)
                {
                    QueryExpression rfrQuery = new QueryExpression("ftp_reasonforrequest")
                    {
                        ColumnSet = new ColumnSet("ftp_reason", "ftp_entitlementtemplateid", "ftp_display", "statecode"),
                        Criteria =
                        {
                            Conditions =
                            {
                                new ConditionExpression("ftp_entitlementtemplateid", ConditionOperator.NotNull),
                                new ConditionExpression("ftp_display", ConditionOperator.Equal, true),
                                new ConditionExpression("statecode", ConditionOperator.Equal, "Active")
                            }
                        },
                        LinkEntities =
                        {
                            new LinkEntity()
                            {
                                EntityAlias = "ET",
                                LinkFromEntityName = "ftp_reasonforrequest",
                                LinkFromAttributeName = "ftp_entitlementtemplateid",
                                LinkToEntityName = "entitlementtemplate",
                                LinkToAttributeName = "entitlementtemplateid",
                                Columns = new ColumnSet(true),
                                LinkEntities = 
                                {
                                    new LinkEntity()
                                    {
                                        EntityAlias = "SLAforET",
                                        LinkFromEntityName = "entitlementtemplate",
                                        LinkFromAttributeName = "slaid",
                                        LinkToEntityName = "sla",
                                        LinkToAttributeName = "slaid",
                                        Columns = new ColumnSet("statecode")
                                    }
                                }
                            },
                            new LinkEntity()
                            {
                                EntityAlias = "RFRFacilityIntersectEntity",
                                LinkFromEntityName = "ftp_reasonforrequest",
                                LinkFromAttributeName = "ftp_reasonforrequestid",
                                LinkToEntityName = "ftp_ftp_reasonforrequest_ftp_facility_nn",
                                LinkToAttributeName = "ftp_reasonforrequestid",
                                LinkCriteria = 
                                {
                                    Conditions = 
                                    {
                                        new ConditionExpression("ftp_facilityid", ConditionOperator.Equal, filterFacility.Id)
                                    }
                                }
                            }
                        }
                    };

                    try
                    {
                        EntityCollection retrievedRFRsWithEntitlementTemplates = service.RetrieveMultiple(rfrQuery);
                        foreach (Entity thisRFR in retrievedRFRsWithEntitlementTemplates.Entities)
                        {
                            if (thisRFR.Contains("ET.slaid") && thisRFR.Contains("SLAforET.statecode") && ((OptionSetValue)((AliasedValue)thisRFR["SLAforET.statecode"]).Value).Value == 1 /*Active*/)
                            {
                                QueryByAttribute entitlementQBA = new QueryByAttribute("entitlement");
                                entitlementQBA.AddAttributeValue("entitlementtemplateid", ((AliasedValue)thisRFR["ET.entitlementtemplateid"]).Value);
                                entitlementQBA.AddAttributeValue("customerid", retrievedContact.ToEntityReference().Id);
                                entitlementQBA.AddAttributeValue("statecode", "Active");
                                //include sla in query to make sure we get newest version, in case template has changed
                                entitlementQBA.AddAttributeValue("slaid", ((EntityReference)((AliasedValue)thisRFR["ET.slaid"]).Value).Id);
                                try
                                {
                                    EntityCollection retrievedVeteranEntitlementsWithThisTemplate = service.RetrieveMultiple(entitlementQBA);
                                    if (retrievedVeteranEntitlementsWithThisTemplate.Entities.Count == 0)
                                    {
                                        List<string> fields = new List<string>()
                                        {
                                            "name", "slaid", "startdate", "enddate", "restrictcasecreation", "description", "allocationtypecode", "decreaseremainingon", "totalterms"
                                        };

                                        Entity newEntitlement = new Entity("entitlement");
                                        foreach (string thisField in fields)
                                        {
                                            if (thisRFR.Contains("ET." + thisField))
                                            {
                                                if (thisField == "name")
                                                {
                                                    string entitlementName = (string)retrievedContact["fullname"] + " " + (string)((AliasedValue)(thisRFR["ET." + thisField])).Value;
                                                    if (entitlementName.Length > 100) entitlementName = entitlementName.Substring(0, 97) + "...";
                                                    newEntitlement.Attributes.Add(thisField, entitlementName);
                                                }
                                                else
                                                {
                                                    newEntitlement.Attributes.Add(thisField, ((AliasedValue)(thisRFR["ET." + thisField])).Value);
                                                }
                                            }
                                        }
                                        newEntitlement.Attributes.Add("customerid", retrievedContact.ToEntityReference());
                                        newEntitlement.Attributes.Add("entitlementtemplateid", new EntityReference("entitlementtemplate", (Guid)((AliasedValue)thisRFR["ET.entitlementtemplateid"]).Value));
                                        try
                                        {
                                            newEntitlement.Id = service.Create(newEntitlement);
                                        }
                                        catch (Exception ex)
                                        {
                                            //error creating the new veteran entitlement
                                            throw ex;
                                        }

                                        SetStateRequest ssr = new SetStateRequest()
                                        {
                                            State = new OptionSetValue(1), /*Active*/
                                            Status = new OptionSetValue(1), /*Active*/
                                            EntityMoniker = newEntitlement.ToEntityReference()
                                        };
                                        try
                                        {
                                            SetStateResponse resp = (SetStateResponse)service.Execute(ssr);
                                        }
                                        catch (Exception ex)
                                        {
                                            //error activating the new veteran entitlement
                                            throw ex;
                                        }
                                    }
                                    else
                                    {
                                        //do not create new Entitlement if one meeting query's conditions already exists
                                        tracer.Trace("Veteran already has an active entitlement with this entitlement template. Skipping entitlement creation for this reason for request, '{0}'.", thisRFR["ftp_reason"]);
                                    }
                                }
                                catch (Exception ex)
                                {
                                    throw ex;
                                }
                            }
                            else
                            {
                                tracer.Trace("The entitlement template for the reason for request, '{0}', does not have an active SLA. Did not create veteran entitlement for this reason for request.", thisRFR["ftp_reason"]);
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        throw ex;
                    }
                }
                else
                {
                    throw new InvalidPluginExecutionException("Could not determine facility to use for entitlement generation.");
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
