﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Text;
using System.Data;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.TBI
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class TBIReferralManager : BaseBO
    {
        private UserAccountManager _user = (UserAccountManager)System.Threading.Thread.CurrentPrincipal;
        private RegistriesCommonManager _commonManager = new RegistriesCommonManager();

        public TBIReferralManager() 
        {
            _defaultSortField = "REFERRAL_ID";
            
        }

        private IQueryable<TBI_REFERRAL> LinqAll()
        {
           

            //populate LinqAll
            //Referral linq class has a subtype of TBI_REFERRAL.  The Registry association is specified in the dbml.
            //Therefore adding the OfType in the following query will only return TBI_REFERRRAL objects.
            IQueryable<TBI_REFERRAL> tempLinq = (from e in _db.REFERRALs.OfType<TBI_REFERRAL>()
                                                 select e);
            
            //add all business filtering rules 
            if (!_user.IsSuperUser)
            {
                //rule: add filter to return only items within the list of institutions accessible by the user
                int[] accessibleInstitutions = (from a in 
                      _commonManager.GetActiveUserRoles(_user.RegistryId, _user.UserId) 
                      select a.STD_INSTITUTION_ID).Distinct().ToArray();

                //TO DO : Move the IQueryable calls above to userAccount Manager

                tempLinq = (from e in tempLinq
                            //join r in activeUserRoles on e.STD_INSTITUTION_ID equals r.STD_INSTITUTION_ID
                            where accessibleInstitutions.Contains(e.STD_INSTITUTION_ID.GetValueOrDefault())
                            select e).Distinct();

            }
            //TO DO : need to get this to filter on role properly
                  
            return tempLinq;

        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
            lo.LoadWith<REFERRAL>(e => e.STD_REFERRALST);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_INSTITUTION_PARENT);
            lo.LoadWith<REFERRAL>(e => e.REFERRAL_DETAILs);
            lo.LoadWith<REFERRAL>(e => e.WKF_CASEs);
            lo.LoadWith<WKF_CASE>(e => e.TBI_TRACKINGs);
            lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_TBIEVALDELAYTYPE);
            lo.LoadWith<TBI_TRACKING>(e => e.STD_TBITEAMTYPE);
            lo.LoadWith<TBI_TRACKING>(e => e.STD_TBIREFERRALST);
            
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;


        }

        private void SetLoadWithForListViews(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
            lo.LoadWith<REFERRAL>(e => e.STD_REFERRALST);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_INSTITUTION_PARENT);
            
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        #region Method for Check Existence
        
        public bool ReferralExists(int id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.REFERRALs where e.REFERRAL_ID == id select e).Count();
                
                if (count > 0)
                {
                    return true;
                }
                else 
                {
                    return false;
                }
            }
        }

        #endregion

        #region Methods for Select

        public IEnumerable<TBI_REFERRAL> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                List<TBI_REFERRAL> entities;
                entities = SelectLinqFilter().OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
                 
            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectLinqFilter().Count();
            }
        }

        private IQueryable<TBI_REFERRAL> SelectLinqFilter()
        {
            IQueryable<TBI_REFERRAL> linqFilter = LinqAll();
            return linqFilter;
        }

        #endregion

        #region Methods for SelectByID

        public TBI_REFERRAL SelectByID(int id)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                TBI_REFERRAL referral = SelectByIDLinqFilter(id).SingleOrDefault();

                if (referral != null)
                {
                    referral.SetAsChangeTrackingRoot(true);

                    int status = referral.EnforceBusinessRules(false, _db);
                }
                
                return referral;

            }

        }

        private IQueryable<TBI_REFERRAL> SelectByIDLinqFilter(int id)
        {
            IQueryable<TBI_REFERRAL> linqFilter = LinqAll();
            
            linqFilter = from t in linqFilter where t.REFERRAL_ID == id select t;

            return linqFilter;
        }

        #endregion

        #region Methods for SelectByStatus

        public IEnumerable<TBI_REFERRAL> SelectByStatus(int id, string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }
            
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                IEnumerable<TBI_REFERRAL> entities;
                entities = SelectByStatusLinqFilter(id).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        
        public int SelectByStatusCount(int id, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectByStatusLinqFilter(id).Count();
            }
        }

        private IQueryable<TBI_REFERRAL> SelectByStatusLinqFilter(int id)
        {
            IQueryable<TBI_REFERRAL> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.STD_REFERRALSTS_ID == id select t;

            return linqFilter;
        }

        #endregion
        
        #region Methods for SelectByPatient

        public IEnumerable<TBI_REFERRAL> SelectByPatient(int id, string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                IEnumerable<TBI_REFERRAL> entities;
                entities = SelectByPatientLinqFilter(id).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }


        public int SelectByPatientCount(int id, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectByPatientLinqFilter(id).Count();
            }
        }

        private IQueryable<TBI_REFERRAL> SelectByPatientLinqFilter(int id)
        {
            IQueryable<TBI_REFERRAL> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.PATIENT_ID == id select t;

            return linqFilter;
        }

        #endregion

        #region Methods for Update

        public int Update(TBI_REFERRAL referral, TBI_REFERRAL original_referral)
        {

            using (_db = GetDataContext())
            {

                _db.DeferredLoadingEnabled = false;

                _db.REFERRALs.Attach(referral, original_referral);
                _db.WKF_CASEs.AttachAll(referral.WKF_CASEs,true);
                
                
                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                }
                catch
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }
            }
            return referral.REFERRAL_ID;
        }

        public int Update(TBI_REFERRAL referral)
        {
            int returnValue = 0;
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                referral.EnforceBusinessRules(true, _db);
                
                //create an instance of the custom eventArgs in order to populate the id selected
                BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

                if (referral.BrokenRules.Count == 0)
                {
                    referral.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                        returnValue = 
                        eventArgs.SavedItemId = referral.REFERRAL_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                        returnValue =
                        eventArgs.SavedItemId = referral.REFERRAL_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    }
                    catch (Exception e)
                    {
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                    }
                }
                RaiseSaveEvent(this, eventArgs);
                return returnValue;
            }
        }

        #endregion

        #region Methods to change foreign key references

        /// <summary>
        /// Method to return a new selected OtherConsultStatus (STD_TBIREFERRALST) in order to modify the linq entity.
        /// </summary>
        public STD_TBIREFERRALST getNewOtherConsultStatus(int? id)
        {
            STD_TBIREFERRALST newEntity = null;

            if (id.HasValue)
            {

                using (_db = GetDataContext())
                {
                    newEntity = (from se in _db.STD_TBIREFERRALSTs where se.ID == id select se).FirstOrDefault();
                }
            }

            return newEntity;

        }

        /// <summary>
        /// Method to return a new selected Referral Status (STD_REFERRALSTs) in order to modify the linq entity.
        /// </summary>
        public STD_REFERRALST getNewReferralStatus(int id)
        {
            STD_REFERRALST newEntity;

            using (_db = GetDataContext())
            {
                newEntity = (from se in _db.STD_REFERRALSTs where se.ID == id select se).FirstOrDefault();
            }

            return newEntity;

        }

        /// <summary>
        /// Method to return a new selected Schedule Status (STD_TBISCHEDULESTs) in order to modify the linq entity.
        /// </summary>
        public STD_TBISCHEDULEST getNewTBIScheduleStatus(int? id)
        {
            STD_TBISCHEDULEST newEntity;

            using (_db = GetDataContext())
            {
                newEntity = (from se in _db.STD_TBISCHEDULESTs where se.ID == id select se).FirstOrDefault();
            }

            return newEntity;

        }

        /// <summary>
        /// Method to return a new selected Delay Type Status (STD_TBIEVALDELAYTYPEs) in order to modify the linq entity.
        /// </summary>
        public STD_TBIEVALDELAYTYPE getNewTBIEvalDelayTypeStatus(int? id)
        {
            STD_TBIEVALDELAYTYPE newEntity;

            using (_db = GetDataContext())
            {
                newEntity = (from se in _db.STD_TBIEVALDELAYTYPEs where se.ID == id select se).FirstOrDefault();
            }

            return newEntity;

        }
        
        /// <summary>
        /// Method to return a new selected TBI Evaluation Team (STD_TBITEAMTYPEs) in order to modify the linq entity.
        /// </summary>
        public STD_TBITEAMTYPE getNewTBIEvalTeam(int? id)
        {
            STD_TBITEAMTYPE newEntity;

            using (_db = GetDataContext())
            {
                newEntity = (from se in _db.STD_TBITEAMTYPEs where se.ID == id select se).FirstOrDefault();
            }

            return newEntity;

        }

        

        #endregion


        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref TBI_REFERRAL referral) 
        {
            using (_db = GetDataContext())
            {
                int status = referral.EnforceBusinessRules(false, _db);
            }

            return 0;
        }


        #endregion


    }
}
