﻿using System;
using System.Data;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Globalization;
using System.Threading;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager.MDWS;
using System.Data.Linq;
using System.Web.Configuration;
using VeteransAffairs.Registries.BusinessManager.TBIInstrumentVIAsrv;
using System.Web;
using System.ServiceModel;

namespace VeteransAffairs.Registries.BusinessManager.TBI
{

    [Serializable()]
    [System.ComponentModel.DataObject]
    public class TBIInstrumentsManager : BaseBO
    {
        public static readonly string LookupStationCode = "442";
        private CprsManager _cprsManager = new CprsManager();
        public TBIServiceTO TbiServiceTo { get; set; } // 01-Nov-2010 jps  replace [internal set] with [set] per JH;
        RegistriesLinqDataContext _RLDC = new RegistriesLinqDataContext();
        ufn_findM2PINoteTypesResult _findPreviousNoteTypes = new ufn_findM2PINoteTypesResult();

        #region MDWS Variable Declarations

        private EmrSvc _tbiService;
        private FaultTO _faultTO;

        private PersonsTO _personsTO;
        public PersonsTO GetPersonTO()
        {
            return this._personsTO;
        }

        #endregion

        #region VIA Variable Declarations

        private TbiServiceInterfaceClient _VIAService;
        private personsTO _personsTOVIA;

        protected string _loginSiteCode = null;
        private string _userID = String.Empty;
        private string _tokenizedDUZ = String.Empty;
        private string _localPid = String.Empty;
        private string _savedItemID = String.Empty;
        private string _providerName = String.Empty;

        #endregion

        public TBIInstrumentsManager()
        {
            _defaultSortField = "SURVEYS_ID";
            if (_loginSiteCode == null)
            {
                _loginSiteCode = LookupStationCode;
            }
        }

        ~TBIInstrumentsManager()
        {
            if (_tbiService != null)
            {
                TaggedTextArray _result = _tbiService.disconnect();
            }
        }

        #region MDWS Service Methods

        /// <summary>
        /// Loads MSSR Service (MDWS) and calls addDataSourceMSSR
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        public bool LoadTbiService(TBIServiceTO tbiServiceTo)
        {
            _tbiService = new EmrSvc();
            _tbiService.CookieContainer = new System.Net.CookieContainer();

            this.TbiServiceTo = tbiServiceTo;

            _faultTO = addDataSourceTBI(tbiServiceTo);
            if (_faultTO != null)
            {
                this.TbiServiceTo = null;
            }

            _personsTO = CPRSLaunch(tbiServiceTo);
            if (_personsTO == null || _personsTO.fault != null)
            {
                this.TbiServiceTo = null;
            }

            if (this.TbiServiceTo != null && _personsTO != null)
            {
                this.TbiServiceTo.PatientName = _personsTO.patient.patientName;
                this.TbiServiceTo.PatientDOB = _personsTO.patient.dob;
                this.TbiServiceTo.PatientDOD = _personsTO.patient.deceasedDate;
                this.TbiServiceTo.PatientGender = _personsTO.patient.gender;
                this.TbiServiceTo.PatientSnum = _personsTO.patient.ssn;
                this.TbiServiceTo.PatientICN = _personsTO.patient.mpiPid;
                this.TbiServiceTo.PatientDFN = _personsTO.patient.localPid;
                this.TbiServiceTo.STA6N = _personsTO.user.siteId;
                this.TbiServiceTo.PatientAge = _personsTO.patient.age.ToString();
                this.TbiServiceTo.PatientEthnicity = _personsTO.patient.ethnicity;

                // look for demographic data
                if (_personsTO.patient.demographics != null)
                {
                    if (_personsTO.patient.demographics.Length > 0)
                    {
                        DemographicSetTO demo = _personsTO.patient.demographics[0];

                        if (demo.addresses != null)
                        {
                            if (demo.addresses.Length > 0)
                            {
                                AddressTO addy = demo.addresses[0];
                                if (addy != null)
                                {
                                    if (!String.IsNullOrEmpty(addy.city))
                                    {
                                        this.TbiServiceTo.PatientCity = addy.city.ToUpper();
                                    }

                                    if (!String.IsNullOrEmpty(addy.state))
                                    {
                                        this.TbiServiceTo.PatientState = addy.state.ToUpper();
                                    }
                                }
                            }
                        }
                    }
                }

                this.TbiServiceTo.UserName = _personsTO.user.name;

                this.TbiServiceTo.PatientAdmitTimeStamp = _personsTO.patient.admitTimestamp;

                if (_personsTO.patient.location != null)
                {
                    this.TbiServiceTo.PatientLocationId = _personsTO.patient.location.id;

                    if (_personsTO.patient.location.facility != null)
                    {
                        this.TbiServiceTo.InstitutionName = _personsTO.patient.location.facility.name;
                    }

                }

                // TODO: PCP - would need to store in MSSR_PATIENT_DEMS
                //TaggedText patientPCP = GetPatientPCP(this.TbiServiceTo);
                //if (patientPCP != null) {

                //}
            }


            if (this.TbiServiceTo == null)
            {
                return false;
            }
            else
            {
                return true;
            }

        }

        private bool? logFullVIAResponse;
        private bool LogFullVIAResponse
        {
            get
            {
                if (!logFullVIAResponse.HasValue)
                    logFullVIAResponse = "true".Equals(Helpers.GetConfigEntry("LogFullVIAResponse"), StringComparison.OrdinalIgnoreCase);
                return logFullVIAResponse.Value;
            }
        }

        /// <summary>
        /// MDWS: calls addDataSourceMSSR to tell MDWS what server to connect to.
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        private FaultTO addDataSourceTBI(TBIServiceTO tbiServiceTo)
        {
            FaultTO ret = null;

            if (tbiServiceTo.siteCode == "931")
            {
                SiteTO result = null;

                MDWSLog mdwsLog = new MDWSLog();
                MDWS_LOG _mdwsLog = new MDWS_LOG();

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "addDataSource";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { tbiServiceTo.siteCode, "TESTSITE", tbiServiceTo.ServerIP, tbiServiceTo.ServerPort, "HIS", "VISTA", "11" });

                try
                {
                    result = _tbiService.addDataSource(tbiServiceTo.siteCode, "TESTSITE", tbiServiceTo.ServerIP, tbiServiceTo.ServerPort, "HIS", "VISTA", "11");
                    if (result != null)
                    {
                        if (LogFullVIAResponse)
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        else
                            _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                        if (result.fault != null)
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                catch (Exception exc)
                {
                    Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "addDatatSourceTBI()");
                    _mdwsLog.ERROR_LEVEL = 2;
                }

                mdwsLog.LogMDWSCall(_mdwsLog);

                ret = result.fault;
            }

            return ret;
        }

        /// <summary>
        /// MDWS: CPRSLauch Method
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        private PersonsTO CPRSLaunch(TBIServiceTO tbiServiceTo)
        {
            PersonsTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsLaunch";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", tbiServiceTo.siteCode, tbiServiceTo.DUZ, tbiServiceTo.DFN });
            //*** DO NOT OVERRIDE THE siteCode IN THIS METHOD
            try
            {
                result = _tbiService.cprsLaunch("TBI-PWD", tbiServiceTo.siteCode, tbiServiceTo.DUZ, tbiServiceTo.DFN);
                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "CPRSLaunch()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: GetPatientPCP Service Method.
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        private TaggedText GetPatientPCP(TBIServiceTO tbiServiceTo)
        {
            TaggedText result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getPcpForPatient";
            _mdwsLog.PARAMETERS = tbiServiceTo.DFN;

            try
            {
                result = _tbiService.getPcpForPatient(tbiServiceTo.DFN);

                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "GetPatientPCP()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Disconnect site service call.
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        public TaggedTextArray DisconnectSite()
        {
            TaggedTextArray result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "disconnect";
            _mdwsLog.PARAMETERS = string.Empty;

            try
            {
                result = _tbiService.disconnect();
                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "DisconnectSite()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Retrieve consults
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getConsults(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            TaggedConsultArrays ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getConsultsForPatient";
            _mdwsLog.PARAMETERS = string.Empty;

            try
            {
                ret = _tbiService.getConsultsForPatient();
                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = ret.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == ret) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.arrays[0].consults == null))
                        {
                            var consultsArray = ret.arrays.SelectMany(s => s.consults).ToList();
                            result = (from e in consultsArray orderby e.timestamp descending select new KeyValuePair<string, string>(e.id, DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + " (" + e.status + ") " + e.title + " " + "Consult #: " + e.id)).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getConsults()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: getHospitalAdmissions (getAdmissions) service call
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getHospitalAdmissions(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            TaggedInpatientStayArray ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getAdmissions";
            _mdwsLog.PARAMETERS = string.Empty;

            try
            {
                ret = _tbiService.getAdmissions();
                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.stays == null))
                        {
                            result = (from e in ret.stays orderby e.admitTimestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.admitTimestamp + ";H", e.location.name + " (" + DateTime.ParseExact(e.admitTimestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + ")")).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getHospitalAdmissions()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: getVisits service call
        /// </summary>
        /// <param name="fromDate"></param>
        /// <param name="toDate"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getVisits(DateTime fromDate, DateTime toDate, out string error)
        {
            TaggedVisitArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            string formattedFromDate = FormatDateForMDWS(fromDate);
            string formattedToDate = FormatDateForMDWS(toDate);

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getVisits";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { formattedFromDate, formattedToDate });

            try
            {
                ret = _tbiService.getVisits(formattedFromDate, formattedToDate);
                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        error = null;
                        result = (from e in ret.visits orderby e.timestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.timestamp + ";A", DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("g") + " " + e.location.name + " " + e.status)).ToList();
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getVisits()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Utility method for formatting date.
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForMDWS(DateTime dateValue)
        {
            // CPRS is expecting date in the following format "20000101.000000";
            return String.Format("{0:yyyyMMdd.HHmmss}", dateValue);
        }

        /// <summary>
        /// MDWS: This will get the list of CPRS users for the correct MDWS connection
        /// </summary>
        /// <param name="target">Beginning of first user name in the list</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> cprsUserLookup(string target, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            UserArray ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsUserLookup";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { target });

            try
            {
                ret = _tbiService.cprsUserLookup(target);

                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.users == null))
                        {
                            result = (from e in ret.users orderby e.name select new KeyValuePair<string, string>(e.DUZ, e.name)).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "addDatatSourceTBI()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: This will get the locations for the correct MDWS connection
        /// </summary>
        /// <param name="filter">Use this to filter the results returned from MDWS</param>
        /// <param name="target">Purpose and expected values Unknown </param>
        /// <param name="direction">MDWS expects a value of 1 or -1</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getLocations(string filter, string target, string direction, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            TaggedHospitalLocationArray ret = null;
            string searchParameter;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            if (filter.Trim() == String.Empty)
            {
                searchParameter = target;
            }
            else
            {
                searchParameter = filter.Trim().ToUpper();
            }

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getLocations";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { searchParameter, direction });

            try
            {
                ret = _tbiService.getLocations(searchParameter, direction);

                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.locations == null))
                        {
                            result = (from e in ret.locations where e.name.ToLower().StartsWith(filter.ToLower()) orderby e.name select new KeyValuePair<string, string>(e.id, e.name)).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getLocations()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Find STD_INSTITUTION_ID by STA3N and data source provider
        /// </summary>
        /// <param name="sta3n"></param>
        /// <param name="dataSourceProvider"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public System.Nullable<int> GetStdInstitutionId(string sta3n, string dataSourceProvider)
        {
            System.Nullable<int> result;
            int sta3nInteger = VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(sta3n);

            if (sta3nInteger != VeteransAffairs.Registries.Business.Utilities.Helpers.InvalidSta3n)
            {
                if (sta3nInteger > 900)   // test site sta3n
                {
                    MDWSLog mdwsLog = new MDWSLog();
                    MDWS_LOG _mdwsLog = new MDWS_LOG();
                    RegionArray ret = null;

                    _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                    _mdwsLog.SERVICE_NAME = "getVHA";
                    _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { dataSourceProvider });

                    try
                    {
                        if (_tbiService == null)
                        {
                            _tbiService = new EmrSvc();
                            _tbiService.CookieContainer = new System.Net.CookieContainer();
                        }
                        ret = _tbiService.getVHA();
                        if (ret != null)
                        {
                            if (LogFullVIAResponse)
                                _mdwsLog.RETURNED_DATA = ret.Serialize();
                            else
                                _mdwsLog.RETURNED_DATA = (null == ret) ? "No Result" : "Valid Result";

                            if (ret.fault != null)
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                            }
                            else
                            {
                                bool sta3nFound = false;
                                for (int i = 0; i < ret.regions.Length; i++)
                                {
                                    RegionTO region = ret.regions[i];

                                    for (int j = 0; j < region.sites.count; j++)
                                    {
                                        SiteTO site = region.sites.sites[j];
                                        for (int k = 0; k < site.dataSources.count; k++)
                                        {
                                            DataSourceTO dataSource = site.dataSources.items[k];
                                            if (dataSource.provider.ToLower() == dataSourceProvider.ToLower())
                                            {
                                                sta3n = dataSource.siteId.tag;
                                                sta3nFound = true;
                                                break;
                                            }
                                        }

                                        if (sta3nFound)
                                        {
                                            break;
                                        }
                                    }

                                    if (sta3nFound)
                                    {
                                        break;
                                    }
                                }
                            }
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 2;
                        }
                    }
                    catch (Exception exc)
                    {
                        Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getStdInstitutionId()");
                        _mdwsLog.ERROR_LEVEL = 2;
                        throw;
                    }

                    mdwsLog.LogMDWSCall(_mdwsLog);
                }


            }
            TbiServiceTo.StationNumber = sta3n;
            InstitutionManager institutionManager = new InstitutionManager();
            result = institutionManager.getInstitutionIDByStationNumber(sta3n);

            return result;
        }

        /// <summary>
        /// MDWS: getNoteTitles service call
        /// </summary>
        /// <param name="target"></param>
        /// <param name="direction"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getNoteTitles(string target, string direction, out string error)
        {
            TaggedTextArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getNoteTitles";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { target, direction });

            try
            {
                ret = _tbiService.getNoteTitles(target, direction);
                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        error = null;
                        result = (from e in ret.results.FirstOrDefault().taggedResults orderby e.text select new KeyValuePair<string, string>(e.tag, e.textArray[0])).ToList();
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getNoteTitles()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: writeNote service call.
        /// </summary>
        /// <param name="tbiServiceTO"></param>
        /// <returns></returns>
        public NoteResultTO writeNote(TBIServiceTO tbiServiceTO)
        {
            NoteResultTO _writeNoteResult = null;
            TextTO _closeNoteResult;
            string titleIEN = "";
            string encounterString = "";
            string noteText = "";
            string authorDUZ = "";
            string cosignerDUZ = "";
            string consultIEN = "";
            string prfIEN = "";

            //set the values for the writeNote web service based on the noteType
            switch (tbiServiceTO.SelectedNoteType.Id)
            {
                case (int)NoteTypeEnum.ScheduledClinicAppointment:  //Scheduled Clinic Appointment

                    //if the note is for current visit or previous visit, call getVisits and have the user pick one . 
                    //Then encounterString = visit.location.id + ‘;’ + visit.timestamp + ‘;A’ 

                    //already formated from the original call to MDWS
                    encounterString = tbiServiceTO.SelectedVisitId;

                    break;
                case (int)NoteTypeEnum.HospitalAdmission:  //Hospital Admission

                    //if the note is for previous stay, call getAdmissions and have the user pick one. 
                    //Then encounterString = stay.location.id + ‘;’ + stay.admitTimestamp + ‘;H’ 

                    //already formated from the original call to MDWS
                    encounterString = tbiServiceTO.SelectedHospitalAdmissionId;


                    break;
                case (int)NoteTypeEnum.CurrentStay:  //Current Stay

                    //if the note is for current stay, encounterString = patient.locationId + ‘;’ + patient.admitTimestamp + ‘;H’ 

                    encounterString = tbiServiceTO.PatientLocationId + ";" + tbiServiceTO.PatientAdmitTimeStamp + ";H";

                    break;
                case (int)NoteTypeEnum.UnscheduledOrNewVisit: //Unscheduled or New Visit

                    if (!tbiServiceTO.isHistorical)
                    {
                        //if the note is for a new visit, but not hisorical, call getLocations and have the user pick one. 
                        //Then encounterString = location.id + ‘;’ + 20080515.1500 (today's date) + ‘;A’ 

                        encounterString = tbiServiceTO.SelectedLocationID + ";" + FormatDateForMDWS(DateTime.Now) + ";A";
                    }
                    else
                    {
                        //if the note is for a new visit, and is historical, 
                        //then previous step but end with ‘;E’ instead of ‘;A’. 

                        encounterString = tbiServiceTO.SelectedLocationID + ";" + FormatDateForMDWS(DateTime.Now) + ";E";

                    }

                    break;

            }
            Logging.WriteLogToFile(tbiServiceTO.SelectedNoteType.Id.ToString() + "::" + encounterString);

            //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
            noteText = tbiServiceTO.NoteText.Replace("<br>", "|").Replace(Environment.NewLine, "|").Replace("\v", "|");

            authorDUZ = tbiServiceTO.DUZ;

            if (!String.IsNullOrEmpty(tbiServiceTO.SelectedConsultId))
            {
                consultIEN = tbiServiceTO.SelectedConsultId;
            }

            if (tbiServiceTO.SelectedNoteTitleId != "" && tbiServiceTO.SelectedNoteTitleId != null)
            {
                titleIEN = tbiServiceTO.SelectedNoteTitleId;
            }

            if (!String.IsNullOrEmpty(tbiServiceTO.SelectedCosignerID))
            {
                cosignerDUZ = tbiServiceTO.SelectedCosignerID;
            }

            prfIEN = "";

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            bool savedFlag = false;

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "writeNote";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN });

            try
            {
                _writeNoteResult = _tbiService.writeNote(titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN);
                if (_writeNoteResult != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = _writeNoteResult.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == _writeNoteResult) ? "No Result" : "Valid Result";

                    if (!String.IsNullOrEmpty(_writeNoteResult.id))
                    {
                        mdwsLog.LogMDWSCall(_mdwsLog);
                        savedFlag = true;

                        CloseNote(_writeNoteResult.id, consultIEN);

                        DisconnectSite();

                    }

                    if (_writeNoteResult.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "writeNote()");
                Logging.WriteLogToFile(tbiServiceTO.Serialize());
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            if (!savedFlag)
            {
                mdwsLog.LogMDWSCall(_mdwsLog);

                Logging.WriteLogToFile(tbiServiceTO.Serialize());
            }
            return _writeNoteResult;
        }

        /// <summary>
        /// MDWS: CloseNote service call
        /// </summary>
        /// <param name="noteIEN"></param>
        /// <param name="consultIEN"></param>
        /// <returns></returns>
        private TextTO CloseNote(string noteIEN, string consultIEN)
        {
            TextTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "closeNote";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteIEN, consultIEN });

            try
            {
                result = _tbiService.closeNote(noteIEN, consultIEN);
                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "CloseNote()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: isCosignerRequired service call
        /// </summary>
        /// <param name="tbiServiceTO"></param>
        /// <param name="errMsg"></param>
        /// <returns></returns>
        public bool isCosignerRequired(TBIServiceTO tbiServiceTO, out string errMsg)
        {
            bool ret = false;
            TextTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            string noteDefinitionIEN = "";
            string authorDUZ = "";

            errMsg = "Service isCosignerRequired returned NULL value"; //Default error message

            noteDefinitionIEN = tbiServiceTO.SelectedNoteTitleId;
            authorDUZ = tbiServiceTO.DUZ;

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "isCosignerRequired";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteDefinitionIEN, authorDUZ });

            try
            {
                result = _tbiService.isCosignerRequired(noteDefinitionIEN, authorDUZ);
                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        errMsg = result.fault.message;
                    }
                    else
                    {
                        errMsg = String.Empty;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "isCosignerRequired()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            if (!String.IsNullOrEmpty(result.text))
            {
                switch (result.text.ToUpper())
                {
                    case "Y":
                        ret = true;
                        break;
                    case "N":
                        break;
                    default:
                        errMsg = "Service isCosignerRequired returned uexpected value " + result.text;
                        break;
                }
            }

            return ret;
            //return true; //Test data
        }

        #endregion

        #region VIA Service Methods

        // TODO: Add logging similar to MDWS Log

        #region VIA Client From DB

        public static TbiServiceInterfaceClient CreateVIAClientFromDB()
        {
            try
            {
                string viaServiceURL = Helpers.GetDBConfigEntry("VIA.ServiceURL");
                Uri uri = new Uri(viaServiceURL);
                bool secure = uri.Scheme.Equals("https", StringComparison.OrdinalIgnoreCase);
                System.ServiceModel.Channels.Binding binding = BuildBinding(secure);
                EndpointAddress address = new EndpointAddress(uri);
                return new TbiServiceInterfaceClient(binding, address);
            }
            catch (Exception x)
            {
                Logging.WriteLogToFile(x);
            }
            return null;
        }

        private static System.ServiceModel.Channels.Binding BuildBinding(bool secure)
        {
            //eventually need to get all of this from DB
            System.ServiceModel.BasicHttpBinding result = new System.ServiceModel.BasicHttpBinding(secure ? BasicHttpSecurityMode.Transport : BasicHttpSecurityMode.None);
            result.CloseTimeout = GetSettingFromDB("VIA.CloseTimeout", TimeSpan.Parse("00:01:00"));
            result.OpenTimeout = GetSettingFromDB("VIA.OpenTimeout", TimeSpan.Parse("00:01:00"));
            result.ReceiveTimeout = GetSettingFromDB("VIA.ReceiveTimeout", TimeSpan.Parse("00:10:00"));
            result.SendTimeout = GetSettingFromDB("VIA.SendTimeout", TimeSpan.Parse("00:01:00"));
            result.AllowCookies = GetSettingFromDB("VIA.AllowCookies", false);
            result.BypassProxyOnLocal = GetSettingFromDB("VIA.BypassProxyOnLocal", false);
            result.HostNameComparisonMode = GetSettingFromDB("VIA.HostNameComparisonMode", HostNameComparisonMode.StrongWildcard);
            result.MaxBufferSize = GetSettingFromDB("VIA.MaxBufferSize", 65536);
            result.MaxBufferPoolSize = GetSettingFromDB("VIA.MaxBufferPoolSize", 524288L);
            result.MaxReceivedMessageSize = GetSettingFromDB("VIA.MaxReceivedMessageSize", 65536L);
            result.MessageEncoding = GetSettingFromDB("VIA.MessageEncoding", WSMessageEncoding.Text);
            result.TextEncoding = GetSettingFromDB("VIA.TextEncoding", Encoding.UTF8);
            result.TransferMode = GetSettingFromDB("VIA.TransferMode", TransferMode.Buffered);
            result.UseDefaultWebProxy = GetSettingFromDB("VIA.UseDefaultWebProxy", true);
            result.ReaderQuotas.MaxDepth = GetSettingFromDB("VIA.ReaderQuotas.MaxDepth", 32);
            result.ReaderQuotas.MaxStringContentLength = GetSettingFromDB("VIA.ReaderQuotas.MaxStringContentLength", 8192);
            result.ReaderQuotas.MaxArrayLength = GetSettingFromDB("VIA.ReaderQuotas.MaxArrayLength", 16384);
            result.ReaderQuotas.MaxBytesPerRead = GetSettingFromDB("VIA.ReaderQuotas.MaxBytesPerRead", 4096);
            result.ReaderQuotas.MaxNameTableCharCount = GetSettingFromDB("VIA.ReaderQuotas.MaxNameTableCharCount", 16384);
            result.Security.Transport.ClientCredentialType = GetSettingFromDB("VIA.Security.Transport.ClientCredentialType", HttpClientCredentialType.None);
            result.Security.Transport.ProxyCredentialType = GetSettingFromDB("VIA.Security.Transport.ProxyCredentialType", HttpProxyCredentialType.None);
            result.Security.Transport.Realm = GetSettingFromDB("VIA.Security.Transport.Realm", "");
            result.Security.Message.ClientCredentialType = GetSettingFromDB("VIA.Security.Message.ClientCredentialType", BasicHttpMessageCredentialType.UserName);
            result.Security.Message.AlgorithmSuite = GetSettingFromDB("VIA.Security.Message.AlgorithmSuite", System.ServiceModel.Security.SecurityAlgorithmSuite.Default);
            //Only supports two modes: Transport for https and None for http
            result.Security.Mode = secure ? BasicHttpSecurityMode.Transport : BasicHttpSecurityMode.None;//use URL to determine
            return result;
        }

        private static T GetSettingFromDB<T>(string key, T defaultValue)
        {
            System.Diagnostics.Debug.Assert(null != defaultValue, "Null default values are not supported for configuration items.");
            try
            {
                string temp = Helpers.GetDBConfigEntry(key);
                Type generic = typeof(T);
                //If the generic type is string then null or white space is not an error
                if (generic == typeof(string))
                    return (T)(object)temp;
                //If the generic is not a string then null or white space should fail, return the provided default value
                if (string.IsNullOrEmpty(temp))
                    return defaultValue;
                if (generic.IsEnum)
                {
                    return (T)Enum.Parse(generic, temp);
                }
                if (generic == typeof(int))
                {
                    return (T)(object)int.Parse(temp);
                }
                if (generic == typeof(long))
                {
                    return (T)(object)long.Parse(temp);
                }
                if (generic == typeof(TimeSpan))
                {
                    return (T)(object)TimeSpan.Parse(temp);
                }
                if (generic == typeof(bool))
                {
                    return (T)(object)bool.Parse(temp);
                }
            }
            catch (Exception x)
            {
                Logging.WriteLogToFile(x);
            }
            return defaultValue;
        }
        #endregion VIA Client From DB

        /// <summary>
        /// VIA: LoadMSSR_VIAService - Load VIA Service Method
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        public bool LoadTBI_VIAService(TBIServiceTO tbiServiceTo)
        {
            try
            {
                _VIAService = CreateVIAClientFromDB();
                if (_VIAService == null)
                    _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
                if (HttpContext.Current.Session["ViaEndpoint"] == null)
                {
                    Logging.WriteLogToFile("LoadTBI_VIAService - Intrument Manager: Using VIA endpoint from config: " + _VIAService.Endpoint.Address.ToString());
                    HttpContext.Current.Session["ViaEndpoint"] = _VIAService.Endpoint.Address.ToString();
                }

                this.TbiServiceTo = tbiServiceTo;

                //if (!String.IsNullOrEmpty(tbiServiceTo.siteCode)) {
                //    _loginSiteCode = tbiServiceTo.siteCode;

                //    // If siteCode = 931, then we are using a test site.  VIA is setup (pre-prod) url to use 442 as the site code.
                //    if (tbiServiceTo.siteCode == "931") {
                //        tbiServiceTo.siteCode = "442";
                //        _loginSiteCode = "442";
                //    }
                //}

                string error = string.Empty;
                //DO NOT GET THIS//textTO textSiteId = getSiteIdVIA(tbiServiceTo.ServerIP, out error);

                if (error == "SUCCESS") { error = null; }

                if (!string.IsNullOrEmpty(error))
                {
                    return false;
                }
                else
                {
                    //_loginSiteCode = tbiServiceTo.siteCode = textSiteId.text;
                    if (!string.IsNullOrEmpty(StationDivision))
                        _loginSiteCode = tbiServiceTo.siteCode = StationDivision;

                    _personsTOVIA = CPRSLaunchVIA(tbiServiceTo);

                    if (_personsTOVIA == null)
                    {
                        this.TbiServiceTo = null;
                    }

                    if (this.TbiServiceTo != null)
                    {
                        if (_personsTOVIA != null)
                        {
                            this.TbiServiceTo.StationNumber = _loginSiteCode;
                            this.TbiServiceTo.PatientName = _personsTOVIA.patient.patientName;
                            this.TbiServiceTo.PatientDOB = _personsTOVIA.patient.dob;
                            this.TbiServiceTo.PatientGender = _personsTOVIA.patient.gender;
                            this.TbiServiceTo.PatientSnum = _personsTOVIA.patient.ssn;
                            this.TbiServiceTo.PatientICN = _personsTOVIA.patient.mpiPid;
                            this.TbiServiceTo.PatientDFN = _personsTOVIA.patient.localPid;
                            this.TbiServiceTo.PatientAdmitTimeStamp = _personsTOVIA.patient.admitTimestamp;
                        }

                        if (_personsTOVIA.user != null)
                        {
                            this.TbiServiceTo.STA6N = _personsTOVIA.user.siteId;
                            this.TbiServiceTo.UserName = _personsTOVIA.user.name;
                            _providerName = _personsTOVIA.user.name;
                            _tokenizedDUZ = _personsTOVIA.user.DUZ;

                            if (HttpContext.Current.Session["TokenizedDUZ"] == null)
                            {
                                HttpContext.Current.Session["TokenizedDUZ"] = _tokenizedDUZ;
                            }
                            if (HttpContext.Current.Session["ProviderName"] == null)
                            {
                                HttpContext.Current.Session["ProviderName"] = _providerName;
                            }
                        }

                        if (_personsTOVIA.patient.location != null)
                        {
                            this.TbiServiceTo.PatientLocationId = _personsTOVIA.patient.location.id;

                            if (_personsTOVIA.patient.location.facility != null)
                            {
                                this.TbiServiceTo.InstitutionName = _personsTOVIA.patient.location.facility.name;
                            }
                        }
                    }

                    if (this.TbiServiceTo == null)
                    {
                        return false;
                    }
                    else
                    {
                        return true;
                    }
                }
            }
            catch (Exception ex)
            {
                Helpers.HandleTBIException(ex, "InstrumentManager.cs", "", "LoadTBI_VIAService()");

                return false;
            }
        }

        public bool TestVIALogin(TBIServiceTO tbiServiceTo, out string error)
        {
            try
            {
                if (_VIAService == null)
                {
                    _VIAService = CreateVIAClientFromDB();
                    if (_VIAService == null)
                        _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
                }
                //DO NOT OVERIDE siteCode
                _loginSiteCode = tbiServiceTo.siteCode;
                var temp = CPRSLaunchVIAWithError(tbiServiceTo);
                TestResultFault(temp);
                error = null;
                return true;
            }
            catch (Exception x)
            {
                Helpers.HandleTBIException(x, "InstrumentManager.cs", "", "TestViaLogin");
                error = x.Message;
                return false;
            }
        }

        private void TestResultFault(abstractTO to)
        {
            if (null == to)
            {
                throw new NullReferenceException("No result returned from call to VIA service.");
            }
            if (to.fault != null && !string.IsNullOrEmpty(to.fault.message))
            {
                throw new Exception(to.fault.message);
            }
        }

        /// <summary>
        /// This method was created to allow the exception to bubble up instead of being discarded.
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        private personsTO CPRSLaunchVIAWithError(TBIServiceTO tbiServiceTo)
        {
            Exception x = null;

            personsTO result = null;
            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsLaunchVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", tbiServiceTo.siteCode, tbiServiceTo.DUZ, tbiServiceTo.DFN });
            //*** DO NOT OVERRIDE THE _loginSiteCode IN THIS METHOD
            try
            {
                provider _providerbean = new provider();
                patient _patientbean = new patient();
                queryBean _bean = new queryBean();

                _userID = tbiServiceTo.DUZ;
                _localPid = tbiServiceTo.DFN;
                HttpContext.Current.Session["ProviderDFN"] = _localPid;

                _bean.criteria = "TBI_PWD";
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                //patient attributes
                _patientbean.localSiteId = _loginSiteCode;
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _bean.patient = _patientbean;
                //provider attributes
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = _userID;
                _bean.provider = _providerbean;

                result = _VIAService.cprsLaunch(_bean);

                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }

            }
            catch (System.ServiceModel.Security.SecurityNegotiationException sne)
            {
                x = sne;
                _mdwsLog.ERROR_LEVEL = 2;
            }
            catch (Exception ex)
            {
                x = ex;
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);
            if (null != x)
            {
                Helpers.HandleTBIException(x, "InstrumentManager.cs", "", "CPRSLaunchVIAWithError()");
                throw x;
            }
            return result;
        }


        /// <summary>
        /// VIA: CPRSLaunchVIA service method
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        private personsTO CPRSLaunchVIA(TBIServiceTO tbiServiceTo)
        {
            try
            {
                personsTO result = CPRSLaunchVIAWithError(tbiServiceTo);
                return result;
            }
            catch (Exception x)
            {
                Helpers.HandleTBIException(x, "InstrumentManager.cs", "", "CPRSLaunchVIA()");
            }
            return null;
        }

        /// <summary>
        /// VIA: getNoteTitlesFromVIA service method
        /// </summary>
        /// <param name="target"></param>
        /// <param name="direction"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getNoteTitlesFromVIA(string target, string direction, out string error)
        {
            taggedTextArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            string searchParameter = String.Empty;

            if (target.Trim() == String.Empty)
            {
                searchParameter = "T";//defaults to tbi note get
            }
            else
            {
                searchParameter = target.Trim().ToUpper();
            }

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG(); 
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getNoteTitlesVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { target, direction });

            try
            {
                if (_VIAService != null)
                {
                    queryBean _bean = new queryBean();
                    patient _patientbean = new patient();
                    provider _providerbean = new provider();

                    if (!string.IsNullOrEmpty(StationDivision))
                        this.TbiServiceTo.StationNumber = StationDivision;
                    _loginSiteCode = this.TbiServiceTo.StationNumber;

                    _bean.target = searchParameter;
                    _bean.direction = "1";
                    _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                    _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                    _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                    _patientbean.localSiteId = _loginSiteCode;
                    _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                    _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                    _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                    _providerbean.loginSiteCode = _loginSiteCode;
                    _bean.patient = _patientbean;
                    _bean.provider = _providerbean;

                    ret = _VIAService.getNoteTitles(_bean);

                    if (ret != null)
                    {
                        if (LogFullVIAResponse)
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        else
                            _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                        if (ret.fault != null)
                        {
                            error = ret.fault.message;
                            result = null;

                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                        else
                        {
                            if (ret.count > 0)
                            {
                                _mdwsLog.ERROR_LEVEL = 0;
                                error = "SUCCESS";
                                result = (from e in ret.results.FirstOrDefault().taggedResults orderby e.text select new KeyValuePair<string, string>(e.tag, e.textArray[0])).ToList();
                            }
                            else
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                                error = null;
                            }
                        }
                    }
                    else
                    {
                        error = "VIA Service call get Note Titles returned null";
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                else
                {
                    error = null;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getNoteTitlesFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;

        }

        /// <summary>
        /// VIA: getConsultsFromVIA service method
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getConsultsFromVIA(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedConsultArrays ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getConsultsForPatientVIA";
            _mdwsLog.PARAMETERS = string.Empty;

            try
            {
                if (_VIAService != null)
                {
                    queryBean _bean = new queryBean();
                    patient _patientbean = new patient();
                    provider _providerbean = new provider();

                    if (!string.IsNullOrEmpty(StationDivision))
                        this.TbiServiceTo.StationNumber = StationDivision;
                    _loginSiteCode = this.TbiServiceTo.StationNumber;

                    _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                    _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                    _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                    _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                    _patientbean.localSiteId = _loginSiteCode;
                    _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                    _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                    _providerbean.loginSiteCode = _loginSiteCode;
                    _bean.patient = _patientbean;
                    _bean.provider = _providerbean;

                    ret = _VIAService.getConsultsForPatient(_bean);

                    if (ret != null)
                    {
                        if (LogFullVIAResponse)
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        else
                            _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                        if (ret.fault != null)
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = ret.fault.message;
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            // VIA modified the getConsultsForPatient method to return an empty consult around 7/2016. This broke our code because
                            // we check the consults array for null, and assume no consults if this array was null. After their first update, the empty
                            // consult caused our LINQ code to fail because there was no timestamp to format since the empty consult did not have a 
                            // timestamp. An apparently recent VIA update that we are seeing in PPD now (7/2017) has the consults array as non-null, however 
                            // it is now an empty array, thus the reference to ret.arrays[0].consults[0] throws an exception. I reverted the main
                            // If statement back to the original, then added the nested If statement to account for both of their updates since there's
                            // no way we can be sure which one we'll be up against.

                            //if (!(ret.count == 0 || ret.arrays[0].consults == null || ret.arrays[0].consults[0].timestamp == null))
                            if (!(ret.count == 0 || ret.arrays[0].consults == null))
                            {
                                if (ret.arrays[0].consults.Count() > 0 && ret.arrays[0].consults[0].timestamp != null)
                                {
                                    var consultsArray = ret.arrays.SelectMany(s => s.consults).ToList();
                                    result = (from e in consultsArray orderby e.timestamp descending select new KeyValuePair<string, string>(e.id, DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + " (" + e.status + ") " + e.title + " " + "Consult #: " + e.id)).ToList();
                                }
                                else
                                {
                                    _mdwsLog.ERROR_LEVEL = 1;
                                    error = null;
                                }
                            }
                            else
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                                error = null;
                            }
                        }
                    }
                    else
                    {
                        error = "VIA Service call get Consults returned null";
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                else
                {
                    error = null;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getConsultsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: getHospitalAdmissionsFromVIA service call
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getHospitalAdmissionsFromVIA(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedInpatientStayArray ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getAdmissionsVIA";
            _mdwsLog.PARAMETERS = string.Empty;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    this.TbiServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");

                ret = _VIAService.getAdmissions(_bean);

                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.stays == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.stays orderby e.admitTimestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + FormatDateForVIA2(DateTime.ParseExact(e.admitTimestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture)) + ";H", e.location.name + " (" + DateTime.ParseExact(e.admitTimestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + ")")).ToList();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Hospital Admissions returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getHospitalAdmissionsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: getVisitsFromVIA service call
        /// </summary>
        /// <param name="fromDate"></param>
        /// <param name="toDate"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getVisitsFromVIA(DateTime fromDate, DateTime toDate, out string error)
        {
            taggedVisitArray ret = null;
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();

            string formattedFromDate = FormatDateForVIA(fromDate);
            string formattedToDate = FormatDateForVIA(toDate);

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getVisitsVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { formattedFromDate, formattedToDate });

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    this.TbiServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _bean.recordSiteCode = _loginSiteCode;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.startDate = formattedFromDate;
                _bean.endDate = formattedToDate;
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();

                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                ret = _VIAService.getVisits(_bean);

                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        if (ret.count > 0)
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.visits orderby e.timestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + FormatDateForVIA2(DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture)) + ";A", DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("g") + " " + e.location.name + " " + e.status)).ToList();
                        }
                        else
                        {
                            error = null;
                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Visits returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getVisitsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// This will get the list of CPRS users for the correct VIA connection
        /// </summary>
        /// <param name="target">Beginning of first user name in the list</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> cprsUserLookupFromVIA(string target, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            userArray ret = null;

            if (string.IsNullOrEmpty(target)) { target = "*"; }

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsUserLookupVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { target });

            try
            {
                queryBean _bean = new queryBean();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    TbiServiceTo.StationNumber = StationDivision;
                _loginSiteCode = TbiServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.target = target;
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.provider = _providerbean;

                ret = _VIAService.cprsUserLookup(_bean);

                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.users == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.users orderby e.name select new KeyValuePair<string, string>(e.DUZ, e.name)).ToList();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call CPRS User Lookup returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "cprsUserLookupFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// This will get the locations for the correct VIA connection
        /// </summary>
        /// <param name="filter">Use this to filter the results returned from MDWS</param>
        /// <param name="target">Purpose and expected values Unknown </param>
        /// <param name="direction">MDWS expects a value of 1 or -1</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getLocationsFromVIA(string filter, string target, string direction, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedHospitalLocationArray ret = null;
            string searchParameter = String.Empty;

            if (filter.Trim() == String.Empty)
            {
                searchParameter = target;
            }
            else
            {
                searchParameter = filter.Trim().ToUpper();
            }

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getLocationsVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { searchParameter, direction });

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    TbiServiceTo.StationNumber = StationDivision;
                _loginSiteCode = TbiServiceTo.StationNumber;
                _bean.target = searchParameter;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.direction = direction;
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                ret = _VIAService.getLocations(_bean);

                if (ret != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.locations == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.locations where e.name.ToLower().StartsWith(filter.ToLower()) orderby e.name select new KeyValuePair<string, string>(e.id, e.name)).ToList();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Locations returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "getLocationsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: writeNoteWithVIA write note method
        /// </summary>
        /// <param name="tbiServiceTO"></param>
        /// <returns></returns>
        public noteResultTO writeNoteWithVIA(TBIServiceTO tbiServiceTO)
        {
            _VIAService = CreateVIAClientFromDB();
            if (_VIAService == null)
                _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");

            noteResultTO _writeNoteResult = new noteResultTO();
            //textTO _closeNoteResult = null;
            string titleIEN = "";
            string encounterString = "";
            string noteText = "";
            string authorDUZ = "";
            string cosignerDUZ = "";
            string consultIEN = "";
            string prfIEN = "";

            //set the values for the writeNote web service based on the noteType
            switch (tbiServiceTO.SelectedNoteType.Id)
            {
                case (int)NoteTypeEnum.ScheduledClinicAppointment:  //Scheduled Clinic Appointment
                    //if the note is for current visit or previous visit, call getVisits and have the user pick one . 
                    //Then encounterString = visit.location.id + ‘;’ + visit.timestamp + ‘;A’ 

                    //already formated when list box data items are created
                    encounterString = tbiServiceTO.SelectedVisitId;
                    break;
                case (int)NoteTypeEnum.HospitalAdmission:  //Hospital Admission
                    //if the note is for previous stay, call getAdmissions and have the user pick one. 
                    //Then encounterString = stay.location.id + ‘;’ + stay.admitTimestamp + ‘;H’ 

                    //already formated when list box data items are created
                    encounterString = tbiServiceTO.SelectedHospitalAdmissionId;
                    break;
                case (int)NoteTypeEnum.CurrentStay:  //Current Stay
                    //if the note is for current stay, encounterString = patient.locationId + ‘;’ + patient.admitTimestamp + ‘;H’ -- needs conversion to Fileman date
                    encounterString = tbiServiceTO.PatientLocationId + ";" + FormatDateForVIA2(DateTime.ParseExact(tbiServiceTO.PatientAdmitTimeStamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture)) + ";H";
                    break;
                case (int)NoteTypeEnum.UnscheduledOrNewVisit: //Unscheduled or New Visit

                    if (!tbiServiceTO.isHistorical)
                    {
                        //if the note is for a new visit, but not hisorical, call getLocations and have the user pick one. 
                        //Then encounterString = location.id + ‘;’ + 20080515.1500 (today's date) + ‘;A’ 
                        encounterString = tbiServiceTO.SelectedLocationID + ";" + FormatDateForVIA2(DateTime.Now) + ";A";
                    }
                    else
                    {
                        //if the note is for a new visit, and is historical, 
                        //then previous step but end with ‘;E’ instead of ‘;A’. 
                        encounterString = tbiServiceTO.SelectedLocationID + ";" + FormatDateForVIA2(DateTime.Now) + ";E";
                    }
                    break;
            }
            Logging.WriteLogToFile(tbiServiceTO.SelectedNoteType.Id.ToString() + "::" + encounterString);

            //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
            noteText = tbiServiceTO.NoteText.Replace("<br>", "|").Replace(Environment.NewLine, "|").Replace("\v", "|");

            authorDUZ = tbiServiceTO.DUZ;

            if (!String.IsNullOrEmpty(tbiServiceTO.SelectedConsultId))
            {
                consultIEN = tbiServiceTO.SelectedConsultId;
            }

            if (tbiServiceTO.SelectedNoteTitleId != "" && tbiServiceTO.SelectedNoteTitleId != null)
            {
                titleIEN = tbiServiceTO.SelectedNoteTitleId;
            }

            if (!String.IsNullOrEmpty(tbiServiceTO.SelectedCosignerID))
            {
                cosignerDUZ = tbiServiceTO.SelectedCosignerID;
            }

            prfIEN = "";

            bool savedFlag = false;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "writeNoteVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN });

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    tbiServiceTO.StationNumber = StationDivision;
                _loginSiteCode = tbiServiceTO.StationNumber;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                //_writenote1.queryBean.patient.localSiteId = _loginSiteCode;
                //_writenote1.queryBean.patient.vistaLocations = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                //_writenote1.queryBean.provider.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();

                //_bean = _writenote1.queryBean;
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                _writeNoteResult = _VIAService.writeNote(titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN, _bean);

                if (_writeNoteResult != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = _writeNoteResult.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == _writeNoteResult) ? "No Result" : "Valid Result";

                    if (!String.IsNullOrEmpty(_writeNoteResult.id))
                    {

                        mdwsLog.LogMDWSCall(_mdwsLog); // log via entry
                        savedFlag = true;

                        CloseNoteWithVIA(titleIEN, consultIEN);
                    }

                    if (_writeNoteResult.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "writeNoteWithVIA()");
                Logging.WriteLogToFile(tbiServiceTO.Serialize());
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            if (!savedFlag)
            {
                mdwsLog.LogMDWSCall(_mdwsLog);
                Logging.WriteLogToFile(tbiServiceTO.Serialize());
            }

            return _writeNoteResult;
        }

        /// <summary>
        /// VIA: CloseNote service method call
        /// </summary>
        /// <param name="noteIEN"></param>
        /// <param name="consultIEN"></param>
        /// <returns></returns>
        private textTO CloseNoteWithVIA(string noteIEN, string consultIEN)
        {
            textTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "closeNoteVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteIEN, consultIEN });

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();
                if (!string.IsNullOrEmpty(StationDivision))
                    this.TbiServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.recordSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.provider = _providerbean;

                result = _VIAService.closeNote(noteIEN, consultIEN, _bean);

                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 1;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "CloseNoteWithVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: isCosignerRequired service method call
        /// </summary>
        /// <param name="tbiServiceTO"></param>
        /// <param name="errMsg"></param>
        /// <returns></returns>
        public bool isCosignerRequiredVIA(TBIServiceTO tbiServiceTO, out string errMsg)
        {
            bool ret = false;
            textTO result = null;
            string noteDefinitionIEN = "";
            string authorDUZ = "";

            errMsg = "Service isCosignerRequiredVIA returned NULL value"; //Default error message

            noteDefinitionIEN = tbiServiceTO.SelectedNoteTitleId;
            authorDUZ = tbiServiceTO.DUZ;
            if (!string.IsNullOrEmpty(StationDivision))
                tbiServiceTO.StationNumber = StationDivision;
            _loginSiteCode = tbiServiceTO.StationNumber;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "isCosignerRequiredVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteDefinitionIEN, authorDUZ });

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _bean.itemId = noteDefinitionIEN;
                _bean.recordSiteCode = _loginSiteCode;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.criteria = authorDUZ;
                _patientbean.mpiPid = tbiServiceTO.PatientICN;

                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                result = _VIAService.isCosignerRequired(_bean);

                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        errMsg = result.fault.message;
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                        errMsg = String.Empty;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "isCosignerRequiredVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            if (!String.IsNullOrEmpty(result.text))
            {
                switch (result.text.ToUpper())
                {
                    case "Y":
                        ret = true;
                        break;
                    case "N":
                        break;
                    default:
                        errMsg = "VIA Service isCosignerRequiredVIA returned uexpected value " + result.text;
                        break;
                }
            }

            return ret;
        }

        public string LookupSiteCodeByServer(string vistaServer, out string error)
        {
            if (_VIAService == null)
            {
                _VIAService = CreateVIAClientFromDB();
                if (_VIAService == null)
                    _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
            }
            textTO temp = getSiteIdVIA(vistaServer, out error);
            if (null != temp)
                return temp.text;
            return null;
        }

        public textTO getSiteIdVIA(string vistaServer, out string error)
        {
            textTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();
            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getSiteIdVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { vistaServer });

            try
            {
                result = _VIAService.getSiteId(vistaServer);

                if (result != null)
                {
                    if (LogFullVIAResponse)
                        _mdwsLog.RETURNED_DATA = result.Serialize();
                    else
                        _mdwsLog.RETURNED_DATA = (null == result) ? "No Result" : "Valid Result";

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = result.fault.message;
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                        error = "SUCCESS";
                    }
                }
                else
                {
                    error = "VIA Service call getSiteId returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception ex)
            {
                Helpers.HandleTBIException(ex, "InstrumentManager.cs", "", "getSiteIdVIA()");
                error = ex.Message;
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: FormatDateForVIA date formatting
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForVIA(DateTime dateValue)
        {
            // CPRS is expecting date in the following format "20000101.000000";
            //  [yyyy -1700]MMdd.HHmmss.  For example, 3130715.163242 represents the date and time of JUL 15, 2013 at 16:32:42
            //return String.Format("{0:yyyyMMdd}", dateValue);

            if (dateValue != null)
            {
                //return (dateValue.Year - 1700) + "" + dateValue.ToString("MM") + dateValue.ToString("dd") + "." + dateValue.ToString("HH") + dateValue.ToString("mm") + dateValue.ToString("ss");
                return (dateValue.Year) + "" + dateValue.ToString("MM") + dateValue.ToString("dd");
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// VIA: FormatDateForVIA date formatting
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForVIA2(DateTime dateValue)
        {
            // CPRS is expecting date in the following format "20000101.000000";
            //  [yyyy -1700]MMdd.HHmmss.  For example, 3130715.163242 represents the date and time of JUL 15, 2013 at 16:32:42
            //return String.Format("{0:yyyyMMdd}", dateValue);

            if (dateValue != null)
            {
                return (dateValue.Year - 1700) + "" + dateValue.ToString("MM") + dateValue.ToString("dd") + "." + dateValue.ToString("HH") + dateValue.ToString("mm") + dateValue.ToString("ss");
            }
            else
            {
                return null;
            }
        }

        #endregion

        #region LINQ / Utility Methods

        /// <summary>
        /// Method used to retrieve all "active" instruments from the database.  Replaces legacy systems usage of
        /// TbiServiceTo.Instruments (static) list.
        /// </summary>
        /// <returns></returns>
        public List<InstrumentType> GetActiveInstrumentList()
        {
            using (_db = GetDataContext())
            {
                // default to TBI if nothing passed.
                int tbiRegistryId = _db.STD_REGISTRies.FirstOrDefault(R => R.CODE == "TBI").ID;
                return convertStdSurveyTypeListToInstrumentType(_db.STD_SURVEY_TYPEs.Where(S => S.REGISTRY_ID == tbiRegistryId && S.IS_ACTIVE == true).OrderBy(S => S.NAME).ToList());
            }
        }

        /// <summary>
        /// Method used to retrieve all instruments from the database.  Replaces legacy systems usage of
        /// TbiServiceTo.Instruments (static) list.
        /// </summary>
        /// <returns></returns>
        public List<InstrumentType> GetAllInstrumentList()
        {
            using (_db = GetDataContext())
            {
                int tbiRegistryId = _db.STD_REGISTRies.FirstOrDefault(R => R.CODE == "TBI").ID;
                return convertStdSurveyTypeListToInstrumentType(_db.STD_SURVEY_TYPEs.Where(S => S.REGISTRY_ID == tbiRegistryId).OrderBy(S => S.NAME).ToList());
            }
        }

        /// <summary>
        /// Set Instrument Active Status by ID
        /// </summary>
        /// <param name="surveyTypeId"></param>
        /// <param name="isActive"></param>
        public void SetInstrumentActiviationStatus(int surveyTypeId, bool isActive)
        {
            using (_db = GetDataContext())
            {
                STD_SURVEY_TYPE surveyType = _db.STD_SURVEY_TYPEs.FirstOrDefault(S => S.ID == surveyTypeId);
                if (surveyType != null)
                {
                    surveyType.IS_ACTIVE = isActive;

                    _db.SubmitChanges();
                }
            }
        }

        /// <summary>
        /// Returns the Instrument Status - is active.
        /// </summary>
        /// <param name="surveyTypeId"></param>
        /// <param name="isActive"></param>
        public bool IsInstrumentActive(int surveyTypeId)
        {
            using (_db = GetDataContext())
            {
                STD_SURVEY_TYPE surveyType = _db.STD_SURVEY_TYPEs.FirstOrDefault(S => S.ID == surveyTypeId);
                if (surveyType != null)
                {
                    return surveyType.IS_ACTIVE;
                }
            }

            return false;
        }

        /// <summary>
        /// SetLoadWith method - private
        /// </summary>
        /// <param name="db"></param>
        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY>(e => e.SURVEY_RESULTs);
            lo.LoadWith<SURVEY_RESULT>(e => e.STD_QUESTION_CHOICE);
            lo.LoadWith<STD_QUESTION_CHOICE>(e => e.STD_QUESTION);
            lo.LoadWith<SURVEY>(e => e.STD_SURVEY_TYPE);
            lo.LoadWith<SURVEY>(e => e.SURVEY_NOTEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// SetSurveyLoadWith private method
        /// </summary>
        /// <param name="db"></param>
        private void SetSurveyLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY>(e => e.SURVEY_RESULTs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// SetQuestionnaireLoadWith private method
        /// </summary>
        /// <param name="db"></param>
        private void SetQuestionnaireLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY_RESULT>(e => e.SURVEY);
            lo.LoadWith<SURVEY_RESULT>(e => e.STD_QUESTION_CHOICE);
            lo.LoadWith<STD_QUESTION>(e => e.STD_QUESTION_CHOICEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        /// <summary>
        /// SetQuestionsLoadWith private method
        /// </summary>
        /// <param name="db"></param>
        private void SetQuestionsLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<STD_QUESTION>(e => e.STD_QUESTION_CHOICEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        /// <summary>
        /// LinqAll
        /// </summary>
        /// <returns></returns>
        private IQueryable<SURVEY> LinqAll()
        {
            //populate LinqAll
            IQueryable<SURVEY> tempLinq = (from e in _db.SURVEYs
                                           select e);

            //TODO - add all business filtering rules 

            return tempLinq;
        }


        public DataTable PreviousM2PINoteTypes(string ICN)
        {
            //call to server to get function results.
            using (_db = GetDataContext())
            {
            IQueryable<ufn_findM2PINoteTypesResult> rtn = _db.ufn_findM2PINoteTypes(ICN);
            DataTable PrevM2PINotes = new DataTable();
            PrevM2PINotes.Columns.Add("Surveys_ID", typeof(string));
            PrevM2PINotes.Columns.Add("prevM2PINoteInfo", typeof(string));
                foreach (var col in rtn)
                {
                    string _data = col.SurveyDate.ToString("d") + ":     "
                            + col.InterimCount.ToString() + " Interim Notes, "
                            + col.DischargeCount.ToString() + " Discharge Note, "
                            + col.FollowUpCount.ToString() + " Follow Up Notes.";
                    string _data2 = col.SurveysID.ToString() + ","
                            + col.InterimCount.ToString() + ","
                            + col.DischargeCount.ToString() + ","
                            + col.FollowUpCount.ToString();
                    PrevM2PINotes.Rows.Add(_data2, _data);
                    _data = "";
                } 

            return PrevM2PINotes;                 
            }
        }
        /// <summary>
        /// SelectByPatientICNFilter
        /// </summary>
        /// <param name="ICN"></param>
        /// <returns></returns>
        private IQueryable<SURVEY> SelectByPatientICNFilter(string ICN)
        {
            IQueryable<SURVEY> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.patient_ICN == ICN select t;

            return linqFilter;
        }

        /// <summary>
        /// Get a survey by a surveyID.
        /// Return a survey object with all the survey results
        /// </summary>
        /// <param name="surveyId"></param>
        /// <returns></returns>
        public SURVEY GetSurveyWithAllAnswers(int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetSurveyLoadWith(_db);

                var survey = (from t in _db.SURVEYs
                              where t.SURVEYS_ID == surveyId
                              select t).FirstOrDefault();
                return survey;
            }
        }

        /// <summary>
        /// GetChoicesByQuestionNumber - returns a list of survey results for question number
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyId"></param>
        /// <returns></returns>
        public List<SURVEY_RESULT> GetChoicesByQuestionNumber(string questionNumber, int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var choices = (from t in _db.SURVEY_RESULTs
                               where t.SURVEYS_ID == surveyId
                               where t.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER == questionNumber
                               select t).ToList();

                return choices;
            }

        }

        public List<SURVEY_RESULT> GetChoicesBySurveyID(int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var choices = (from t in _db.SURVEY_RESULTs
                               where t.SURVEYS_ID == surveyId
                               where t.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER != null
                               select t).ToList();

                return choices;
            }

        }

        /// <summary>
        /// GetSurveyTypeID
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyTypeId"></param>
        /// <returns></returns>
        public string GetSurveyTypeID(string SurveyName)
        {
            using (_db = GetDataContext())
            {
                string ret = "";

                SetQuestionnaireLoadWith(_db);
                var SurveyTypeID = (from t in _db.STD_SURVEY_TYPEs
                                    where t.CODE == SurveyName
                                    select t).ToList();
               // var question = (from t in _db.STD_QUESTIONs
               //                 where t.STD_SURVEY_TYPE_ID == surveyTypeId
               //                 where t.QUESTION_NUMBER == questionNumber
               //                 select t).ToList();

                if (SurveyTypeID.Count() > 0)
                {
                    ret = SurveyTypeID.First().ID.ToString();
                }

                return ret;
            }

        }

        /// <summary>
        /// GetQuestionbyNumberAndSurveyType
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyTypeId"></param>
        /// <returns></returns>
        public string GetQuestionbyNumberAndSurveyType(string questionNumber, int surveyTypeId)
        {
            using (_db = GetDataContext())
            {
                string ret = "";

                SetQuestionnaireLoadWith(_db);

                var question = (from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE_ID == surveyTypeId
                                where t.QUESTION_NUMBER == questionNumber
                                select t).ToList();

                if (question.Count() > 0)
                {
                    ret = question.First().QUESTION_TEXT;
                }

                return ret;
            }

        }

        /// <summary>
        /// IsInitialNoteType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="surveyType"></param>
        /// <returns></returns>
        public bool IsInitialNoteType(string patientICN, int surveyType)
        {
            bool ret = false;

            //Get note type of latest completed survey
            string lastNoteType = GetLastCompletedNoteTypeByPatientICNAndSurveyType(patientICN, surveyType);

            if (String.IsNullOrEmpty(lastNoteType) ||
                String.Equals(lastNoteType, "Initial", StringComparison.OrdinalIgnoreCase))
            {
                ret = true;
            }

            return ret;
        }

        /// <summary>
        /// GetMapiNoteTypeChoices
        /// </summary>
        /// <returns></returns>
        public List<STD_QUESTION_CHOICE> GetMapiNoteTypeChoices()
        {
            return GetChoicesByQuestionNumberandSurveyType("-1", "MPAI");
        }

        /// <summary>
        /// GetMapiPersonReportingChoices
        /// </summary>
        /// <returns></returns>
        public List<STD_QUESTION_CHOICE> GetMapiPersonReportingChoices()
        {
            return GetChoicesByQuestionNumberandSurveyType("0", "MPAI");
        }

        /// <summary>
        /// Returns the Instrument with the passed ID - updated for new database approach.
        /// </summary>
        /// <param name="instrumentID"></param>
        /// <returns></returns>
        public InstrumentType GetInstrumentByID(int instrumentID)
        {
            return this.GetAllInstrumentList().FirstOrDefault(I => I.Id == instrumentID);
        }

        /// <summary>
        /// GetChoicesByQuestionNumberandSurveyType
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyTypeCode"></param>
        /// <returns></returns>
        public List<STD_QUESTION_CHOICE> GetChoicesByQuestionNumberandSurveyType(string questionNumber, string surveyTypeCode)
        {
            using (_db = GetDataContext())
            {
                
                    SetQuestionsLoadWith(_db);

                    var question = (from t in _db.STD_QUESTIONs
                                    where t.STD_SURVEY_TYPE.CODE == surveyTypeCode
                                    where t.QUESTION_NUMBER == questionNumber
                                    where t.INACTIVE_FLAG == false
                                    select t).ToList();

                    var choices = (from t in _db.STD_QUESTION_CHOICEs
                                   where t.STD_QUESTION_ID == question.FirstOrDefault().ID
                                   where t.INACTIVE_FLAG == false
                                   select t).OrderBy("CHOICE_SORT_ORDER").ToList();

                    return choices;
            }
        }

        /// <summary>
        /// SelectByPatientICNAndSurveyType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <param name="followUp"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectByPatientICNAndSurveyType(string patientICN, int typeId, bool followUp)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.STD_SURVEY_TYPE_ID == typeId
                          select t;

                if (followUp)

                    surveys = from s in surveys where s.SURVEY_STATUS == "inprocess" select s;

                return surveys.ToList();
            }

        }

        /// <summary>
        /// SelectLatestByPatientICNAndSurveyType: Returns the latest (most recent) survey for the given patient and survey type.  If null, no survey was found.
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public SURVEY SelectLatestByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientICN).Where(T => T.STD_SURVEY_TYPE_ID == typeId).OrderByDescending(T => T.SURVEY_DATE).FirstOrDefault();

                return survey;

                //SURVEY survey;

                //survey = (from t in SelectByPatientICNFilter(patientICN)
                //          where t.STD_SURVEY_TYPE_ID == typeId
                //          select t).Take(1).to;

                //return survey;
            }
        }

        /// <summary>
        /// SelectLatestCompletedByPatientICNAndSurveyType: Returns the latest (most recent) complete survey for the given patient and survey type.  If null, no survey was found.
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public SURVEY SelectLatestCompletedByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientICN).Where(T => T.STD_SURVEY_TYPE_ID == typeId && T.SURVEY_STATUS.ToLower() == "completed").OrderByDescending(T => T.SURVEY_DATE).FirstOrDefault();

                return survey;
            }
        }

        /// <summary>
        /// SelectLastCompletedSurveyByPatientICNAndSurveyType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectLastCompletedSurveyByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = SelectByPatientICNAndSurveyType(patientICN, typeId, false);
                surveys = (from s in surveys
                           where
                               String.Equals(s.SURVEY_STATUS, "completed", StringComparison.OrdinalIgnoreCase)
                           orderby s.UPDATED descending
                           select s).Take(1).ToList();


                return surveys.ToList();
            }
        }

        /// <summary>
        /// SelectByStatus
        /// </summary>
        /// <param name="statusId"></param>
        /// <param name="patientICN"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectByStatus(string statusId, string patientICN)
        {

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.SURVEY_STATUS == statusId
                          select t;


                return surveys;
            }
        }

        /// <summary>
        /// GetLastCompletedNoteTypeByPatientICNAndSurveyType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public string GetLastCompletedNoteTypeByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            string noteType = String.Empty;

            IEnumerable<SURVEY> survey;

            survey = SelectLastCompletedSurveyByPatientICNAndSurveyType(patientICN, typeId);

            if (survey.Count() == 1)
            {
                int surveyID = survey.First().SURVEYS_ID;

                try
                {
                    List<SURVEY_RESULT> lastCompletedSurveyResults = GetChoicesByQuestionNumber("-1", surveyID);

                    if (lastCompletedSurveyResults.Count() > 0)
                    {
                        noteType = lastCompletedSurveyResults.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    }

                    //List<SURVEY_RESULT> lastCompletedSurveyResultsNote1 = GetChoicesByQuestionNumber("Initial", surveyID);
                    //if (lastCompletedSurveyResultsNote1.Count() > 0)
                    //{
                    //    noteType = lastCompletedSurveyResultsNote1.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    //}
                    //List<SURVEY_RESULT> lastCompletedSurveyResultsNote2 = GetChoicesByQuestionNumber("Interim", surveyID);
                    //if (lastCompletedSurveyResultsNote2.Count() > 0)
                    //{
                    //    noteType = lastCompletedSurveyResultsNote2.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    //}
                    //List<SURVEY_RESULT> lastCompletedSurveyResultsNote3 = GetChoicesByQuestionNumber("Discharge Note", surveyID);
                    //if (lastCompletedSurveyResultsNote3.Count() > 0)
                    //{
                    //    noteType = lastCompletedSurveyResultsNote3.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    //}
                    
                    //List<SURVEY_RESULT> lastCompletedSurveyResultsNote4 = GetChoicesByQuestionNumber("Follow Up Note", surveyID);
                    //if (lastCompletedSurveyResultsNote4.Count() > 0)
                    //{
                    //    noteType = lastCompletedSurveyResultsNote4.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    //}
                }
                catch (Exception exc)
                {
                    Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "GetLastCompletedNoteTypeByPatientICNAndSurveyType()");
                    //Ignore error
                }
            }

            return noteType;
        }

        /// <summary>
        /// SelectByStatusCount
        /// </summary>
        /// <param name="statusId"></param>
        /// <param name="patientICN"></param>
        /// <returns></returns>
        public int SelectByStatusCount(string statusId, string patientICN)
        {
            using (_db = GetDataContext())
            {
                return SelectByPatientICNFilter(patientICN).Where(c => c.SURVEY_STATUS == statusId).Count();
            }
        }

        /// <summary>
        /// SelectAllQuestions
        /// </summary>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public IEnumerable<STD_QUESTION> SelectAllQuestions(int typeId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var questions = from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE_ID == typeId && t.INACTIVE_FLAG == false
                                select t;

                return questions.ToList();
            }

        }

        /// <summary>
        /// Update
        /// </summary>
        /// <param name="survey"></param>
        /// <returns></returns>
        public Int32 Update(SURVEY survey)
        {
            Int32 returnStatus = 0;

            survey.AUTHOR_DUZ = TbiServiceTo.DUZ;
            survey.PROVIDER_DUZ = TbiServiceTo.DUZ;
            if (!string.IsNullOrEmpty(StationDivision))
                TbiServiceTo.siteCode = StationDivision;

            if (!string.IsNullOrEmpty(TbiServiceTo.siteCode) && !string.IsNullOrEmpty(TbiServiceTo.ServerIP))
            {

                survey.STD_INSTITUTION_ID = GetStdInstitutionId(TbiServiceTo.siteCode, TbiServiceTo.ServerIP);
                if (!string.IsNullOrEmpty(TbiServiceTo.StationNumber))
                {
                    UsersManager userManager = new UsersManager();
                    if (!string.IsNullOrEmpty(TbiServiceTo.SelectedCosignerID))
                    {
                        survey.PROVIDER_DUZ = TbiServiceTo.SelectedCosignerID;
                    }
                    int sta3n = VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(TbiServiceTo.StationNumber);
                    survey.AUTHOR_ID = userManager.GetProviderIdByStaffIenAndSta3n(survey.AUTHOR_DUZ, sta3n);
                    survey.PROVIDER_ID = survey.AUTHOR_ID;
                }
            }

            using (_db = GetDataContext())
            {
                _db.Connection.Open();
                using (var transaction = _db.Connection.BeginTransaction())
                {
                    try
                    {
                        _db.Transaction = transaction;
                        _db.DeferredLoadingEnabled = false;

                        survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        transaction.Commit();

                        //create an instance of the custom eventArgs in order to populate the id selected
                        BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                        eventArgs.SavedItemId = survey.SURVEYS_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;

                        returnStatus = survey.SURVEYS_ID;

                        RaiseSaveEvent(this, eventArgs);

                    }
                    catch (Exception ex)
                    {
                        transaction.Rollback();
                        Helpers.HandleTBIException(ex, "InstrumentManager.cs", "", "Update(SURVEY survey)");
                        Console.Write(ex.Message);
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    }
                    return returnStatus;
                }
            }
        }

        /// <summary>
        /// DeleteAllSurveyResults
        /// </summary>
        /// <param name="survey"></param>
        public void DeleteAllSurveyResults(SURVEY survey)
        {
            if (survey != null)
            {
                if (survey.SURVEY_RESULTs.Count > 0)
                {
                    survey.SetAsChangeTrackingRoot();

                    foreach (var result in survey.SURVEY_RESULTs)
                    {
                        result.SetAsDeleteOnSubmit();
                    }

                    using (_db = GetDataContext())
                    {
                        _db.DeferredLoadingEnabled = false;

                        survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                        try
                        {
                            _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                            //create an instance of the custom eventArgs in order to populate the id selected
                            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                            eventArgs.SavedItemId = survey.SURVEYS_ID;
                            eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        }
                        catch (Exception exc)
                        {
                            Helpers.HandleTBIException(exc, "InstrumentManager.cs", "", "DeleteAllSurveyResults()");
                            _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        }
                    }
                }
            }
        }

        /// <summary>
        /// Accepts a List<STD_SURVEY_TYPE> objects, converts and returns the equivalent List<InstrumentType>
        /// </summary>
        /// <returns></returns>
        private List<InstrumentType> convertStdSurveyTypeListToInstrumentType(List<STD_SURVEY_TYPE> surveyTypes)
        {
            List<InstrumentType> returnList = new List<InstrumentType>();

            foreach (STD_SURVEY_TYPE survey in surveyTypes)
            {
                // for Mayo-Portland Adaptability Inventory-4 Participation Index and TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care - custom titles required

                InstrumentType instrumentType = new InstrumentType();
                instrumentType.Id = survey.ID;
                instrumentType.Name = survey.NAME;
                instrumentType.IsActive = survey.IS_ACTIVE;

                if (survey.CODE == "MPAI")
                {
                    List<string> titleMpai = new List<string>();
                    titleMpai.Add("Mayo-Portland Adaptability Inventory-4");
                    titleMpai.Add("Participation Index (M2PI)");
                    titleMpai.Add("Muriel D. Lezak, PhD, ABPP & James F. Malec, PhD, ABPP");
                    titleMpai.Add("");
                    titleMpai.Add("Used as VA Interdisciplinary Team Assessment of Community Functioning");

                    instrumentType.Title = titleMpai;

                }
                else if (survey.CODE == "CAREPLAN")
                {
                    List<string> titleCarePlan = new List<string>();
                    titleCarePlan.Add("TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care");

                    instrumentType.Title = titleCarePlan;
                }
                else
                {
                    instrumentType.Title = new List<string>();
                    instrumentType.Title.Add(survey.NAME);
                }

                returnList.Add(instrumentType);
            }

            return returnList;
        }

        #endregion

        private string StationDivision
        {
            get
            {
                return System.Web.HttpContext.Current.Session["StationDivision"] as string;
            }
            //Fortify - Unused Method issue: Commented out set method
            //set
            //{
            //    System.Web.HttpContext.Current.Session["StationDivision"] = value;
            //}
        }

    }

    [Serializable()]
    public class TBIServiceTO
    {
        #region Properties

        public string siteCode { get; set; }
        public string DUZ { get; set; }
        public string DFN { get; set; }

        public string ServerIP { get; set; }
        public string ServerPort { get; set; }

        public string UserName { get; set; }
        public string PatientName { get; set; }
        public string PatientDOB { get; set; }
        public string PatientDOD { get; set; } // Deceased Date
        public string PatientGender { get; set; }
        public string PatientSnum { get; set; }
        public string PatientLocationId { get; set; }
        public string PatientICN { get; set; }
        public string PatientDFN { get; set; }
        public string STA6N { get; set; }
        public string PatientAdmitTimeStamp { get; set; }
        public string InstitutionName { get; set; }

        // Demographics
        public string PatientAge { get; set; }
        public string PatientCity { get; set; }
        public string PatientState { get; set; }
        public string PatientMaritalStatus { get; set; }
        public string PatientReligion { get; set; }
        public int PatientRegion { get; set; }
        public string PatientRace { get; set; }
        public string PatientEthnicity { get; set; }

        //public string PatientPCP { get; set; }

        public int CprsSentTBIEvaluationNoteId { get; set; }
        public int CprsSentTBIFollowUpNoteId { get; set; }

        public NoteType SelectedNoteType { get; set; }
        public string NoteText { get; set; }

        public InstrumentType SelectedInstrumentType { get; set; }

        public string SelectedNoteTitleId { get; set; }
        public string SelectedConsultId { get; set; }
        public string SelectedVisitId { get; set; }
        public string SelectedHospitalAdmissionId { get; set; }
        public string SelectedLocationID { get; set; }
        public string SelectedNewVisitId { get; set; }

        public string SelectedCosignerID { get; set; }

        public bool isHistorical { get; set; }

        public List<InstrumentType> Instruments;

        public string StationNumber { get; set; }

        #endregion

        public TBIServiceTO() { }
    }

    public enum NoteTypeEnum : int
    {
        ScheduledClinicAppointment = 0,
        HospitalAdmission = 1,
        CurrentStay = 2,
        UnscheduledOrNewVisit = 3
    }

    [Serializable()]
    public class NoteType
    {
        public int Id { get; set; }
        public string Name { get; set; }
    }

    [Serializable()]
    public class InstrumentType
    {
        public int Id { get; set; }
        public string Name { get; set; }
        public string Status { get; set; }
        public List<string> Title { get; set; }
        public Boolean IsActive { get; set; }
    }
}