import {
  Component,
  OnInit,
  OnDestroy,
  Input,
  Output,
  EventEmitter,
  ViewChild,
  OnChanges,
  SimpleChanges,
  Inject,
  ElementRef
} from '@angular/core';
import { DOCUMENT } from '@angular/platform-browser';
import { Subscription } from 'rxjs/Subscription';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';

// DomSanitizer and Sanitizer needed to inject xml files into the viewer iframe.
import { DomSanitizer } from '@angular/platform-browser';
import { Sanitizer } from '@angular/core/src/security';

import { Search275Service } from '../../search275/search275.service';
import { AttachmentViewerService } from './attachment-viewer.service';

@Component({
  selector: 'app-attachment-viewer',
  templateUrl: './attachment-viewer.component.html',
  styleUrls: ['./attachment-viewer.component.scss']
})
export class AttachmentViewerComponent implements OnInit, OnChanges, OnDestroy {
  // boolean to control whether to show/hide the attachment viewer.
  @Input() open = false;
  // list of attachments returned from search results passed by search275 component.
  @Input() attachmentList: string[] = [];
  // attachment that was clicked by end user.
  @Input() selectedAttachment: string = null;
  cleanSelectedAttachment;
  // notifies the parent component that the viewer needs to close;
  @Input() attachmentId: string = null;
  @Output() notify: EventEmitter<{}> = new EventEmitter();
  // path is a reference to a hidden field to copy the attachment path to the clipboard.
  @ViewChild('path') path: ElementRef = new ElementRef('');
  associateView = false;
  fileExtension: string = null;
  subscription: Subscription = new Subscription();
  selectedAttachmentName: string = null;
  modalView = false;
  // need to replace the following with the message component.
  tempMessage: string = null;
  showMessage = false;
  form: FormGroup;
  // following variables are for mapping the response from "Manual matching of unmatched 837 claim"
  success: boolean;
  failed: boolean;
  errorDescription: string = null;
  attachmentModified = false;

  constructor(
    private search275Service: Search275Service,
    private sanitizer: DomSanitizer,
    @Inject(DOCUMENT) private document,
    private fb: FormBuilder,
    private attachmentViewerService: AttachmentViewerService
  ) {}
  roles;
  permissions;
  ngOnInit() {
    this.formInit();
    if (sessionStorage.getItem('additionalInformationPermissions')) {
      this.roles = JSON.parse(sessionStorage.getItem('userRole'));
      this.permissions = JSON.parse(sessionStorage.getItem('permissions'));
      // console.log('roles', this.roles, 'permissions', this.permissions);
    }
  }
  ngOnChanges(change: SimpleChanges) {
    if (change.selectedAttachment && change.selectedAttachment.currentValue) {
      this.cleanSelectedAttachment = this.sanitizer.bypassSecurityTrustResourceUrl(
        this.selectedAttachment
      );
      this.selectedAttachmentName = change.selectedAttachment.currentValue
        .split('/')
        .pop();
    }
    if (change.open && change.open.currentValue) {
      this.document.body.style.overflow = 'hidden';
    }
  }
  formInit() {
    this.form = this.fb.group({
      unmatched837Id: [
        null,
        [Validators.required, Validators.pattern('[0-9]*')]
      ]
    });
  }
  onSubmit() {
    // console.log({...this.form.value, ...{attachmentId: this.attachmentId.toString()}});
    this.subscription = this.attachmentViewerService
      .match({
        ...this.form.value,
        ...{ attachmentId: this.attachmentId.toString() }
      })
      .subscribe(data => {
        // console.log('attachmentComponent ', data);
        this.attachmentModified = true;
        if (data.response) {
          this.success = true;
          this.associateView = false;
          setTimeout(() => {
            this.success = false;
          }, 7500);
        } else {
          this.failed = true;
        }
        if (data.errorCode) {
          this.errorDescription = data.message;
          setTimeout(() => {
            (this.errorDescription = ''), (this.failed = false);
          }, 7500);
        }
      });
  }
  closeViewer(): void {
    const viewerNotify = {
      open: false,
      updateTable: this.attachmentModified
    };
    this.fileExtension = null;
    this.cleanSelectedAttachment = null;
    this.document.body.style.overflow = 'visible';
    this.attachmentModified = false;
    this.associateView = false;
    this.form.reset();
    this.notify.emit(viewerNotify);
  }
  nextAttachment(): void {
    const length = this.attachmentList.length;
    const index = this.attachmentList.indexOf(this.selectedAttachment);
    if (index + 1 < length) {
      this.selectedAttachment = this.attachmentList[index + 1];
      this.setfileExtension(this.selectedAttachment);
    }
  }
  previousAttachment(): void {
    const length = this.attachmentList.length;
    const index = this.attachmentList.indexOf(this.selectedAttachment);
    if (index - 1 >= 0) {
      this.selectedAttachment = this.attachmentList[index - 1];
      this.setfileExtension(this.selectedAttachment);
    }
  }
  setfileExtension(attachment): void {
    if (attachment) {
      this.selectedAttachmentName = attachment.split('/').pop();
      this.fileExtension = attachment
        .split('.')
        .pop()
        .toUpperCase();
    }
  }
  // method to enable xml into the iframe.
  onXmlFile() {
    return this.sanitizer.bypassSecurityTrustResourceUrl(
      this.selectedAttachment
    );
  }
  viewAssociateForm(): void {
    this.associateView = !this.associateView;
    if (!this.associateView) {
      this.form.controls['unmatched837Id'].setValue(null);
    }
  }
  copyToClipboard(): void {
    this.associateView = false;
    this.path.nativeElement.select();
    this.document.execCommand('copy');
  }
  // modal related methods.
  onArchiveView(): void {
    this.modalView = true;
  }
  onArchive() {
    console.log(
      'onArchive() this will be completed once the rest call is done.'
    );
  }
  onModalClose(): void {
    this.modalView = false;
  }
  // method to show the user a successful message once a path has been copied to the clipboard.
  clipBoardMessage(): void {
    this.tempMessage = 'The attachment path was copied to your clipboard.';
    this.showMessage = true;
    setTimeout(() => {
      this.showMessage = false;
    }, 7500);
  }

  ngOnDestroy() {
    this.subscription.unsubscribe();
  }
}
