import { Subscription } from 'rxjs/Subscription';
import {
  Component,
  OnInit,
  ViewChild,
  ElementRef,
  OnDestroy
} from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  FormControl,
  Validators
} from '@angular/forms';
import { Search275Service } from './search275.service';
import {
  Search275Model,
  ResponseModel,
  FormModel
} from './search275.component.model';
import { ScriptService } from './../shared/scriptLoader/script.service';
import {
  TableColumnModel,
  TableSettings,
  PaginationSettings
} from './../shared/table/table.component.model';
import { atLeastOne } from './../rfai-lookup/rfai-lookup.atleastoneValidator';
import { AttachmentViewerComponent } from './../shared/attachment-viewer/attachment-viewer.component';

@Component({
  selector: 'app-search275',
  templateUrl: './search275.component.html',
  styleUrls: ['./search275.component.scss'],
  providers: [ScriptService]
})
export class Search275Component implements OnInit, OnDestroy {
  constructor(
    private fb: FormBuilder,
    private search275Service: Search275Service,
    private script: ScriptService
  ) {
    this.script
      .load('datepicker')
      .then(data => {})
      .catch(error => console.log(error));
  }
  @ViewChild(AttachmentViewerComponent) attachmentViewerComponent;
  @ViewChild('serviceStartDate')
  serviceStartDate: ElementRef = new ElementRef('');
  @ViewChild('serviceEndDate') serviceEndDate: ElementRef = new ElementRef('');
  tableColumns: TableColumnModel[] = [
    new TableColumnModel('Attch. ID', 'attachId'),
    new TableColumnModel('Attachment Ctrl.', 'attachCtrNumber'),
    new TableColumnModel('Patient Name', 'patientName'),
    new TableColumnModel('Patient ID', 'patientIdentifier'),
    new TableColumnModel('Patient Control #', 'patientCtrNumber'),
    new TableColumnModel('Provider Name', 'providerName'),
    new TableColumnModel('Provider NPI', 'providerNpi'),
    new TableColumnModel('Payer Ctrl.', 'payerControlNumber'),
    new TableColumnModel('Claim ID', 'externalClaimId'),
    new TableColumnModel('Svc. Start Date', 'claimServiceStartDate'),
    new TableColumnModel('Rpt. Type', 'reportTypeCode'),
    new TableColumnModel('Status', 'status'),
    new TableColumnModel('Med. Record', 'medicalRecordNumber')
    // new TableColumnModel('Attch. Type', 'attachType'),
    // new TableColumnModel('View', '')
  ];
  tableSettings: TableSettings = new TableSettings();
  paginationSettings: PaginationSettings = new PaginationSettings();
  errorStatus = false;
  errorMsg = '';
  successStatus = false;
  successMsg = '';
  showResults = false;
  requestModel: Search275Model = new Search275Model();
  loading = true;
  searchSubscription: Subscription;
  reportCodes = [
    'All ',
    '03 Report Justifying Treatment Beyond Utilization Guidelines',
    '04 Drugs Administered',
    '05 Treatment Diagnosis',
    '06 Initial Assessment',
    '07 Functional Goals',
    '08 Plan of Treatment',
    '09 Progress Report',
    '10 Continued Treatment',
    '11 Chemical Analysis',
    '13 Certified Test Report',
    '15 Justification for Admission',
    '21 Recovery Plan',
    'A3 Allergies/Sensitivities Document',
    'A4 Autopsy Report',
    'AM Ambulance Certification',
    'AS Admission Summary',
    'B2 Prescription',
    'B3 Physician Order',
    'B4 Referral Form',
    'BR Benchmark Testing Results',
    'BS Baseline',
    'BT Blanket Test Results',
    'CB Chiropractic Justification',
    'CK Consent Form(s)',
    'CT Certification',
    'D2 Drug Profile Document',
    'DA Dental Models',
    'DB Durable Medical Equipment Prescription',
    'DG Diagnostic Report',
    'DJ Discharge Monitoring Report',
    'DS Discharge Summary',
    'EB Explanation of Benefits (Coordination of Benefits or Medicare Secondary Payer)',
    'HC Health Certificate',
    'HR Health Clinic Records',
    'I5 Immunization Record',
    'IR State School Immunization Records',
    'LA Laboratory Results',
    'M1 Medical Record Attachment',
    'MT Models',
    'NN Nursing Notes',
    'OB Operative Note',
    'OC Oxygen Content Averaging Report',
    'OD Orders and Treatments Document',
    'OE Objective Physical Examination (including vital signs) Document',
    'OX Oxygen Therapy Certification',
    'OZ Support Data for Claim',
    'P4 Pathology Report',
    'P5 Patient Medical History Document',
    'PE Parenteral or Enteral Certification',
    'PN Physical Therapy Notes',
    'PO Prosthetics or Orthotic Certification',
    'PQ Paramedical Results',
    'PY Physician’s Report',
    'PZ Physical Therapy Certification',
    'RB Radiology Films',
    'RR Radiology Reports',
    'RT Report of Tests and Analysis Report',
    'RX Renewable Oxygen Content Averaging Report',
    'SG Symptoms'
  ];
  searchResults: ResponseModel[];
  form: FormGroup;
  statuses = ['Active', 'Inactive'];
  validDateRange = true;
  // test data list
  selectedAttachmentId: string;
  fileList: string[] = [];
  selectedAttachment: string = null;
  displayAttachmentViewer = false;
  ngOnInit() {
    this.formInit();
    this.getUserData();
    this.tableInit();
    // this.search275Service
    //   .getSampleData()
    //   .subscribe(fileList => (this.fileList = fileList));
  }

  formInit() {
    this.form = this.fb.group(
      {
        attachId: [this.requestModel.attachId, Validators.pattern('[0-9]*')],
        attachCtrNumber: [this.requestModel.attachCtrNumber],
        claimId: [this.requestModel.claimId, Validators.pattern('[0-9]*')],
        status: this.requestModel.status,
        patientLastName: this.requestModel.patientLastName,
        patientFirstName: this.requestModel.patientFirstName,
        patientCtrNumber: [this.requestModel.patientCtrNumber],
        patientIdentifier: [
          this.requestModel.patientIdentifier,
          Validators.pattern('[0-9]*')
        ],
        providerName: this.requestModel.providerName,
        providerNpi: [
          this.requestModel.providerNpi,
          Validators.pattern('[0-9]*')
        ],
        medicalRecordNumber: [
          this.requestModel.medicalRecordNumber,
          Validators.pattern('[0-9]*')
        ],
        payerControlNumber: [this.requestModel.payerControlNumber],
        reportTypeCode: this.requestModel.reportTypeCode,
        claimServiceStartDate: [
          this.serviceStartDate.nativeElement.value,
          [
            Validators.pattern(
              '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
            )
          ]
        ],
        claimServiceEndDate: [
          this.serviceEndDate.nativeElement.value,
          [
            Validators.pattern(
              '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
            )
          ]
        ]
      }
      // { validator: atLeastOne(Validators.required) }
    );
  }
  tableInit() {
    // Initializes the search results table

    this.tableSettings.imgColumn = true;
    this.tableSettings.extImages = true; // test code
    // added to accomodate digits on the attachment images.
    this.tableSettings.imgCounterView = true;
    // this.tableSettings.imageUrls = [
    //   '../../assets/images/ic_attachCount_blue_24.svg'
    // ];
    this.tableSettings.imageColumnNames = [' '];

    // this.tableSettings.buttonName = 'View';
    // this.tableSettings.buttonShow = true;
  }
  getUserData() {
    // if (sessionStorage.getItem('additionalInformationPermissions')) {
    //   this.additionalInfoPermissions = JSON.parse(
    //     sessionStorage.getItem('additionalInformationPermissions')
    //   );
    //   this.disablePopulateInfo = !this.additionalInfoPermissions[
    //     'populateInfo'
    //   ];
    //   this.disableSubmitInfo = !this.additionalInfoPermissions['submitInfo'];
    // }
    // this.roles = JSON.parse(sessionStorage.getItem('userRole'));
  }
  resetForm() {
    this.form.reset();
    this.requestModel = new Search275Model();
    this.form.get('reportTypeCode').setValue(this.requestModel.reportTypeCode);
    this.form.get('status').setValue(this.requestModel.status);
    this.form.updateValueAndValidity();
    this.serviceStartDate.nativeElement.value = '';
    this.serviceEndDate.nativeElement.value = '';
    this.showResults = false;
    this.errorStatus = false;
    this.errorMsg = '';
    this.successStatus = false;
    this.successMsg = '';
    this.validDateRange = true;
  }
  search() {
    this.paginationSettings = new PaginationSettings();
    this.fetchData();
  }
  fetchData() {
    this.requestModel.pageSize = this.paginationSettings.pageSize;
    this.requestModel.pageNumber = this.paginationSettings.currentPage;
    this.requestModel.descending = this.paginationSettings.descending;
    this.requestModel.sortColumn = this.paginationSettings.sortColumn;
    this.searchSubscription = this.search275Service
      .search(this.requestModel)
      .subscribe(data => {
        if (data.errorCode) {
          this.paginationSettings = new PaginationSettings();
          this.errorStatus = true;
          this.errorMsg = 'Error in processing search request.';
          this.searchResults = [];
          this.showResults = false;
        } else {
          this.searchResults = data.response;
          this.showResults = true;
          this.errorStatus = false;
          this.errorMsg = '';
          this.paginationSettings = {
            currentPage: data.pageNumber,
            pageSize: data.pageSize,
            totalPages: Math.ceil(data.totalNumberOfResults / data.pageSize),
            totalResults: data.totalNumberOfResults,
            sortColumn: data.sortColumn,
            descending: this.requestModel.descending
          };
        }
        this.loading = false;
      });
  }
  updateTable() {
    this.fetchData();
  }
  imgClicked(event: any) {}
  // validate the date range between the two date fields. fromDate should be less than toDate.
  datesValidator() {
    if (
      this.serviceStartDate.nativeElement.value.length === 10 &&
      this.serviceEndDate.nativeElement.value.length === 10
    ) {
      this.form
        .get('serviceStartDate')
        .setValue(this.serviceStartDate.nativeElement.value);
      this.form
        .get('serviceEndDate')
        .setValue(this.serviceEndDate.nativeElement.value);
      const fromDate = new Date(this.form.get('serviceStartDate').value);
      const toDate = new Date(this.form.get('serviceEndDate').value);
      if (toDate.valueOf() < fromDate.valueOf()) {
        this.validDateRange = false;
        this.form.get('serviceStartDate').setValue('');
        this.form.get('serviceEndDate').setValue('');
      }
    } else {
      this.validDateRange = true;
    }
  }
  onDisplayAttachmentViewer(attachment) {
    this.selectedAttachmentId = attachment.attachId;
    this.fileList = attachment.pathAddress;
    this.selectedAttachment = this.fileList[0];
    this.attachmentViewerComponent.fileExtension = this.selectedAttachment
      .split('.')
      .pop()
      .toUpperCase();
    // console.log(
    //   'attachment extension',
    //   this.attachmentViewerComponent.fileExtension
    // );
    // console.log('attachment', this.selectedAttachment);
    this.displayAttachmentViewer = true;
  }
  onAttachmentViewerClose(attachmentOption) {
    this.displayAttachmentViewer = attachmentOption.open;
    if (attachmentOption.updateTable) {
      this.updateTable();
    }
  }
  ngOnDestroy() {
    if (this.searchSubscription) {
      this.searchSubscription.unsubscribe();
    }
  }
}
