import { Component, OnInit, Input } from '@angular/core';
import {
  SubmissionModel,
  AttachmentModel,
  RelatedRFAIModel,
  LineItemModel,
  UpdateRFAIRequestModel
} from './rfai-submission.model';
import { TableService } from './../shared/table/table.service';
import {
  TableColumnModel,
  TableSettings,
  PaginationSettings
} from './../shared/table/table.component.model';
import { RfaiSubmissionService } from './rfai-submission.service';
import { RfaiLookupService } from './../rfai-lookup/rfai-lookup.service';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';

@Component({
  selector: 'app-rfai-submission',
  templateUrl: './rfai-submission.component.html',
  styleUrls: ['./rfai-submission.component.scss']
})
export class RfaiSubmissionComponent implements OnInit {
  submissionId = '';
  model: SubmissionModel = new SubmissionModel();
  attachments: AttachmentModel[] = [];
  lines: LineItemModel[] = [];
  updateRequest: UpdateRFAIRequestModel = new UpdateRFAIRequestModel();
  showUpdate = true;
  modalClosed = true;
  modalError = false;
  submissionNote = '';
  successMsg = '';
  errorMsg = '';
  errorStatus = false;
  successStatus = false;
  viewSubmissionPermissions = '';
  disablePrint = true;
  disableSave = true;
  disableUpdate = true;
  loading = true;
  rfaiStatusValues = [
    'Pending',
    'Error',
    'Complete',
    'Abandoned',
    'Closed',
    'Manually Closed'
  ];
  // table setup
  tableColumns: TableColumnModel[] = [
    new TableColumnModel('Submit ID', 'submissionId'),
    // new TableColumnModel('PDI / Claim ID', 'claimId'),
    new TableColumnModel('Provider Name', 'providerName'),
    new TableColumnModel('Provider NPI', 'providerNPI'),
    new TableColumnModel('Provider TIN', 'providerTin'),
    // new TableColumnModel('Patient Name', 'patientName'),
    // new TableColumnModel('Patient ID', 'patientId'),
    new TableColumnModel('Patient Control', 'patientControlNumber'),
    new TableColumnModel('Medical Record', 'medicalRecordNumber'),
    new TableColumnModel('Date Submitted', 'startDate'), // Change to correct data source
    new TableColumnModel('Required Response Date', 'endDate') // Change to correct data source
  ];
  tableSettings: TableSettings = new TableSettings();
  paginationSettings: PaginationSettings = new PaginationSettings();
  relatedRFAIs: RelatedRFAIModel[] = [];
  form: FormGroup;
  // end of table setup
  constructor(
    private fb: FormBuilder,
    private rfaiSubmissionService: RfaiSubmissionService,
    private rfaiLookupService: RfaiLookupService
  ) {}

  ngOnInit() {
    this.getUserData();
    this.submissionId = this.getId();
    this.getDetails(this.submissionId);
    // table Settings
    this.tableSettings.pagination = false;
    this.paginationSettings = {
      currentPage: 1,
      pageSize: 10,
      totalPages: 1,
      totalResults: 2,
      sortColumn: 'claimId',
      descending: false
    };
    this.formInit();
  }
  getUserData() {
    if (sessionStorage.getItem('viewSubmissionPermissions')) {
      this.viewSubmissionPermissions = JSON.parse(
        sessionStorage.getItem('viewSubmissionPermissions')
      );
      this.disablePrint = !this.viewSubmissionPermissions['printRfai'];
      this.disableSave = !this.viewSubmissionPermissions['saveUpdates'];
      this.disableUpdate = !this.viewSubmissionPermissions['updateRfai'];
    }
  }
  formInit() {
    this.form = this.fb.group({
      responseDate: [
        this.model.responseDate,
        Validators.pattern(
          '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
        )
      ],
      status: [this.model.rfaiStatus]
    });
  }
  updateTable() {}
  getId(): string {
    return this.rfaiLookupService.getSubmissionId();
  }
  getDetails(id) {
    this.rfaiSubmissionService.getDetails(id).subscribe(data => {
      this.model = data;
      this.attachments = this.model.rfaiAttachmentResponse;
      this.relatedRFAIs = this.model.rfaiRelatedItemResponse;
      this.lines = this.model.rfaiLineItemResponse;
      this.loading = false;
    });
  }
  saveUpdates() {
    if (this.updateRequest.submissionNote) {
      this.model.responseDate = this.updateRequest.responseDate;
      this.model.rfaiStatus = this.updateRequest.submissionStatus;
      this.modalClosed = true;
      this.showUpdate = !this.showUpdate;
      this.updateRequest = {
        submissionId: +this.submissionId,
        submissionStatus: this.model.rfaiStatus,
        submissionNote: this.updateRequest.submissionNote,
        responseDate: this.model.responseDate
      };

      this.rfaiSubmissionService
        .updateRfai(this.updateRequest)
        .subscribe(data => {
          if (data.result) {
            this.successMsg = 'Updated RFAI successfully.';
            this.successStatus = true;
            this.errorStatus = false;
            setTimeout(() => {
              this.successStatus = false;
            }, 8000);
          } else {
            this.errorStatus = true;
            this.successStatus = false;
            this.errorMsg = 'An error occured RFAI update.';
            setTimeout(() => {
              this.errorStatus = false;
            }, 8000);
          }
        });
    } else {
      this.modalError = true;
    }
  }
  print() {
    console.log('print');
  }
  update() {
    this.updateRequest.submissionStatus = this.model.rfaiStatus;
    this.updateRequest.responseDate = this.model.responseDate;
    this.showUpdate = !this.showUpdate;
  }

  onModalOpen() {
    this.modalClosed = false;
  }

  onModalClose(bool: boolean) {
    this.modalClosed = bool;
  }

  cancelModal() {
    this.modalClosed = true;
    this.updateRequest = new UpdateRFAIRequestModel();
    this.showUpdate = !this.showUpdate;
  }
}
