import { Component, OnInit, ViewChild, ElementRef } from '@angular/core';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';

import { ScriptService } from './../shared/scriptLoader/script.service';
import { RfaiFormService } from './rfai-form.service';

import {
  AdditionalInformationRequestModel,
  PDIInfoModel,
  LineModel,
  RfaiInfoModel
} from './rfai-form.model';

import { CodeModel } from './../rfai-admin/rfai-admin.model';

@Component({
  selector: 'app-rfai-form',
  templateUrl: './rfai-form.component.html',
  styleUrls: ['./rfai-form.component.scss'],
  providers: [ScriptService]
})
export class RfaiFormComponent implements OnInit {
  autoFillform: FormGroup;
  manualForm: FormGroup;
  lines: LineModel[] = [];
  modalClosed = true;
  selectedLines = [];
  enableAddLine = false;
  errorMsg = '';
  errorStatus = false;
  successStatus = false;
  successMsg = '';
  populateInfoClicked = false;
  prepopulatedInfo: PDIInfoModel = new PDIInfoModel();
  additionalInfoPermissions: any;
  disablePopulateInfo = false;
  disableSubmitInfo = false;
  roles: string[];

  // Initial Values for the dropdown
  healthCareClaimStatusCodes: CodeModel[] = [];
  loincCodes: CodeModel[] = [];
  loincCodeModifiers: CodeModel[] = [];
  autoFormModel: AdditionalInformationRequestModel = new AdditionalInformationRequestModel();
  manualFormModel: AdditionalInformationRequestModel = new AdditionalInformationRequestModel();
  pdiInfoModel: PDIInfoModel = new PDIInfoModel();
  line: LineModel = new LineModel();

  // viewChild is required here due to the implementation of the calendar.
  @ViewChild('responseDueDateAuto')
  responseDueDateAuto: ElementRef = new ElementRef('');

  @ViewChild('responseDueDateManual')
  responseDueDateManual: ElementRef = new ElementRef('');

  @ViewChild('serviceFromDate')
  serviceFromDate: ElementRef = new ElementRef('');

  @ViewChild('serviceToDate') serviceToDate: ElementRef = new ElementRef('');
  @ViewChild('selectAllLines') selectAllLines: ElementRef = new ElementRef('');

  constructor(
    private fb: FormBuilder,
    private rfaiAdditionalInformationService: RfaiFormService,
    private script: ScriptService
  ) {
    this.script
      .load('datepicker')
      .then(data => {})
      .catch(error => console.log(error));
  }

  ngOnInit() {
    this.getCodes();
    this.formInit();
    this.getUserData();
  }

  getCodes() {
    this.rfaiAdditionalInformationService.getHccscCodes().subscribe(data => {
      this.healthCareClaimStatusCodes = data;
    });

    this.rfaiAdditionalInformationService.getLoincCodes().subscribe(data => {
      this.loincCodes = data;
    });

    this.rfaiAdditionalInformationService
      .getLoincCodeModifiers()
      .subscribe(data => {
        this.loincCodeModifiers = data;
      });
  }

  getUserData() {
    if (sessionStorage.getItem('additionalInformationPermissions')) {
      this.additionalInfoPermissions = JSON.parse(
        sessionStorage.getItem('additionalInformationPermissions')
      );

      this.disablePopulateInfo = !this.additionalInfoPermissions[
        'populateInfo'
      ];
      this.disableSubmitInfo = !this.additionalInfoPermissions['submitInfo'];
    }
    this.roles = JSON.parse(sessionStorage.getItem('userRole'));
  }

  formInit() {
    this.autoFillform = this.fb.group({
      claimId: [
        this.autoFormModel.claimId,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      healthCareClaimStatusCode: [
        this.autoFormModel.claimStatus,
        Validators.required
      ],
      loincCode: [this.autoFormModel.loincCode, Validators.required],
      loincCodeModifier: [
        this.autoFormModel.loincCodeModifier
        // Validators.required
      ],
      contactInfo: [
        this.autoFormModel.contactInformation,
        [Validators.required, Validators.email]
      ],
      responseDueDateAuto: [
        this.responseDueDateAuto.nativeElement.value,
        [
          Validators.required,
          Validators.pattern(
            '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
          )
        ]
      ]
    });

    this.manualForm = this.fb.group({
      healthCareClaimStatusCode: [
        this.manualFormModel.claimStatus,
        Validators.required
      ],
      loincCode: [this.manualFormModel.loincCode, Validators.required],
      loincCodeModifier: [
        this.manualFormModel.loincCodeModifier
        // Validators.required
      ],
      contactInfo: [
        this.manualFormModel.contactInformation,
        [Validators.required, Validators.email]
      ],
      responseDueDateManual: [
        this.responseDueDateManual.nativeElement.value,
        [
          Validators.required,
          Validators.pattern(
            '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
          )
        ]
      ],
      payerName: [this.pdiInfoModel.payerName, Validators.required],
      payerID: [
        this.pdiInfoModel.payerId,
        [Validators.required, Validators.pattern('[0-9]{0,15}')]
      ],
      payerClaimControlNumber: [
        this.pdiInfoModel.payerClaimControlNumber,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      billingProvider: [
        this.pdiInfoModel.providerInformation,
        Validators.required
      ],
      providerNpi: [
        this.pdiInfoModel.providerNpi,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      billingProviderTIN: [
        this.pdiInfoModel.providerTin,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      lastName: [this.pdiInfoModel.patientLastname, Validators.required],
      firstName: [this.pdiInfoModel.patientFirstName, Validators.required],
      patientID: [
        this.pdiInfoModel.patientIdentifier,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      patientControlNumber: [
        this.pdiInfoModel.patientcontrolNumber,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      informationReceiver: [
        this.pdiInfoModel.informationReceiver,
        Validators.required
      ],

      billType: [this.pdiInfoModel.billType, Validators.required],
      clearingHouseID: [
        this.pdiInfoModel.clearingHouseId,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      medicalRecordNumber: [
        this.pdiInfoModel.medicalRecordNumber,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      level: [this.pdiInfoModel.level, Validators.required],
      lines: this.lines,
      serviceFromDate: [
        this.serviceFromDate.nativeElement.value,
        Validators.pattern(
          '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
        )
      ],
      serviceToDate: [
        this.serviceToDate.nativeElement.value,
        Validators.pattern(
          '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
        )
      ]
    });

    this.manualForm.get('level').valueChanges.subscribe(level => {
      if (level === 'C') {
        this.manualForm
          .get('serviceFromDate')
          .setValidators([
            Validators.required,
            Validators.pattern(
              '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
            )
          ]);
        this.manualForm
          .get('serviceToDate')
          .setValidators([
            Validators.required,
            Validators.pattern(
              '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
            )
          ]);
        this.manualForm.get('serviceFromDate').updateValueAndValidity();
        this.manualForm.get('serviceToDate').updateValueAndValidity();
      } else {
        this.manualForm.get('serviceToDate').clearValidators();
        this.manualForm.get('serviceFromDate').clearValidators();
        // this.manualForm.get('serviceToDate').setValidators('');
        // this.manualForm.get('serviceFromDate').setValidators('');
      }
    });
  }

  onSubmitPrepopulated() {
    this.prepopulatedInfo.rfaiLineItemResponse = this.prepopulatedInfo.rfaiLineItemResponse.filter(
      line => line.selected
    );
    this.prepopulatedInfo.responseDate = this.responseDueDateAuto.nativeElement.value;
    let rfaiInfoModel: RfaiInfoModel = new RfaiInfoModel();
    rfaiInfoModel = Object.assign(
      {},
      this.autoFormModel,
      this.prepopulatedInfo
    );
    rfaiInfoModel.claimId = this.autoFormModel.claimId
      ? +this.autoFormModel.claimId
      : null;

    this.submitData(rfaiInfoModel);
  }

  onSubmitManual() {
    console.log('contactInfo', this.manualFormModel.contactInformation);
    if (this.pdiInfoModel.level === 'C') {
      this.pdiInfoModel.serviceFromDate = this.serviceFromDate.nativeElement.value;
      this.pdiInfoModel.serviceToDate = this.serviceToDate.nativeElement.value;
      this.pdiInfoModel.rfaiLineItemResponse = [];
    }
    this.pdiInfoModel.responseDate = this.responseDueDateManual.nativeElement.value;
    this.selectedLines = this.lines.filter(line => line.selected);
    this.pdiInfoModel.rfaiLineItemResponse = this.selectedLines;

    let rfaiInfoModel: RfaiInfoModel = new RfaiInfoModel();
    rfaiInfoModel = Object.assign({}, this.manualFormModel, this.pdiInfoModel);
    console.log(rfaiInfoModel);
    rfaiInfoModel.claimId = this.manualFormModel.claimId
      ? +this.manualFormModel.claimId
      : +rfaiInfoModel.payerClaimControlNumber;
    this.submitData(rfaiInfoModel);
  }

  submitData(rfaiInfoModel: RfaiInfoModel) {
    // console.log('Submitted to service with rfaiInfoModel', rfaiInfoModel);
    this.rfaiAdditionalInformationService
      .submitRfai(rfaiInfoModel)
      .subscribe(data => {
        console.log('data', data);
        if (data.result) {
          this.onResetManual();
          this.onResetPrepopulated();
          this.successMsg = '277 RFAI successfully submitted.';
          this.successStatus = true;
          this.errorStatus = false;

          setTimeout(() => {
            this.successStatus = false;
          }, 8000);
        } else {
          this.errorStatus = true;
          this.successStatus = false;
          this.errorMsg = 'An error occured during 277 RFAI submission.';
          setTimeout(() => {
            this.errorStatus = false;
          }, 8000);
        }
      });
  }

  resetCommon() {
    this.lines = [new LineModel()];
    this.errorStatus = false;
    this.successStatus = false;
    this.errorMsg = '';
    this.successMsg = '';
    this.pdiInfoModel.level = 'C';
    this.formInit();
  }

  onResetPrepopulated() {
    this.populateInfoClicked = false;
    this.autoFormModel = new AdditionalInformationRequestModel();
    this.resetCommon();
  }

  // Common code to be resetted on both forms
  onResetManual() {
    this.pdiInfoModel = new PDIInfoModel();
    this.manualFormModel = new AdditionalInformationRequestModel();
    this.serviceFromDate.nativeElement.value = '';
    this.serviceToDate.nativeElement.value = '';
    this.resetCommon();
  }

  addNewInitialLine() {
    if (this.lines.length === 0) {
      this.addNewLine();
    }
  }

  addNewLine() {
    const line: LineModel = new LineModel();
    this.lines.push(line);
    this.enableAddLine = false;
  }
  
  populateInfo() {
    if (this.autoFormModel.claimId) {
      this.populateInfoClicked = true;
      this.errorStatus = false;
      this.errorMsg = '';
      this.rfaiAdditionalInformationService
        .populateInfo(this.autoFormModel.claimId)
        .subscribe(data => {
          if (data.errorCode) {
            this.errorMsg = data.message;
            this.errorStatus = true;
            this.prepopulatedInfo = new PDIInfoModel();
          } else {
            this.prepopulatedInfo = data;
            this.errorStatus = false;
            this.errorMsg = '';
          }
          this.selectAllLines.nativeElement.checked = false;
        });
    } else {
      this.errorMsg = 'Enter PDI or Claim ID.';
      this.errorStatus = true;
    }
  }

  selectAll(bool: boolean) {
    this.prepopulatedInfo.rfaiLineItemResponse.map(
      line => (line.selected = bool)
    );
  }

  saveLine() {
    console.log('before Lines', this.lines);

    this.lines.forEach(line => {
      if (line.serviceLineId) {
        line.selected = true;
        line.serviceLineId = +line.serviceLineId;
        line.chargeAmount = +line.chargeAmount;
        this.enableAddLine = true;
        console.log('After Lines', this.lines);
      } else {
        this.enableAddLine = false;
      }
    });
  }

  enableAddLineButton() {
    this.enableAddLine = true;
  }
}
