import { RfaiFormComponent } from './rfai-form.component';
import { RfaiFormService } from './rfai-form.service';
import { FormBuilder } from '@angular/forms';
import { ElementRef } from '@angular/core';

import { ScriptService } from './../shared/scriptLoader/script.service';

import {
  AdditionalInformationRequestModel,
  PDIInfoModel,
  LineModel,
  RfaiInfoModel
} from './rfai-form.model';

import { Observable } from 'rxjs/Observable';
import { Observer } from 'rxjs/Observer';
import { CodeModel } from './../rfai-admin/rfai-admin.model';

describe('RfaiAdditionalInformationComponent', () => {
  let rfaiAdditionalInformationService: RfaiFormService;
  let component: RfaiFormComponent;
  let fb: FormBuilder;
  let script: ScriptService;
  let healthCareClaimStatusCodes: CodeModel[];
  let loincCodes: CodeModel[];
  let loincCodeModifiers: CodeModel[];
  let additionalInfoPermissions: any;
  let roles: string[];

  beforeEach(() => {
    healthCareClaimStatusCodes = [
      { cd: 'HEALTH_CODE_1', stcCd: 'HEALTH_STC_CODE_1', stcCdDesc: 'HEALTH_STC_CODE_DESC_1', flag: false, editedOrNewFlag: false },
      { cd: 'HEALTH_CODE_2', stcCd: 'HEALTH_STC_CODE_2', stcCdDesc: 'HEALTH_STC_CODE_DESC_2', flag: false, editedOrNewFlag: true },
      { cd: 'HEALTH_CODE_3', stcCd: 'HEALTH_STC_CODE_3', stcCdDesc: 'HEALTH_STC_CODE_DESC_3', flag: true, editedOrNewFlag: false },
      { cd: 'HEALTH_CODE_4', stcCd: 'HEALTH_STC_CODE_4', stcCdDesc: 'HEALTH_STC_CODE_DESC_4', flag: true, editedOrNewFlag: true }
    ];

    loincCodes = [
      { cd: 'LOINC_CODE_1', stcCd: 'LOINC_STC_CODE_1', stcCdDesc: 'LOINC_STC_CODE_DESC_1', flag: false, editedOrNewFlag: false },
      { cd: 'LOINC_CODE_2', stcCd: 'LOINC_STC_CODE_2', stcCdDesc: 'LOINC_STC_CODE_DESC_2', flag: false, editedOrNewFlag: true },
      { cd: 'LOINC_CODE_3', stcCd: 'LOINC_STC_CODE_3', stcCdDesc: 'LOINC_STC_CODE_DESC_3', flag: true, editedOrNewFlag: false },
      { cd: 'LOINC_CODE_4', stcCd: 'LOINC_STC_CODE_4', stcCdDesc: 'LOINC_STC_CODE_DESC_4', flag: true, editedOrNewFlag: true }
    ];

    loincCodeModifiers = [
      { cd: 'LOINC_MOD_CODE_1', stcCd: 'LOINC_MOD_STC_CODE_1', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_1', flag: false, editedOrNewFlag: false },
      { cd: 'LOINC_MOD_CODE_2', stcCd: 'LOINC_MOD_STC_CODE_2', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_2', flag: false, editedOrNewFlag: true },
      { cd: 'LOINC_MOD_CODE_3', stcCd: 'LOINC_MOD_STC_CODE_3', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_3', flag: true, editedOrNewFlag: false },
      { cd: 'LOINC_MOD_CODE_4', stcCd: 'LOINC_MOD_STC_CODE_4', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_4', flag: true, editedOrNewFlag: true }
    ];

    roles = [ 'ADMIN' ];

    additionalInfoPermissions = {
      populateInfo: false,
      submitInfo: false
    };

    sessionStorage.setItem('additionalInformationPermissions', JSON.stringify(additionalInfoPermissions));
    sessionStorage.setItem('userRole', JSON.stringify(roles));


    rfaiAdditionalInformationService = new RfaiFormService(null);
    fb = new FormBuilder();
    script = new ScriptService();

    component = new RfaiFormComponent(
      fb,
      rfaiAdditionalInformationService,
      script
    );
  });

  it('component should have been created', () => {
    expect(component).toBeTruthy();
  });

  it('test ngOnInit method', () => {
    spyOn(component, 'getCodes');
    spyOn(component, 'formInit');
    spyOn(component, 'getUserData');

    component.ngOnInit();

    expect(component.getCodes).toHaveBeenCalled();
    expect(component.formInit).toHaveBeenCalled();
    expect(component.getUserData).toHaveBeenCalled();
  });

  it('test ngOnInit method: auto fill form should have been initialized', () => {
    const hccs = spyOn(
      rfaiAdditionalInformationService,
      'getHccscCodes'
    ).and.callFake(() => {
      return Observable.create((observer: Observer<string>) =>
        observer.next('data')
      );
    });

    const loincCode = spyOn(
      rfaiAdditionalInformationService,
      'getLoincCodes'
    ).and.callFake(() => {
      return Observable.create((observer: Observer<string>) =>
        observer.next('data')
      );
    });

    const lioncCodeModifier = spyOn(
      rfaiAdditionalInformationService,
      'getLoincCodeModifiers'
    ).and.callFake(() => {
      return Observable.create((observer: Observer<string>) =>
        observer.next('data')
      );
    });

    component.ngOnInit();
    const formSpy = spyOn(component, 'formInit');

    expect(component.autoFillform).toBeTruthy();
  });

  it('test ngOnInit method: manual entry fill form should have been initialized', () => {
    const hccs = spyOn(
      rfaiAdditionalInformationService,
      'getHccscCodes'
    ).and.callFake(() => {
      return Observable.create((observer: Observer<string>) =>
        observer.next('data')
      );
    });

    const loincCode = spyOn(
      rfaiAdditionalInformationService,
      'getLoincCodes'
    ).and.callFake(() => {
      return Observable.create((observer: Observer<string>) =>
        observer.next('data')
      );
    });

    const lioncCodeModifier = spyOn(
      rfaiAdditionalInformationService,
      'getLoincCodeModifiers'
    ).and.callFake(() => {
      return Observable.create((observer: Observer<string>) =>
        observer.next('data')
      );
    });

    component.ngOnInit();

    const formSpy = spyOn(component, 'formInit');
    expect(component.manualForm).toBeTruthy();
  });

  it('test getCodes method', () => {
    spyOn(rfaiAdditionalInformationService, 'getHccscCodes').and.callFake(() => {
      return new Observable(observer => observer.next(healthCareClaimStatusCodes));
    });

    spyOn(rfaiAdditionalInformationService, 'getLoincCodes').and.callFake(() => {
      return new Observable(observer => observer.next(loincCodes));
    });

    spyOn(rfaiAdditionalInformationService, 'getLoincCodeModifiers').and.callFake(() => {
      return new Observable(observer => observer.next(loincCodeModifiers));
    });

    expect(component.healthCareClaimStatusCodes).toEqual([]);
    expect(component.loincCodes).toEqual([]);
    expect(component.loincCodeModifiers).toEqual([]);

    component.getCodes();

    expect(rfaiAdditionalInformationService.getHccscCodes).toHaveBeenCalled();
    expect(rfaiAdditionalInformationService.getLoincCodes).toHaveBeenCalled();
    expect(rfaiAdditionalInformationService.getLoincCodeModifiers).toHaveBeenCalled();

    expect(component.healthCareClaimStatusCodes).toEqual(healthCareClaimStatusCodes);
    expect(component.loincCodes).toEqual(loincCodes);
    expect(component.loincCodeModifiers).toEqual(loincCodeModifiers);
  });

  it('test getUserData method: all sessionStorage items not present', () => {
    sessionStorage.removeItem('additionalInformationPermissions');
    sessionStorage.removeItem('userRole');

    expect(component.additionalInfoPermissions).toEqual(undefined);
    expect(component.disablePopulateInfo).toEqual(false);
    expect(component.disableSubmitInfo).toEqual(false);
    expect(component.roles).toEqual(undefined);

    component.getUserData();

    expect(component.additionalInfoPermissions).toEqual(undefined);
    expect(component.disablePopulateInfo).toEqual(false);
    expect(component.disableSubmitInfo).toEqual(false);
    expect(component.roles).toEqual(null);
  });

  it('test getUserData method: all sessionStorage items are present', () => {
    expect(component.additionalInfoPermissions).toEqual(undefined);
    expect(component.disablePopulateInfo).toEqual(false);
    expect(component.disableSubmitInfo).toEqual(false);
    expect(component.roles).toEqual(undefined);

    component.getUserData();

    expect(component.additionalInfoPermissions).toEqual(additionalInfoPermissions);
    expect(component.disablePopulateInfo).toEqual(true);
    expect(component.disableSubmitInfo).toEqual(true);
    expect(component.roles).toEqual(roles);
  });

  it('test formInit method: should pass falsy value into healthCareclaimStatusCode and be invalid', () => {
    component.formInit();

    component.autoFillform.get('healthCareClaimStatusCode').setValue('');
    expect(component.autoFillform.get('healthCareClaimStatusCode').invalid).toBeTruthy();

    component.manualForm.get('healthCareClaimStatusCode').setValue('');
    expect(component.manualForm.get('healthCareClaimStatusCode').invalid).toBeTruthy();
  });
  
  it('test formInit method: should pass falsy value into loincCode and be invalid', () => {
    component.formInit();
    
    component.autoFillform.get('loincCode').setValue('');
    expect(component.autoFillform.get('loincCode').invalid).toBeTruthy();

    component.manualForm.get('loincCode').setValue('');
    expect(component.manualForm.get('loincCode').invalid).toBeTruthy();
  });

  it('test formInit method: should pass empty value into loincCodeModifier and still be valid', () => {
    component.formInit();
    
    component.autoFillform.get('loincCodeModifier').setValue('');
    expect(component.autoFillform.get('loincCodeModifier').invalid).toBeFalsy();

    component.manualForm.get('loincCodeModifier').setValue('');
    expect(component.manualForm.get('loincCodeModifier').invalid).toBeFalsy();    
  });

  it('test formInit method: should pass truthy value into healthCareClaimStatusCode and be valid', () => {
    component.formInit();
   
    component.autoFillform.get('healthCareClaimStatusCode').setValue('value');
    expect(component.autoFillform.get('healthCareClaimStatusCode').valid).toBeTruthy();

    component.manualForm.get('healthCareClaimStatusCode').setValue('value');
    expect(component.manualForm.get('healthCareClaimStatusCode').valid).toBeTruthy();    
  });

  it('test formInit method: should pass truthy value into loincCode and be valid', () => {
    component.formInit();
    
    component.autoFillform.get('loincCode').setValue('value');
    expect(component.autoFillform.get('loincCode').valid).toBeTruthy();

    component.manualForm.get('loincCode').setValue('value');
    expect(component.manualForm.get('loincCode').valid).toBeTruthy();    
  });

  it('test formInit method: should pass truthy value into loincCodeModifier and be valid', () => {
    component.formInit();

    component.autoFillform.get('loincCodeModifier').setValue('value');
    expect(component.autoFillform.get('loincCodeModifier').valid).toBeTruthy();

    component.manualForm.get('loincCodeModifier').setValue('value');
    expect(component.manualForm.get('loincCodeModifier').valid).toBeTruthy();
  });

  // ENTIRE FORM

  it('test formInit method: should pass falsy values into validators, form should be invalid', () => {
    component.formInit();
    
    component.autoFillform.get('healthCareClaimStatusCode').setValue('');
    component.autoFillform.get('loincCode').setValue('');
    component.autoFillform.get('loincCodeModifier').setValue('');
    expect(component.autoFillform.invalid).toBeTruthy();

    component.manualForm.get('healthCareClaimStatusCode').setValue('');
    component.manualForm.get('loincCode').setValue('');
    component.manualForm.get('loincCodeModifier').setValue('');
    expect(component.manualForm.invalid).toBeTruthy();    
  });

  it('test formInit method: should pass truthy values into validators, form should be valid', () => {
    component.formInit();
 
    component.autoFillform.get('claimId').setValue('1234');
    component.autoFillform.get('healthCareClaimStatusCode').setValue('value');
    component.autoFillform.get('loincCode').setValue('value');
    component.autoFillform.get('loincCodeModifier').setValue('value');
    component.autoFillform.get('contactInfo').setValue('cindy.lauper@gmail.com');
    component.autoFillform.get('responseDueDateAuto').setValue('11/17/2018');
    expect(component.autoFillform.valid).toBeTruthy();

    component.manualForm.get('healthCareClaimStatusCode').setValue('value');
    component.manualForm.get('loincCode').setValue('value');
    component.manualForm.get('loincCodeModifier').setValue('value');
    component.manualForm.get('contactInfo').setValue('cindy.lauper@gmail.com');
    component.manualForm.get('responseDueDateManual').setValue('11/17/2018');  
    component.manualForm.get('payerName').setValue('Cindy Lauper');
    component.manualForm.get('payerID').setValue('78954678');
    component.manualForm.get('payerClaimControlNumber').setValue('637589004');
    component.manualForm.get('billingProvider').setValue('value');
    component.manualForm.get('providerNpi').setValue('5464789');
    component.manualForm.get('billingProviderTIN').setValue('546478987');
    component.manualForm.get('lastName').setValue('Lauper');
    component.manualForm.get('firstName').setValue('Cindy');
    component.manualForm.get('patientID').setValue('63838495409');
    component.manualForm.get('patientControlNumber').setValue('73849859');
    component.manualForm.get('informationReceiver').setValue('value');
    component.manualForm.get('billType').setValue('value');
    component.manualForm.get('clearingHouseID').setValue('73834949');
    component.manualForm.get('medicalRecordNumber').setValue('63738494');
    component.manualForm.get('level').setValue('value');
    component.manualForm.get('serviceFromDate').setValue('01/01/2018');
    component.manualForm.get('serviceToDate').setValue('11/17/2018');
    expect(component.manualForm.valid).toBeTruthy();
  });

  it('test onSubmitPrepopulated method', () => {
    spyOn(component, 'submitData');

    component.prepopulatedInfo.rfaiLineItemResponse = <LineModel[]> [ 
      { serviceLineId: 1, selected: true },
      { serviceLineId: 2, selected: false },
      { serviceLineId: 3, selected: true },
      { serviceLineId: 4, selected: false }
    ];

    expect(component.prepopulatedInfo.responseDate).toEqual('');
    expect(component.autoFormModel.claimId).toEqual(undefined);

    component.responseDueDateAuto = {
      nativeElement: {
        value: '11/17/2018'
      }
    };

    // let rfaiInfoModel = Object.assign(
    //   {},
    //   component.autoFormModel,
    //   component.prepopulatedInfo.rfaiLineItemResponse.filter(line => line.selected)
    // );
    // rfaiInfoModel.claimId = null;

    component.onSubmitPrepopulated();

    expect(component.submitData).toHaveBeenCalled();
    // expect(component.submitData).toHaveBeenCalledWith(rfaiInfoModel);

    expect(component.prepopulatedInfo.rfaiLineItemResponse.length).toEqual(2);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[0].serviceLineId).toEqual(1);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[0].selected).toEqual(true);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[1].serviceLineId).toEqual(3);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[1].selected).toEqual(true);

    expect(component.prepopulatedInfo.responseDate).toEqual('11/17/2018');
  });   

  it('test onSubmitManual method: pdfInfoModel.level is not equl to "C"', () => {
    component.pdiInfoModel.level = 'D';

    expect(component.pdiInfoModel.serviceFromDate).toEqual('');
    expect(component.pdiInfoModel.serviceToDate).toEqual('');
    expect(component.pdiInfoModel.rfaiLineItemResponse).toEqual([ ]);

    expect(component.pdiInfoModel.responseDate).toEqual('');

    component.responseDueDateManual = {
      nativeElement: {
        value: '11/17/2018'
      }
    };

    component.lines = <LineModel[]> [
      { serviceLineId: 1, selected: false },
      { serviceLineId: 2, selected: true },
      { serviceLineId: 3, selected: false },
      { serviceLineId: 4, selected: true }
    ];

    expect(component.selectedLines).toEqual([ ]);

    spyOn(component, 'submitData');

    component.onSubmitManual();

    expect(component.submitData).toHaveBeenCalled();

    expect(component.pdiInfoModel.serviceFromDate).toEqual('');
    expect(component.pdiInfoModel.serviceToDate).toEqual('');

    expect(component.pdiInfoModel.responseDate).toEqual('11/17/2018');

    expect(component.selectedLines.length).toEqual(2);
    expect(component.selectedLines[0].serviceLineId).toEqual(2);
    expect(component.selectedLines[0].selected).toEqual(true);
    expect(component.selectedLines[1].serviceLineId).toEqual(4);
    expect(component.selectedLines[1].selected).toEqual(true);

    expect(component.pdiInfoModel.rfaiLineItemResponse.length).toEqual(2);
    expect(component.pdiInfoModel.rfaiLineItemResponse[0].serviceLineId).toEqual(2);
    expect(component.pdiInfoModel.rfaiLineItemResponse[0].selected).toEqual(true);
    expect(component.pdiInfoModel.rfaiLineItemResponse[1].serviceLineId).toEqual(4);
    expect(component.pdiInfoModel.rfaiLineItemResponse[1].selected).toEqual(true);    
  });

  it('test submitData method: returned data has "result" property', () => {
    let rfaiInfoModel: RfaiInfoModel = new RfaiInfoModel();

    let returnedData = { result: 'Success'};
    spyOn(rfaiAdditionalInformationService, 'submitRfai').and.callFake(() => {
      return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'onResetManual');
    spyOn(component, 'onResetPrepopulated');

    expect(component.successMsg).toEqual('');
    expect(component.errorMsg).toEqual('');
    expect(component.successStatus).toEqual(false);
    expect(component.errorStatus).toEqual(false);

    component.submitData(rfaiInfoModel);

    expect(rfaiAdditionalInformationService.submitRfai).toHaveBeenCalled();
    expect(rfaiAdditionalInformationService.submitRfai).toHaveBeenCalledWith(rfaiInfoModel);
    expect(component.onResetManual).toHaveBeenCalled();
    expect(component.onResetPrepopulated).toHaveBeenCalled();

    expect(component.successMsg).toEqual('277 RFAI successfully submitted.');
    expect(component.errorMsg).toEqual('');
    expect(component.successStatus).toEqual(true);
    expect(component.errorStatus).toEqual(false);
  }); 

  it('test submitData method: returned data has NO "result" property', () => {
    let rfaiInfoModel: RfaiInfoModel = new RfaiInfoModel();

    let returnedData = { noResult: 'Success'};
    spyOn(rfaiAdditionalInformationService, 'submitRfai').and.callFake(() => {
      return new Observable(observer => observer.next(returnedData));
    });

    expect(component.successMsg).toEqual('');
    expect(component.errorMsg).toEqual('');
    expect(component.successStatus).toEqual(false);
    expect(component.errorStatus).toEqual(false);

    component.submitData(rfaiInfoModel);

    expect(rfaiAdditionalInformationService.submitRfai).toHaveBeenCalled();
    expect(rfaiAdditionalInformationService.submitRfai).toHaveBeenCalledWith(rfaiInfoModel);

    expect(component.successMsg).toEqual('');
    expect(component.errorMsg).toEqual('An error occured during 277 RFAI submission.');
    expect(component.successStatus).toEqual(false);
    expect(component.errorStatus).toEqual(true);
  }); 

  it('test resetCommon method', () => {
    expect(component.lines).toEqual([ ]);

    component.errorStatus = true;
    component.successStatus = true;
    component.errorMsg = 'ZZZZZ';
    component.successMsg = 'XXXX';
    component.pdiInfoModel.level = 'X';

    spyOn(component, 'formInit');

    component.resetCommon();

    expect(component.lines.length).toEqual(1);
    expect(component.errorStatus).toEqual(false);
    expect(component.successStatus).toEqual(false);
    expect(component.errorMsg).toEqual('');
    expect(component.successMsg).toEqual('');
    expect(component.pdiInfoModel.level).toEqual('C');

    expect(component.formInit).toHaveBeenCalled();
  });

  it('test onResetPrepopulated method', () => {
    component.populateInfoClicked = true;
    component.autoFormModel = null;
    spyOn(component, 'resetCommon');

    component.onResetPrepopulated();

    expect(component.populateInfoClicked).toEqual(false);
    expect(component.autoFormModel).toBeDefined();
    expect(component.autoFormModel).not.toBeNull();
    expect(component.resetCommon).toHaveBeenCalled();
  });

  it('test onResetManual method', () => {
    component.pdiInfoModel = null;
    component.manualFormModel = null;

    component.serviceFromDate = {
      nativeElement: {
        value: '10/17/2018'
      }
    };

    component.serviceToDate = {
      nativeElement: { 
        value: '11/17/2018'
      }
    };

    spyOn(component, 'resetCommon');

    component.onResetManual();

    expect(component.pdiInfoModel).not.toBeNull();
    expect(component.pdiInfoModel).toBeDefined();
    expect(component.manualFormModel).not.toBeNull();
    expect(component.manualFormModel).toBeDefined();
    expect(component.serviceFromDate.nativeElement.value).toEqual('');
    expect(component.serviceToDate.nativeElement.value).toEqual('');
    expect(component.resetCommon).toHaveBeenCalled();
  });

  it('test addNewInitialLine method: lines length is equal to 0', () => {
    expect(component.lines.length).toEqual(0);
    spyOn(component, 'addNewLine');

    component.addNewInitialLine();

    expect(component.addNewLine).toHaveBeenCalled();
  });

  it('test addNewInitialLine method: lines length is NOT equal to 0', () => {
    component.lines = [ new LineModel() ];
    spyOn(component, 'addNewLine');

    component.addNewInitialLine();

    expect(component.addNewLine).not.toHaveBeenCalled();
  });

  it('test addNewLine method', () => {
    component.enableAddLine = true;
    expect(component.lines.length).toEqual(0);

    component.addNewLine();

    expect(component.enableAddLine).toBe(false);
    expect(component.lines.length).toBe(1);
  });

  it('test populateInfo method: "claimId" value not present', () => {
    expect(component.errorMsg).toEqual('');
    expect(component.errorStatus).toEqual(false);

    expect(component.populateInfoClicked).toEqual(false);
    expect(component.prepopulatedInfo).not.toEqual(undefined);
    expect(component.autoFormModel.claimId).toEqual(undefined);

    spyOn(rfaiAdditionalInformationService, 'populateInfo');

    component.selectAllLines = {
      nativeElement: {
        checked: true
      }
    };

    component.populateInfo();

    expect(rfaiAdditionalInformationService.populateInfo).not.toHaveBeenCalled();

    expect(component.errorMsg).toEqual('Enter PDI or Claim ID.');
    expect(component.errorStatus).toEqual(true);

    expect(component.populateInfoClicked).toEqual(false);
    expect(component.prepopulatedInfo).not.toEqual(undefined);
    expect(component.prepopulatedInfo.payerName).toEqual('');

    expect(component.selectAllLines.nativeElement.checked).toEqual(true);
  });

  it('test populateInfo method: "claimId" value present and returned data has no errorCode', () => {
    expect(component.errorMsg).toEqual('');
    expect(component.errorStatus).toEqual(false);

    expect(component.populateInfoClicked).toEqual(false);
    expect(component.prepopulatedInfo).not.toEqual(undefined);

    expect(component.autoFormModel.claimId).toEqual(undefined);
    component.autoFormModel.claimId = 123;

    let returnedData = new PDIInfoModel();
    returnedData.payerName = 'Cindy Lauper';
    spyOn(rfaiAdditionalInformationService, 'populateInfo').and.callFake(() => {
      return new Observable(observer => observer.next(returnedData));
    });

    component.selectAllLines = {
      nativeElement: {
        checked: true
      }
    };

    component.populateInfo();

    expect(rfaiAdditionalInformationService.populateInfo).toHaveBeenCalled();
    expect(rfaiAdditionalInformationService.populateInfo).toHaveBeenCalledWith(123);

    expect(component.errorMsg).toEqual('');
    expect(component.errorStatus).toEqual(false);

    expect(component.populateInfoClicked).toEqual(true);
    expect(component.prepopulatedInfo).toEqual(returnedData);

    expect(component.selectAllLines.nativeElement.checked).toEqual(false);
  });

  it('test populateInfo method: "claimId" value present and returned data has errorCode', () => {
    expect(component.errorMsg).toEqual('');
    expect(component.errorStatus).toEqual(false);

    expect(component.populateInfoClicked).toEqual(false);
    expect(component.prepopulatedInfo).not.toEqual(undefined);

    expect(component.autoFormModel.claimId).toEqual(undefined);
    component.autoFormModel.claimId = 123;

    let returnedData = {
      errorCode: 'ERROR_CODE',
      message: 'Error Message'
    }
    spyOn(rfaiAdditionalInformationService, 'populateInfo').and.callFake(() => {
      return new Observable(observer => observer.next(returnedData));
    });

    component.selectAllLines = {
      nativeElement: {
        checked: true
      }
    };

    component.populateInfo();

    expect(rfaiAdditionalInformationService.populateInfo).toHaveBeenCalled();
    expect(rfaiAdditionalInformationService.populateInfo).toHaveBeenCalledWith(123);

    expect(component.errorMsg).toEqual(returnedData.message);
    expect(component.errorStatus).toEqual(true);

    expect(component.populateInfoClicked).toEqual(true);
    expect(component.prepopulatedInfo).not.toEqual(undefined);
    expect(component.prepopulatedInfo.payerName).toEqual('');

    expect(component.selectAllLines.nativeElement.checked).toEqual(false);
  });

  it('test selectAll method: input parameter is boolean true', () => {
    component.prepopulatedInfo.rfaiLineItemResponse = <LineModel[]> [
      { serviceLineId: 1, selected: false },
      { serviceLineId: 2, selected: false },
      { serviceLineId: 3, selected: false }
    ];

    component.selectAll(true);

    expect(component.prepopulatedInfo.rfaiLineItemResponse.length).toEqual(3);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[0].serviceLineId).toEqual(1);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[0].selected).toEqual(true);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[1].serviceLineId).toEqual(2);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[1].selected).toEqual(true);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[2].serviceLineId).toEqual(3);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[2].selected).toEqual(true);
  });

  it('test selectAll method: input parameter is boolean false', () => {
    component.prepopulatedInfo.rfaiLineItemResponse = <LineModel[]> [
      { serviceLineId: 1, selected: true },
      { serviceLineId: 2, selected: true },
      { serviceLineId: 3, selected: true }
    ];

    component.selectAll(false);

    expect(component.prepopulatedInfo.rfaiLineItemResponse.length).toEqual(3);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[0].serviceLineId).toEqual(1);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[0].selected).toEqual(false);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[1].serviceLineId).toEqual(2);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[1].selected).toEqual(false);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[2].serviceLineId).toEqual(3);
    expect(component.prepopulatedInfo.rfaiLineItemResponse[2].selected).toEqual(false);
  });

  it('test saveLine method: no lines present', () => {
    expect(component.lines.length).toEqual(0);
    expect(component.enableAddLine).toEqual(false);

    component.saveLine();

    expect(component.lines.length).toEqual(0);
    expect(component.enableAddLine).toEqual(false);
  });  

  it('test saveLine method: all lines have a serviceLineId', () => {
    component.lines = <LineModel[]> [
      { serviceLineId: 1, selected: false, chargeAmount: 100 },
      { serviceLineId: 2, selected: false, chargeAmount: 200 },
      { serviceLineId: 3, selected: false, chargeAmount: 300 }
    ];

    expect(component.enableAddLine).toEqual(false);

    component.saveLine();

    expect(component.lines.length).toEqual(3);
    
    expect(component.lines[0].serviceLineId).toEqual(1);
    expect(component.lines[0].selected).toEqual(true);
    expect(component.lines[0].chargeAmount).toEqual(100);

    expect(component.lines[1].serviceLineId).toEqual(2);
    expect(component.lines[1].selected).toEqual(true);
    expect(component.lines[1].chargeAmount).toEqual(200);

    expect(component.lines[2].serviceLineId).toEqual(3);
    expect(component.lines[2].selected).toEqual(true);
    expect(component.lines[2].chargeAmount).toEqual(300);

    expect(component.enableAddLine).toEqual(true);
  });  

  it('test saveLine method: not all lines have a serviceLineId', () => {
    component.lines = <LineModel[]> [
      { serviceLineId: 1, selected: false, chargeAmount: 100 },
      { selected: false, chargeAmount: 200 },
      { serviceLineId: 3, selected: false, chargeAmount: 300 },
      { selected: false, chargeAmount: 400 },
    ];

    expect(component.enableAddLine).toEqual(false);

    component.saveLine();

    expect(component.lines.length).toEqual(4);
    
    expect(component.lines[0].serviceLineId).toEqual(1);
    expect(component.lines[0].selected).toEqual(true);
    expect(component.lines[0].chargeAmount).toEqual(100);

    expect(component.lines[1].serviceLineId).toEqual(undefined);
    expect(component.lines[1].selected).toEqual(false);
    expect(component.lines[1].chargeAmount).toEqual(200);

    expect(component.lines[2].serviceLineId).toEqual(3);
    expect(component.lines[2].selected).toEqual(true);
    expect(component.lines[2].chargeAmount).toEqual(300);

    expect(component.lines[3].serviceLineId).toEqual(undefined);
    expect(component.lines[3].selected).toEqual(false);
    expect(component.lines[3].chargeAmount).toEqual(400);    

    expect(component.enableAddLine).toEqual(false);
  });  

  it('test enableAddLineButton method', () => {
    component.enableAddLine = false;
    component.enableAddLineButton();
    expect(component.enableAddLine).toBe(true);
  });

});
