import { Component, OnInit } from '@angular/core';
import { RfaiAdminService } from './rfai-admin.service';
import { CodeModel, EmailModel } from './rfai-admin.model';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';
import { initialValueChanged } from './initial-value-changed.directive';

@Component({
  selector: 'app-rfai-admin',
  templateUrl: './rfai-admin.component.html',
  styleUrls: [
    './../rfai-form/rfai-form.component.scss',
    './rfai-admin.component.scss'
  ]
})
export class RfaiAdminComponent implements OnInit {
  modalClosed = true;
  constructor(
    private fb: FormBuilder,
    private rfaiAdminService: RfaiAdminService
  ) {}
  currentCodeFullName: string;
  healthCodes: CodeModel[] = [];
  loincCodes: CodeModel[] = [];
  loincModCodes: CodeModel[] = [];
  modalErrorCode = false;
  modalErrorDesc = false;
  addClicked = false;
  successMsg = '';
  errorMsg = '';
  errorStatus = false;
  successStatus = false;
  pendingEmailForm: FormGroup;
  submissionErrorEmailForm: FormGroup;
  abondonedEmailForm: FormGroup;
  closedEmailForm: FormGroup;
  manuallyClosedForm: FormGroup;
  newCode: CodeModel = new CodeModel();
  tempCodeObj: CodeModel = new CodeModel();
  tempNewCodeObj: CodeModel = new CodeModel();
  pendingEmail: EmailModel = new EmailModel();
  submissionErrorEmail: EmailModel = new EmailModel();
  abondonedEmail: EmailModel = new EmailModel();
  closedEmail: EmailModel = new EmailModel();
  manuallyClosedEmail: EmailModel = new EmailModel();
  initalObject: EmailModel = new EmailModel();

  // added to test clipboard
  isCopied1=false;
  copyText='';

  ngOnInit() {
    this.getCodes();
    this.getEmailData();
    this.formInit();
  }

  formInit() {
    this.pendingEmailForm = this.fb.group({
      pendingEmailHeading: this.pendingEmail.heading,
      pendingEmailMessage: [
        this.pendingEmail.message
        // initialValueChanged(this.pendingEmail.message)
      ]
    });

    this.submissionErrorEmailForm = this.fb.group({
      submissionErrorEmailHeading: this.submissionErrorEmail.heading,
      submissionErrorEmailMessage: this.submissionErrorEmail.message
    });
    this.abondonedEmailForm = this.fb.group({
      abondonedEmailHeading: this.abondonedEmail.heading,
      abondonedEmailMessage: this.abondonedEmail.message
    });
    this.closedEmailForm = this.fb.group({
      closedEmailHeading: this.closedEmail.heading,
      closedEmailMessage: this.closedEmail.message
    });
    this.manuallyClosedForm = this.fb.group({
      manuallyClosedEmailHeading: this.manuallyClosedEmail.heading,
      manuallyClosedEmailMessage: this.manuallyClosedEmail.message
    });
  }

  getCodes() {
    this.rfaiAdminService.getHccscCode().subscribe(data => {
      this.healthCodes = data;
    });

    this.rfaiAdminService.getLoincCodes().subscribe(data => {
      this.loincCodes = data;
    });

    this.rfaiAdminService.getLoincModifierCodes().subscribe(data => {
      this.loincModCodes = data;
    });
  }

  getEmailData() {
    this.rfaiAdminService.getEmailTemplates().subscribe(data => {
      if (data.emailTemplates) {
        data.emailTemplates.forEach(template => {
          switch (template.templateName) {
            case 'manuallyClosedTemplate':
              this.manuallyClosedEmail = template;
              // console.log('manuallyclosed', this.manuallyClosedEmail);
              this.setCustomValidator(
                this.manuallyClosedForm,
                'manuallyClosedEmailHeading',
                'manuallyClosedEmailMessage',
                { ...template }
              );
              break;
            case 'abandonedTemplate':
              this.abondonedEmail = template;
              this.setCustomValidator(
                this.abondonedEmailForm,
                'abondonedEmailHeading',
                'abondonedEmailMessage',
                { ...template }
              );
              break;
            case 'completeOrClosedTemplate':
              this.closedEmail = template;
              this.setCustomValidator(
                this.closedEmailForm,
                'closedEmailHeading',
                'closedEmailMessage',
                { ...template }
              );
              break;
            case 'errorTemplate':
              this.submissionErrorEmail = template;
              this.setCustomValidator(
                this.submissionErrorEmailForm,
                'submissionErrorEmailHeading',
                'submissionErrorEmailMessage',
                { ...template }
              );
              break;
            case 'pendingTemplate':
              this.pendingEmail = template;
              this.setCustomValidator(
                this.pendingEmailForm,
                'pendingEmailHeading',
                'pendingEmailMessage',
                { ...template }
              );
              break;
            default:
              return;
          }
        });
      }
    });
  }

  setCustomValidator(
    form: FormGroup,
    headingName: string,
    messageName: string,
    initialObject: EmailModel
  ) {
    form.setValidators([
      initialValueChanged(initialObject, headingName, messageName)
    ]);
    form.updateValueAndValidity();
  }

  addCode(currentCodeFullName: string) {
    this.modalErrorCode = false;
    this.modalErrorDesc = false;
    if (this.tempNewCodeObj.stcCd && this.tempNewCodeObj.stcCdDesc) {
      this.tempNewCodeObj.editedOrNewFlag = true;
      switch (currentCodeFullName) {
        case 'Health Care Claim Status Category Code': {
          this.healthCodes.push(this.tempNewCodeObj);
          break;
        }

        case 'LOINC Code': {
          this.loincCodes.push(this.tempNewCodeObj);
          break;
        }

        case 'LOINC Modifier Code': {
          this.loincModCodes.push(this.tempNewCodeObj);
          break;
        }
      }
      this.newCode = new CodeModel();
      this.tempNewCodeObj = new CodeModel();
      this.modalClosed = true;
      this.modalErrorCode = false;
      this.modalErrorDesc = false;
    } else {
      this.tempNewCodeObj.stcCd
        ? (this.modalErrorDesc = true)
        : (this.modalErrorCode = true);
      this.tempNewCodeObj.stcCdDesc
        ? (this.modalErrorCode = true)
        : (this.modalErrorDesc = true);
    }
  }

  editCode(
    newCode: CodeModel,
    oldCode: CodeModel,
    currentCodeFullName: string
  ) {
    this.modalErrorDesc = false;
    this.modalErrorCode = false;

    if (newCode.stcCd && newCode.stcCdDesc) {
      this.modalClosed = true;
      newCode.editedOrNewFlag = true;
      switch (currentCodeFullName) {
        case 'Health Care Claim Status Category Code': {
          const indexElement = this.healthCodes.findIndex(
            code => code.stcCd === oldCode.stcCd
          );
          if (indexElement > -1) {
            this.healthCodes[indexElement] = newCode;
          }
          break;
        }

        case 'LOINC Code': {
          const indexElement = this.loincCodes.findIndex(
            code => code.stcCd === oldCode.stcCd
          );
          if (indexElement > -1) {
            this.loincCodes[indexElement] = newCode;
          }
          break;
        }

        case 'LOINC Modifier Code': {
          const indexElement = this.loincModCodes.findIndex(
            code => code.stcCd === oldCode.stcCd
          );
          if (indexElement > -1) {
            this.loincModCodes[indexElement] = newCode;
          }
          break;
        }
      }
    } else {
      newCode.stcCd
        ? (this.modalErrorDesc = true)
        : (this.modalErrorCode = true);

      newCode.stcCdDesc
        ? (this.modalErrorCode = true)
        : (this.modalErrorDesc = true);
    }
  }

  openAddModal(currentCodeName: string) {
    this.addClicked = true;
    if (currentCodeName === 'HCCS') {
      this.currentCodeFullName = 'Health Care Claim Status Category Code';
      this.tempNewCodeObj = this.newCode;
    }

    if (currentCodeName === 'LOINC') {
      this.currentCodeFullName = 'LOINC Code';
      this.tempNewCodeObj = this.newCode;
    }

    if (currentCodeName === 'LOINCMod') {
      this.currentCodeFullName = 'LOINC Modifier Code';
      this.tempNewCodeObj = this.newCode;
    }
    this.onModalOpen();
  }

  openEditModal(codeObj: CodeModel, currentCodeName: string) {
    this.addClicked = false;
    this.modalErrorCode = false;
    this.modalErrorDesc = false;
    this.onModalOpen();
    this.tempNewCodeObj = { ...codeObj };
    this.tempCodeObj = { ...codeObj };
    if (currentCodeName === 'HCCS') {
      this.currentCodeFullName = 'Health Care Claim Status Category Code';
    }

    if (currentCodeName === 'LOINC') {
      this.currentCodeFullName = 'LOINC Code';
    }

    if (currentCodeName === 'LOINCMod') {
      this.currentCodeFullName = 'LOINC Modifier Code';
    }
  }

  onModalOpen() {
    this.modalClosed = false;
    this.tempNewCodeObj = new CodeModel();
  }

  onModalClose(bool: boolean) {
    this.modalClosed = bool;
    this.addClicked = false;
    this.modalErrorCode = false;
    this.modalErrorDesc = false;
  }

  submit() {
    const requestBody = {
      hccList: this.healthCodes.filter(code => code.editedOrNewFlag === true),
      loincList: this.loincCodes.filter(code => code.editedOrNewFlag === true),
      loincModList: this.loincModCodes.filter(
        code => code.editedOrNewFlag === true
      )
    };
    this.rfaiAdminService.postCodes(requestBody).subscribe(data => {
      // console.log('Adding Codes', data);
      if (data.result) {
        this.successMsg = 'Updated Codes successfully.';
        this.successStatus = true;
        this.errorStatus = false;
        this.getCodes();
        setTimeout(() => {
          this.successStatus = false;
        }, 8000);
      } else {
        this.errorStatus = true;
        this.successStatus = false;
        this.errorMsg = 'An error occured updating Codes.';
        setTimeout(() => {
          this.errorStatus = false;
        }, 8000);
      }
    });
  }

  cancel() {
    this.modalClosed = true;
    this.addClicked = false;
    this.getCodes();
  }

  setEditOrNewFlag(code: CodeModel) {
    code.editedOrNewFlag = true;
  }

  onSubmitEmail(email: EmailModel) {
    email.modifiedBy = sessionStorage.getItem('userName');
    console.log('email submit', email);
    this.rfaiAdminService.submitEmailTemplate(email).subscribe(data => {
      if (data.result) {
        this.successMsg = 'Updated Email Template successfully.';
        this.successStatus = true;
        this.errorStatus = false;
        this.getCodes();
        setTimeout(() => {
          this.successStatus = false;
        }, 8000);
      } else {
        this.errorStatus = true;
        this.successStatus = false;
        this.errorMsg = 'An error occured updating Email Templates.';
        setTimeout(() => {
          this.errorStatus = false;
        }, 8000);
      }
    });
  }

  cancelEmailTemplates(emailType: string) {
    this.rfaiAdminService.getEmailTemplates().subscribe(data => {
      if (data.emailTemplates) {
        data.emailTemplates.forEach(template => {
          switch (template.templateName) {
            case 'manuallyClosedTemplate':
              if (emailType === 'manuallyClosedTemplate') {
                this.manuallyClosedEmail = template;
              }

              // console.log('manuallyclosed', this.manuallyClosedEmail);
              break;
            case 'abandonedTemplate':
              if (emailType === 'abandonedTemplate') {
                this.abondonedEmail = template;
              }
              break;
            case 'completeOrClosedTemplate':
              if (emailType === 'completeOrClosedTemplate') {
                this.closedEmail = template;
              }
              break;
            case 'errorTemplate':
              if (emailType === 'errorTemplate') {
                this.submissionErrorEmail = template;
              }
              break;
            case 'pendingTemplate':
              if (emailType === 'pendingTemplate') {
                this.pendingEmail = template;
              }
              break;
            default:
              return;
          }
        });
      }
    });
  }

}
