import { FormBuilder, Validators, FormGroup } from '@angular/forms';
import { RfaiAdminService } from './rfai-admin.service';
import { CodeModel, EmailModel } from './rfai-admin.model';

import { Observable } from 'rxjs/Observable';
import { Observer } from 'rxjs/Observer';

import { RfaiAdminComponent } from './rfai-admin.component';

describe('RfaiAdminComponent', () => {
  let fb: FormBuilder;
  let rfaiAdminService: RfaiAdminService;
  let component: RfaiAdminComponent;
  let healthCareClaimStatusCodes: CodeModel[];
  let loincCodes: CodeModel[];
  let loincCodeModifiers: CodeModel[];

  beforeEach(() => {
    healthCareClaimStatusCodes = [
        { cd: 'HEALTH_CODE_1', stcCd: 'HEALTH_STC_CODE_1', stcCdDesc: 'HEALTH_STC_CODE_DESC_1', flag: false, editedOrNewFlag: false },
        { cd: 'HEALTH_CODE_2', stcCd: 'HEALTH_STC_CODE_2', stcCdDesc: 'HEALTH_STC_CODE_DESC_2', flag: false, editedOrNewFlag: true },
        { cd: 'HEALTH_CODE_3', stcCd: 'HEALTH_STC_CODE_3', stcCdDesc: 'HEALTH_STC_CODE_DESC_3', flag: true, editedOrNewFlag: false },
        { cd: 'HEALTH_CODE_4', stcCd: 'HEALTH_STC_CODE_4', stcCdDesc: 'HEALTH_STC_CODE_DESC_4', flag: true, editedOrNewFlag: true }
    ];
  
    loincCodes = [
        { cd: 'LOINC_CODE_1', stcCd: 'LOINC_STC_CODE_1', stcCdDesc: 'LOINC_STC_CODE_DESC_1', flag: false, editedOrNewFlag: false },
        { cd: 'LOINC_CODE_2', stcCd: 'LOINC_STC_CODE_2', stcCdDesc: 'LOINC_STC_CODE_DESC_2', flag: false, editedOrNewFlag: true },
        { cd: 'LOINC_CODE_3', stcCd: 'LOINC_STC_CODE_3', stcCdDesc: 'LOINC_STC_CODE_DESC_3', flag: true, editedOrNewFlag: false },
        { cd: 'LOINC_CODE_4', stcCd: 'LOINC_STC_CODE_4', stcCdDesc: 'LOINC_STC_CODE_DESC_4', flag: true, editedOrNewFlag: true }
    ];
  
    loincCodeModifiers = [
        { cd: 'LOINC_MOD_CODE_1', stcCd: 'LOINC_MOD_STC_CODE_1', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_1', flag: false, editedOrNewFlag: false },
        { cd: 'LOINC_MOD_CODE_2', stcCd: 'LOINC_MOD_STC_CODE_2', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_2', flag: false, editedOrNewFlag: true },
        { cd: 'LOINC_MOD_CODE_3', stcCd: 'LOINC_MOD_STC_CODE_3', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_3', flag: true, editedOrNewFlag: false },
        { cd: 'LOINC_MOD_CODE_4', stcCd: 'LOINC_MOD_STC_CODE_4', stcCdDesc: 'LOINC_MOD_STC_CODE_DESC_4', flag: true, editedOrNewFlag: true }
    ];

    fb = new FormBuilder();
    rfaiAdminService = new RfaiAdminService(null);

    component = new RfaiAdminComponent(fb, rfaiAdminService);
  });

  it('should be created', () => {
    expect(component).toBeTruthy();
  });

  it('test ngOnInit method', () => {
    spyOn(component, 'getCodes');
    spyOn(component, 'getEmailData');
    spyOn(component, 'formInit');

    component.ngOnInit();

    expect(component.getCodes).toHaveBeenCalled();
    expect(component.getEmailData).toHaveBeenCalled();
    expect(component.formInit).toHaveBeenCalled();
  });

  it('test formInit method', () => {
    expect(component.pendingEmailForm).toEqual(undefined);
    expect(component.submissionErrorEmailForm).toEqual(undefined);
    expect(component.abondonedEmailForm).toEqual(undefined);
    expect(component.closedEmailForm).toEqual(undefined);
    expect(component.manuallyClosedForm).toEqual(undefined);

    component.formInit();

    expect(component.pendingEmailForm).toBeDefined();
    expect(component.pendingEmailForm).not.toBeNull();

    expect(component.submissionErrorEmailForm).toBeDefined();
    expect(component.submissionErrorEmailForm).not.toBeNull();

    expect(component.abondonedEmailForm).toBeDefined();
    expect(component.abondonedEmailForm).not.toBeNull();

    expect(component.closedEmailForm).toBeDefined();
    expect(component.closedEmailForm).not.toBeNull();

    expect(component.manuallyClosedForm).toBeDefined();
    expect(component.manuallyClosedForm).not.toBeNull();
  });

  it('test getCodes method', () => {
    spyOn(rfaiAdminService, 'getHccscCode').and.callFake(() => {
        return new Observable(observer => observer.next(healthCareClaimStatusCodes));
    });

    spyOn(rfaiAdminService, 'getLoincCodes').and.callFake(() => {
        return new Observable(observer => observer.next(loincCodes));
    });

    spyOn(rfaiAdminService, 'getLoincModifierCodes').and.callFake(() => {
        return new Observable(observer => observer.next(loincCodeModifiers));
    });

    expect(component.healthCodes).toEqual([]);
    expect(component.loincCodes).toEqual([]);
    expect(component.loincModCodes).toEqual([]);

    component.getCodes();

    expect(rfaiAdminService.getHccscCode).toHaveBeenCalled();
    expect(rfaiAdminService.getLoincCodes).toHaveBeenCalled();
    expect(rfaiAdminService.getLoincModifierCodes).toHaveBeenCalled();

    expect(component.healthCodes).toEqual(healthCareClaimStatusCodes);
    expect(component.loincCodes).toEqual(loincCodes);
    expect(component.loincModCodes).toEqual(loincCodeModifiers);
  });

  it('test getEmailData method: returned data has no emailTemplates attribute', () => {
    let returnedData = { };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).not.toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with empty array', () => {
    let returnedData = { emailTemplates: [ ] };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).not.toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having template element whose name is not any recognized template name', () => {
    let returnedData = { 
        emailTemplates: [ 
            { templateName: 'DUMMY' }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).not.toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having template element whose name is "manuallyClosedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(1);

    expect(component.manuallyClosedEmail.heading).toEqual('manuallyClosedTemplate_heading');
    expect(component.manuallyClosedEmail.message).toEqual('manuallyClosedTemplate_message');
    expect(component.manuallyClosedEmail.templateName).toEqual('manuallyClosedTemplate');
    expect(component.manuallyClosedEmail.templateType).toEqual('manuallyClosedTemplate_templateType');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('manuallyClosedTemplate_modifiedBy');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having template element whose name is "abandonedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(1);

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('abandonedTemplate_heading');
    expect(component.abondonedEmail.message).toEqual('abandonedTemplate_message');
    expect(component.abondonedEmail.templateName).toEqual('abandonedTemplate');
    expect(component.abondonedEmail.templateType).toEqual('abandonedTemplate_templateType');
    expect(component.abondonedEmail.modifiedBy).toEqual('abandonedTemplate_modifiedBy');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having template element whose name is "completeOrClosedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(1);

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('completeOrClosedTemplate_heading');
    expect(component.closedEmail.message).toEqual('completeOrClosedTemplate_message');
    expect(component.closedEmail.templateName).toEqual('completeOrClosedTemplate');
    expect(component.closedEmail.templateType).toEqual('completeOrClosedTemplate_templateType');
    expect(component.closedEmail.modifiedBy).toEqual('completeOrClosedTemplate_modifiedBy');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having template element whose name is "errorTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(1);

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('errorTemplate_heading');
    expect(component.submissionErrorEmail.message).toEqual('errorTemplate_message');
    expect(component.submissionErrorEmail.templateName).toEqual('errorTemplate');
    expect(component.submissionErrorEmail.templateType).toEqual('errorTemplate_templateType');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('errorTemplate_modifiedBy');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having template element whose name is "pendingTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(1);

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('pendingTemplate_heading');
    expect(component.pendingEmail.message).toEqual('pendingTemplate_message');
    expect(component.pendingEmail.templateName).toEqual('pendingTemplate');
    expect(component.pendingEmail.templateType).toEqual('pendingTemplate_templateType');
    expect(component.pendingEmail.modifiedBy).toEqual('pendingTemplate_modifiedBy'); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having all the various types of templates', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(5);

    expect(component.manuallyClosedEmail.heading).toEqual('manuallyClosedTemplate_heading');
    expect(component.manuallyClosedEmail.message).toEqual('manuallyClosedTemplate_message');
    expect(component.manuallyClosedEmail.templateName).toEqual('manuallyClosedTemplate');
    expect(component.manuallyClosedEmail.templateType).toEqual('manuallyClosedTemplate_templateType');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('manuallyClosedTemplate_modifiedBy');

    expect(component.abondonedEmail.heading).toEqual('abandonedTemplate_heading');
    expect(component.abondonedEmail.message).toEqual('abandonedTemplate_message');
    expect(component.abondonedEmail.templateName).toEqual('abandonedTemplate');
    expect(component.abondonedEmail.templateType).toEqual('abandonedTemplate_templateType');
    expect(component.abondonedEmail.modifiedBy).toEqual('abandonedTemplate_modifiedBy');

    expect(component.closedEmail.heading).toEqual('completeOrClosedTemplate_heading');
    expect(component.closedEmail.message).toEqual('completeOrClosedTemplate_message');
    expect(component.closedEmail.templateName).toEqual('completeOrClosedTemplate');
    expect(component.closedEmail.templateType).toEqual('completeOrClosedTemplate_templateType');
    expect(component.closedEmail.modifiedBy).toEqual('completeOrClosedTemplate_modifiedBy');

    expect(component.submissionErrorEmail.heading).toEqual('errorTemplate_heading');
    expect(component.submissionErrorEmail.message).toEqual('errorTemplate_message');
    expect(component.submissionErrorEmail.templateName).toEqual('errorTemplate');
    expect(component.submissionErrorEmail.templateType).toEqual('errorTemplate_templateType');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('errorTemplate_modifiedBy');

    expect(component.pendingEmail.heading).toEqual('pendingTemplate_heading');
    expect(component.pendingEmail.message).toEqual('pendingTemplate_message');
    expect(component.pendingEmail.templateName).toEqual('pendingTemplate');
    expect(component.pendingEmail.templateType).toEqual('pendingTemplate_templateType');
    expect(component.pendingEmail.modifiedBy).toEqual('pendingTemplate_modifiedBy'); 
  });

  it('test getEmailData method: returned data has emailTemplates attribute with array having all the various types of templates but first template type is not a known type', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'dummy',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    spyOn(component, 'setCustomValidator');

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    
    component.getEmailData();

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalled();
    expect(component.setCustomValidator).toHaveBeenCalledTimes(5);

    expect(component.manuallyClosedEmail.heading).toEqual('manuallyClosedTemplate_heading');
    expect(component.manuallyClosedEmail.message).toEqual('manuallyClosedTemplate_message');
    expect(component.manuallyClosedEmail.templateName).toEqual('manuallyClosedTemplate');
    expect(component.manuallyClosedEmail.templateType).toEqual('manuallyClosedTemplate_templateType');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('manuallyClosedTemplate_modifiedBy');

    expect(component.abondonedEmail.heading).toEqual('abandonedTemplate_heading');
    expect(component.abondonedEmail.message).toEqual('abandonedTemplate_message');
    expect(component.abondonedEmail.templateName).toEqual('abandonedTemplate');
    expect(component.abondonedEmail.templateType).toEqual('abandonedTemplate_templateType');
    expect(component.abondonedEmail.modifiedBy).toEqual('abandonedTemplate_modifiedBy');

    expect(component.closedEmail.heading).toEqual('completeOrClosedTemplate_heading');
    expect(component.closedEmail.message).toEqual('completeOrClosedTemplate_message');
    expect(component.closedEmail.templateName).toEqual('completeOrClosedTemplate');
    expect(component.closedEmail.templateType).toEqual('completeOrClosedTemplate_templateType');
    expect(component.closedEmail.modifiedBy).toEqual('completeOrClosedTemplate_modifiedBy');

    expect(component.submissionErrorEmail.heading).toEqual('errorTemplate_heading');
    expect(component.submissionErrorEmail.message).toEqual('errorTemplate_message');
    expect(component.submissionErrorEmail.templateName).toEqual('errorTemplate');
    expect(component.submissionErrorEmail.templateType).toEqual('errorTemplate_templateType');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('errorTemplate_modifiedBy');

    expect(component.pendingEmail.heading).toEqual('pendingTemplate_heading');
    expect(component.pendingEmail.message).toEqual('pendingTemplate_message');
    expect(component.pendingEmail.templateName).toEqual('pendingTemplate');
    expect(component.pendingEmail.templateType).toEqual('pendingTemplate_templateType');
    expect(component.pendingEmail.modifiedBy).toEqual('pendingTemplate_modifiedBy'); 
  });

  it('test setCustomValidator method', () => {
    let form = { setValidators: () => true, updateValueAndValidity: () => true };
    let headingName = 'HEADING_NAME';
    let messageName = 'MESSAGE_NAME';
    let initialObject = new EmailModel();

    spyOn(form, 'setValidators');
    spyOn(form, 'updateValueAndValidity');

    component.setCustomValidator(<any> form, headingName, messageName, initialObject);

    expect(form.setValidators).toHaveBeenCalled();
    expect(form.updateValueAndValidity).toHaveBeenCalled();
  });

  it('test addCode method: tempNewCodeObj.stcCd and tempNewCodeObj.stcCdDesc are empty string', () => {
    component.modalErrorCode = false;
    component.modalErrorDesc = false;

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.newCode = null;
    component.modalClosed = false;

    component.addCode('');

    expect(component.modalErrorCode).toEqual(true);
    expect(component.modalErrorDesc).toEqual(true);

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.newCode).toEqual(null);

    expect(component.modalClosed).toEqual(false);
  });

  it('test addCode method: tempNewCodeObj.stcCd is empty string and tempNewCodeObj.stcCdDesc is NOT empty string', () => {
    component.modalErrorCode = false;
    component.modalErrorDesc = false;

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    // expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    component.tempNewCodeObj.stcCdDesc = 'STC_C_DESC';
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.newCode = null;
    component.modalClosed = false;

    component.addCode('');

    expect(component.modalErrorCode).toEqual(true);
    expect(component.modalErrorDesc).toEqual(false);

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('STC_C_DESC');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.newCode).toEqual(null);

    expect(component.modalClosed).toEqual(false);
  });

  it('test addCode method: tempNewCodeObj.stcCd is NOT empty string and tempNewCodeObj.stcCdDesc is empty string', () => {
    component.modalErrorCode = false;
    component.modalErrorDesc = false;

    // expect(component.tempNewCodeObj.stcCd).toEqual('');
    component.tempNewCodeObj.stcCd = 'STC_CD';
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.newCode = null;
    component.modalClosed = false;

    component.addCode('');

    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(true);

    expect(component.tempNewCodeObj.stcCd).toEqual('STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.newCode).toEqual(null);

    expect(component.modalClosed).toEqual(false);
  });

  it('test addCode method: tempNewCodeObj.stcCd and tempNewCodeObj.stcCdDesc are NOT empty string and input parameter is "Health Care Claim Status Category Code"', () => {
    component.modalErrorCode = false;
    component.modalErrorDesc = false;

    component.tempNewCodeObj.stcCd = 'STC_CD';
    component.tempNewCodeObj.stcCdDesc = 'STC_CD_DESC';
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.newCode = null;
    component.modalClosed = false;

    component.addCode('Health Care Claim Status Category Code');

    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes.length).toEqual(1);
    expect(component.healthCodes[0].stcCd).toEqual('STC_CD');
    expect(component.healthCodes[0].stcCdDesc).toEqual('STC_CD_DESC');
    expect(component.healthCodes[0].editedOrNewFlag).toEqual(true);

    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.newCode).not.toEqual(null);
    expect(component.newCode.stcCd).toEqual('');
    expect(component.newCode.stcCdDesc).toEqual('');
    expect(component.newCode.editedOrNewFlag).toEqual(false);

    expect(component.modalClosed).toEqual(true);
  });

  it('test addCode method: tempNewCodeObj.stcCd and tempNewCodeObj.stcCdDesc are NOT empty string and input parameter is "LOINC Code"', () => {
    component.modalErrorCode = false;
    component.modalErrorDesc = false;

    component.tempNewCodeObj.stcCd = 'STC_CD';
    component.tempNewCodeObj.stcCdDesc = 'STC_CD_DESC';
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.newCode = null;
    component.modalClosed = false;

    component.addCode('LOINC Code');

    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);

    expect(component.loincCodes.length).toEqual(1);
    expect(component.loincCodes[0].stcCd).toEqual('STC_CD');
    expect(component.loincCodes[0].stcCdDesc).toEqual('STC_CD_DESC');
    expect(component.loincCodes[0].editedOrNewFlag).toEqual(true);

    expect(component.loincModCodes).toEqual([ ]);

    expect(component.newCode).not.toEqual(null);
    expect(component.newCode.stcCd).toEqual('');
    expect(component.newCode.stcCdDesc).toEqual('');
    expect(component.newCode.editedOrNewFlag).toEqual(false);

    expect(component.modalClosed).toEqual(true);
  });

  it('test addCode method: tempNewCodeObj.stcCd and tempNewCodeObj.stcCdDesc are NOT empty string and input parameter is "LOINC Modifier Code"', () => {
    component.modalErrorCode = false;
    component.modalErrorDesc = false;

    component.tempNewCodeObj.stcCd = 'STC_CD';
    component.tempNewCodeObj.stcCdDesc = 'STC_CD_DESC';
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.newCode = null;
    component.modalClosed = false;

    component.addCode('LOINC Modifier Code');

    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);

    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    
    expect(component.loincModCodes.length).toEqual(1);
    expect(component.loincModCodes[0].stcCd).toEqual('STC_CD');
    expect(component.loincModCodes[0].stcCdDesc).toEqual('STC_CD_DESC');
    expect(component.loincModCodes[0].editedOrNewFlag).toEqual(true);

    expect(component.newCode).not.toEqual(null);
    expect(component.newCode.stcCd).toEqual('');
    expect(component.newCode.stcCdDesc).toEqual('');
    expect(component.newCode.editedOrNewFlag).toEqual(false);

    expect(component.modalClosed).toEqual(true);
  });

  it('test editCode method: newCode.stcCd and newCode.stcCdDesc are empty strings', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    expect(newCode.stcCd).toEqual('');
    expect(newCode.stcCdDesc).toEqual('');

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = '';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('');
    expect(newCode.stcCdDesc).toEqual('');
    expect(newCode.editedOrNewFlag).toEqual(undefined);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(true);
    expect(component.modalErrorCode).toEqual(true);
    expect(component.modalClosed).toEqual(undefined);
  }); 

  it('test editCode method: newCode.stcCd is an empty string and newCode.stcCdDesc is NOT an empty string', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    expect(newCode.stcCd).toEqual('');
    // expect(newCode.stcCdDesc).toEqual('');
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = '';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(undefined);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(true);
    expect(component.modalClosed).toEqual(undefined);
  }); 

  it('test editCode method: newCode.stcCd is NOT an empty string and newCode.stcCdDesc is an empty string', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    // expect(newCode.stcCd).toEqual('');
    newCode.stcCd = 'NEW_CODE_STC_CD';
    expect(newCode.stcCdDesc).toEqual('');

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = '';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('');
    expect(newCode.editedOrNewFlag).toEqual(undefined);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(true);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(undefined);
  }); 

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "Health Care Claim Status Category Code", and (c) healthCodes array has no elements', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'Health Care Claim Status Category Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  });

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "Health Care Claim Status Category Code", and (c) healthCodes array has elements but none whose stcCd is equal to oldCode.stcCd', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.flag = true;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'Health Care Claim Status Category Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    component.healthCodes = [ 
        { cd: 'CODE_1', stcCd: 'STC_CD_1', stcCdDesc: 'STC_CD_DESC_1', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_2', stcCd: 'STC_CD_2', stcCdDesc: 'STC_CD_DESC_2', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_3', stcCd: 'STC_CD_3', stcCdDesc: 'STC_CD_DESC_3', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_4', stcCd: 'STC_CD_4', stcCdDesc: 'STC_CD_DESC_4', flag: undefined, editedOrNewFlag: undefined }
    ];

    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes.length).toEqual(4);

    expect(component.healthCodes[0].cd).toEqual('CODE_1');
    expect(component.healthCodes[0].stcCd).toEqual('STC_CD_1');
    expect(component.healthCodes[0].stcCdDesc).toEqual('STC_CD_DESC_1');
    expect(component.healthCodes[0].flag).toEqual(undefined);
    expect(component.healthCodes[0].editedOrNewFlag).toEqual(undefined);

    expect(component.healthCodes[1].cd).toEqual('CODE_2');
    expect(component.healthCodes[1].stcCd).toEqual('STC_CD_2');
    expect(component.healthCodes[1].stcCdDesc).toEqual('STC_CD_DESC_2');
    expect(component.healthCodes[1].flag).toEqual(undefined);
    expect(component.healthCodes[1].editedOrNewFlag).toEqual(undefined);

    expect(component.healthCodes[2].cd).toEqual('CODE_3');
    expect(component.healthCodes[2].stcCd).toEqual('STC_CD_3');
    expect(component.healthCodes[2].stcCdDesc).toEqual('STC_CD_DESC_3');
    expect(component.healthCodes[2].flag).toEqual(undefined);
    expect(component.healthCodes[2].editedOrNewFlag).toEqual(undefined);

    expect(component.healthCodes[3].cd).toEqual('CODE_4');
    expect(component.healthCodes[3].stcCd).toEqual('STC_CD_4');
    expect(component.healthCodes[3].stcCdDesc).toEqual('STC_CD_DESC_4');
    expect(component.healthCodes[3].flag).toEqual(undefined);
    expect(component.healthCodes[3].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  });

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "Health Care Claim Status Category Code", and (c) healthCodes array has elements with one whose stcCd is equal to oldCode.stcCd', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.flag = true;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE';
    oldCode.stcCd = 'STC_CD_3';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'Health Care Claim Status Category Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    component.healthCodes = [ 
        { cd: 'CODE_1', stcCd: 'STC_CD_1', stcCdDesc: 'STC_CD_DESC_1', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_2', stcCd: 'STC_CD_2', stcCdDesc: 'STC_CD_DESC_2', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_3', stcCd: 'STC_CD_3', stcCdDesc: 'STC_CD_DESC_3', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_4', stcCd: 'STC_CD_4', stcCdDesc: 'STC_CD_DESC_4', flag: undefined, editedOrNewFlag: undefined }
    ];

    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE');
    expect(oldCode.stcCd).toEqual('STC_CD_3');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes.length).toEqual(4);

    expect(component.healthCodes[0].cd).toEqual('CODE_1');
    expect(component.healthCodes[0].stcCd).toEqual('STC_CD_1');
    expect(component.healthCodes[0].stcCdDesc).toEqual('STC_CD_DESC_1');
    expect(component.healthCodes[0].flag).toEqual(undefined);
    expect(component.healthCodes[0].editedOrNewFlag).toEqual(undefined);

    expect(component.healthCodes[1].cd).toEqual('CODE_2');
    expect(component.healthCodes[1].stcCd).toEqual('STC_CD_2');
    expect(component.healthCodes[1].stcCdDesc).toEqual('STC_CD_DESC_2');
    expect(component.healthCodes[1].flag).toEqual(undefined);
    expect(component.healthCodes[1].editedOrNewFlag).toEqual(undefined);

    expect(component.healthCodes[2].cd).toEqual('NEW_CODE');
    expect(component.healthCodes[2].stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.healthCodes[2].stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.healthCodes[2].flag).toEqual(true);
    expect(component.healthCodes[2].editedOrNewFlag).toEqual(true);

    expect(component.healthCodes[3].cd).toEqual('CODE_4');
    expect(component.healthCodes[3].stcCd).toEqual('STC_CD_4');
    expect(component.healthCodes[3].stcCdDesc).toEqual('STC_CD_DESC_4');
    expect(component.healthCodes[3].flag).toEqual(undefined);
    expect(component.healthCodes[3].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  });
  
  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "LOINC Code", and (c) loincCodes array has no elements', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'LOINC Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  }); 

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "LOINC Code", and (c) loincCodes array has elements but none whose stcCd is equal to oldCode.stcCd', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'LOINC Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);

    component.loincCodes = [ 
        { cd: 'CODE_1', stcCd: 'STC_CD_1', stcCdDesc: 'STC_CD_DESC_1', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_2', stcCd: 'STC_CD_2', stcCdDesc: 'STC_CD_DESC_2', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_3', stcCd: 'STC_CD_3', stcCdDesc: 'STC_CD_DESC_3', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_4', stcCd: 'STC_CD_4', stcCdDesc: 'STC_CD_DESC_4', flag: undefined, editedOrNewFlag: undefined }
    ];

    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);

    expect(component.loincCodes.length).toEqual(4);

    expect(component.loincCodes[0].cd).toEqual('CODE_1');
    expect(component.loincCodes[0].stcCd).toEqual('STC_CD_1');
    expect(component.loincCodes[0].stcCdDesc).toEqual('STC_CD_DESC_1');
    expect(component.loincCodes[0].flag).toEqual(undefined);
    expect(component.loincCodes[0].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes[1].cd).toEqual('CODE_2');
    expect(component.loincCodes[1].stcCd).toEqual('STC_CD_2');
    expect(component.loincCodes[1].stcCdDesc).toEqual('STC_CD_DESC_2');
    expect(component.loincCodes[1].flag).toEqual(undefined);
    expect(component.loincCodes[1].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes[2].cd).toEqual('CODE_3');
    expect(component.loincCodes[2].stcCd).toEqual('STC_CD_3');
    expect(component.loincCodes[2].stcCdDesc).toEqual('STC_CD_DESC_3');
    expect(component.loincCodes[2].flag).toEqual(undefined);
    expect(component.loincCodes[2].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes[3].cd).toEqual('CODE_4');
    expect(component.loincCodes[3].stcCd).toEqual('STC_CD_4');
    expect(component.loincCodes[3].stcCdDesc).toEqual('STC_CD_DESC_4');
    expect(component.loincCodes[3].flag).toEqual(undefined);
    expect(component.loincCodes[3].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  }); 

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "LOINC Code", and (c) loincCodes array has elements but with one whose stcCd is equal to oldCode.stcCd', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.flag = true;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'STC_CD_3';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'LOINC Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);

    component.loincCodes = [ 
        { cd: 'CODE_1', stcCd: 'STC_CD_1', stcCdDesc: 'STC_CD_DESC_1', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_2', stcCd: 'STC_CD_2', stcCdDesc: 'STC_CD_DESC_2', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_3', stcCd: 'STC_CD_3', stcCdDesc: 'STC_CD_DESC_3', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_4', stcCd: 'STC_CD_4', stcCdDesc: 'STC_CD_DESC_4', flag: undefined, editedOrNewFlag: undefined }
    ];

    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('STC_CD_3');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);

    expect(component.loincCodes.length).toEqual(4);

    expect(component.loincCodes[0].cd).toEqual('CODE_1');
    expect(component.loincCodes[0].stcCd).toEqual('STC_CD_1');
    expect(component.loincCodes[0].stcCdDesc).toEqual('STC_CD_DESC_1');
    expect(component.loincCodes[0].flag).toEqual(undefined);
    expect(component.loincCodes[0].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes[1].cd).toEqual('CODE_2');
    expect(component.loincCodes[1].stcCd).toEqual('STC_CD_2');
    expect(component.loincCodes[1].stcCdDesc).toEqual('STC_CD_DESC_2');
    expect(component.loincCodes[1].flag).toEqual(undefined);
    expect(component.loincCodes[1].editedOrNewFlag).toEqual(undefined);

    expect(component.loincCodes[2].cd).toEqual('NEW_CODE');
    expect(component.loincCodes[2].stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.loincCodes[2].stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.loincCodes[2].flag).toEqual(true);
    expect(component.loincCodes[2].editedOrNewFlag).toEqual(true);

    expect(component.loincCodes[3].cd).toEqual('CODE_4');
    expect(component.loincCodes[3].stcCd).toEqual('STC_CD_4');
    expect(component.loincCodes[3].stcCdDesc).toEqual('STC_CD_DESC_4');
    expect(component.loincCodes[3].flag).toEqual(undefined);
    expect(component.loincCodes[3].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  }); 

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "LOINC Modifier Code", and (c) loincModCodes array has no elements', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'LOINC Modifier Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);
    expect(component.loincModCodes).toEqual([ ]);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  }); 

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "LOINC Modifier Code", and (c) loincModCodes array has elements but none whose stcCd is equal to oldCode.stcCd', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'OLD_CODE_STC_CD';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'LOINC Modifier Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);

    component.loincModCodes = [
        { cd: 'CODE_1', stcCd: 'STC_CD_1', stcCdDesc: 'STC_CD_DESC_1', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_2', stcCd: 'STC_CD_2', stcCdDesc: 'STC_CD_DESC_2', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_3', stcCd: 'STC_CD_3', stcCdDesc: 'STC_CD_DESC_3', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_4', stcCd: 'STC_CD_4', stcCdDesc: 'STC_CD_DESC_4', flag: undefined, editedOrNewFlag: undefined }
    ];

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('OLD_CODE_STC_CD');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);

    expect(component.loincModCodes.length).toEqual(4);

    expect(component.loincModCodes[0].cd).toEqual('CODE_1');
    expect(component.loincModCodes[0].stcCd).toEqual('STC_CD_1');
    expect(component.loincModCodes[0].stcCdDesc).toEqual('STC_CD_DESC_1');
    expect(component.loincModCodes[0].flag).toEqual(undefined);
    expect(component.loincModCodes[0].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes[1].cd).toEqual('CODE_2');
    expect(component.loincModCodes[1].stcCd).toEqual('STC_CD_2');
    expect(component.loincModCodes[1].stcCdDesc).toEqual('STC_CD_DESC_2');
    expect(component.loincModCodes[1].flag).toEqual(undefined);
    expect(component.loincModCodes[1].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes[2].cd).toEqual('CODE_3');
    expect(component.loincModCodes[2].stcCd).toEqual('STC_CD_3');
    expect(component.loincModCodes[2].stcCdDesc).toEqual('STC_CD_DESC_3');
    expect(component.loincModCodes[2].flag).toEqual(undefined);
    expect(component.loincModCodes[2].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes[3].cd).toEqual('CODE_4');
    expect(component.loincModCodes[3].stcCd).toEqual('STC_CD_4');
    expect(component.loincModCodes[3].stcCdDesc).toEqual('STC_CD_DESC_4');
    expect(component.loincModCodes[3].flag).toEqual(undefined);
    expect(component.loincModCodes[3].editedOrNewFlag).toEqual(undefined);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  }); 

  it('test editCode method: (a) newCode.stcCd and newCode.stcCdDesc are NOT empty string, (b) currentCodeFullName is "LOINC Modifier Code", and (c) loincModCodes array has elements with one whose stcCd is equal to oldCode.stcCd', () => {
    let newCode = new CodeModel();
    newCode.cd = 'NEW_CODE';
    newCode.editedOrNewFlag = undefined;
    newCode.stcCd = 'NEW_CODE_STC_CD';
    newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let oldCode = new CodeModel();
    oldCode.cd = 'OLD_CODE_CD';
    oldCode.stcCd = 'STC_CD_3';
    oldCode.stcCdDesc = 'OLD_CODE_STC_CD_DESC';

    let currentCodeFullName = 'LOINC Modifier Code';

    component.modalErrorDesc = undefined;
    component.modalErrorCode = undefined;
    component.modalClosed = undefined;

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);

    component.loincModCodes = [
        { cd: 'CODE_1', stcCd: 'STC_CD_1', stcCdDesc: 'STC_CD_DESC_1', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_2', stcCd: 'STC_CD_2', stcCdDesc: 'STC_CD_DESC_2', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_3', stcCd: 'STC_CD_3', stcCdDesc: 'STC_CD_DESC_3', flag: undefined, editedOrNewFlag: undefined },
        { cd: 'CODE_4', stcCd: 'STC_CD_4', stcCdDesc: 'STC_CD_DESC_4', flag: undefined, editedOrNewFlag: undefined }
    ];

    component.editCode(newCode, oldCode, currentCodeFullName);

    expect(newCode.cd).toEqual('NEW_CODE');
    expect(newCode.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(newCode.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(newCode.editedOrNewFlag).toEqual(true);

    expect(oldCode.cd).toEqual('OLD_CODE_CD');
    expect(oldCode.stcCd).toEqual('STC_CD_3');
    expect(oldCode.stcCdDesc).toEqual('OLD_CODE_STC_CD_DESC');

    expect(component.healthCodes).toEqual([ ]);
    expect(component.loincCodes).toEqual([ ]);

    expect(component.loincModCodes.length).toEqual(4);

    expect(component.loincModCodes[0].cd).toEqual('CODE_1');
    expect(component.loincModCodes[0].stcCd).toEqual('STC_CD_1');
    expect(component.loincModCodes[0].stcCdDesc).toEqual('STC_CD_DESC_1');
    expect(component.loincModCodes[0].flag).toEqual(undefined);
    expect(component.loincModCodes[0].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes[1].cd).toEqual('CODE_2');
    expect(component.loincModCodes[1].stcCd).toEqual('STC_CD_2');
    expect(component.loincModCodes[1].stcCdDesc).toEqual('STC_CD_DESC_2');
    expect(component.loincModCodes[1].flag).toEqual(undefined);
    expect(component.loincModCodes[1].editedOrNewFlag).toEqual(undefined);

    expect(component.loincModCodes[2].cd).toEqual('NEW_CODE');
    expect(component.loincModCodes[2].stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.loincModCodes[2].stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.loincModCodes[2].flag).toEqual(true);
    expect(component.loincModCodes[2].editedOrNewFlag).toEqual(true);

    expect(component.loincModCodes[3].cd).toEqual('CODE_4');
    expect(component.loincModCodes[3].stcCd).toEqual('STC_CD_4');
    expect(component.loincModCodes[3].stcCdDesc).toEqual('STC_CD_DESC_4');
    expect(component.loincModCodes[3].flag).toEqual(undefined);
    expect(component.loincModCodes[3].editedOrNewFlag).toEqual(undefined);

    expect(component.modalErrorDesc).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalClosed).toEqual(true);
  }); 

  it('test openAddModal method: currentCodeName input parameter is not equal to HCCS, LOINC, nor LOINCMod', () => {
    component.addClicked = undefined;
    component.currentCodeFullName = undefined;
    component.tempNewCodeObj = undefined;

    spyOn(component, 'onModalOpen')

    let currentCodeName = 'DUMMY';
    component.openAddModal(currentCodeName);

    expect(component.addClicked).toEqual(true);
    expect(component.currentCodeFullName).toEqual(undefined);
    expect(component.tempNewCodeObj).toEqual(undefined);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openAddModal method: currentCodeName input parameter is equal to HCCS', () => {
    component.addClicked = undefined;
    component.currentCodeFullName = undefined;
    component.tempNewCodeObj = undefined;

    component.newCode.cd = 'NEW_CODE_CD';
    component.newCode.stcCd = 'NEW_CODE_STC_CD';
    component.newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    spyOn(component, 'onModalOpen')

    let currentCodeName = 'HCCS';
    component.openAddModal(currentCodeName);

    expect(component.addClicked).toEqual(true);
    expect(component.currentCodeFullName).toEqual('Health Care Claim Status Category Code');

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openAddModal method: currentCodeName input parameter is equal to LOINC', () => {
    component.addClicked = undefined;
    component.currentCodeFullName = undefined;
    component.tempNewCodeObj = undefined;

    component.newCode.cd = 'NEW_CODE_CD';
    component.newCode.stcCd = 'NEW_CODE_STC_CD';
    component.newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    spyOn(component, 'onModalOpen')

    let currentCodeName = 'LOINC';
    component.openAddModal(currentCodeName);

    expect(component.addClicked).toEqual(true);
    expect(component.currentCodeFullName).toEqual('LOINC Code');

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openAddModal method: currentCodeName input parameter is equal to LOINCMod', () => {
    component.addClicked = undefined;
    component.currentCodeFullName = undefined;
    component.tempNewCodeObj = undefined;

    component.newCode.cd = 'NEW_CODE_CD';
    component.newCode.stcCd = 'NEW_CODE_STC_CD';
    component.newCode.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    spyOn(component, 'onModalOpen')

    let currentCodeName = 'LOINCMod';
    component.openAddModal(currentCodeName);

    expect(component.addClicked).toEqual(true);
    expect(component.currentCodeFullName).toEqual('LOINC Modifier Code');

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openEditModal method: currentCodeName input parameter is not equal to HCCS, LOINC, nor LOINCMod', () => {
    component.addClicked = undefined;
    component.modalErrorCode = undefined;
    component.modalErrorDesc = undefined;
    component.tempNewCodeObj = undefined;
    component.tempCodeObj = undefined;
    component.currentCodeFullName = undefined;

    spyOn(component, 'onModalOpen');

    let codeObj = new CodeModel();
    codeObj.cd = 'NEW_CODE_CD';
    codeObj.stcCd = 'NEW_CODE_STC_CD';
    codeObj.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let currentCodeName = 'DUMMY';
    component.openEditModal(codeObj, currentCodeName);

    expect(component.addClicked).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);
    expect(component.currentCodeFullName).toEqual(undefined);

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.tempCodeObj).not.toEqual(undefined);
    expect(component.tempCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempCodeObj.flag).toEqual(true);
    expect(component.tempCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openEditModal method: currentCodeName input parameter is not equal to HCCS', () => {
    component.addClicked = undefined;
    component.modalErrorCode = undefined;
    component.modalErrorDesc = undefined;
    component.tempNewCodeObj = undefined;
    component.tempCodeObj = undefined;
    component.currentCodeFullName = undefined;

    spyOn(component, 'onModalOpen');

    let codeObj = new CodeModel();
    codeObj.cd = 'NEW_CODE_CD';
    codeObj.stcCd = 'NEW_CODE_STC_CD';
    codeObj.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let currentCodeName = 'HCCS';
    component.openEditModal(codeObj, currentCodeName);

    expect(component.addClicked).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);
    expect(component.currentCodeFullName).toEqual('Health Care Claim Status Category Code');

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.tempCodeObj).not.toEqual(undefined);
    expect(component.tempCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempCodeObj.flag).toEqual(true);
    expect(component.tempCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openEditModal method: currentCodeName input parameter is not equal to LOINC', () => {
    component.addClicked = undefined;
    component.modalErrorCode = undefined;
    component.modalErrorDesc = undefined;
    component.tempNewCodeObj = undefined;
    component.tempCodeObj = undefined;
    component.currentCodeFullName = undefined;

    spyOn(component, 'onModalOpen');

    let codeObj = new CodeModel();
    codeObj.cd = 'NEW_CODE_CD';
    codeObj.stcCd = 'NEW_CODE_STC_CD';
    codeObj.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let currentCodeName = 'LOINC';
    component.openEditModal(codeObj, currentCodeName);

    expect(component.addClicked).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);
    expect(component.currentCodeFullName).toEqual('LOINC Code');

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.tempCodeObj).not.toEqual(undefined);
    expect(component.tempCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempCodeObj.flag).toEqual(true);
    expect(component.tempCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test openEditModal method: currentCodeName input parameter is not equal to LOINCMod', () => {
    component.addClicked = undefined;
    component.modalErrorCode = undefined;
    component.modalErrorDesc = undefined;
    component.tempNewCodeObj = undefined;
    component.tempCodeObj = undefined;
    component.currentCodeFullName = undefined;

    spyOn(component, 'onModalOpen');

    let codeObj = new CodeModel();
    codeObj.cd = 'NEW_CODE_CD';
    codeObj.stcCd = 'NEW_CODE_STC_CD';
    codeObj.stcCdDesc = 'NEW_CODE_STC_CD_DESC';

    let currentCodeName = 'LOINCMod';
    component.openEditModal(codeObj, currentCodeName);

    expect(component.addClicked).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);
    expect(component.currentCodeFullName).toEqual('LOINC Modifier Code');

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempNewCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.tempCodeObj).not.toEqual(undefined);
    expect(component.tempCodeObj.cd).toEqual('NEW_CODE_CD');
    expect(component.tempCodeObj.stcCd).toEqual('NEW_CODE_STC_CD');
    expect(component.tempCodeObj.stcCdDesc).toEqual('NEW_CODE_STC_CD_DESC');
    expect(component.tempCodeObj.flag).toEqual(true);
    expect(component.tempCodeObj.editedOrNewFlag).toEqual(false);

    expect(component.onModalOpen).toHaveBeenCalled();
  });

  it('test onModalOpen method', () => {
    component.modalClosed = undefined;
    component.tempNewCodeObj = undefined;

    component.onModalOpen();

    expect(component.modalClosed).toEqual(false);

    expect(component.tempNewCodeObj).not.toEqual(undefined);
    expect(component.tempNewCodeObj.cd).toEqual('');
    expect(component.tempNewCodeObj.stcCd).toEqual('');
    expect(component.tempNewCodeObj.stcCdDesc).toEqual('');
    expect(component.tempNewCodeObj.flag).toEqual(true);
    expect(component.tempNewCodeObj.editedOrNewFlag).toEqual(false);    
  });

  it('test onModalClose method: input parameter is boolean false', () => {
    component.modalClosed = undefined;
    component.addClicked = undefined;
    component.modalErrorCode = undefined;
    component.modalErrorDesc = undefined;

    component.onModalClose(false);

    expect(component.modalClosed).toEqual(false);
    expect(component.addClicked).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);
  });

  it('test onModalClose method: input parameter is boolean true', () => {
    component.modalClosed = undefined;
    component.addClicked = undefined;
    component.modalErrorCode = undefined;
    component.modalErrorDesc = undefined;

    component.onModalClose(true);

    expect(component.modalClosed).toEqual(true);
    expect(component.addClicked).toEqual(false);
    expect(component.modalErrorCode).toEqual(false);
    expect(component.modalErrorDesc).toEqual(false);
  });

  it('test submit method: returned data has "result" property', () => {
    let returnedData = { result: 'SUCCESS' };
    spyOn(rfaiAdminService, 'postCodes').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    const requestBody = {
        hccList: component.healthCodes.filter(code => code.editedOrNewFlag === true),
        loincList: component.loincCodes.filter(code => code.editedOrNewFlag === true),
        loincModList: component.loincModCodes.filter(
          code => code.editedOrNewFlag === true
        )
    };

    component.successMsg = undefined;
    component.successStatus = undefined;
    component.errorStatus = undefined;
    component.errorMsg = undefined;

    spyOn(component, 'getCodes');

    component.submit();

    expect(component.getCodes).toHaveBeenCalled();
    expect(rfaiAdminService.postCodes).toHaveBeenCalled();
    expect(rfaiAdminService.postCodes).toHaveBeenCalledWith(requestBody);

    expect(component.successMsg).toEqual('Updated Codes successfully.');
    expect(component.successStatus).toEqual(true);
    expect(component.errorStatus).toEqual(false);
    expect(component.errorMsg).toEqual(undefined);
  });

  it('test submit method: returned data has NO "result" property', () => {
    let returnedData = { noResult: 'NOT-SUCCESS' };
    spyOn(rfaiAdminService, 'postCodes').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    const requestBody = {
        hccList: component.healthCodes.filter(code => code.editedOrNewFlag === true),
        loincList: component.loincCodes.filter(code => code.editedOrNewFlag === true),
        loincModList: component.loincModCodes.filter(
          code => code.editedOrNewFlag === true
        )
    };

    component.successMsg = undefined;
    component.successStatus = undefined;
    component.errorStatus = undefined;
    component.errorMsg = undefined;

    spyOn(component, 'getCodes');

    component.submit();

    expect(component.getCodes).not.toHaveBeenCalled();
    expect(rfaiAdminService.postCodes).toHaveBeenCalled();
    expect(rfaiAdminService.postCodes).toHaveBeenCalledWith(requestBody);

    expect(component.successMsg).toEqual(undefined);
    expect(component.successStatus).toEqual(false);
    expect(component.errorStatus).toEqual(true);
    expect(component.errorMsg).toEqual('An error occured updating Codes.');
  });

  it('test cancel method', () => {
    component.modalClosed = undefined;
    component.addClicked = undefined;
    spyOn(component, 'getCodes');

    component.cancel();

    expect(component.modalClosed).toEqual(true);
    expect(component.addClicked).toEqual(false);
    expect(component.getCodes).toHaveBeenCalled();
  });

  it('test setEditOrNewFlag method', () => {
    let code = new CodeModel();
    code.editedOrNewFlag = undefined;

    component.setEditOrNewFlag(code);

    expect(code.editedOrNewFlag).toEqual(true);
  });

  it('test onSubmitEmail method: returned data has "result" property', () => {
    let returnedData = { result: 'SUCCESS' };
    spyOn(rfaiAdminService, 'submitEmailTemplate').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    sessionStorage.setItem('userName', 'USER_NAME');

    component.successMsg = undefined;
    component.successStatus = undefined;
    component.errorStatus = undefined;
    component.errorMsg = undefined;

    spyOn(component, 'getCodes');

    let email = new EmailModel();
    component.onSubmitEmail(email);

    expect(component.getCodes).toHaveBeenCalled();
    expect(rfaiAdminService.submitEmailTemplate).toHaveBeenCalled();
    expect(rfaiAdminService.submitEmailTemplate).toHaveBeenCalledWith(email);

    expect(email.modifiedBy).toEqual('USER_NAME');

    expect(component.successMsg).toEqual('Updated Email Template successfully.');
    expect(component.successStatus).toEqual(true);
    expect(component.errorStatus).toEqual(false);
    expect(component.errorMsg).toEqual(undefined);
  });

  it('test onSubmitEmail method: returned data has NO "result" property', () => {
    let returnedData = { noResult: 'NOT SUCCESS' };
    spyOn(rfaiAdminService, 'submitEmailTemplate').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    sessionStorage.setItem('userName', 'USER_NAME');

    component.successMsg = undefined;
    component.successStatus = undefined;
    component.errorStatus = undefined;
    component.errorMsg = undefined;

    spyOn(component, 'getCodes');

    let email = new EmailModel();
    component.onSubmitEmail(email);

    expect(component.getCodes).not.toHaveBeenCalled();
    expect(rfaiAdminService.submitEmailTemplate).toHaveBeenCalled();
    expect(rfaiAdminService.submitEmailTemplate).toHaveBeenCalledWith(email);

    expect(email.modifiedBy).toEqual('USER_NAME');

    expect(component.successMsg).toEqual(undefined);
    expect(component.successStatus).toEqual(false);
    expect(component.errorStatus).toEqual(true);
    expect(component.errorMsg).toEqual('An error occured updating Email Templates.');
  });

  it('test cancelEmailTemplates: input paramter emailType is "manuallyClosedTemplate" but returned data has no "emailTemplates" attribute', () => {
    let returnedData = { };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('manuallyClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "manuallyClosedTemplate" but returned data has "emailTemplates" attribute with empty array', () => {
    let returnedData = { emailTemplates: [ ]};
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('manuallyClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "manuallyClosedTemplate" but returned data has "emailTemplates" attribute with array that does not contain "manuallyClosedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('manuallyClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "manuallyClosedTemplate" but returned data has "emailTemplates" attribute with array that does contain "manuallyClosedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('manuallyClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('manuallyClosedTemplate_heading');
    expect(component.manuallyClosedEmail.message).toEqual('manuallyClosedTemplate_message');
    expect(component.manuallyClosedEmail.templateName).toEqual('manuallyClosedTemplate');
    expect(component.manuallyClosedEmail.templateType).toEqual('manuallyClosedTemplate_templateType');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('manuallyClosedTemplate_modifiedBy');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "abandonedTemplate" but returned data has no "emailTemplates" attribute', () => {
    let returnedData = { };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('abandonedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "abandonedTemplate" but returned data has "emailTemplates" attribute with empty array', () => {
    let returnedData = { emailTemplates: [ ]};
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('abandonedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "abandonedTemplate" but returned data has "emailTemplates" attribute with array that does not contain "abandonedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('abandonedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "abandonedTemplate" but returned data has "emailTemplates" attribute with array that does contain "abandonedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('abandonedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('abandonedTemplate_heading');
    expect(component.abondonedEmail.message).toEqual('abandonedTemplate_message');
    expect(component.abondonedEmail.templateName).toEqual('abandonedTemplate');
    expect(component.abondonedEmail.templateType).toEqual('abandonedTemplate_templateType');
    expect(component.abondonedEmail.modifiedBy).toEqual('abandonedTemplate_modifiedBy');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "completeOrClosedTemplate" but returned data has no "emailTemplates" attribute', () => {
    let returnedData = { };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('completeOrClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "completeOrClosedTemplate" but returned data has "emailTemplates" attribute with empty array', () => {
    let returnedData = { emailTemplates: [ ]};
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('completeOrClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "completeOrClosedTemplate" but returned data has "emailTemplates" attribute with array that does not contain "completeOrClosedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('completeOrClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "completeOrClosedTemplate" but returned data has "emailTemplates" attribute with array that does contain "completeOrClosedTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('completeOrClosedTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('completeOrClosedTemplate_heading');
    expect(component.closedEmail.message).toEqual('completeOrClosedTemplate_message');
    expect(component.closedEmail.templateName).toEqual('completeOrClosedTemplate');
    expect(component.closedEmail.templateType).toEqual('completeOrClosedTemplate_templateType');
    expect(component.closedEmail.modifiedBy).toEqual('completeOrClosedTemplate_modifiedBy');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "errorTemplate" but returned data has no "emailTemplates" attribute', () => {
    let returnedData = { };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('errorTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "errorTemplate" but returned data has "emailTemplates" attribute with empty array', () => {
    let returnedData = { emailTemplates: [ ]};
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('errorTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "errorTemplate" but returned data has "emailTemplates" attribute with array that does not contain "errorTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('errorTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "errorTemplate" but returned data has "emailTemplates" attribute with array that does contain "errorTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('errorTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('errorTemplate_heading');
    expect(component.submissionErrorEmail.message).toEqual('errorTemplate_message');
    expect(component.submissionErrorEmail.templateName).toEqual('errorTemplate');
    expect(component.submissionErrorEmail.templateType).toEqual('errorTemplate_templateType');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('errorTemplate_modifiedBy');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "pendingTemplate" but returned data has no "emailTemplates" attribute', () => {
    let returnedData = { };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('pendingTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "pendingTemplate" but returned data has "emailTemplates" attribute with empty array', () => {
    let returnedData = { emailTemplates: [ ]};
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('pendingTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "pendingTemplate" but returned data has "emailTemplates" attribute with array that does not contain "pendingTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('pendingTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual('');    
  });

  it('test cancelEmailTemplates: input paramter emailType is "pendingTemplate" but returned data has "emailTemplates" attribute with array that does contain "pendingTemplate"', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('pendingTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('pendingTemplate_heading');
    expect(component.pendingEmail.message).toEqual('pendingTemplate_message');
    expect(component.pendingEmail.templateName).toEqual('pendingTemplate');
    expect(component.pendingEmail.templateType).toEqual('pendingTemplate_templateType');
    expect(component.pendingEmail.modifiedBy).toEqual('pendingTemplate_modifiedBy'); 
  });

  it('test cancelEmailTemplates: input paramter emailType is not a valie type but returned data has "emailTemplates" attribute with array that does contain all the valid template types', () => {
    let returnedData = { 
        emailTemplates: [ 
            { 
                heading: 'manuallyClosedTemplate_heading',
                message: 'manuallyClosedTemplate_message',
                templateName: 'manuallyClosedTemplate',
                templateType: 'manuallyClosedTemplate_templateType',
                modifiedBy: 'manuallyClosedTemplate_modifiedBy'
            },
            { 
                heading: 'abandonedTemplate_heading',
                message: 'abandonedTemplate_message',
                templateName: 'abandonedTemplate',
                templateType: 'abandonedTemplate_templateType',
                modifiedBy: 'abandonedTemplate_modifiedBy'
            },
            { 
                heading: 'completeOrClosedTemplate_heading',
                message: 'completeOrClosedTemplate_message',
                templateName: 'completeOrClosedTemplate',
                templateType: 'completeOrClosedTemplate_templateType',
                modifiedBy: 'completeOrClosedTemplate_modifiedBy'
            },
            { 
                heading: 'pendingTemplate_heading',
                message: 'pendingTemplate_message',
                templateName: 'pendingTemplate',
                templateType: 'pendingTemplate_templateType',
                modifiedBy: 'pendingTemplate_modifiedBy'
            },
            { 
                heading: 'errorTemplate_heading',
                message: 'errorTemplate_message',
                templateName: 'errorTemplate',
                templateType: 'errorTemplate_templateType',
                modifiedBy: 'errorTemplate_modifiedBy'
            }
        ] 
    };
    spyOn(rfaiAdminService, 'getEmailTemplates').and.callFake(() => {
        return new Observable(observer => observer.next(returnedData));
    });

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
    

    component.cancelEmailTemplates('inavlidDummyTemplate');

    expect(rfaiAdminService.getEmailTemplates).toHaveBeenCalled();

    expect(component.manuallyClosedEmail.heading).toEqual('');
    expect(component.manuallyClosedEmail.message).toEqual('');
    expect(component.manuallyClosedEmail.templateName).toEqual('');
    expect(component.manuallyClosedEmail.templateType).toEqual('');
    expect(component.manuallyClosedEmail.modifiedBy).toEqual('');

    expect(component.abondonedEmail.heading).toEqual('');
    expect(component.abondonedEmail.message).toEqual('');
    expect(component.abondonedEmail.templateName).toEqual('');
    expect(component.abondonedEmail.templateType).toEqual('');
    expect(component.abondonedEmail.modifiedBy).toEqual('');

    expect(component.closedEmail.heading).toEqual('');
    expect(component.closedEmail.message).toEqual('');
    expect(component.closedEmail.templateName).toEqual('');
    expect(component.closedEmail.templateType).toEqual('');
    expect(component.closedEmail.modifiedBy).toEqual('');

    expect(component.submissionErrorEmail.heading).toEqual('');
    expect(component.submissionErrorEmail.message).toEqual('');
    expect(component.submissionErrorEmail.templateName).toEqual('');
    expect(component.submissionErrorEmail.templateType).toEqual('');
    expect(component.submissionErrorEmail.modifiedBy).toEqual('');

    expect(component.pendingEmail.heading).toEqual('');
    expect(component.pendingEmail.message).toEqual('');
    expect(component.pendingEmail.templateName).toEqual('');
    expect(component.pendingEmail.templateType).toEqual('');
    expect(component.pendingEmail.modifiedBy).toEqual(''); 
  });

});
