import { Router } from '@angular/router';
import { Location } from '@angular/common';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';
import { AuthenticationService } from './login.service';

import { Observable } from 'rxjs/Observable';
import { Observer } from 'rxjs/Observer';

import { LoginComponent } from './login.component';
import { DomSanitizer } from '@angular/platform-browser';

describe('LoginComponent', () => {
  let router;
  let location: Location;
  let fb: FormBuilder;
  let authenticationService: AuthenticationService;

  let component: LoginComponent;
  let sanitizer = null;

  beforeEach(() => {
    router = { navigate: () => true };
    // location = new Location(null);
    fb = new FormBuilder();
    authenticationService = new AuthenticationService(null);

    component = new LoginComponent(router,null, fb, authenticationService, sanitizer, null);
  });

  it('should be created', () => {
    expect(component).toBeTruthy();
  });

  // it('test ngOnInit method', () => {
  //   spyOn(component, 'getDropDown');
  //   spyOn(component, 'formInit');
  //   spyOn(authenticationService, 'sso').and.callFake(() => {
  //     return new Observable(observer => observer.next({u:'u', c:'c',s:'s' }))
  //   });
  //   sanitizer = {}
  //   sanitizer.bypassSecurityTrustUrl = ''
  //   component.safeUrl = {}
  //   component.ngOnInit();
    
  //   expect(component.getDropDown).toHaveBeenCalled();
  //   expect(component.formInit).toHaveBeenCalled();
  //   expect(authenticationService.sso).toHaveBeenCalled();
  // });

  it('test formInit method: check loginForm initlialized', () => {
    expect(component.loginForm).toEqual(undefined);

    component.formInit();

    expect(component.loginForm).toBeDefined();
    expect(component.loginForm).not.toEqual(null);
  });

  it('test formInit method: verify loginForm initially in invalid state', () => {
    component.formInit();

    expect(component.loginForm.get('userName').invalid).toEqual(true);
    expect(component.loginForm.invalid).toEqual(true);
  });

  it('test formInit method: verify loginForm in invalid state if userName set to empty string', () => {
    component.formInit();

    component.loginForm.get('userName').setValue('');

    expect(component.loginForm.get('userName').invalid).toEqual(true);
    expect(component.loginForm.invalid).toEqual(true);
  });  

  it('test formInit method: verify loginForm in valid state if userName set to non-empty string', () => {
    component.formInit();

    component.loginForm.get('userName').setValue('USERADMIN');

    expect(component.loginForm.get('userName').valid).toEqual(true);
    expect(component.loginForm.valid).toEqual(true);
  });  

  it('test getDropDown method', () => {
    let dropDownInfoResponse = 'USERADMIN';
    spyOn(authenticationService, 'getDropDownInfo').and.callFake(() => {
      return new Observable(observer => observer.next(dropDownInfoResponse));
    });

    expect(component.userName).toEqual('');

    component.getDropDown();

    expect(authenticationService.getDropDownInfo).toHaveBeenCalled();
    expect(component.userName).toEqual(dropDownInfoResponse);
  });

  it('test accept method', () => {
    component.formInit();
    component.loginForm.get('userName').setValue('USERSUPERVISOR');
    spyOn(component, 'getUsrInfo');

    component.accept();

    expect(component.getUsrInfo).toHaveBeenCalled();
    expect(component.getUsrInfo).toHaveBeenCalledWith('USERSUPERVISOR');
  });

  it('test getUsrInfo method', () => {
    let inputUserName = 'MY_USER_NAME';
    
    let responseData = { theData: 'THE_DATA' };
    spyOn(authenticationService, 'authenticateAndGetUserInfo').and.callFake(() => {
      return new Observable(observer => observer.next(responseData));
    });   
    
    spyOn(component, 'redirect');

    component.getUsrInfo(inputUserName);

    expect(authenticationService.authenticateAndGetUserInfo).toHaveBeenCalled();
    expect(authenticationService.authenticateAndGetUserInfo).toHaveBeenCalledWith(inputUserName);
    expect(component.redirect).toHaveBeenCalled();

    expect(component.userInfo).toEqual(responseData);
  });

  it('test redirect method: userInfo.userRoles is undefined', () => {
    component.userInfo = {
      userRoles: undefined
    }

    spyOn(component, 'setSessionValues');
    spyOn(router, 'navigate');

    component.redirect();

    expect(component.setSessionValues).not.toHaveBeenCalled();
    expect(router.navigate).not.toHaveBeenCalled();
  });

  it('test redirect method: userInfo.userRoles is array having ADMIN, BASE_USER, and EWV roles', () => {
    component.userInfo = {
      userRoles: [ 'ADMIN', 'BASE_USER', 'EWV', 'ADMIN', 'BASE_USER', 'EWV', 'DUMMY_1', 'DUMMY_2' ]
    }

    spyOn(component, 'setSessionValues');
    spyOn(router, 'navigate');

    component.redirect();

    expect(component.setSessionValues).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalledWith(['search275']);
  });

  it('test redirect method: userInfo.userRoles is array having only BASE_USER and EWV roles', () => {
    component.userInfo = {
      userRoles: [ 'BASE_USER', 'EWV', 'BASE_USER', 'EWV', 'DUMMY_1', 'DUMMY_2' ]
    }

    spyOn(component, 'setSessionValues');
    spyOn(router, 'navigate');

    component.redirect();

    expect(component.setSessionValues).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalledWith(['search275']);
  });

  it('test redirect method: userInfo.userRoles is array having only EWV roles', () => {
    component.userInfo = {
      userRoles: [ 'EWV', 'EWV', 'DUMMY_1', 'DUMMY_2' ]
    }

    spyOn(component, 'setSessionValues');
    spyOn(router, 'navigate');

    component.redirect();

    expect(component.setSessionValues).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalledWith(['ewvSearch']);
  });

  it('test redirect method: userInfo.userRoles is array having roles that are not ADMIN, BASE_USER, and EWV', () => {
    component.userInfo = {
      userRoles: [ 'DUMMY_1', 'DUMMY_2' ]
    }

    spyOn(component, 'setSessionValues');
    spyOn(router, 'navigate');

    component.redirect();

    expect(component.setSessionValues).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalled();
    expect(router.navigate).toHaveBeenCalledWith(['/']);
  });

  it('test setSessionValues method', () => {
    component.userInfo = {
      userRoles: [ 'BASE_USER', 'EWV', 'BASE_USER', 'EWV', 'DUMMY_1', 'DUMMY_2' ],
      userName: 'USER_NAME',
      lastLogin: 'LAST_LOGIN',
      permissions: {
        additionalInformation: 'ADDITIONAL_INFORMATION',
        lookup: 'LOOKUP_INFO',
        viewSubmission: 'VIEW_SUBMISSION',
        admin: 'ADMIN_INFO',
        ewvSearch: 'EWV_SEARCH'
      }
    };

    component.setSessionValues();

    expect(sessionStorage.getItem('userInfo')).toEqual(JSON.stringify(component.userInfo));
    expect(sessionStorage.getItem('role')).toEqual(component.userInfo.userRoles.length.toString())
    expect(sessionStorage.getItem('userName')).toEqual(component.userInfo.userName);
    expect(sessionStorage.getItem('lastLogin')).toEqual(component.userInfo.lastLogin);
    expect(sessionStorage.getItem('userRole')).toEqual(JSON.stringify(component.userInfo.userRoles));

    expect(sessionStorage.getItem('permissions')).toEqual(JSON.stringify(component.userInfo.permissions));
    expect(sessionStorage.getItem('additionalInformationPermissions')).toEqual(
      JSON.stringify(component.userInfo.permissions.additionalInformation));

    expect(sessionStorage.getItem('lookupPermissions')).toEqual(
      JSON.stringify(component.userInfo.permissions.lookup));
    expect(sessionStorage.getItem('viewSubmissionPermissions')).toEqual(
      JSON.stringify(component.userInfo.permissions.viewSubmission));
    expect(sessionStorage.getItem('adminPermissions')).toEqual(
      JSON.stringify(component.userInfo.permissions.admin));
    expect(sessionStorage.getItem('ewvSearch')).toEqual(
      JSON.stringify(component.userInfo.permissions.ewvSearch));
    
  });

});
