import { Component, OnInit, ViewChild, ElementRef } from '@angular/core';
import { Chart } from 'chart.js';

import { ArsStorageService } from './ars-storage.service';
import { AttachmentDetailsModel, MetadataDetailsModel } from './ars-storage.model';
import { getDates } from './date-util';

@Component({
  selector: 'app-ars-storage',
  templateUrl: './ars-storage.component.html',
  styleUrls: ['./ars-storage.component.scss']
})
export class ArsStorageComponent implements OnInit {
  attachmentStoragePieChart;
  attachmentStorageComboChart;
  metadataStoragePieChart;
  metadataStorageComboChart;

  metadataDetails$: MetadataDetailsModel = new MetadataDetailsModel();
  attachmentDetails$: AttachmentDetailsModel = new AttachmentDetailsModel();


  constructor(private storageService: ArsStorageService) { }

  @ViewChild('attachmentStorageCanvasLeft')
  attachmentStorageCanvasLeft: ElementRef;
  @ViewChild('attachmentStorageCanvasRight')
  attachmentStorageCanvasRight: ElementRef;
  @ViewChild('metadataStorageCanvasLeft')
  metadataStorageCanvasLeft: ElementRef;
  @ViewChild('metadataStorageCanvasRight')
  metadataStorageCanvasRight: ElementRef;

  ngOnInit() {
    // methods to create charts.
    this.createAttachmentStoragePieChart();
    this.createAttachmentStorageComboChart();
    this.createMetadataStoragePieChart();
    this.createMetadataStorageComboChart();
    // initial rest calls to get chart data.
    this.storageService.getAttachmentDetails()
      .subscribe(attachmentDetails => {
        this.attachmentDetails$ = attachmentDetails;

        // load attachmentStorage data into pie chart
        this.attachmentStoragePieChart.data.datasets[0].data[0] = this.attachmentDetails$.availableStorage;
        this.attachmentStoragePieChart.data.datasets[0].data[1] = this.attachmentDetails$.usedStorage;

        // load attachmentStorage data into combo chart
        this.attachmentStorageComboChart.data.datasets[0].data = this.attachmentDetails$.usedStorageStatistic // usage
        this.attachmentStorageComboChart.data.datasets[1].data = this.attachmentDetails$.totalStorageStatistic // line
        this.attachmentStorageComboChart.data.labels = getDates(this.attachmentDetails$.fromDate,this.attachmentDetails$.toDate);

        // update charts.
        this.attachmentStoragePieChart.update();
        this.attachmentStorageComboChart.update();
      });
    this.storageService.getMetadataDetails()
      .subscribe(metadataDetails => {
        this.metadataDetails$ = metadataDetails;
        // load metadataStorage data into pie chart
        this.metadataStoragePieChart.data.datasets[0].data[0] = this.metadataDetails$.availableStorage;
        this.metadataStoragePieChart.data.datasets[0].data[1] = this.metadataDetails$.usedStorage;
        // load metadataStorage data into combo chart
        this.metadataStorageComboChart.data.datasets[0].data = this.metadataDetails$.usedStorageStatistic // usage
        this.metadataStorageComboChart.data.datasets[1].data = this.metadataDetails$.totalStorageStatistic // line
        this.metadataStorageComboChart.data.labels = getDates(this.metadataDetails$.fromDate,this.metadataDetails$.toDate);

        // update charts.
        this.metadataStoragePieChart.update();
        this.metadataStorageComboChart.update();
      });

    
    
  }
  createAttachmentStoragePieChart() {
    this.attachmentStoragePieChart = new Chart(
      this.attachmentStorageCanvasLeft.nativeElement,
      {
        type: 'pie',
        data: {
          datasets: [
            {
              data: [100, 0], // [avail, used]
              backgroundColor: ['rgba(0,115,167,0.2)', 'rgba(0,115,167,1.0)'],
              
               //data: [20, 80]
            }
          ],
          labels: ['Available', 'Used']
        },
        options: {
          layout: {
            padding: {
              bottom: 18
            }
          },
          legend: {
            position: 'bottom',
            display: false,
            labels: {
              boxWidth: 24,
              fontFamily: 'Roboto',
              fontColor: '#000',
              fontSize: 12
            }
          }
        }
      }
    );
  }
  createAttachmentStorageComboChart() {
    this.attachmentStorageComboChart = new Chart(
      this.attachmentStorageCanvasRight.nativeElement,
      {
        type: 'bar',
        data: {
          datasets: [
            {
              backgroundColor: 'rgb(0,115,167)',
              label: 'Attachments',
              data: [10, 20, 30, 40, 45, 50, 60]
            },
            {
              label: 'Limit',
              data: [90, 88, 80, 88, 90, 80, 82],
              type: 'line',
              borderColor: 'rgb(170,32,39)'
            }
          ],
          labels: ['1/28', '1/29', '1/30', '1/31', '2/1', '2/2', '2/3']
        },
        options: {
          scales: {
            yAxes: [
              {
                gridLines: {
                  color: 'rgb(204,204,204)'
                }
              }
            ],
            xAxes: [
              {
                display: true,
                gridLines: {
                  display: false
                }
              }
            ]
          },
          legend: {
            position: 'bottom',
            display: false,
            labels: {
              boxWidth: 24,
              fontFamily: 'Roboto',
              fontColor: '#000',
              fontSize: 12
            }
          }
        }
      }
    );
  }
  createMetadataStoragePieChart() {
    this.metadataStoragePieChart = new Chart(
      this.metadataStorageCanvasLeft.nativeElement,
      {
        type: 'pie',
        data: {
          datasets: [
            {
              backgroundColor: ['rgba(77,124,24,0.3)', 'rgba(77,124,24,1.0)'],
              data: [40, 60]
            }
          ],
          labels: ['Available', 'Used']
        },
        options: {
          layout: {
            padding: {
              bottom: 18
            }
          },
          legend: {
            position: 'bottom',
            display: false,
            labels: {
              boxWidth: 24,
              fontFamily: 'Roboto',
              fontColor: '#000',
              fontSize: 12
            }
          }
        }
      }
    );
  }
  createMetadataStorageComboChart() {
    this.metadataStorageComboChart = new Chart(
      this.metadataStorageCanvasRight.nativeElement,
      {
        type: 'bar',
        data: {
          datasets: [
            {
              backgroundColor: 'rgb(77,124,24)',
              label: 'Metadata',
              data: [10, 20, 30, 40, 45, 50, 55]
            },
            {
              label: 'Limit',
              data: [70, 75, 69, 78, 71, 68, 77],
              type: 'line',
              borderColor: 'rgb(170,32,39)'
            }
          ],
          labels: ['1/28', '1/29', '1/30', '1/31', '2/1', '2/2', '2/3']
        },
        options: {
          scales: {
            yAxes: [
              {
                gridLines: {
                  color: 'rgb(204,204,204)'
                }
              }
            ],
            xAxes: [
              {
                display: true,
                gridLines: {
                  display: false
                }
              }
            ]
          },
          legend: {
            position: 'bottom',
            display: false,
            labels: {
              boxWidth: 24,
              fontFamily: 'Roboto',
              fontColor: '#000',
              fontSize: 12
            }
          }
        }
      }
    );
  }
}
