import { Subscription } from 'rxjs/Subscription';
import {
  Component,
  OnInit,
  OnDestroy,
  ViewChild,
  ElementRef
} from '@angular/core';
import {
  ARSReportsModelReqRangeDate,
  ARSReportResponse,
  ResponseModel
} from './ars-reports.model';
import {
  FormBuilder,
  FormGroup,
  FormControl,
  Validators
} from '@angular/forms';
import { ArsReportsService } from './ars-reports.service';
import {
  TableColumnModel,
  TableSettings,
  PaginationSettings
} from './../shared/table/table.component.model';
import { ScriptService } from './../shared/scriptLoader/script.service';

@Component({
  selector: 'app-ars-reports',
  templateUrl: './ars-reports.component.html',
  styleUrls: ['./ars-reports.component.scss'],
  providers: [ScriptService]
})
export class ArsReportsComponent implements OnInit, OnDestroy {
  arsReportsModelReqRangeDate: ARSReportsModelReqRangeDate = new ARSReportsModelReqRangeDate();
  subscription: Subscription = new Subscription();
  form: FormGroup;
  searchResults: ResponseModel[];
  response: ARSReportResponse;
  showResults = false;
  validDateRange = true;
  errorMsg = '';
  errorStatus = false;
  loading = true;
  displayMatUnMat = 'Matched';
  showStorageWarning = false;

  @ViewChild('startDate') startDate: ElementRef = new ElementRef('');
  @ViewChild('endDate') endDate: ElementRef = new ElementRef('');
  tableColumns: TableColumnModel[] = [
    new TableColumnModel('Attch. ID', 'attachmentId'),
    // new TableColumnModel('Facility', 'facility'),
    new TableColumnModel('Provider', 'providerName'),
    new TableColumnModel('Prov. ZIP', 'providerZip'),
    new TableColumnModel('Prov. NPI', 'providerNpi'),
    new TableColumnModel('Patient Name', 'patientName'),
    new TableColumnModel('Patient ID', 'patientIdentifierNumber'),
    // new TableColumnModel('Pat. DOB', 'patientIdentifierNumber'),
    new TableColumnModel('Pat. Ctrl', 'patientControlNumber'),
    new TableColumnModel('Svc. Start', 'claimServiceStartDate'),
    new TableColumnModel('Svc. End', 'claimServiceEndDate'),
    new TableColumnModel('Attch. Ctrl', 'attachmentControlNumber'),
    //  new TableColumnModel('Report', 'attachmentControlNumber'),
    new TableColumnModel('Status', 'status'),
    new TableColumnModel('Date Rec.', 'dateReceived')
  ];
  tableSettings: TableSettings = new TableSettings();
  paginationSettings: PaginationSettings = new PaginationSettings();
  constructor(
    private fb: FormBuilder,
    private arsReportsService: ArsReportsService,
    private script: ScriptService
  ) {
    this.script
      .load('datepicker')
      .then(data => {})
      .catch(error => console.log(error));
  }

  ngOnInit() {
    this.formInit();
    this.setDefaultDate();
  }

  setDefaultDate() {
    const temp = new Date();
    let currentDate = '';
    const month =
      temp.getMonth() > 9 ? '' : '0' + (temp.getMonth() + 1).toString();

    const day = temp.getDate() > 9 ? '' : '0' + temp.getDate().toString();
    const year = temp.getFullYear().toString();
    currentDate = month + '/' + day + '/' + year;
    this.form.get('startDate').setValue(currentDate);
    this.form.get('endDate').setValue(currentDate);
    this.startDate.nativeElement.value = currentDate;
    this.endDate.nativeElement.value = currentDate;
    this.arsReportsModelReqRangeDate.startDate = currentDate;
    this.arsReportsModelReqRangeDate.endDate = currentDate;
  }
  formInit() {
    this.form = this.fb.group({
      payerId: [
        this.arsReportsModelReqRangeDate.payerId,
        Validators.compose([
          Validators.required,
          // Validators.minLength(2),
          Validators.pattern('[0-9]*')
        ])
      ],
      reportType: this.arsReportsModelReqRangeDate.reportType,
      // last24Hours: this.arsReportsModelReqRangeDate.last24Hours,
      startDate: [
        this.startDate.nativeElement.value,
        [
          Validators.required,
          Validators.pattern(
            '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
          )
        ]
      ],
      endDate: [
        this.endDate.nativeElement.value,
        [
          Validators.required,
          Validators.pattern(
            '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
          )
        ]
      ]
    });
  }

  assignStartValue() {
    this.form.get('startDate').setValue(this.startDate.nativeElement.value);
  }
  assignEndValue() {
    this.form.get('endDate').setValue(this.endDate.nativeElement.value);
  }
  onGenerate() {
    this.loading = true;
    this.searchResults = [];
    if (
      this.arsReportsModelReqRangeDate.reportType &&
      this.tableColumns[1].property !== 'externalClaimId'
    ) {
      this.tableColumns.splice(
        1,
        0,
        new TableColumnModel('Claim ID', 'externalClaimId')
      );
      this.tableColumns.splice(
        2,
        0,
        new TableColumnModel('Facility', 'facility')
      );
      this.tableColumns.splice(
        8,
        0,
        new TableColumnModel('Patient DOB', 'patientDob')
      );
      this.tableColumns.splice(
        13,
        0,
        new TableColumnModel('Rpt. Type', 'reportCode')
      );
    }
    if (
      !this.arsReportsModelReqRangeDate.reportType &&
      this.tableColumns[1].property === 'externalClaimId'
    ) {
      this.tableColumns.splice(13, 1);
      this.tableColumns.splice(8, 1);
      this.tableColumns.splice(2, 1);
      this.tableColumns.splice(1, 1);
    }

    this.arsReportsModelReqRangeDate.endDate = this.endDate.nativeElement.value;
    this.arsReportsModelReqRangeDate.startDate = this.startDate.nativeElement.value;
    this.paginationSettings = new PaginationSettings();
    this.arsReportsModelReqRangeDate.pageSize = this.paginationSettings.pageSize;
    this.arsReportsModelReqRangeDate.pageNumber = this.paginationSettings.currentPage;
    this.arsReportsModelReqRangeDate.descending = this.paginationSettings.descending;
    this.arsReportsModelReqRangeDate.sortColumn = this.paginationSettings.sortColumn;
    //  console.log(this.arsReportsModelReqRangeDate);
    // subscribing to generate reports
    this.fetchData();
  }
  fetchData() {
    const generateSubscription = this.arsReportsService
      .generateReportRange(this.arsReportsModelReqRangeDate)
      .subscribe(data => {
        if (data.errorCode) {
          this.paginationSettings = new PaginationSettings();
          this.errorStatus = true;
          this.errorMsg = 'Error in Generating Report.';
          this.searchResults = [];
          this.showResults = false;
        } else {
          this.response = data;
          this.searchResults = this.response.response;
          this.showResults = true;
          this.errorStatus = false;
          this.errorMsg = '';
          this.loading = false;
          this.paginationSettings = {
            currentPage: this.response.pageNumber,
            pageSize: this.response.pageSize,
            totalPages: Math.ceil(
              this.response.totalNumberOfResults / this.response.pageSize
            ),
            totalResults: this.response.totalNumberOfResults,
            sortColumn: this.response.sortColumn,
            descending: this.arsReportsModelReqRangeDate.descending
          };
        }
      });
    this.subscription.add(generateSubscription);
    this.tableSettings.linkColumn = true;
    this.showResults = true;
    if (this.arsReportsModelReqRangeDate.reportType) {
      this.displayMatUnMat = 'Matched';
    } else {
      this.displayMatUnMat = 'Unmatched';
    }
  }
  resetAttachReports() {
    this.arsReportsModelReqRangeDate = new ARSReportsModelReqRangeDate();
    this.startDate.nativeElement.value = '';
    this.endDate.nativeElement.value = '';
    this.showResults = false;
    this.errorStatus = false;
    this.errorMsg = '';
    this.validDateRange = true;
    this.displayMatUnMat = 'Matched';
    this.formInit();
    this.setDefaultDate();
  }

  datesValidator() {
    if (
      this.startDate.nativeElement.value.length === 10 &&
      this.endDate.nativeElement.value.length === 10
    ) {
      this.form.get('startDate').setValue(this.startDate.nativeElement.value);
      this.form.get('endDate').setValue(this.endDate.nativeElement.value);
      const fromDate = new Date(this.form.get('startDate').value);
      const toDate = new Date(this.form.get('endDate').value);
      if (toDate.valueOf() < fromDate.valueOf()) {
        this.validDateRange = false;
        this.form.get('startDate').setValue('');
        this.form.get('endDate').setValue('');
      }
    } else {
      this.validDateRange = true;
    }
  }
  updateTable() {
    this.arsReportsModelReqRangeDate.pageSize = this.paginationSettings.pageSize;
    this.arsReportsModelReqRangeDate.pageNumber = this.paginationSettings.currentPage;
    this.arsReportsModelReqRangeDate.descending = this.paginationSettings.descending;
    this.arsReportsModelReqRangeDate.sortColumn = this.paginationSettings.sortColumn;
    this.fetchData();
  }
  print() {}

  export() {}

  resetArsStorage() {
    this.showStorageWarning = false;
  }

  ngOnDestroy() {
    this.subscription.unsubscribe();
  }
}
