package gov.va.med.ars.service.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;

import gov.va.med.ars.dao.ars.IRfaiLookupRepository;
import gov.va.med.ars.dao.ars.RfaiLookupRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.request.RfaiLookupRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.RfaiLookUpResponse;
import gov.va.med.domain.ars.ClaimSubmission;

@RunWith(MockitoJUnitRunner.class)
public class RfaiLookupServiceImplTest {

	@InjectMocks
	RfaiLookupServiceImpl rfaiLookupService;

	@Mock
	RfaiLookupRepository simpleSearchRfaiRepository;

	@Mock
	IRfaiLookupRepository advSearchRfaiRepository;

	@Test
	public void testMockIsWorking() {
		assertNotNull("RfaiService Injection Failed", rfaiLookupService);
	}

	@SuppressWarnings("unchecked")
	@Test
	public void getRfaiSimpleSearchResults_numberAsSearchParam_success() throws GenericException {
		List<ClaimSubmission> pdiClaimList = new ArrayList<>();
		ClaimSubmission cs1 = new ClaimSubmission();
		cs1.setHl3nm103svcbillprvnm("Allen");
		// submitted date jan 1, 2018, response.getsubmitted should be 01-01-2018
		cs1.setDatesubmitted(new Date(1514808000000l));
		cs1.setHl4nm103ptlstnm("lastName");
		cs1.setHl4nm104ptftnm("firstName");
		pdiClaimList.add(cs1);

		List<ClaimSubmission> submissionIdList = new ArrayList<>();
		ClaimSubmission cs2 = new ClaimSubmission();
		cs2.setHl3nm103svcbillprvnm("Zed");
		// response.getrequiredresponsedate should be 03-01-2018
		cs2.setDtp203responseduedate(new Date(1519905600000l));
		submissionIdList.add(cs2);

		ClaimSubmission cs3 = new ClaimSubmission();
		cs3.setHl3nm103svcbillprvnm("Mark");
		cs3.setRef302medrecnmbr("medicalRecordNumber");
		submissionIdList.add(cs3);

		RfaiLookupRequest lookupRequest = new RfaiLookupRequest();
		lookupRequest.setNumber("12345");
		Long lookupRequestNumber = Long.valueOf(lookupRequest.getNumber());
		lookupRequest.setSortColumn("providerName");
		lookupRequest.setDescending(false);
		lookupRequest.setPageNumber(1);
		lookupRequest.setPageSize(5);

		when(simpleSearchRfaiRepository.findBytrn02fppsClaimId(lookupRequestNumber)).thenReturn(pdiClaimList);
		when(simpleSearchRfaiRepository.findBysubmissionid(lookupRequestNumber)).thenReturn(submissionIdList);

		GenericResponse result = rfaiLookupService.getRfaiSimpleSearchResults(lookupRequest);
		assertNotNull(result);
		assertEquals(1, result.getpageNumber());
		assertEquals(5, result.getpageSize());
		assertEquals("providerName", result.getsortColumn());
		assertNotNull(result.getResponse());
		assertEquals(3, result.getResponse().size());

		List<RfaiLookUpResponse> resultResponseList = (List<RfaiLookUpResponse>) result.getResponse();
		assertEquals("Allen", resultResponseList.get(0).getProviderName());
		assertEquals("Mark", resultResponseList.get(1).getProviderName());
		assertEquals("Zed", resultResponseList.get(2).getProviderName());
		assertEquals("01/01/2018", resultResponseList.get(0).getDatesubmitted());
		assertEquals("03/01/2018", resultResponseList.get(2).getRequiredResponseDate());
		assertEquals("lastName, firstName", resultResponseList.get(0).getPatientName());
		assertEquals("medicalRecordNumber", resultResponseList.get(1).getMedicalRecordNumber());
	}

	@SuppressWarnings("unchecked")
	@Test
	public void getRfaiSimpleSearchResults_nonNumberAsSearchParam_success() throws GenericException {
		List<ClaimSubmission> patientIdList = new ArrayList<>();
		List<ClaimSubmission> providerNpiList = new ArrayList<>();

		RfaiLookupRequest lookupRequest = new RfaiLookupRequest();
		lookupRequest.setNumber("non_number");
		lookupRequest.setSortColumn("providerName");
		lookupRequest.setDescending(false);
		lookupRequest.setPageNumber(1);
		lookupRequest.setPageSize(5);

		when(simpleSearchRfaiRepository.findByhl4nm109ptid(lookupRequest.getNumber())).thenReturn(patientIdList);
		when(simpleSearchRfaiRepository.findByhl3nm109prvnpi(lookupRequest.getNumber())).thenReturn(providerNpiList);

		ClaimSubmission cs1 = new ClaimSubmission();
		cs1.setHl3nm103svcbillprvnm("Allen");
		// submitted date jan 1, 2018, response.getsubmitted should be 01-01-2018
		cs1.setDatesubmitted(new Date(1514808000000l));
		cs1.setHl4nm103ptlstnm("lastName");
		cs1.setHl4nm104ptftnm("firstName");
		patientIdList.add(cs1);

		ClaimSubmission cs2 = new ClaimSubmission();
		cs2.setHl3nm103svcbillprvnm("Zed");
		// response.getrequiredresponsedate should be 03-01-2018
		cs2.setDtp203responseduedate(new Date(1519905600000l));
		providerNpiList.add(cs2);

		ClaimSubmission cs3 = new ClaimSubmission();
		cs3.setHl3nm103svcbillprvnm("Mark");
		cs3.setRef302medrecnmbr("medicalRecordNumber");
		providerNpiList.add(cs3);
		GenericResponse result = rfaiLookupService.getRfaiSimpleSearchResults(lookupRequest);

		List<RfaiLookUpResponse> resultResponseList = (List<RfaiLookUpResponse>) result.getResponse();
		assertEquals("Allen", resultResponseList.get(0).getProviderName());
		assertEquals("Mark", resultResponseList.get(1).getProviderName());
		assertEquals("Zed", resultResponseList.get(2).getProviderName());
		assertEquals("01/01/2018", resultResponseList.get(0).getDatesubmitted());
		assertEquals("03/01/2018", resultResponseList.get(2).getRequiredResponseDate());
		assertEquals("lastName, firstName", resultResponseList.get(0).getPatientName());
		assertEquals("medicalRecordNumber", resultResponseList.get(1).getMedicalRecordNumber());
	}

	@SuppressWarnings("unchecked")
	@Test
	public void getRfaiAdvResults_success() throws GenericException {
		// arrange
		RfaiLookupRequest lookupRequest = new RfaiLookupRequest();
		lookupRequest.setPageSize(10);
		lookupRequest.setSortColumn("");
		lookupRequest.setDescending(false);
		lookupRequest.setPageNumber(1);
		lookupRequest.setProviderName("Allen");

		List<ClaimSubmission> claimSubmissionList = new ArrayList<>();
		ClaimSubmission cs1 = new ClaimSubmission();
		cs1.setHl3nm103svcbillprvnm("Allen");
		// submitted date jan 1, 2018, response.getsubmitted should be 01-01-2018
		cs1.setDatesubmitted(new Date(1514808000000l));
		cs1.setHl4nm103ptlstnm("lastName");
		cs1.setHl4nm104ptftnm("firstName");
		claimSubmissionList.add(cs1);

		Page<ClaimSubmission> claimSubmissionPage = new PageImpl<ClaimSubmission>(claimSubmissionList);

		when(advSearchRfaiRepository.count(any(Specification.class))).thenReturn(1l);
		when(advSearchRfaiRepository.findAll(any(Specification.class), any(PageRequest.class)))
				.thenReturn(claimSubmissionPage);

		// act
		GenericResponse result = rfaiLookupService.getRfaiAdvResults(lookupRequest);

		// assert
		verify(advSearchRfaiRepository, times(1)).count(any(Specification.class));
		verify(advSearchRfaiRepository, times(1)).findAll(any(Specification.class), any(PageRequest.class));

		List<RfaiLookUpResponse> resultList = (List<RfaiLookUpResponse>) result.getResponse();

		assertEquals(1, resultList.size());
		assertEquals("Allen", resultList.get(0).getProviderName());
		assertEquals("lastName, firstName", resultList.get(0).getPatientName());
		assertEquals("01/01/2018", resultList.get(0).getDatesubmitted());

	}

}
