package gov.va.med.ars.service.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;

import gov.va.med.ars.dao.ars.IReportsViewRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.request.SearchReportRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.ReportResponse;
import gov.va.med.domain.ars.ReportsView;

@RunWith(MockitoJUnitRunner.class)
public class ReportServiceImplTest {

	@Mock
	private IReportsViewRepository reportsRepository;

	@InjectMocks
	private ReportServiceImpl reportsService;

	@Test
	public void testMockIsWorking() {
		assertNotNull("ReportServiceImpl Injection Failed", reportsService);
	}

	@SuppressWarnings("unchecked")
	@Test
	public void getReports_getMatchedReport_Success() throws GenericException {

		// arrange
		SearchReportRequest searchRequest = new SearchReportRequest();
		searchRequest.setPageNumber(1);
		searchRequest.setPageSize(10);
		searchRequest.setDescending(false);
		searchRequest.setSortColumn("attachmentId");
		searchRequest.setStartDate("03/16/2018");
		searchRequest.setEndDate("03/16/2018");
		searchRequest.setPayerId("121150001");

		boolean searchForMatched = false;

		List<ReportsView> reportsList = new ArrayList<>();
		ReportsView report1 = new ReportsView();
		report1.setAttachmentId(999l);
		report1.setPatientFirstName("firstName");
		report1.setPatientLastName("lastName");
		reportsList.add(report1);

		Page<ReportsView> pagedReports = new PageImpl<>(reportsList);

		when(reportsRepository.count(any(Specification.class))).thenReturn(1l);
		when(reportsRepository.findAll(any(Specification.class), any(PageRequest.class))).thenReturn(pagedReports);

		// act
		GenericResponse repoResponse = reportsService.getReports(searchRequest, searchForMatched);

		// assert
		verify(reportsRepository, times(1)).count(any(Specification.class));
		verify(reportsRepository, times(1)).findAll(any(Specification.class), any(PageRequest.class));

		List<ReportResponse> repoResponseList = (List<ReportResponse>) repoResponse.getResponse();
		assertEquals(10, repoResponse.getpageSize());
		assertEquals(new Long(1), repoResponse.gettotalNumberOfResults());
		assertEquals(10, repoResponse.getpageSize());
		assertEquals(999, repoResponseList.get(0).getAttachmentId());
		assertEquals("firstName", repoResponseList.get(0).getPatientFirstName());
		assertEquals("lastName", repoResponseList.get(0).getPatientLastName());

	}

	@SuppressWarnings("unchecked")
	@Test
	public void getReports_getMatchedReport_Failure() throws GenericException {

		// arrange
		SearchReportRequest searchRequest = new SearchReportRequest();
		searchRequest.setPageNumber(1);
		searchRequest.setPageSize(10);
		searchRequest.setDescending(false);
		searchRequest.setSortColumn("attachmentId");
		searchRequest.setStartDate("03/16/2018");
		searchRequest.setEndDate("03/16/2018");
		searchRequest.setPayerId("121150001");
		boolean searchForMatched = false;

		try {
			// act
			reportsService.getReports(searchRequest, searchForMatched);
		} catch (GenericException ge) {
			// assert
			assertEquals("At least one field information needs to be added", ge.getMessage());
		}

		// assert
		verify(reportsRepository, never()).findAll(any(Specification.class), any(PageRequest.class));
	}
}
